/*
 * This file is part of FastPTPAI, which is an AI for OpenTTD
 *
 * FastPTPAI is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License
 *
 * FastPTPAI is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with FastPTPAI; If not, see <http://www.gnu.org/licenses/> or
 * write to the Free Software Foundation, Inc., 51 Franklin Street, 
 * Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

// This class is simply to increase the speed of the pathfinder, because that is part of what TrainAI is about
// Much of the code has been copied over from the origional Rail Pathfinder, only with my modifications in overriden functions
class CustomPathFinder extends RailPathFinder
{
	speed_factor = null;
	constructor()
	{
		// these are tailored for the AI since they are here (I know I am evil)
		//this._max_cost = 10000000;
		//this._cost_tile = 50;
		//this._cost_diagonal_tile = 200;
		//this._cost_turn = 100;
		//this._cost_slope = 100;
		//this._cost_bridge_per_tile = 750;
		//this._cost_tunnel_per_tile = 350;
		//this._cost_coast = 50;
		//this._max_bridge_length = 6;
		//this._max_tunnel_length = 6;
		this._max_cost = 2000000000;
		this._cost_tile = 100;
		this._cost_diagonal_tile = 100; // 70
		this._cost_turn = 100;
		this._cost_slope = 200;
		this._cost_bridge_per_tile = 150; // 200
		this._cost_tunnel_per_tile = 120; // 120
		this._cost_coast = 20;
		this._max_bridge_length = 6;
		this._max_tunnel_length = 6;
		this._pathfinder = this._aystar_class(this._Cost, this._Estimate, this._Neighbours, this._CheckDirection, this, this, this, this);

		this.cost = this.Cost(this);
		this._running = false;
		this.speed_factor = 1.0;
	}
	
	// TODO: Potentially add _Cost funciton which tries to build next to existing rail for lower cost!!!
	function _Cost(path, new_tile, new_direction, self) {
		/* path == null means this is the first node of a path, so the cost is 0. */
		if (path == null) return 0;

		local prev_tile = path.GetTile();

		/* If the new tile is a bridge / tunnel tile, check whether we came from the other
		 *  end of the bridge / tunnel or if we just entered the bridge / tunnel. */
		if (AIBridge.IsBridgeTile(new_tile)) {
			if (AIBridge.GetOtherBridgeEnd(new_tile) != prev_tile) {
				local cost = path.GetCost() + self._cost_tile;
				if (path.GetParent() != null && path.GetParent().GetTile() - prev_tile != prev_tile - new_tile) cost += self._cost_turn;
				return cost;
			}
			return path.GetCost() + AIMap.DistanceManhattan(new_tile, prev_tile) * self._cost_tile + self._GetBridgeNumSlopes(new_tile, prev_tile) * self._cost_slope;
		}
		if (AITunnel.IsTunnelTile(new_tile)) {
			if (AITunnel.GetOtherTunnelEnd(new_tile) != prev_tile) {
				local cost = path.GetCost() + self._cost_tile;
				if (path.GetParent() != null && path.GetParent().GetTile() - prev_tile != prev_tile - new_tile) cost += self._cost_turn;
				return cost;
			}
			return path.GetCost() + AIMap.DistanceManhattan(new_tile, prev_tile) * self._cost_tile;
		}

		/* If the two tiles are more then 1 tile apart, the pathfinder wants a bridge or tunnel
		 *  to be build. It isn't an existing bridge / tunnel, as that case is already handled. */
		if (AIMap.DistanceManhattan(new_tile, prev_tile) > 1) {
			/* Check if we should build a bridge or a tunnel. */
			local cost = path.GetCost();
			if (AITunnel.GetOtherTunnelEnd(new_tile) == prev_tile) {
				cost += AIMap.DistanceManhattan(new_tile, prev_tile) * (self._cost_tile + self._cost_tunnel_per_tile);
			} else {
				cost += AIMap.DistanceManhattan(new_tile, prev_tile) * (self._cost_tile + self._cost_bridge_per_tile) + self._GetBridgeNumSlopes(new_tile, prev_tile) * self._cost_slope;
			}
			if (path.GetParent() != null && path.GetParent().GetParent() != null &&
					path.GetParent().GetParent().GetTile() - path.GetParent().GetTile() != max(AIMap.GetTileX(prev_tile) - AIMap.GetTileX(new_tile), AIMap.GetTileY(prev_tile) - AIMap.GetTileY(new_tile)) / AIMap.DistanceManhattan(new_tile, prev_tile)) {
				cost += self._cost_turn;
			}
			return cost;
		}

		/* Check for a turn. We do this by substracting the TileID of the current
		 *  node from the TileID of the previous node and comparing that to the
		 *  difference between the tile before the previous node and the node before
		 *  that. */
		local cost = self._cost_tile;
		if (path.GetParent() != null && AIMap.DistanceManhattan(path.GetParent().GetTile(), prev_tile) == 1 && path.GetParent().GetTile() - prev_tile != prev_tile - new_tile) cost = self._cost_diagonal_tile;
		if (path.GetParent() != null && path.GetParent().GetParent() != null &&
				AIMap.DistanceManhattan(new_tile, path.GetParent().GetParent().GetTile()) == 3 &&
				path.GetParent().GetParent().GetTile() - path.GetParent().GetTile() != prev_tile - new_tile) {
			cost += self._cost_turn;
		}

		/* Check if the new tile is a coast tile. */
		if (AITile.IsCoastTile(new_tile)) {
			cost += self._cost_coast;
		}

		/* Check if the last tile was sloped. */
		if (path.GetParent() != null && !AIBridge.IsBridgeTile(prev_tile) && !AITunnel.IsTunnelTile(prev_tile) &&
				self._IsSlopedRail(path.GetParent().GetTile(), prev_tile, new_tile)) {
			cost += self._cost_slope;
		}
		
		/* Check if any nearby tiles have rails. */
		if(IsAlongAnyRail(new_tile)) cost *= 0.3;

		/* We don't use already existing rail, so the following code is unused. It
		 *  assigns if no rail exists along the route. */
		/*
		if (path.GetParent() != null && !AIRail.AreTilesConnected(path.GetParent().GetTile(), prev_tile, new_tile)) {
			cost += self._cost_no_existing_rail;
		}
		*/

		return path.GetCost() + cost;
	}
	
	
	function _Estimate(cur_tile, cur_direction, goal_tiles, self)
	{
		local min_cost = self._max_cost;
		/* As estimate we multiply the lowest possible cost for a single tile with
		*  with the minimum number of tiles we need to traverse. */
		foreach (tile in goal_tiles) {
			local dx = abs(AIMap.GetTileX(cur_tile) - AIMap.GetTileX(tile[0]));
			local dy = abs(AIMap.GetTileY(cur_tile) - AIMap.GetTileY(tile[0]));
			min_cost = min(min_cost, min(dx, dy) * self._cost_diagonal_tile * 2 + (max(dx, dy) - min(dx, dy)) * self._cost_tile);
		}
		return min_cost * self.speed_factor;
	}
}

class Pathfinder
{
	pathfinder = null;
	path = null;
	start = null;
	end = null;
	blacklist = null;
	signals = null;
	constructor(start, end, signals, blacklist, speedfactor = 1.0) {
		pathfinder = CustomPathFinder();
		pathfinder.speed_factor = speedfactor;
		this.start = start;
		this.end = end;
		this.blacklist = blacklist;
		this.signals = signals;
	}
	function FindPath();
	function BuildPath();
	// TODO: DemolishPath();
	function IsAlongPath();
}

function Pathfinder::FindPath()
{
	if(!start || !end) {
		Log.Warning("Not all arguments specified");
		return false;
	}
	if(!AIMap.IsValidTile(start[0]) || !AIMap.IsValidTile(end[0])) { // TODO: Check all tiles
		Log.Warning("Invalid tiles passed!");
		return false;
	}
	pathfinder.cost.max_bridge_length = 30;
	pathfinder.cost.max_tunnel_length = 30;
	//pathfinder.cost.diagonal_tile = 200; // for some reason this creates an error
	// Swap start tiles
	//local temp = start[1];
	//start[1] = start[0];
	//start[0] = temp;
	if(blacklist == null) pathfinder.InitializePath([start],[end]);
	else pathfinder.InitializePath([start],[end], blacklist);
	path = false;
	//while(path == false) {
		path = pathfinder.FindPath(4000);
		//AIController.Sleep(1);
	//}
	if(!path) {
		Log.Warning("Path could not be found :(");
		return false;
	}
	return true;
}

// The following function was from Convoy

function Pathfinder::BuildPath()
{
	local path = this.path; // So that the path can be built multiple times
	local prev = null;
	local prevprev = null;
	local signalCount = 0;
	while (path != null) {
		if (prevprev != null) {
			if (AIMap.DistanceManhattan(prev, path.GetTile()) > 1) {
				if (AITunnel.GetOtherTunnelEnd(prev) == path.GetTile()) {
					if(!AITunnel.BuildTunnel(AIVehicle.VT_RAIL, prev)) {
						DemolishPath();
						return "retry";
					}
				} else {
					local bridge_list = AIBridgeList_Length(AIMap.DistanceManhattan(path.GetTile(), prev) + 1);
					bridge_list.Valuate(AIBridge.GetMaxSpeed);
					bridge_list.Sort(AIList.SORT_BY_VALUE, false);
					if(!AIBridge.BuildBridge(AIVehicle.VT_RAIL, bridge_list.Begin(), prev, path.GetTile())) {
						DemolishPath();
						return "retry";
					}
				}
				prevprev = prev;
				prev = path.GetTile();
				path = path.GetParent();
			} else {
				if(!AIRail.BuildRail(prevprev, prev, path.GetTile())) {
					DemolishPath();
					// For now assume that the route can be rebuilt
					return "retry";
				}
				if(signals) {
					signalCount++;
					if(signalCount >= 3) {
						AIRail.BuildSignal(prev, path.GetTile(), AIRail.SIGNALTYPE_PBS_ONEWAY); // No big deal if the rare case of a signal build failure occours (but it is still something to look into for the future)
						signalCount = 0;
					}
					
				}
			}
		}
		if (path != null) {
			prevprev = prev;
			prev = path.GetTile();
			path = path.GetParent();
		}
	}
	return "success";
}

function Pathfinder::DemolishPath() {
	local path = this.path; // So that the path can be used multiple times
	// Go through the path from beginning to end, and if there is a rail track on that tile, demolish.
	while(path != null) {
		if(AIRail.IsRailTile(path.GetTile()) || AIBridge.IsBridgeTile(path.GetTile()) || AITunnel.IsTunnelTile(path.GetTile())) AITile.DemolishTile(path.GetTile());
		path = path.GetParent();
	}
}

function Pathfinder::IsAlongPath(tile) {
	local tiles = Tile.GetNeighbours4MainDir(tile);
	tiles.Valuate(AIRail.IsRailTile);
	tiles.KeepValue(1);
	tiles.Valuate(AITile.GetSlope);
	tiles.KeepValue(0);
	tiles.Valuate(AIRail.IsRailStationTile);
	tiles.KeepValue(0);
	tiles.Valuate(AIBridge.IsBridgeTile);
	tiles.KeepValue(0);
	//tiles.Valuate(AIRail.GetRailTracks);
	foreach(t, value in tiles) {
		if(AIRail.GetRailTracks(t) != AIRail.RAILTRACK_NE_SW && AIRail.GetRailTracks(t) != AIRail.RAILTRACK_NW_SE) tiles.SetValue(t, 0);
		else tiles.SetValue(t, 1);
	}
	tiles.KeepValue(1);
	if(tiles.IsEmpty()) return false;
	// See if it is along the path
	foreach(t, value in tiles) {
		local path = this.path;
		local found = false;
		while(path != null) {
			if(path.GetTile() == t) {
				found = true;
				break;
			}
			path = path.GetParent();
		}
		if(!found) tiles.SetValue(t, 0);
		else tiles.SetValue(t, 1);
	}
	tiles.KeepValue(1);
	
	return !tiles.IsEmpty();
}
