import("pathfinder.road", "RoadPathFinder", 3);

enum RoadTileTypes {
    ROAD,
    TUNNEL,
    BRIDGE
}

class RoadTile {
    m_tile = null;
    m_type = null;
    m_bridge_id = null;

    constructor(tile, type, bridgeId = -1) {
        m_tile = tile;
        m_type = type;
        m_bridge_id = bridgeId;
    }
    
    function saveRoadTile() {
        local tile = [];
        
        tile.append(m_tile);
        tile.append(m_type);
        tile.append(m_bridge_id);
        
        return tile;
    }
    
    function loadBuildTile(data) {
        local tile = data[0];
        local type = data[1];
        local bridgeId = data[2];
        
        return BuldRoadTile(tile, type, bridgeId);
    }
}

class BuildManager {

    m_cityFrom = -1;
    m_cityTo = -1;
    m_stationFrom = -1;
    m_stationTo = -1;
    m_depotTile = -1;
    m_cargoClass = -1;

    constructor() {

    }

    function buildTownStation(town, cargoClass, stationTile);
    function expandRectangle(town, topCornerTile, bottomCornerTile, edge);
    function estimateTownRectangle(town);
    function buildRoad(fromTile, toTile, oneWayRoad);
    function findRoadTileDepot(tile);
    function buildDepotOnRoad(roadArray);
    function buildRoadFromArray(roadArray);
    function repairRoad(tile);
    function saveBuildManager();
    function buildRoute(cityFrom, cityTo, cargoClass);

    function hasUnfinishedRoute() {
        if(m_cityFrom != -1 && m_cityTo != -1 && m_cargoClass != -1) {
            return 1;
        }

        return 0;
    }

    function setRouteFinish() {
        m_cityFrom = -1;
        m_cityTo = -1;
        m_stationFrom = -1;
        m_stationTo = -1;
        m_depotTile = -1;
        m_cargoClass = -1;
    }

    function buildRoute(cityFrom, cityTo, cargoClass) {
        m_cityFrom = cityFrom;
        m_cityTo = cityTo;
        m_cargoClass = cargoClass;

        if(m_stationFrom == -1) {
            m_stationFrom = buildTownStation(m_cityFrom, m_cargoClass, null);
            if (m_stationFrom == null) {
                setRouteFinish();
                return null;
            }
        }

        if(m_stationTo == -1) {
            m_stationTo = buildTownStation(m_cityTo, cargoClass, null);
            if (m_stationTo == null) {
                if (m_stationFrom != null) {
                    AITile.DemolishTile(m_stationFrom);
                    repairRoad(m_stationFrom);
                }
                setRouteFinish();
                return null;
            }
        }

        if(m_depotTile == -1) {
            local roadArray = buildRoad(m_stationFrom, m_stationTo, 0);
            m_depotTile = BuildManager().buildDepotOnRoad(roadArray);
        }

        if((m_depotTile == null)) {

            if (m_stationFrom != null) {
                AITile.DemolishTile(m_stationFrom);
                repairRoad(m_stationFrom);
            }

            if (m_stationTo != null) {
                AITile.DemolishTile(m_stationTo);
                repairRoad(m_stationTo);
            }

            setRouteFinish();
            return null;
        }

        return Route(m_cityFrom, m_cityTo, m_stationFrom, m_stationTo, m_depotTile, m_cargoClass);
    }

    function buildTownStation(town, cargoClass, stationTile) {
        local stationId = (stationTile == null) ?  AIStation.STATION_NEW : AIStation.GetStationID(stationTile);
        local vehicleType = (cargoClass == AICargo.CC_MAIL) ? AIRoad.ROADVEHTYPE_TRUCK : AIRoad.ROADVEHTYPE_BUS;

        local tileList = AITileList();
        if(stationTile == null) {
            //stationTile = AIStation.STATION_NEW;
            //build square around @town and find suitable tiles for truck stops
            local rectangleCoordinates = estimateTownRectangle(town);

            tileList.AddRectangle(rectangleCoordinates[0],
                rectangleCoordinates[1]);

            //valuate and sort by the number of cargo tiles the station covers
            tileList.Valuate(Utils.valuateTruckStopTile, cargoClass, stationId);
            tileList.RemoveValue(0);
            tileList.Sort(AIList.SORT_BY_VALUE, false); //starts from corners if without sort
        }
        else {
            //expanding existing station
            if(!AIStation.IsValidStation(stationId)) {
                return null;
            }

            local squareSize = 5;

            tileList.AddRectangle(Utils.getOffsetTile(stationTile, -1 * squareSize, -1 * squareSize),
                Utils.getOffsetTile(stationTile, squareSize, squareSize));

            tileList.Valuate(Utils.valuateTruckStopTile, cargoClass, stationId);
            tileList.RemoveValue(0);
        }

        for (local tile = tileList.Begin(); tileList.HasNext(); tile = tileList.Next()) {
            //get adjacent tiles
            local adjTileList = Utils.getAdjacentTiles(tile);
            local adjRoadTiles = AITileList();
            adjRoadTiles.AddList(adjTileList);
            adjRoadTiles.Valuate(AIRoad.IsRoadTile);
            adjRoadTiles.KeepValue(1);

            AIRoad.SetCurrentRoadType(AIRoad.ROADTYPE_ROAD);
            local adjRoadCount = adjRoadTiles.Count();

            switch (adjRoadCount) {
            //case where station tile has no adjacent road tiles
                case 0:
                //connect station tile with center of its town
                    if (buildRoad(tile, AITown.GetLocation(town), 0) != null) {
                        //station tile now has adjacent road tiles
                        for (local adjTile = adjTileList.Begin(); adjTileList.HasNext(); adjTile = adjTileList.Next()) {
                            if (AIRoad.IsRoadTile(adjTile)) {
                                //since new road includes station tile we need to demolish the road in order to build the station
//                                if(!AITile.DemolishTile(tile)) {
//                                    continue;
//                                }

                                if (!AIRoad.BuildRoadStation(tile, adjTile, vehicleType, stationId)) {
                                    //if((AIError.GetLastErrorString() != "ERR_ALREADY_BUILT") && (AIError.GetLastErrorString() != "ERR_PRECONDITION_FAILED")) {
                                        //AILog.Warning("Couldnt build station! " + AIError.GetLastErrorString());
                                    //}
                                    continue;
                                }
                                else {
                                    AILog.Info("Station built in " + AITown.GetName(town) + "!");
                                    //AISign.BuildSign(tile, "" + adjRoadCount);
                                    return tile;
                                }
                            }
                        }
                    }
                    else {
                        AILog.Warning("Couldnt find path! " + AIError.GetLastErrorString());
                        continue;
                    }

                    break;

                    case 1:
                        if(AIRoad.IsRoadTile(tile)) {
                            continue;
                        }

                        local adjTile = adjRoadTiles.Begin();
                        local heighDifference = abs(AITile.GetMaxHeight(tile) - AITile.GetMaxHeight(adjTile));

                        if(heighDifference != 0) {
                            continue;
                        }

                        if(AIRoad.IsRoadTile(tile)) {
                            local counter = 0;
                            do {
                                if(!AITile.DemolishTile(tile)) {
                                    ++counter;
                                    continue;
                                }
                                else {
                                    break;
                                }
                            } while(counter < 500);

                            if(counter == 500) {
                                continue;
                            }
                        }

                        if (!AIRoad.BuildRoadStation(tile, adjTile, vehicleType, stationId)) {
                            if(AIError.GetLastErrorString() != "ERR_LOCAL_AUTHORITY_REFUSES") {
                                //AILog.Warning("Couldnt build station! " + AIError.GetLastErrorString());
                            }
                            continue;
                        }
                        else {
                            local counter = 0;
                            do {
                                if(!AIRoad.BuildRoad(tile, adjTile) && AIError.GetLastErrorString() != "ERR_ALREADY_BUILT") {
                                    ++counter;
                                    continue;
                                }
                                else {
                                    break;
                                }
                            } while(counter < 500);

                            if(counter == 500) {
                                AITile.DemolishTile(tile);
                                repairRoad(tile);
                                continue;
                            }
                            else {
                                AILog.Info("Station built in " + AITown.GetName(town) + "!");
                                //AISign.BuildSign(tile, "" + adjRoadCount);
                                return tile;
                            }
                        }

                    break;

                case 2:
                    local adjTile = adjRoadTiles.Begin();
                    local nextAdjTile = adjRoadTiles.Next();

                    //dont build drivethrough station next to regular station
                    adjTileList.RemoveItem(adjTile);
                    adjTileList.RemoveItem(nextAdjTile);
                    local found = false;
                    for(local t = adjTileList.Begin(); adjTileList.HasNext(); t = adjTileList.Next()) {
                        if(AITile.IsStationTile(t)) {
                            found = true;
                            break;
                        }
                    }
                    if(found) {
                        continue;
                    }

                    local temp

                    local heighDifference = abs(AITile.GetMaxHeight(nextAdjTile) - AITile.GetMaxHeight(adjTile));
                    if(heighDifference != 0) {
                        continue;
                    }


                     //check whether adjacent tiles are opposite
                    local opposite = false;
                    if((AIMap.GetTileX(adjTile) == AIMap.GetTileX(nextAdjTile))
                        || (AIMap.GetTileY(adjTile) == AIMap.GetTileY(nextAdjTile))) {
                            opposite = true;
                    }

                    local destroyed = false;
                    if(AIRoad.IsRoadTile(tile)) {
                            if(!opposite) {
                                continue;
                            }

                        local counter = 0;
                        do {
                            if(!AITile.DemolishTile(tile)) {
                                ++counter;
                                continue;
                            }
                            else {
                                destroyed = true;
                                break;
                            }
                        } while(counter < 500);

                        if(counter == 500) {
                            continue;
                        }

                    }

                    if(opposite) {
                        if (!AIRoad.BuildDriveThroughRoadStation(tile, adjTile, vehicleType, stationId)) {
                            if((AIError.GetLastErrorString() != "ERR_ALREADY_BUILT") && (AIError.GetLastErrorString() != "ERR_PRECONDITION_FAILED")) {
                                if(AIError.GetLastErrorString() != "ERR_LOCAL_AUTHORITY_REFUSES") {
                                    //AILog.Warning("Couldnt build station! " + AIError.GetLastErrorString());
                                }
                                if(destroyed) {
                                    repairRoad(tile);
                                }
                            }
                            continue;
                        }
                        else {
                            local counter = 0;
                            do {
                                if(!AIRoad.BuildRoad(adjTile, nextAdjTile) && AIError.GetLastErrorString() != "ERR_ALREADY_BUILT") {
                                    ++counter;
                                    continue;
                                }
                                else {
                                    break;
                                }
                            } while(counter < 500);

                            if(counter == 500) {
                                AITile.DemolishTile(tile);
                                if(destroyed) {
                                    repairRoad(tile);
                                }
                                continue;
                            }
                            else {
                                AILog.Info("Drivethrough station built in " + AITown.GetName(town) + "!");
                                //AISign.BuildSign(tile, "" + adjRoadCount);
                                return tile;
                            }
                        }
                    }
                    //similar to case 1 if adjacent tiles are not opposite
                    else {
                        if (!AIRoad.BuildRoadStation(tile, adjTile, vehicleType, stationId)) {
                            if((AIError.GetLastErrorString() != "ERR_ALREADY_BUILT") && (AIError.GetLastErrorString() != "ERR_PRECONDITION_FAILED")) {
                                if(AIError.GetLastErrorString() != "ERR_LOCAL_AUTHORITY_REFUSES") {
                                   // AILog.Warning("Couldnt build station! " + AIError.GetLastErrorString());
                                }
                            }
                            continue;
                        }
                        else {
                            local counter = 0;
                            do {
                                if(!AIRoad.BuildRoad(tile, adjTile)) {
                                    ++counter;
                                    continue;
                                }
                                else {
                                    break;
                                }
                            } while(counter < 500);

                            if(counter == 500) {
                                continue;
                            }
                            else {
                                AILog.Info("Station built in " + AITown.GetName(town) + "!");
                                //AISign.BuildSign(tile, "" + adjRoadCount);
                                return tile;
                            }
                        }
                    }

                    break;

                case 3:
                case 4:
                    //similar to case 2 but always builds drivethrough station
                    AIRoad.SetCurrentRoadType(AIRoad.ROADTYPE_ROAD);

                    if(AIRoad.IsRoadTile(tile)) {
                        continue;
                    }

                    local adjTile = adjRoadTiles.Begin();
                    local nextAdjTile = adjRoadTiles.Next();

                    //check whether adjacent tiles are opposite
                    if(!((AIMap.GetTileX(adjTile) == AIMap.GetTileX(nextAdjTile))
                        || (AIMap.GetTileY(adjTile) == AIMap.GetTileY(nextAdjTile)))) {
                        nextAdjTile = adjRoadTiles.Next();
                    }

                    local heighDifference = abs(AITile.GetMaxHeight(nextAdjTile) - AITile.GetMaxHeight(adjTile));
                    if(heighDifference != 0) {
                        continue;
                    }

                    if (!AIRoad.BuildDriveThroughRoadStation(tile, adjTile, AIRoad.ROADVEHTYPE_TRUCK, stationId)) {
                        if((AIError.GetLastErrorString() != "ERR_ALREADY_BUILT") && (AIError.GetLastErrorString() != "ERR_PRECONDITION_FAILED")) {
                            if(AIError.GetLastErrorString() != "ERR_LOCAL_AUTHORITY_REFUSES") {
                                //AILog.Warning("Couldnt build station! " + AIError.GetLastErrorString());
                            }
                        }
                        continue;
                    }
                    else {
                        local counter = 0;
                        do {
                            if(!AIRoad.BuildRoad(adjTile, nextAdjTile) && AIError.GetLastErrorString() != "ERR_ALREADY_BUILT") {
                                ++counter;
                                continue;
                            }
                            else {
                                break;
                            }
                        } while(counter < 500);

                        if(counter == 500) {
                            AITile.DemolishTile(tile);
                            continue;
                        }
                        else {
                            AILog.Info("Drivethrough station built in " + AITown.GetName(town) + "!");
                            //AISign.BuildSign(tile, "" + adjRoadCount);
                            return tile;
                        }
                    }

                    break;

                default:
                    break;
            }
        }

        return null;
    }

    function estimateTownRectangle(town) {
        local townId = town;
        local townLocation = AITown.GetLocation(town);
        local rectangleIncreaseKoeficient = 1;

        local topCornerTile = AITown.GetLocation(town);
        local bottomCornerTile = AITown.GetLocation(town);

        local isMaxExpanded = false;
        while(!isMaxExpanded) {
            local maxExpandedCounter = 0;
            for(local i = 0; i < 4; ++i) {
                switch(i) {
                    case 0:
                        local offsetTile = Utils.getOffsetTile(topCornerTile, (-1) * rectangleIncreaseKoeficient, 0);

                        if(offsetTile == AIMap.TILE_INVALID) {
                            ++maxExpandedCounter;
                            continue;
                        }

                        if(AITown.IsWithinTownInfluence(town, offsetTile)) {
                            topCornerTile = offsetTile;
                        }
//                        else if(expandRectangle(town, topCornerTile, bottomCornerTile, "NE")) {
//                            topCornerTile = Utils.getOffsetTile(topCornerTile, (-1) * rectangleIncreaseKoeficient, 0);
//                        }
                        else {
                            ++maxExpandedCounter;
                            continue;
                        }
                        break;

                    case 1:
                        local offsetTile = Utils.getOffsetTile(bottomCornerTile, 0, rectangleIncreaseKoeficient);

                        if(offsetTile == AIMap.TILE_INVALID) {
                            ++maxExpandedCounter;
                            continue;
                        }

                        if(AITown.IsWithinTownInfluence(town, offsetTile)) {
                            bottomCornerTile = offsetTile;
                        }
//                        else if(expandRectangle(town, topCornerTile, bottomCornerTile, "SE")) {
//                            bottomCornerTile = Utils.getOffsetTile(bottomCornerTile, 0, rectangleIncreaseKoeficient);
//                        }
                        else {
                            ++maxExpandedCounter;
                            continue;
                        }
                        break;

                    case 2:
                        local offsetTile = Utils.getOffsetTile(bottomCornerTile, rectangleIncreaseKoeficient, 0);

                        if(offsetTile == AIMap.TILE_INVALID) {
                            ++maxExpandedCounter;
                            continue;
                        }

                        if(AITown.IsWithinTownInfluence(town, offsetTile)) {
                            bottomCornerTile = offsetTile;
                        }
//                        else if(expandRectangle(town, topCornerTile, bottomCornerTile, "SW")) {
//                            bottomCornerTile = Utils.getOffsetTile(bottomCornerTile, rectangleIncreaseKoeficient, 0);
//                        }
                        else {
                            ++maxExpandedCounter;
                            continue;
                        }
                        break;

                    case 3:
                        local offsetTile = Utils.getOffsetTile(topCornerTile, 0, (-1) * rectangleIncreaseKoeficient);

                        if(offsetTile == AIMap.TILE_INVALID) {
                            ++maxExpandedCounter;
                        }

                        if(AITown.IsWithinTownInfluence(town, offsetTile)) {
                            topCornerTile = offsetTile;
                        }
//                        else if(expandRectangle(town, topCornerTile, bottomCornerTile, "NW")) {
//                            topCornerTile = Utils.getOffsetTile(topCornerTile, 0, (-1) * rectangleIncreaseKoeficient);
//                        }
                        else {
                            ++maxExpandedCounter;
                        }
                        break;

                    default:
                        break;
                }
            }

            if(maxExpandedCounter == 4) {
                isMaxExpanded = true;
            }
        }

        return [topCornerTile, bottomCornerTile];
    }

    function expandRectangle(town, topCornerTile, bottomCornerTile, edge) {
        switch(edge) {
            case "NE":
                local topX = AIMap.GetTileX(topCornerTile);
                local botY = AIMap.GetTileY(bottomCornerTile);

                for(local topY = AIMap.GetTileY(topCornerTile); topY <= botY; ++topY) {
                    local checkTile = AIMap.GetTileIndex(topX - 1, topY);

                    if(AITown.IsWithinTownInfluence(town, checkTile)) {
                        return 1;
                    }
                }
                break;

            case "SE":
                local botX = AIMap.GetTileX(bottomCornerTile);
                local botY = AIMap.GetTileY(bottomCornerTile);

                for(local topX = AIMap.GetTileX(topCornerTile); topX <= botX; ++topX) {
                    local checkTile = AIMap.GetTileIndex(topX, botY + 1);

                    if(AITown.IsWithinTownInfluence(town, checkTile)) {
                        return 1;
                    }
                }
                break;

            case "SW":
                local botX = AIMap.GetTileX(bottomCornerTile);
                local botY = AIMap.GetTileY(bottomCornerTile);

                for(local topY = AIMap.GetTileY(topCornerTile); topY <= botY; ++topY) {
                    local checkTile = AIMap.GetTileIndex(botX + 1, topY);

                    if(AITown.IsWithinTownInfluence(town, checkTile)) {
                        return 1;
                    }
                }
                break;

            case "NW":
                local topY = AIMap.GetTileY(topCornerTile);
                local botX = AIMap.GetTileX(bottomCornerTile);

                for(local topX = AIMap.GetTileX(topCornerTile); topX <= botX; ++topX) {
                    local checkTile = AIMap.GetTileIndex(topX, topY - 1);

                    if(AITown.IsWithinTownInfluence(town, checkTile)) {
                        return 1;
                    }
                }
                break;
        }

        return 0;
    }

    //find road way between fromTile and toTile, builds one way road if oneWayRoad is true
    function buildRoad(fromTile, toTile, oneWayRoad = false) {
        //can store road tiles into array
        local builtTiles = [];

        if (fromTile != toTile) {
            //* Print the names of the towns we'll try to connect. */
            AILog.Info("Connecting " + AITown.GetName(AITile.GetClosestTown(fromTile)) + " and " + AITown.GetName(AITile.GetClosestTown(toTile)));

            // Tell OpenTTD we want to build normal road (no tram tracks). */
            AIRoad.SetCurrentRoadType(AIRoad.ROADTYPE_ROAD);

            // Create an instance of the pathfinder. */
            local pathfinder = RoadPathFinder();

            // Set the cost for making a turn extreme high. */
            //pathfinder.cost.turn = 5000000;

            // Give the source and goal tiles to the pathfinder. */
            pathfinder.InitializePath([fromTile], [toTile]);
            local path = false;
            local counter = 0;
            while (path == false && counter < 250) {
                ++counter;
                path = pathfinder.FindPath(100);
                AIController.Sleep(1);
            }

            if (path == false) {
                // No path was found. */
                AILog.Error("pathfinder: FindPath return null");
                return null;
            }

            AILog.Info("Path found! Building road...");

            local temp = 0;
            // If a path was found, build a road over it. */
            while (path != null) {
                local par = path.GetParent();
                if (par != null) {
                    local last_node = path.GetTile();

                    if (AIMap.DistanceManhattan(path.GetTile(), par.GetTile()) == 1) {
                        if (oneWayRoad) {
                            if (!AIRoad.BuildOneWayRoad(par.GetTile(), path.GetTile())) {
                                if(!AIRoad.BuildRoad(par.GetTile(), path.GetTile())) {
                                }
                            }
                        }
                        else {
                            local counter = 0;
                            do {
                                if(!AIRoad.BuildRoad(path.GetTile(), par.GetTile())) {
                                    if(AIError.GetLastErrorString() == "ERR_ALREADY_BUILT") {
                                        break;
                                    }
                                    else {
                                        ++counter;
                                    }
                                }
                                else {
                                    break;
                                }

                                AIController.Sleep(1);
                            } while(counter < 500);

                            if(counter == 500) {
                                AILog.Warning("Couldnt build road! " + AIError.GetLastErrorString());
                                return null;
                            }
                        }

                        //add road piece into the road array
                        builtTiles.append(RoadTile(path.GetTile(), RoadTileTypes.ROAD));
                    }
                    else {
                        // Build a bridge or tunnel. */
                        if (!AIBridge.IsBridgeTile(path.GetTile()) && !AITunnel.IsTunnelTile(path.GetTile())) {
                            // If it was a road tile, demolish it first. Do this to work around expended roadbits. */
                            if (AIRoad.IsRoadTile(path.GetTile())) AITile.DemolishTile(path.GetTile());
                            if (AITunnel.GetOtherTunnelEnd(path.GetTile()) == par.GetTile()) {
                                local counter = 0;
                                do {
                                    if (!AITunnel.BuildTunnel(AIVehicle.VT_ROAD, path.GetTile())) {
                                        if(AIError.GetLastErrorString() == "ERR_ALREADY_BUILT") {
                                            break;
                                        }
                                        else {
                                            ++counter;
                                        }
                                    }
                                    else {
                                        break;
                                    }

                                    AIController.Sleep(1);
                                } while(counter < 500);

                                if(counter == 500) {
                                    AILog.Warning("Couldnt build tunnel! " + AIError.GetLastErrorString());
                                    return null;
                                }

                                builtTiles.append(RoadTile(path.GetTile(), RoadTileTypes.TUNNEL));

                            }
                            else {
                                local bridge_list = AIBridgeList_Length(AIMap.DistanceManhattan(path.GetTile(), par.GetTile()) + 1);
                                bridge_list.Valuate(AIBridge.GetMaxSpeed);
                                bridge_list.Sort(AIAbstractList.SORT_BY_VALUE, false);
                                local counter = 0;
                                do {
                                    if (!AIBridge.BuildBridge(AIVehicle.VT_ROAD, bridge_list.Begin(), path.GetTile(), par.GetTile())) {
                                        if(AIError.GetLastErrorString() == "ERR_ALREADY_BUILT") {
                                            break;
                                        }
                                        else {
                                            ++counter;
                                        }
                                    }
                                    else {
                                        break;
                                    }

                                    AIController.Sleep(1);
                                } while(counter < 500);

                                if(counter == 500) {
                                    AILog.Warning("Couldnt build bridge! " + AIError.GetLastErrorString());
                                    return null;
                                }

                                builtTiles.append(RoadTile(path.GetTile(), RoadTileTypes.BRIDGE, bridge_list.Begin()));
                                builtTiles.append(RoadTile(par.GetTile(), RoadTileTypes.ROAD));
                            }
                        }
                    }
                }
                path = par;
            }
        }

        AILog.Info("Road built!");
        return builtTiles;
    }

    //valuator for buildDepotOnRoad(roadArray)
    function findRoadTileDepot(tile) {
        if(!AIRoad.IsRoadTile(tile)) {
            return null;
        }

        local depotTile = null;
        local adjacentTiles = Utils.getAdjacentTiles(tile);
        for (local adjacentTile = adjacentTiles.Begin(); adjacentTiles.HasNext(); adjacentTile = adjacentTiles.Next()) {
            if (!AITile.IsBuildable(adjacentTile) || AITile.GetSlope(adjacentTile) != AITile.SLOPE_FLAT) {
                continue;
            }
            else {
                depotTile = adjacentTile;
                break;
            }
        }

        return depotTile;
    }

    function buildDepotOnRoad(roadArray) {
        if(roadArray == null) {
            return null;
        }

        local depotTile = null;
        local depotFrontTile = null;

        local size = roadArray.len();

        //check from middle to end
        for (local index = roadArray.len() / 2; (depotTile == null) && (index < roadArray.len()); ++index) {
            local square = AITileList();
            local squareSize = 1;
            local depotNearby = false;
            square.AddRectangle(Utils.getOffsetTile(roadArray[index].m_tile, (-1) * squareSize, (-1) * squareSize),
                Utils.getOffsetTile(roadArray[index].m_tile, squareSize, squareSize));

            //if other station is nearby return 0
            for(local tile = square.Begin(); square.HasNext(); tile = square.Next()) {
                if(AIRoad.IsRoadDepotTile(tile)) {
                    //can return tile to reuse old depot
                    depotNearby = true;
                    break;
                }
            }
            if(depotNearby) {
                continue;
            }

            depotTile = findRoadTileDepot(roadArray[index].m_tile);
            depotFrontTile = roadArray[index].m_tile;

            if(depotTile != null) {
                AIRoad.BuildRoad(depotTile, depotFrontTile);
                if(!AIRoad.AreRoadTilesConnected(depotTile, depotFrontTile)) {
                    depotTile = null;
                    depotFrontTile = null;
                }
                else {
                    break;
                }
            }
        }

        //from middle to start
        if(depotTile == null) {
            for (local index = 0; index < (roadArray.len() / 2) - 1; ++index) {
                local square = AITileList();
                local squareSize = 1;
                local depotNearby = false;
                square.AddRectangle(Utils.getOffsetTile(roadArray[index].m_tile, (-1) * squareSize, (-1) * squareSize),
                    Utils.getOffsetTile(roadArray[index].m_tile, squareSize, squareSize));

                //if other station is nearby return 0
                for(local tile = square.Begin(); square.HasNext(); tile = square.Next()) {
                    if(AIRoad.IsRoadDepotTile(tile)) {
                        //can return tile to reuse old depot
                        depotNearby = true;
                        break;
                    }
                }
                if(depotNearby) {
                    continue;
                }

                depotTile = findRoadTileDepot(roadArray[index].m_tile);
                depotFrontTile = roadArray[index].m_tile;

                if(depotTile != null) {
                    AIRoad.BuildRoad(depotTile, depotFrontTile);
                    if(!AIRoad.AreRoadTilesConnected(depotTile, depotFrontTile)) {
                        depotTile = null;
                        depotFrontTile = null;
                    }
                    else {
                        //todo check
                        if(!AIRoad.BuildRoadDepot(depotTile, depotFrontTile)) {
                            AILog.Warning("Couldnt built road depot!");
                            return null;
                        }
                    }
                }
            }
        }
        else {
            if(!AIRoad.BuildRoadDepot(depotTile, depotFrontTile)) {
                AILog.Warning("Couldnt built road depot!");
                return null;
            }
        }

        return depotTile;
    }

    //@param array of RoadTile type
    function buildRoadFromArray(roadArray) {
        for(local i = 0; i < roadArray.len() - 1; ++i) {
            switch(roadArray[i].m_type) {
                case 0:
                    AIRoad.BuildRoad(roadArray[i].m_tile, roadArray[i+1].m_tile);
                    break;

                case 1:
                    AITunnel.BuildTunnel(AIVehicle.VT_ROAD, roadArray[i].m_tile);
                    break;

                case 2:
                    AIBridge.BuildBridge(AIVehicle.VT_ROAD, roadArray[i].m_bridge_id, roadArray[i].m_tile, roadArray[i+1].m_tile);
                    break
            }
        }
        return;
    }

    //connets adjacent tiles
    function repairRoad(tile) {
        local adjTiles = Utils.getAdjacentTiles(tile);
        adjTiles.Valuate(AIRoad.IsRoadTile);
        adjTiles.KeepValue(1);

        for (local adjTile = adjTiles.Begin(); adjTiles.HasNext(); adjTile = adjTiles.Next()) {
            local counter = 0;
            do {
                if(!AIRoad.BuildRoad(tile, adjTile) && AIError.GetLastErrorString() != "ERR_ALREADY_BUILT") {
                    ++counter;
                    continue;
                }
                else {
                    break;
                }
            } while(counter < 500);
        }

        return;
    }

    function saveBuildManager() {
        local route = [];

        if(m_cityFrom == null) {
            m_cityFrom = -1;
        }

        if(m_cityTo == null) {
            m_cityTo = -1;
        }

        if(m_stationFrom == null) {
            m_stationFrom = -1;
        }

        if(m_stationTo == null) {
            m_stationTo = -1;
        }

        if(m_depotTile == null) {
            m_depotTile = -1;
        }


//        local roadArray;
//        if(m_roadArray == null || m_roadArray == -1) {
//            m_roadArray = -1;
//            roadArray = -1;
//        }
//        else {
//            roadArray = [];
//            for(local i = 0; i < m_roadArray.len(); ++i) {
//                roadArray.append(m_roadArray[i].saveRoadTile());
//            }
//        }

        route.append(m_cityFrom);
        route.append(m_cityTo);
        route.append(m_stationFrom);
        route.append(m_stationTo);
        route.append(m_depotTile);
        route.append(m_cargoClass);

        return route;
    }

    function loadBuildManager(data) {
        m_cityFrom = data[0];
        m_cityTo = data[1];
        m_stationFrom = data[2];
        m_stationTo = data[3];
        m_depotTile = data[4];

//        local roadArray;
//        if(data[5] == null || data[5] == -1) {
//            roadArray = -1;
//        }
//        else {
//            roadArray = [];
//            for (local i = 0; i < data[5].len(); ++i) {
//
//                local tile = data[5][i][0];
//                local type = data[5][i][1];
//                local bridgeId = data[5][i][2];
//
//                roadArray.append(RoadTile(tile, type, bridgeId));
//            }
//        }
//
//        m_roadArray = roadArray;
        m_cargoClass = data[5];

        return;
    }
}

