require("RouteManager.nut");

class Route extends RouteManager {
    MIN_VEHICLE_START_COUNT = 5;
    MAX_VEHICLE_COUNT = 25;
    START_VEHICLE_COUNT = 10;

    m_cityFrom = null;
    m_cityTo = null;
    m_stationFrom = null;
    m_stationTo = null;
    m_depotTile = null;
    m_cargoClass = null;

    m_engine = null;
    m_lastEngineUpdate = null;
    m_lastVehicleAdded = null;
    m_lastVehicleRemoved = null;

    m_sentToDepotList = null;
    m_renewList = null;

    m_updateStationCount = null;
    m_expandedFromCount = null;
    m_expandedToCount = null;


    constructor(cityFrom, cityTo, stationFrom, stationTo, depotTile, cargoClass, isLoaded = 0) {
        m_cityFrom = cityFrom;
        m_cityTo = cityTo;
        m_stationFrom = stationFrom;
        m_stationTo = stationTo;
        m_depotTile = depotTile;
        m_cargoClass = cargoClass;

        m_engine = GetTruckEngine(cargoClass);
        m_lastEngineUpdate = AIDate.GetCurrentDate();

        m_lastVehicleAdded = 0;
        m_lastVehicleRemoved = AIDate.GetCurrentDate();;

        m_sentToDepotList = AIList();
        m_renewList = AIList();

        m_updateStationCount = 0;
        m_expandedFromCount = 0;
        m_expandedToCount = 0;

        if(!isLoaded) {
            addVehiclesToNewRoute();
        }
    }

    function updateEngine();
    function addVehicle();
    function addVehiclesToNewRoute();
    function GetEngineList(cargoClass);
    function GetTruckEngine(cargoClass);

    function GetEngineList(cargoClass) {
        local engineList = AIEngineList(AIVehicle.VT_ROAD);
        engineList.Valuate(AIEngine.GetRoadType);
        engineList.KeepValue(AIRoad.ROADTYPE_ROAD);

        engineList.Valuate(AIEngine.GetCargoType)
        engineList.KeepValue(cargoClass);

        return engineList;
    }

    function GetTruckEngine(cargoClass) {

        local cargoList = AICargoList();
        cargoList.Valuate(AICargo.HasCargoClass, cargoClass);
        cargoList.KeepValue(1);

        local engineList = GetEngineList(cargoList.Begin());

        engineList.Valuate(AIEngine.GetMaxSpeed);
        engineList.Sort(AIList.SORT_BY_VALUE, true);

        local engine = engineList.KeepBottom(1);
        if(engine != m_engine) {
            LuDiAI.MAX_DISTANCE += 15;
            LuDiAI.MIN_DISTANCE += 10;
        }

        return engineList.Begin();
    }

    function updateEngine() {
        if((AIDate.GetCurrentDate() - m_lastEngineUpdate) >= 365) {
            m_engine = GetTruckEngine(m_cargoClass);
            m_lastEngineUpdate = AIDate.GetCurrentDate();
        }
    }

    function addVehicle() {
        if(AIVehicleList_Depot(this.m_depotTile).Count() > this.MAX_VEHICLE_COUNT) {
            return null;
        }

        local vehicle = AIVehicle.BuildVehicle(this.m_depotTile, this.m_engine);

        if(AIVehicle.IsValidVehicle(vehicle)) {
            AIOrder.AppendOrder(vehicle, m_depotTile, AIOrder.AIOF_SERVICE_IF_NEEDED);
            AIOrder.AppendOrder(vehicle, m_stationFrom, AIOrder.AIOF_NON_STOP_INTERMEDIATE);
            AIOrder.AppendOrder(vehicle, m_stationTo, AIOrder.AIOF_NON_STOP_INTERMEDIATE);

            AIVehicle.StartStopVehicle(vehicle);
        }
        else {
            return null;
        }

        m_lastVehicleAdded = AIDate.GetCurrentDate();

        return 1;
    }

    function addVehiclesToNewRoute() {
        if(AIVehicleList_Depot(this.m_depotTile).Count() > this.MAX_VEHICLE_COUNT) {
            return null;
        }

        local stationDistance = AITile.GetDistanceManhattanToTile(m_stationFrom, m_stationTo);
        //local maxPossibleDistance = LuDiAI.MAX_DISTANCE * LuDiAI.MAX_DISTANCE_INCREASE;
        local buyVehicleCount = START_VEHICLE_COUNT;

        buyVehicleCount += stationDistance / 20;
        if(buyVehicleCount > (MAX_VEHICLE_COUNT).tointeger()) {
            buyVehicleCount = (MAX_VEHICLE_COUNT).tointeger();
        }

        local temp;
        for(local i = 0; i < buyVehicleCount; ++i) {
            if(i % 2 == 1) {
                temp = m_stationFrom;
                m_stationFrom = m_stationTo;
                m_stationTo = temp;
            }

            m_lastVehicleAdded = 0; //to allow addVehiclesToNewRoute() add vehicles even with m_lastVehicleAdded restriction
            if(addVehicle()) {
                AILog.Info("Added vehicle on route from " + AITown.GetName(m_cityFrom)
                    + " to " + AITown.GetName(m_cityTo) + "!");
            }
        }

        //swap back
        local temp = m_stationFrom;
        m_stationFrom = m_stationTo;
        m_stationTo = temp;
    }

    function sendVehicleToDepot(vehicle) {
        if ((!m_sentToDepotList.HasItem(vehicle)) && (!m_renewList.HasItem(vehicle))) {

            AIVehicle.SendVehicleToDepot(vehicle);

            m_lastVehicleRemoved = AIDate.GetCurrentDate();

            AIOrder.AppendOrder(vehicle, m_depotTile, AIOrder.AIOF_STOP_IN_DEPOT);

            AILog.Info("Vehicle on route from " + AITown.GetName(m_cityFrom)
                + " to " + AITown.GetName(m_cityTo) + " has been sent to its depot!");

            return 1;
        }

        return 0;
    }

    function sendNegativeProfitVehiclesToDepot() {
        local vehicleList = AIVehicleList_Depot(m_depotTile);
        vehicleList.Valuate(AIVehicle.GetAge);
        vehicleList.KeepAboveValue(2 * 365);

        for(local vehicle = vehicleList.Begin(); vehicleList.HasNext(); vehicle = vehicleList.Next()) {
            AIController.Sleep(1);
            if((AIVehicle.GetProfitLastYear(vehicle)) < 0) {
                if(sendVehicleToDepot(vehicle)) {
                    m_sentToDepotList.AddItem(vehicle, 0);
                }
            }
        }
    }

    function sendLowProfitVehiclesToDepot() {
        local roadVehicleList = AIVehicleList();
        roadVehicleList.Valuate(AIVehicle.GetVehicleType);
        roadVehicleList.KeepValue(AIVehicle.VT_ROAD);

        if(LuDiAI.MAX_TOWN_VEHICLES - 50 > roadVehicleList.Count()) {
            return;
        }

        if((AIDate.GetCurrentDate() - m_lastVehicleRemoved) > 60) {
            local vehicleList = AIVehicleList_Depot(m_depotTile);
            vehicleList.Valuate(AIVehicle.GetAge);
            vehicleList.KeepAboveValue(2 * 365);


            for (local vehicle = vehicleList.Begin(); vehicleList.HasNext(); vehicle = vehicleList.Next()) {
                AIController.Sleep(1);
                local cargoId = Utils.getCargoId(m_cargoClass);
                local cargoWaiting = AIStation.GetCargoWaiting(AIStation.GetStationID(m_stationTo), cargoId) +
                    AIStation.GetCargoWaiting(AIStation.GetStationID(m_stationFrom), cargoId);

                if ((cargoWaiting < 150) ||
                    (AIVehicle.GetProfitLastYear(vehicle) < (highestProfitLastYear() / 6))) {

                    if(sendVehicleToDepot(vehicle)) {
                        m_sentToDepotList.AddItem(vehicle, 0);
                    }
                }
            }
        }
    }

    function sellVehiclesInDepot() {
        for(local vehicle = m_sentToDepotList.Begin(); m_sentToDepotList.HasNext(); vehicle = m_sentToDepotList.Next()) {
            if(AIVehicle.IsStoppedInDepot(vehicle)) {

                m_sentToDepotList.RemoveItem(vehicle);
                AIVehicle.SellVehicle(vehicle);

                AILog.Info("Vehicle on route from " + AITown.GetName(m_cityFrom)
                    + " to " + AITown.GetName(m_cityTo) + " has been sold!");

            }
        }

        for(local vehicle = m_renewList.Begin(); m_renewList.HasNext(); vehicle = m_renewList.Next()) {
            if(AIVehicle.IsStoppedInDepot(vehicle)) {

                m_renewList.RemoveItem(vehicle);
                AIVehicle.SellVehicle(vehicle)

                if(addVehicle()) {
                    AILog.Info("Vehicle on route from " + AITown.GetName(m_cityFrom)
                        + " to " + AITown.GetName(m_cityTo) + " has been renewed!");

                }
            }
        }
    }

    function addVehicleToRoute() {
        if(AIDate.GetCurrentDate() -  m_lastVehicleAdded < 90) {
            return null;
        }
        
        local cargoId = Utils.getCargoId(m_cargoClass);
        local cargoWaiting1 = AIStation.GetCargoWaiting(AIStation.GetStationID(m_stationTo), cargoId);
        local cargoWaiting2 = AIStation.GetCargoWaiting(AIStation.GetStationID(m_stationFrom), cargoId);

        if((cargoWaiting1 > 50 && cargoWaiting2 > 50) || (cargoWaiting1 + cargoWaiting2 > 300)) {
            if(addVehicle()) {
                AILog.Info("Added vehicle on route from " + AITown.GetName(m_cityFrom)
                    + " to " + AITown.GetName(m_cityTo) + "!");
            }
        }
    }

    function renewVehicles() {
        local vehicleList = AIVehicleList_Depot(m_depotTile);
        vehicleList.Valuate(AIVehicle.GetAgeLeft);
        vehicleList.KeepBelowValue(365);

        for(local vehicle = vehicleList.Begin(); vehicleList.HasNext(); vehicle = vehicleList.Next()) {
            AIController.Sleep(1);
            if(sendVehicleToDepot(vehicle)) {
                m_renewList.AddItem(vehicle, 0);
            }
        }
    }

    function expandStations() {
        local population = AITown.GetPopulation(m_cityFrom);
        local result = 0;
        if((population / 1000).tointeger() > m_expandedFromCount + 1) {
            if(BuildManager().buildTownStation(m_cityFrom, m_cargoClass, m_stationFrom) != null) {
                ++m_expandedFromCount;
                result = 1;
            }
        }

        population = AITown.GetPopulation(m_cityTo);

        if((population / 1000).tointeger() > m_expandedToCount + 1) {
            local tile = BuildManager().buildTownStation(m_cityTo, m_cargoClass, m_stationTo);
            if(tile != null) {
                //AISign.BuildSign(tile, "exp");
                ++m_expandedToCount;
                result = 1;
            }
        }

        return result;
    }

    function saveRoute() {
        local route = [];

        route.append(m_cityFrom);
        route.append(m_cityTo);
        route.append(m_stationFrom);
        route.append(m_stationTo);
        route.append(m_depotTile);

        route.append(m_cargoClass);

        route.append(m_lastVehicleAdded);
        route.append(m_lastVehicleRemoved);
        route.append(m_updateStationCount);

        return route;
    }

    function loadRoute(data) {
        local cityFrom = data[0];
        local cityTo = data[1];
        local stationFrom = data[2];
        local stationTo = data[3];
        local depotTile = data[4];

        local cargoClass = data[5];
        local route = Route(cityFrom, cityTo, stationFrom, stationTo, depotTile, cargoClass, 1);

        route.m_lastVehicleAdded = data[6];
        route.m_lastVehicleRemoved = data[7];
        route.m_updateStationCount = data[8];

        return route;
    }
}