//https://wiki.openttd.org/AI:WrightAI (from: https://bananas.openttd.org/en/ai/)
//modified by lukin_


class WrightAI extends AIController {
	name = null;
	towns_used = null;
	route_1 = null;
	route_2 = null;
	distance_of_route = {};
	vehicle_to_depot = {};
	delay_build_airport_route = 1000;

	cargoId = null;

	constructor(cargoClass) {
		this.towns_used = AIList();
		this.route_1 = AIList();
		this.route_2 = AIList();

		cargoId = Utils.getCargoId(cargoClass);
	}
};

/**
 * Check if we have enough money (via loan and on bank).
 */
function WrightAI::HasMoney(money)
{
	if (AICompany.GetBankBalance(AICompany.COMPANY_SELF) + (AICompany.GetMaxLoanAmount() - AICompany.GetLoanAmount()) > money) return true;
	return false;
}

/**
 * Get the amount of money requested, loan if needed.
 */
function WrightAI::GetMoney(money)
{
	if (!this.HasMoney(money)) return;
	if (AICompany.GetBankBalance(AICompany.COMPANY_SELF) > money) return;

	local loan = money - AICompany.GetBankBalance(AICompany.COMPANY_SELF) + AICompany.GetLoanInterval() + AICompany.GetLoanAmount();
	loan = loan - loan % AICompany.GetLoanInterval();
	AILog.Info("Need a loan to get " + money + ": " + loan);
	AICompany.SetLoanAmount(loan);
}

/**
 * Build an airport route. Find 2 cities that are big enough and try to build airport in both cities.
 *  Then we can build an aircraft and make some money.
 */
function WrightAI::BuildAirportRoute()
{
	local airport_type = (AIAirport.IsValidAirportType(AIAirport.AT_LARGE) ? AIAirport.AT_LARGE : AIAirport.AT_SMALL);

	AILog.Info("Trying to build an airport route");

	local tile_1 = this.FindSuitableAirportSpot(airport_type, 0);
	if (tile_1 < 0) return -1;
	local tile_2 = this.FindSuitableAirportSpot(airport_type, tile_1);
	if (tile_2 < 0) {
		this.towns_used.RemoveValue(tile_1);
		return -2;
	}

	local adjacentStationId = checkAdjacentStation(tile_1);

	/* Build the airports for real */
	if (!AIAirport.BuildAirport(tile_1, airport_type, adjacentStationId)) {
		if (!AIAirport.BuildAirport(tile_2, airport_type, AIStation.STATION_NEW)) {
			AILog.Warning("Although the testing told us we could build 2 airports, it still failed on the first airport at tile " + tile_1 + ".");
			this.towns_used.RemoveValue(tile_1);
			this.towns_used.RemoveValue(tile_2);
			return -3;
		}
	}

	adjacentStationId = checkAdjacentStation(tile_2);

	if (!AIAirport.BuildAirport(tile_2, airport_type, adjacentStationId)) {
		if (!AIAirport.BuildAirport(tile_2, airport_type, AIStation.STATION_NEW)) {
			AILog.Warning("Although the testing told us we could build 2 airports, it still failed on the second airport at tile " + tile_2 + ".");
			AIAirport.RemoveAirport(tile_1);
			this.towns_used.RemoveValue(tile_1);
			this.towns_used.RemoveValue(tile_2);
			return -4;
		}
	}

	local ret = this.BuildAircraft(tile_1, tile_2);
	if (ret < 0) {
		AIAirport.RemoveAirport(tile_1);
		AIAirport.RemoveAirport(tile_2);
		this.towns_used.RemoveValue(tile_1);
		this.towns_used.RemoveValue(tile_2);
		return ret;
	}

	AILog.Info("Done building a route");
	return ret;
}

/**
 * Build an aircraft with orders from tile_1 to tile_2.
 *  The best available aircraft of that time will be bought.
 */
function WrightAI::BuildAircraft(tile_1, tile_2)
{
	/* Build an aircraft */
	local hangar = AIAirport.GetHangarOfAirport(tile_1);
	local engine = null;

	local engine_list = AIEngineList(AIVehicle.VT_AIR);

	/* When bank balance < 300000, buy cheaper planes */
	local balance = AICompany.GetBankBalance(AICompany.COMPANY_SELF);
	engine_list.Valuate(AIEngine.GetPrice);
	engine_list.KeepBelowValue(balance < 300000 ? 50000 : (balance < 1000000 ? 300000 : 1000000));

	engine_list.Valuate(AIEngine.GetCargoType);
	engine_list.KeepValue(this.cargoId);

	engine_list.Valuate(AIEngine.GetCapacity);
	engine_list.KeepTop(1);

	engine = engine_list.Begin();

	if (!AIEngine.IsValidEngine(engine)) {
		AILog.Error("Couldn't find a suitable engine");
		return -5;
	}
	local vehicle = AIVehicle.BuildVehicle(hangar, engine);
	if (!AIVehicle.IsValidVehicle(vehicle)) {
		AILog.Error("Couldn't build the aircraft");
		return -6;
	}
	/* Send him on his way */
	AIOrder.AppendOrder(vehicle, tile_1, AIOrder.AIOF_NONE);
	AIOrder.AppendOrder(vehicle, tile_2, AIOrder.AIOF_NONE);
	AIVehicle.StartStopVehicle(vehicle);
	this.distance_of_route.rawset(vehicle, AIMap.DistanceManhattan(tile_1, tile_2));
	this.route_1.AddItem(vehicle, tile_1);
	this.route_2.AddItem(vehicle, tile_2);

	AILog.Info("Done building an aircraft");

	return 0;
}

/**
 * Find a suitable spot for an airport, walking all towns hoping to find one.
 *  When a town is used, it is marked as such and not re-used.
 */
function WrightAI::FindSuitableAirportSpot(airport_type, center_tile)
{
	local airport_x, airport_y, airport_rad;

	airport_x = AIAirport.GetAirportWidth(airport_type);
	airport_y = AIAirport.GetAirportHeight(airport_type);
	airport_rad = AIAirport.GetAirportCoverageRadius(airport_type);

	local town_list = AITownList();
	/* Remove all the towns we already used */
	town_list.RemoveList(this.towns_used);

	town_list.Valuate(AITown.GetPopulation);
	town_list.KeepAboveValue(500);
	/* Keep the best 10, if we can't find 2 stations in there, just leave it anyway */
	town_list.KeepTop(10);
	town_list.Valuate(AIBase.RandItem);

	/* Now find 2 suitable towns */
	for (local town = town_list.Begin(); town_list.HasNext(); town = town_list.Next()) {
		/* Don't make this a CPU hog */
		Sleep(1);

		local tile = AITown.GetLocation(town);

		/* Create a 30x30 grid around the core of the town and see if we can find a spot for a small airport */
		local list = AITileList();
		/* XXX -- We assume we are more than 15 tiles away from the border! */
		list.AddRectangle(tile - AIMap.GetTileIndex(15, 15), tile + AIMap.GetTileIndex(15, 15));
		list.Valuate(AITile.IsBuildableRectangle, airport_x, airport_y);
		list.KeepValue(1);
		if (center_tile != 0) {
			/* If we have a tile defined, we don't want to be within 25 tiles of this tile */
			list.Valuate(AITile.GetDistanceSquareToTile, center_tile);
			list.KeepAboveValue(625);
		}
		/* Sort on acceptance, remove places that don't have acceptance */
		list.Valuate(AITile.GetCargoAcceptance, this.cargoId, airport_x, airport_y, airport_rad);
		list.RemoveBelowValue(10);

		/* Couldn't find a suitable place for this town, skip to the next */
		if (list.Count() == 0) continue;
		/* Walk all the tiles and see if we can build the airport at all */
		{
			local test = AITestMode();
			local good_tile = 0;

			for (tile = list.Begin(); list.HasNext(); tile = list.Next()) {
				Sleep(1);
				if (!AIAirport.BuildAirport(tile, airport_type, AIStation.STATION_NEW)) continue;
				good_tile = tile;
				break;
			}

			/* Did we found a place to build the airport on? */
			if (good_tile == 0) continue;
		}

		AILog.Info("Found a good spot for an airport in town " + town + " at tile " + tile);

		/* Make the town as used, so we don't use it again */
		this.towns_used.AddItem(town, tile);

		return tile;
	}

	AILog.Info("Couldn't find a suitable town to build an airport in");
	return -1;
}

function WrightAI::ManageAirRoutes()
{
	local list = AIVehicleList();
	list.Valuate(AIVehicle.GetVehicleType);
	list.KeepValue(AIVehicle.VT_AIR);
	list.Valuate(AIVehicle.GetAge);
	/* Give the plane at least 2 years to make a difference */
	list.KeepAboveValue(365 * 2);
	list.Valuate(AIVehicle.GetProfitLastYear);

	for (local i = list.Begin(); list.HasNext(); i = list.Next()) {
		local profit = list.GetValue(i);
		/* Profit last year and this year bad? Let's sell the vehicle */
		if (profit < 10000 && AIVehicle.GetProfitThisYear(i) < 10000) {
			/* Send the vehicle to depot if we didn't do so yet */
			if (!vehicle_to_depot.rawin(i) || vehicle_to_depot.rawget(i) != true) {
				AILog.Info("Sending " + i + " to depot as profit is: " + profit + " / " + AIVehicle.GetProfitThisYear(i));
				AIVehicle.SendVehicleToDepot(i);
				vehicle_to_depot.rawset(i, true);
			}
		}
		/* Try to sell it over and over till it really is in the depot */
		if (vehicle_to_depot.rawin(i) && vehicle_to_depot.rawget(i) == true) {
			if (AIVehicle.SellVehicle(i)) {
				AILog.Info("Selling " + i + " as it finally is in a depot.");
				/* Check if we are the last one serving those airports; else sell the airports */
				local list2 = AIVehicleList_Station(AIStation.GetStationID(this.route_1.GetValue(i)));
				if (list2.Count() == 0) this.SellAirports(i);
				vehicle_to_depot.rawdelete(i);
			}
		}
	}

	/* Don't try to add planes when we are short on cash */
	if (!this.HasMoney(50000)) return;

	list = AIStationList(AIStation.STATION_AIRPORT);
	list.Valuate(AIStation.GetCargoWaiting, cargoId);
	list.KeepAboveValue(250);

	for (local i = list.Begin(); list.HasNext(); i = list.Next()) {
		local list2 = AIVehicleList_Station(i);
		list2.Valuate(AIVehicle.GetVehicleType);
		list2.KeepValue(AIVehicle.VT_AIR);
		/* No vehicles going to this station, abort and sell */
		if (list2.Count() == 0) {
			this.SellAirports(i);
			continue;
		};

		/* Find the first vehicle that is going to this station */
		local v = list2.Begin();
		local dist = this.distance_of_route.rawget(v);

		list2.Valuate(AIVehicle.GetAge);
		list2.KeepBelowValue(dist);
		/* Do not build a new vehicle if we bought a new one in the last DISTANCE days */
		if (list2.Count() != 0) continue;

		AILog.Info("Station " + i + " (" + AIStation.GetLocation(i) + ") has too many cargo, adding a new vehicle for the route.");

		return this.BuildAircraft(this.route_1.GetValue(v), this.route_2.GetValue(v));
	}
}

/**
  * Sells the airports from route index i
  * Removes towns from towns_used list too
  */
function WrightAI::SellAirports(i) {
	/* Remove the airports */
	AILog.Info("Removing airports as nobody serves them anymore.");
	AIAirport.RemoveAirport(this.route_1.GetValue(i));
	AIAirport.RemoveAirport(this.route_2.GetValue(i));
	/* Free the towns_used entries */
	this.towns_used.RemoveValue(this.route_1.GetValue(i));
	this.towns_used.RemoveValue(this.route_2.GetValue(i));
	/* Remove the route */
	this.route_1.RemoveItem(i);
	this.route_2.RemoveItem(i);
}

function WrightAI::HandleEvents()
{
	while (AIEventController.IsEventWaiting()) {
		local e = AIEventController.GetNextEvent();
		switch (e.GetEventType()) {
			case AIEvent.AI_ET_VEHICLE_CRASHED: {
				local ec = AIEventVehicleCrashed.Convert(e);
				local v = ec.GetVehicleID();
				if(AIVehicle.GetVehicleType(v) != AIVehicle.VT_AIR) {
					break;
				}
				AILog.Info("We have a crashed vehicle (" + v + "), buying a new one as replacement");
				this.BuildAircraft(this.route_1.GetValue(v), this.route_2.GetValue(v));
				this.route_1.RemoveItem(v);
				this.route_2.RemoveItem(v);
			} break;

			default:
				break;
		}
	}
}

function WrightAI::BuildAirRoute() {
	this.BuildAirportRoute();
	this.ManageAirRoutes();
	this.HandleEvents();
}

function WrightAI::checkAdjacentStation(airportTile) {
	local tileList = AITileList();
	tileList.AddRectangle(Utils.getOffsetTile(airportTile, (-7), (-7)),
						  Utils.getOffsetTile(airportTile, 7, 7));

	tileList.Valuate(Utils.isTileMyStation);
	tileList.KeepValue(1);
	tileList.Valuate(AIStation.GetStationID);

	local stationList = AIList();
	for(local tile = tileList.Begin(); tileList.HasNext(); tileList.Next()) {
		stationList.AddItem(tileList.GetValue(tile), 0);
	}

	//stationList.Valuate(AIStation.GetCargoWaiting, cargoId);
	//stationList.Sort(AIList.SORT_BY_VALUE, true);

	stationList.Valuate(AITown.GetDistanceManhattanToTile, airportTile);
	stationList.Sort(AIList.SORT_BY_VALUE, true);

	local adjacentStation = AIStation.STATION_NEW;
	if(stationList.Count()) {
		adjacentStation = stationList.Begin();
	}

	return adjacentStation;
}

function WrightAI::save() {
	local array = [];

	array.append(name);
	array.append(cargoId);

	array.append(distance_of_route);
	array.append(vehicle_to_depot);

	local usedTownsTable = {};
		for(local town = this.towns_used.Begin(), i = 0; this.towns_used.HasNext(); town = this.towns_used.Next(), ++i) {
		usedTownsTable.rawset(i, [town, towns_used.GetValue(town)]);
	}

	array.append(usedTownsTable);

	local route1Table = {};
	for(local route = this.route_1.Begin(), i = 0; this.route_1.HasNext(); route = this.route_1.Next(), ++i) {
		route1Table.rawset(i, [route, route_1.GetValue(route)]);
	}

	array.append(route1Table);

	local route2Table = {};
	for(local route = this.route_2.Begin(), i = 0; this.route_2.HasNext(); route = this.route_2.Next(), ++i) {
		route2Table.rawset(i, [route, route_2.GetValue(route)]);
	}

	array.append(route2Table);

	return array;
}

function WrightAI::load(data) {
	if(towns_used == null) {
		usedTownsTable = AIList();
	}

	if(route_1 == null) {
		route_1 = AIList();
	}

	if(route_2 == null) {
		route_2 = AIList();
	}

	name = data[0];
	cargoId= data[1];

	distance_of_route = data[2];
	vehicle_to_depot = data[3];

	local table = data[4];

	local i = 0;
	while(table.rawin(i)) {
		local town = table.rawget(i);
		towns_used.AddItem(town[0], town[1]);

		++i;
	}

	table = data[5];

	i = 0;
	while(table.rawin(i)) {
		local route = table.rawget(i);
		this.route_1.AddItem(route[0], route[1]);

		++i;
	}

	table = data[6];

	i = 0;
	while(table.rawin(i)) {
		local route = table.rawget(i);
		this.route_2.AddItem(route[0], route[1]);

		++i;
	}
}
