/*
 * This file is part of TownCars, which is an AI for OpenTTD
 * Copyright (C) 2009  Leif Linse
 *
 * TownCars is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License
 *
 * TownCars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with TownCars; If not, see <http://www.gnu.org/licenses/> or
 * write to the Free Software Foundation, Inc., 51 Franklin Street, 
 * Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

/*
 * Much of the code of this AI has been taken from the PAXLink AI,
 * also written by me (Zuu / Leif Linse). 
 */

// TownCars only use the Helper library from SuperLib
import("util.superlib", "SuperLib", 6);
Helper <- SuperLib.Helper;


class TownCars extends AIController {

	loaded_from_save = false;

	constructor()
	{
		this.loaded_from_save = false;
	}

	function Start();
	function Save();
	function Load(version, data);
}

function TownCars::SetCompanyName()
{
	if(!AICompany.SetName("TownCars"))
	{
		local i = 2;
		while(!AICompany.SetName("TownCars #" + i))
		{
			i = i + 1;
			if(i > 255)
				break;
		}
	}

	AICompany.SetPresidentName("Bob")
}

function GetTileRelative(relative_to_tile, delta_x, delta_y)
{
	local tile_x = AIMap.GetTileX(relative_to_tile);
	local tile_y = AIMap.GetTileY(relative_to_tile);

	local new_x = tile_x + delta_x;
	local new_y = tile_y + delta_y;

	return AIMap.GetTileIndex(new_x, new_y);
}

function GetNeighbours4(tile_id)
{
	local list = AITileList();

	if(!AIMap.IsValidTile(tile_id))
		return list;

	local tile_x = AIMap.GetTileX(tile_id);
	local tile_y = AIMap.GetTileY(tile_id);

	list.AddTile(GetTileRelative(tile_id, -1, 0));
	list.AddTile(GetTileRelative(tile_id, 0, -1));
	list.AddTile(GetTileRelative(tile_id, 1, 0));
	list.AddTile(GetTileRelative(tile_id, 0, 1));

	return list;
}

function CanConnectToRoad(road_tile, adjacent_tile_to_connect)
{
	local neighbours = GetNeighbours4(road_tile);
	
	neighbours.RemoveTile(adjacent_tile_to_connect);

	// This function requires that road_tile is connected with at least one other road tile.
	neighbours.Valuate(AIRoad.IsRoadTile);
	if(Helper.ListValueSum(neighbours) < 1)
		return false;
	
	for(local neighbour_tile_id = neighbours.Begin(); neighbours.HasNext(); neighbour_tile_id = neighbours.Next())
	{
		if(AIRoad.IsRoadTile(neighbour_tile_id))
		{
			local ret = AIRoad.CanBuildConnectedRoadPartsHere(road_tile, neighbour_tile_id, adjacent_tile_to_connect);
			if(ret == 0 || ret == -1)
				return false;
		}
	}

	return true;
}

function TownCars::BuildRoadStation(tile_id, depot)
{
	//AISign.BuildSign(tile_id, "Bld stn");
	AIRoad.SetCurrentRoadType(AIRoad.ROADTYPE_ROAD);

	if(AITile.IsWaterTile(tile_id))
		return false;

	// Don't even try if the tile has a steep slope
	local slope = AITile.GetSlope(tile_id);
	if(slope != 0 && AITile.IsSteepSlope(slope))
		return false;

	local neighbours = GetNeighbours4(tile_id);
	neighbours.Valuate(AIRoad.IsRoadTile);
	neighbours.KeepValue(1);

	neighbours.Valuate(CanConnectToRoad, tile_id);
	neighbours.KeepValue(1);

	//MyValuate(neighbours, AITile.IsSteepSlope); // steep sloped neighbours can't be used
	//neighbours.KeepValue(0);

	neighbours.Valuate(AITile.GetMaxHeight);
	neighbours.KeepValue(AITile.GetMaxHeight(tile_id));

	// Return false if none of the neighbours has road
	if(neighbours.Count() == 0)
		return false;

	local front = neighbours.Begin();


	if(!CanConnectToRoad(front, tile_id))
		return false;

	if(!AITile.IsBuildable(tile_id) && !AIRoad.IsRoadTile(tile_id) && !AICompany.IsMine(AITile.GetOwner(tile_id)))
	{
		AITile.DemolishTile(tile_id);
	}

	// Build road
	// Keep trying until we get another error than vehicle in the way
	while(!AIRoad.BuildRoad(tile_id, front) && AIError.GetLastError() == AIError.ERR_VEHICLE_IN_THE_WAY); 
	
	if(!AIRoad.AreRoadTilesConnected(tile_id, front))
		return false;

	Helper.SetSign(tile_id, "tile");
	Helper.SetSign(front, "front");

	// Build station/depot
	// Keep trying until we get another error than vehicle in the way
	local ret = false; // for some strange reason this variable declaration can't be inside the loop
	do
	{
		if(depot)
			ret = AIRoad.BuildRoadDepot(tile_id, front);
		else
			ret = AIRoad.BuildRoadStation(tile_id, front, AIRoad.ROADVEHTYPE_BUS, AIStation.STATION_JOIN_NEW);
	}
	while(!ret && AIError.GetLastError() == AIError.ERR_VEHICLE_IN_THE_WAY);

	return ret;
}
function BuildDepotAtTileScoreValuator(tile_id)
{
	local score = 0;

	local road_neighbour_count = AIRoad.GetNeighbourRoadCount(tile_id);

	if(AIRoad.IsRoadTile(tile_id))
	{
		// give score if tile has road, but only one neighbor has that, as that means it is a road stump
		if(road_neighbour_count == 1)
			score += 1000;
	}
	else
	{
		// better build it along an existing road, than hijacking a road stump
		if(road_neighbour_count > 0)
			score += 2000;
	}

	if(AITile.GetSlope(tile_id) == 0)
		score += 500;

	if(AITile.IsBuildable(tile_id))
	{
		score += 100;
	}
	else
	{
		local testmode = AITestMode();
		local acount = AIAccounting();

		AITile.DemolishTile(tile_id);
		
		score -= acount.GetCosts();
	}

	return score;
}

function TownCars::BuildDepot(town_id)
{
	local depot_placement_tiles = AITileList();
	local town_loc = AITown.GetLocation(town_id);
	depot_placement_tiles.AddRectangle(town_loc - AIMap.GetTileIndex(15, 15), town_loc + AIMap.GetTileIndex(15, 15));

	AILog.Info("Building new depot");

	// Only consider tiles that are buildable
	depot_placement_tiles.Valuate(AITile.IsBuildable);
	depot_placement_tiles.KeepValue(1);

	// search from town center and outwards
	depot_placement_tiles.Valuate(AIMap.DistanceManhattan, town_loc);
	depot_placement_tiles.Sort(AIAbstractList.SORT_BY_VALUE, true); // highest value last

	// try with all tiles until a working one is found
	local depot_tile = depot_placement_tiles.Begin();
	while(depot_placement_tiles.HasNext())
	{
		//Helper.SetSign(depot_tile, "tile");
		//AILog.Info("Trying to build depot at tile " + depot_tile);
		//AISign.BuildSign(depot_tile, "build depot");

		local depot = true;
		if(BuildRoadStation(depot_tile, depot))
			break;			

		depot_tile = depot_placement_tiles.Next();
	}

	if(!AIRoad.IsRoadDepotTile(depot_tile))
		return -1;

	// depot built likely is equal to no buses yet in town.
	return depot_tile;
}

function TownCars::BuildCars(town_id, car_count, depot_tile)
{
	local engine_list = AIEngineList(AIVehicle.VT_ROAD);

	if(AIController.GetSetting("build_only_zero_price_rvs") == 1)
	{
		engine_list.Valuate(AIEngine.GetPrice);
		engine_list.KeepValue(0);
	}
	if(AIController.GetSetting("build_only_zero_running_cost_rvs") == 1)
	{
		engine_list.Valuate(AIEngine.GetRunningCost);
		engine_list.KeepValue(0);
	}
	if(AIController.GetSetting("build_only_zero_capacity_rvs") == 1)
	{
		engine_list.Valuate(AIEngine.GetCapacity);
		engine_list.KeepValue(0);
	}
	if(AIController.GetSetting("build_only_if_cargo_type_is_pax") == 1)
	{
		engine_list.Valuate(AIEngine.GetCargoType);
		engine_list.KeepValue(Helper.GetPAXCargo());
	}

	if(engine_list.Count() == 0)
	{
		AILog.Error("There are no vehicles (engines) that the AI is allowed to buy.");
		AILog.Info("1| Check in the AI settings that you have defined "); 
		AILog.Info("2| correctly which road vehicles that should be built. ");
		AILog.Info("3| To have cars you will need to load a GRF containing cars, ");
		AILog.Info("4| and tell the AI how to distinguish cars from buses etc. ");
		AILog.Info("5| in the AI settings.");

//		AILog.Error("There is no vehicle that the AI is allowed to buy.");
//		AILog.Info("Checkout in the AI settings that you have defined");
//		AILog.Info("  correctly which road vehicles that should be built.");
//		AILog.Info("To have cars you will need to load a GRF containing cars,");
//		AILog.Info("  and tell the AI how to distinguish cars from buses etc.");
//		AILog.Info("  in the AI settings.");
		return false;
	}

	local engine_id = engine_list.Begin();
	for(local i = 0; i < car_count; i++)
	{
		local vehicle_id = AIVehicle.BuildVehicle(depot_tile, engine_id);
		if(!AIVehicle.IsValidVehicle(vehicle_id))
		{
			AILog.Error("failed to buy vehicle");
			return false;
		}

		AIVehicle.StartStopVehicle(vehicle_id);

		if(engine_list.HasNext())
			engine_id = engine_list.Next();
		else
			engine_id = engine_list.Begin();
	}

	return true;
}

function TownCars::Start()
{
	try
	{
		AILog.Info("--------------------------------------------------------------------------");
		AILog.Info("  TownCars  version 5");
		AILog.Info("--------------------------------------------------------------------------");
		AILog.Info("Hello, this is TownCars - an OpenTTD AI.");
		AILog.Info("I will spawn road vehicles that will drive randomly on the streets.");
		AILog.Info("However, I don't build any money making services, so if you want");
		AILog.Info("me to stay alive for long, use the cheats to give me some extra money.");
		AILog.Info(": Ctrl+Alt+C => Change company to company of this AI => Use money cheat =>");
		AILog.Info(": => Change back to your human player company");
		AILog.Info("");
		AILog.Info("If you find any bugs, please go to http://junctioneer.net/o-ai/TCAR and report them."); 
		AILog.Info("Further help can also be found on the same address."); 
		AILog.Info("--------------------------------------------------------------------------");
		AILog.Info("");

		this.Sleep(1);
		this.SetCompanyName();
		this.Sleep(10);

		AICompany.SetLoanAmount(AICompany.GetMaxLoanAmount());

		if(this.loaded_from_save)
		{
			AILog.Info("A game was loaded. No attempt will be made to");
			AILog.Info("  detect if not 'all' cars was generated before");
			AILog.Info("  game was saved. Instead the AI will be put into");
			AILog.Info("  sleep mode now.");
		}
		else
		{

			// Buy the cars
			local non_used_towns = AITownList();
			non_used_towns.Valuate(AITown.GetPopulation);
			non_used_towns.Sort(AIAbstractList.SORT_BY_VALUE, false); // highest value first

			local num_cars = 0;

			local max_num_cars = AIController.GetSetting("num_cars_total");
			local num_cars_per_town = AIController.GetSetting("num_cars_per_town");
			while(num_cars < max_num_cars && non_used_towns.Count() > 0)
			{
				local town_id = non_used_towns.Begin();
				non_used_towns.RemoveTop(1);

				AILog.Info("Now working in " + AITown.GetName(town_id));

				local depot_tile = BuildDepot(town_id);
				if(AIMap.IsValidTile(depot_tile))
				{
					AILog.Info("Depot built");

					local num_cars_to_buy = Helper.Min(num_cars_per_town, max_num_cars - num_cars);
					if(!BuildCars(depot_tile, num_cars_to_buy, depot_tile))
					{
						AILog.Warning("Building cars failed, aborting.. (no more cars will be built)");
						break;
					}
					num_cars += num_cars_to_buy;

					AILog.Info("Cars built");

					local front = AIRoad.GetRoadDepotFrontTile(depot_tile);
					while(!AITile.DemolishTile(depot_tile))
					{
						this.Sleep(10);
					}
					while(!AIRoad.RemoveRoad(depot_tile, front) && AIError.GetLastError() == AIError.ERR_VEHICLE_IN_THE_WAY); // clear the road stump from the depot

					AILog.Info("Depot cleared");
				}
				else
				{
					AILog.Warning("Failed to build depot");
				}
			}

			AILog.Info("");

			if(non_used_towns.Count() == 0)
			{
				AILog.Info("Stop buying more cars as there are no more towns to visit (built " + num_cars + " cars in total)."); 
				AILog.Info("The AI will be put into sleep forever.");
			}
			else
			{
				AILog.Info("Bought enough cars now (" + num_cars + " cars). The AI will be put into sleep forever.");
			}

			AILog.Info("");
		}

		this.Sleep(10);
		AILog.Info("Zzzz");

		while(true)
		{
			// Keep the company alive
			this.Sleep(1000);
		}

	}
	catch(e)
	{
		AILog.Info("");
		AILog.Warning("=== Ooops.. TownCars Crashed ===");
		AILog.Info("Error reports from users are very useful to fix issues like the one you just experienced.");
		AILog.Info("Please take a screenshot of the error message above. Use the button at the lower right to enlarge the window");
		AILog.Info("before taking a screenshot to include the entire error message in red above. It is important to get the full");
		AILog.Info("width of the error messages.");
	}
}

function TownCars::Save()
{
	local table = {};
	return table;
}

function TownCars::Load(version, data)
{
	this.loaded_from_save = true;
	AILog.Info("Previously saved with AI version " + version);
}

