 /*
 *	Copyright  2008 George Weller
 *	
 *	This file is part of PathZilla.
 *	
 *	PathZilla is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, either version 2 of the License, or
 *	(at your option) any later version.
 *	
 *	PathZilla is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *	
 *	You should have received a copy of the GNU General Public License
 *	along with PathZilla.  If not, see <http://www.gnu.org/licenses/>.
 *
 * PathWrapper.nut
 * 
 * Wrapper for the (modified) library road pathfinder. This class handles
 * general path finding a road construction with robust construction and a
 * path reassesment mechanism sensetive to changes in the map. This class also
 * implements a number of cost decorator functions, which are explained in the 
 * FindPath method below.
 * 
 * Author:  George Weller (Zutty)
 * Created: 15/01/2009
 * Version: 1.2
 */

class PathWrapper {
	// Feature constants
	FEAT_ROAD_LOOP = 1;
	FEAT_SEPARATE_ROAD_TYPES = 2;
	FEAT_GRID_LAYOUT = 3;
	FEAT_DEPOT_ALIGN = 4;
	FEAT_SHORT_SCOPE = 5;
	FEAT_NO_WORMHOLES = 6;
	FEAT_COUNTRY_LANE = 7;
	FEAT_CHK_CONNECTION = 8
	
	// Costs
	COST_ROAD_LOOP = 600;
	COST_SEPARATE_ROAD_TYPES = 7220;
	COST_PARALLEL_BONUS = 7590;
	COST_GRID_LAYOUT = 40;
	COST_DEPOT_ALIGN = 3000;
	COST_COUNTRY_LANE = 350;
	COST_CHK_CONNECTION = 0;	

	constructor() {
	}
}

/*
 * Find a path between the two specified tiles and then attempt to build it. All
 * parameters are passed up to the FindPath and TryBuldPath methods. 
 */
function PathWrapper::BuildRoad(fromTile, toTile, roadType, ignoreTiles = [], demolish = true, features = [],service=null) {
	// First, try to find a path
	local path = PathWrapper.FindPath(fromTile, toTile, roadType, ignoreTiles, demolish, features);

	// If the path could not be found then there is nothing left to try
	if(path == null) {
(::german)?AILog.Error("Kann keinen Pfad finden!"):AILog.Error("Could not find a path!");
		return false;
	}
	
	return PathWrapper.TryBuildPath(path, fromTile, toTile, roadType, ignoreTiles , demolish, features,service);
}
	
/*
 * Try to build ithe specified path with the specified road type up to
 * PathZilla.MAX_REPATH_TRIES times using the BuildPath method.
 */
function PathWrapper::TryBuildPath(path, fromTile, toTile, roadType, ignoreTiles = [], demolish = true, features = [],service=null) {
	if (path==null||path==false)return false;

	local tries = 0;
	local success = -1;
	// Try to build the path
	do {
		success = PathWrapper.BuildPath(path, roadType);
			if (path==null||path==false){
success=-1;
break;
}
		// If we failed, try to find the path from the point it went wrong
		if(success != 0) {
if (AICompany.GetBankBalance(AICompany.COMPANY_SELF)<222222) FinanceManager.Borrow(AICompany.GetLoanInterval()*2);
if (tries%3==2)demolish=(!demolish);
			path = PathWrapper.FindPath(fromTile, success, roadType, ignoreTiles, demolish, features);
		}
	} while(success > 0 && tries++ < PathZilla.MAX_REPATH_TRIES);
	
	// If we still failed after a number of attempts, show an error message
	if(success != 0) {
(::german)?AILog.Error("Strasse konnte nicht gebaut werden!"):AILog.Error("Road cannot be built.");
		return false;
	}
	if (path==null||path==false)return false;
	return true;
}

/*
 * Find a path between the specified tiles and return it. The path will be of 
 * type roadType, will go around any tiles specified in the ignoreTiles array.
 * If demolish is true then the path may go through town houses. Additionally
 * a number of features may be specified to control the layout of the eventual
 * path. Any of the following can be specified...
 *
 * FEAT_ROAD_LOOP - Build a loop around the first tile in ignoreTiles
 * FEAT_SEPARATE_ROAD_TYPES - Split road types apart to run in parallel
 * FEAT_GRID_LAYOUT - Snap roads to x2/3x3 town layouts
 * FEAT_DEPOT_ALIGN - Join a road to the entrace of a depot not its side
 * FEAT_SHORT_SCOPE - Avoid wasting time on paths that are known to be short
 * FEAT_NO_WORMHOLES - Disallow bridges and tunnels
 */
function PathWrapper::FindPath(fromTile, toTile, roadType, ignoreTiles = [], demolish = false, features = [],second=0) {
	
	// Ignore traffic black spots
	ignoreTiles.extend(ListToArray(::trafficBlackSpots));

	// Initialise the pathfinder
	local pathfinder = Road();

	pathfinder.cost.allow_demolition = demolish;
	pathfinder.cost.demolition = 9980;
	pathfinder.cost.no_existing_road = 3380;
	pathfinder.cost.max_bridge_length = PathZilla.MAX_BRIDGE_LENGTH;
	pathfinder.cost.bridge_per_tile = 383;
	pathfinder.cost.tunnel_per_tile = 343;
	pathfinder.InitializePath([fromTile], [toTile], 1, 20, ignoreTiles);
	local stpj=0;
	// Add on any additional features
	foreach(feat in features) {
		switch(feat) {
			case PathWrapper.FEAT_ROAD_LOOP:
/*				local sideRoadList = LandManager.GetAdjacentTileList(ignoreTiles[0]);
				sideRoadList.RemoveTile(toTile);
				sideRoadList.RemoveTile(LandManager.GetApproachTile(ignoreTiles[0], toTile));
				local sideRoads = ListToArray(sideRoadList);

				pathfinder.RegisterCostCallback(function (tile, prevTile, sideRoads) {
					return (tile == sideRoads[0] || tile == sideRoads[1]) ? PathWrapper.COST_ROAD_LOOP : 0;
				}, sideRoads);
*/			break;
			case PathWrapper.FEAT_SEPARATE_ROAD_TYPES:
				pathfinder.RegisterCostCallback(function (tile, prevTile, roadType) {
					local diff = AIMap.GetMapSizeY() / (tile - prevTile);
					local parrl = (AIRoad.IsRoadTile(tile + diff) && !AIRoad.HasRoadType(tile + diff, roadType))
								|| (AIRoad.IsRoadTile(tile - diff) && !AIRoad.HasRoadType(tile - diff, roadType));
					return ((AIRoad.IsRoadTile(tile) && !AIRoad.HasRoadType(tile, roadType)) ? PathWrapper.COST_SEPARATE_ROAD_TYPES : 0) + ((parrl) ? 0 : PathWrapper.COST_PARALLEL_BONUS);
				}, roadType);
			break;
			case PathWrapper.FEAT_GRID_LAYOUT:
				local n = AIGameSettings.GetValue("economy.town_layout").tointeger();
				if((n >= 2 && n <= 4) && roadType == AIRoad.ROADTYPE_ROAD) {
					local towns = AITownList();
					
					// Find the nearest town to the "from" tile
					towns.Valuate(AITown.GetDistanceManhattanToTile, fromTile);
					towns.Sort(AIAbstractList.SORT_BY_VALUE, true);
					local fTown = towns.Begin();
					local fTile = AITown.GetLocation(fTown);
					
					// Get some details about this town
					local fType = AITown.GetRoadLayout(fTown);
					local fSize = AITown.GetPopulation(fTown).tofloat(); 
					local fn = (fType == AITown.ROAD_LAYOUT_2x2) ? 3 : ((fType == AITown.ROAD_LAYOUT_3x3) ? 4 : 0);
					local fx = fTile % AIMap.GetMapSizeY();
					local fy = fTile / AIMap.GetMapSizeY();

					// Find the nearest town to the "to" tile
					towns.Valuate(AITown.GetDistanceManhattanToTile, toTile);
					towns.Sort(AIAbstractList.SORT_BY_VALUE, true);
					local tTown = towns.Begin();
					local tTile = AITown.GetLocation(tTown);
					
					// If both towns are the same then we only need to check one grid
					if(fTown == tTown && fn > 0) {
					//deactivated this because of too big roadloops for town stations sometimes
						return 0;
						pathfinder.RegisterCostCallback(function (tile, prevTile, n, fx, fy) {
							local x = tile % AIMap.GetMapSizeY();
							local y = tile / AIMap.GetMapSizeY();
							local dx = abs(x - fx) % n;
							local dy = abs(y - fy) % n;
							local len = AIMap.DistanceManhattan(tile, prevTile);
							
							if(len > 1) {
								local px = prevTile % AIMap.GetMapSizeY();
								local py = prevTile / AIMap.GetMapSizeY();
								//local pdx = abs(px - fx) % n;
								local pdy = abs(py - fy) % n;
								
								if((x == px && dx == 0) || (y == py && dy == 0)) return 0;

								local m = 0;
								if(dy == pdy) {
									m = ((dx == 0) ? 1 : 0) + (len / n);
								} else {
									m = ((dy == 0) ? 1 : 0) + (len / n);
								}
								
								return PathWrapper.COST_GRID_LAYOUT * (len - m);
							} else {
								return (dx == 0 || dy == 0) ? 0 : PathWrapper.COST_GRID_LAYOUT;
							}
						}, fn, fx, fy);
					} else if(fTown != tTown) {
						// Otherwise get details about the other town
						local tType = AITown.GetRoadLayout(tTown);
						local tSize = AITown.GetPopulation(tTown).tofloat(); 
						local tn = (tType == AITown.ROAD_LAYOUT_2x2) ? 3 : ((tType == AITown.ROAD_LAYOUT_3x3) ? 4 : 0);
						local tx = tTile % AIMap.GetMapSizeY();
						local ty = tTile / AIMap.GetMapSizeY();
						
						// Calculate grid weights for each town based on population
						local fWeight = min(0.8, max(0.2, fSize / (fSize + tSize))); 
						local tWeight = min(0.8, max(0.2, tSize / (fSize + tSize))); 

						// Calculate the distance between towns and initialise
						local totalDist = AITile.GetDistanceManhattanToTile(fTile, tTile).tofloat();
						local stx = (tx - fx).tofloat() / totalDist;
						local sty = (ty - fy).tofloat() / totalDist;
						local fRad = null;
						local tRad = null;
						
						// Find the "radii" of the two towns, by plotting a line between them
						for(local i = 0; i < totalDist; i++) {
							local tile = (fx + (stx * i.tofloat()) +  ((fy + (sty * i.tofloat())).tointeger() * AIMap.GetMapSizeY())).tointeger();
				
							if(!AITown.IsWithinTownInfluence(fTown, tile) && fRad == null) {
								fRad = i - 1;
								if(tRad != null) break;
							}
							if(AITown.IsWithinTownInfluence(tTown, tile) && tRad == null) {
								tRad = totalDist - i;
								if(fRad != null) break;
							}
						}
						if(fRad == null) fRad = totalDist;

						// If either town has a grid road layout then interpolate between the two
						if(fn > 0 || tn > 0) {
							pathfinder.RegisterCostCallback(function (tile, prevTile, fTile, fWeight, fRad, fn, fx, fy, tTile, tWeight, tRad, tn, tx, ty) {
								local x = tile % AIMap.GetMapSizeY();
								local y = tile / AIMap.GetMapSizeY();
								local fdx, fdy, tdx, tdy;
								if(fn > 0) {
									fdx = abs(x - fx) % fn;
									fdy = abs(y - fy) % fn;
								}
								if(tn > 0) {
									tdx = abs(x - tx) % tn;
									tdy = abs(y - ty) % tn;
								}

								local len = AIMap.DistanceManhattan(tile, prevTile);
								local fCost = 0;
								local tCost = 0;
		
								// Account for bridges and tunnels
								if(len > 1) {
									local px = prevTile % AIMap.GetMapSizeY();
									local py = prevTile / AIMap.GetMapSizeY();
									local fpdy = (fn > 0) ? abs(py - fy) % fn : 0;
									local tpdy = (tn > 0) ? abs(py - ty) % tn : 0;
									
									local fm = 0;
									if(fn > 0) {
										if(fdy == fpdy) {
											fm = ((fdx == 0) ? 1 : 0) + (len / fn);
										} else {
											fm = ((fdy == 0) ? 1 : 0) + (len / fn);
										}
									}

									local tm = 0;
									if(tn > 0) {
										if(tdy == fpdy) {
											tm = ((tdx == 0) ? 1 : 0) + (len / tn);
										} else {
											tm = ((tdy == 0) ? 1 : 0) + (len / tn);
										}
									}
									
									fCost = (fn == 0 || (x == px && fdx == 0) || (y == py && fdy == 0)) ? 0.0 : PathWrapper.COST_GRID_LAYOUT * (len - fm - 0.0);
									tCost = (tn == 0 || (x == px && tdx == 0) || (y == py && tdy == 0)) ? 0.0 : PathWrapper.COST_GRID_LAYOUT * (len - tm - 0.0);
								} else {
									fCost = (fn == 0 || fdx == 0 || fdy == 0) ? 0.0 : PathWrapper.COST_GRID_LAYOUT.tofloat();
									tCost = (tn == 0 || tdx == 0 || tdy == 0) ? 0.0 : PathWrapper.COST_GRID_LAYOUT.tofloat();
								}
		
								local fDist = AITile.GetDistanceManhattanToTile(tile, fTile).tofloat();
								local tDist = AITile.GetDistanceManhattanToTile(tile, tTile).tofloat();
								local total = (fDist + tDist) - (fRad + tRad);
								local GRID_MIX_DEAD_SPOT = 0.05;

								// Calculate the balancing weights based on our location within either town
								local fBal, tBal;
								if(fDist < fRad) {
									// If inside the "from" town, only apply the "from" grid
									fBal = 1.0;
									tBal = 0.0;
								} else if(tDist < tRad){
									// If inside the "to" town, only apply the "to" grid
									fBal = 0.0;
									tBal = 1.0;
								} else {
									// Otherwise mix the two grids based on a weighted distance
									fBal = max(0.0, ((tDist - tRad) / (total * (fWeight + GRID_MIX_DEAD_SPOT))) - ((1 / (fWeight + GRID_MIX_DEAD_SPOT)) - 1));
									tBal = max(0.0, ((fDist - fRad) / (total * (tWeight + GRID_MIX_DEAD_SPOT))) - ((1 / (tWeight + GRID_MIX_DEAD_SPOT)) - 1));
								}

								return ((fBal * fCost) + (tBal * tCost)).tointeger();
							}, fTile, fWeight, fRad, fn, fx, fy, tTile, tWeight, tRad, tn, tx, ty);
						}
					}
				}
			break;
			case PathWrapper.FEAT_DEPOT_ALIGN:
				local sideTileList = LandManager.GetAdjacentTileList(toTile);
				sideTileList.Valuate(function (tile, roadType,toTile) {
					return AIRoad.IsRoadTile(tile) && AIRoad.HasRoadType(tile, roadType)&&AIRoad.AreRoadTilesConnected(tile,toTile)&&(!AIRoad.IsDriveThroughRoadStationTile(toTile));
				}, roadType,toTile);
				sideTileList.KeepValue(0);
				local sideTiles = ListToArray(sideTileList);
				if(sideTiles.len() < 4) {
					pathfinder.RegisterCostCallback(function (tile, prevTile, sideTiles) {
						local misaligned = false;
						if(sideTiles.len() >= 3) misaligned = misaligned || (tile == sideTiles[2]);
						if(sideTiles.len() >= 2) misaligned = misaligned || (tile == sideTiles[1]);
						if(sideTiles.len() >= 1) misaligned = misaligned || (tile == sideTiles[0]);
						return (misaligned) ? PathWrapper.COST_DEPOT_ALIGN : 0;
					}, sideTiles);
				}
			break;
			case PathWrapper.FEAT_SHORT_SCOPE:
	pathfinder.cost.no_existing_road = 7880;
				pathfinder.cost.max_cost = 3000000;
			break;
			case PathWrapper.FEAT_NO_WORMHOLES:
				pathfinder.cost.max_tunnel_length = 1;
				pathfinder.cost.max_bridge_length = 1;
				pathfinder.cost.demolition = 14980;
				pathfinder.cost.turn=19150;
				pathfinder.cost.no_existing_road = 28880;
				stpj=32;
			break;
			case PathWrapper.FEAT_COUNTRY_LANE:
				if(Settings.EnableCountryLanes() && roadType == AIRoad.ROADTYPE_ROAD) {
					local towns = AITownList();
					
					// Find the nearest town to the "from" tile
					towns.Valuate(AITown.GetDistanceManhattanToTile, fromTile);
					towns.Sort(AIAbstractList.SORT_BY_VALUE, true);
					local fSize = AITown.GetPopulation(towns.Begin());
					local fType = AITown.GetRoadLayout(towns.Begin());
					local fGrid = (fType == AITown.ROAD_LAYOUT_2x2 || fType == AITown.ROAD_LAYOUT_3x3);
					
					// Find the nearest town to the "to" tile
					towns.Valuate(AITown.GetDistanceManhattanToTile, toTile);
					towns.Sort(AIAbstractList.SORT_BY_VALUE, true);
					local tSize = AITown.GetPopulation(towns.Begin());
					local tType = AITown.GetRoadLayout(towns.Begin());
					local tGrid = (tType == AITown.ROAD_LAYOUT_2x2 || tType == AITown.ROAD_LAYOUT_3x3);
					
					// Get the year
					local year = AIDate.GetYear(AIDate.GetCurrentDate());
					local LATE_YEAR = 1950;
					local EARLY_YEAR = 1940;
					
					// If the towns are small and simple enough, add cost for trees and rough terrain, to make the road "windy"
					if(year < LATE_YEAR && ((year >= EARLY_YEAR && !fGrid && !tGrid) || year < EARLY_YEAR) && fSize < 1000 && tSize < 1000) {
						pathfinder.RegisterCostCallback(function (tile, prevTile) {
							local len = AIMap.DistanceManhattan(tile, prevTile);
							if(len > 1) return PathWrapper.COST_COUNTRY_LANE * len;
							local slope = AITile.GetSlope(tile);
							local unevenTerrain = !(slope == AITile.SLOPE_FLAT || slope == AITile.SLOPE_NW || slope == AITile.SLOPE_SW || slope == AITile.SLOPE_NE || slope == AITile.SLOPE_SE);
							return (unevenTerrain || AITile.HasTreeOnTile(tile) || AITile.IsFarmTile(tile) || AITile.IsRockTile(tile) || AITile.IsRoughTile(tile)) ? PathWrapper.COST_COUNTRY_LANE : 0;
						});
					}
				}
			break;
			case PathWrapper.FEAT_CHK_CONNECTION:
			pathfinder.cost.allow_demolition = false;
			pathfinder.cost.demolition = 10000000;
			pathfinder.cost.no_existing_road = 10000000;
			pathfinder.cost.max_bridge_length = PathZilla.MAX_BRIDGE_LENGTH;
			pathfinder.cost.bridge_per_tile = 10000000;
			pathfinder.cost.tunnel_per_tile = 10000000;
			break;			

		}
	}

(::german)?AILog.Info("    Suche einen Pfad von [" + AIMap.GetTileX(fromTile) + ", " + AIMap.GetTileY(fromTile) + "] nach [" + AIMap.GetTileX(toTile) + ", " + AIMap.GetTileY(toTile) + "]..."):AILog.Info("    Trying to find a path between [" + AIMap.GetTileX(fromTile) + ", " + AIMap.GetTileY(fromTile) + "] and [" + AIMap.GetTileX(toTile) + ", " + AIMap.GetTileY(toTile) + "]...");

	// Make the necessary preparations
	FinanceManager.EnsureFundsAvailable(PathZilla.FLOAT*3);
	AIRoad.SetCurrentRoadType(roadType);
	// If we are very poor, do not attempt to build tunnels
	if(!FinanceManager.CanAfford(PathZilla.FLOAT)) {
		pathfinder.cost.max_tunnel_length = 1;
	}
	// Run the pathfinder
	local path = false;
	local steps = 0;
        local zlr=0;
	local fpf=((AIMap.DistanceManhattan(fromTile,toTile)+33));
	local wd=22+AIMap.DistanceManhattan(fromTile,toTile)*2;
if (stpj>0&&second==0){
pathfinder.cost.allow_demolition=false;
second=2;
}
if (stpj>0&&second==1){
pathfinder.cost.allow_demolition=true;
stpj=18;
}
if (wd>555)return null;
	while (path == false&&zlr++<wd) {
if (stpj>0)stpj=stpj-2;
		path = pathfinder.FindPath(fpf-stpj);
if(zlr%19==8&&AICompany.GetLoanAmount()*3<AICompany.GetMaxLoanAmount())::pz.CloneVehicles(true);
	}

if (path!= false &&path !=null&&zlr<wd-1&&path.GetParent()!=null) {
(::german)?AILog.Info("      Pfad gefunden!"):AILog.Info("      Done finding path.");

	// Return the finished path
	
	return path;
}else{
if (second==2){
return PathWrapper.FindPath(fromTile, toTile, roadType, ignoreTiles, demolish, features,1);
}
(::german)?AILog.Warning("      Kein Pfad gefunden!"):AILog.Warning("      No path found!");
return null;
}
}


function PathWrapper::OptimizePath(path,fromTile, toTile, roadType, ignoreTiles = [], demolish = false, features = [],rek=0) {

if (path == null||path==false||AIMap.DistanceManhattan(fromTile,toTile)>=path.GetLength())return path;
local plen=path.GetLength()+1;
local p=path;
local xm=AIList();
local xp=AIList();
local ym=AIList();
local yp=AIList();
local bad=AIList();
local my=AIMap.GetMapSizeX();
local z=0;
local dir=array(plen);
local cost=array(plen);
local tile=array(plen);
local par=array(plen);
local len=array(plen);
local ptile=path.GetTile();
while (path != null&&path!=false) {
		tile[z] = ptile;
		par[z] = path.GetParent();
		len[z]=path.GetLength();
		cost[z]=path.GetCost();
		dir[z]=path.GetDirection();
//AISign.BuildSign(tile[z],"O");
if (par[z]!=null){
                ptile=par[z].GetTile();
(tile[z]-ptile==1)?xm.AddItem(ptile,z):(tile[z]-ptile==-1)?xp.AddItem(ptile,z):(tile[z]-ptile==my)?ym.AddItem(ptile,z):(tile[z]-ptile==-my)?yp.AddItem(ptile,z):0;
}
path=par[z];
z++;
}
(xm.Count()<=xp.Count())?bad.AddList(xm):(xp.Count()<=xm.Count())?bad.AddList(xp):0;
(ym.Count()<=yp.Count())?bad.AddList(ym):(yp.Count()<=ym.Count())?bad.AddList(yp):0;
if(bad.IsEmpty())return p;
if (rek==0)(::german)?AILog.Info("      Pfadoptimierung wird durchgefuehrt..."):AILog.Info("      Trying to improve path...");
bad.Sort(AIAbstractList.SORT_BY_VALUE, false);
local zl=bad.GetValue(bad.Begin());
bad.Sort(AIAbstractList.SORT_BY_VALUE, true);
local ze=bad.GetValue(bad.Begin());
local xmax=AIMap.GetMapSizeX();
local xmin=0;
local ymax=AIMap.GetMapSizeY();
local ymin=0;
local xzmin=xmax;
local xzmax=xmin;
local yzmin=ymax;
local yzmax=ymin;
for(local i=ze;i<=zl;i++){
xzmax=max(xzmax,min(AIMap.GetTileX(tile[i]),xmax));
xzmin=min(xzmin,max(AIMap.GetTileX(tile[i]),xmin));
yzmax=max(yzmax,min(AIMap.GetTileY(tile[i]),ymax));
yzmin=min(yzmin,max(AIMap.GetTileY(tile[i]),ymin));
}
local nf=toTile;
local nt=fromTile;
local onz=z-1;
local ofz=0;
for(local i=1;i<min(z-2,ze);i++){
local tx=AIMap.GetTileX(tile[i]);
local ty=AIMap.GetTileY(tile[i]);
//AISign.BuildSign(tile[i],"ze"+i.tostring()+"|"+tx+"/"+ty+"|"+xzmin+"|"+xzmax+"|"+yzmin+"|"+yzmax);
if(((tx>xzmax||tx<xzmin)&&(ty>yzmax||ty<yzmin))){
if(AITile.GetSlope(tile[i])!=AITile.SLOPE_FLAT||AIMap.DistanceManhattan(tile[i],tile[i-1])>1||AIMap.DistanceManhattan(tile[i],tile[i+1])>1||AITunnel.IsTunnelTile(tile[i-1])||AITunnel.IsTunnelTile(tile[i+1])||AIBridge.IsBridgeTile(tile[i-1])||AIBridge.IsBridgeTile(tile[i+1]))continue;
nf=tile[i];
ofz=i;
}else{
//if(!(AITile.GetSlope(tile[i])!=AITile.SLOPE_FLAT||AIMap.DistanceManhattan(tile[i],tile[i-1])>1||AIMap.DistanceManhattan(tile[i],tile[i+1])>1))nt=tile[i];
break;
}
}
for(local i=z-2;i>max(1,zl);i--){
local tx=AIMap.GetTileX(tile[i]);
local ty=AIMap.GetTileY(tile[i]);
//AISign.BuildSign(tile[i],"zl"+i.tostring()+"|"+tx+"/"+ty+"|"+xzmin+"|"+xzmax+"|"+yzmin+"|"+yzmax);
if(((tx>xzmax||tx<xzmin)&&(ty>yzmax||ty<yzmin))){
if(AITile.GetSlope(tile[i])!=AITile.SLOPE_FLAT||AIMap.DistanceManhattan(tile[i],tile[i-1])>1||AIMap.DistanceManhattan(tile[i],tile[i+1])>1||AITunnel.IsTunnelTile(tile[i-1])||AITunnel.IsTunnelTile(tile[i+1])||AIBridge.IsBridgeTile(tile[i-1])||AIBridge.IsBridgeTile(tile[i+1]))continue;
nt=tile[i];
onz=i;
}else{
//if(!(AITile.GetSlope(tile[i])!=AITile.SLOPE_FLAT||AIMap.DistanceManhattan(tile[i],tile[i-1])>1||AIMap.DistanceManhattan(tile[i],tile[i+1])>1))nt=tile[i];
break;
}
}
//AISign.BuildSign(nf,"from"+z);
//AISign.BuildSign(nt,"to"+z);
local pp=PathWrapper.FindPath(nf, nt, roadType, ignoreTiles, demolish, features);
//AILog.Error(pp.GetLength()+">="+(onz-ofz).tostring());
if (pp==null||pp==false||pp.GetParent()==null||pp.GetLength()>=onz-ofz){
if (rek==0)(::german)?AILog.Info("      Pfad konnte nicht verkuerzt werden..."):AILog.Info("      No better path was found...");
return p;
}
local opt=PathWrapper.OptimizePath(pp,nf, nt, roadType, ignoreTiles, demolish, features,rek+1);
local final= clone pp;
final._prev=par[onz];
final._tile=tile[onz];
final._direction=255;
final._cost=cost[onz];
final._length=onz;
local old=final._prev;
local ii=final.GetLength()+1;
local lopt=opt.GetParent()
while (lopt != null&&lopt!=false) {
local dop=lopt.GetDirection();
dop=((dop==1)?2:(dop==2)?1:(dop==4)?8:(dop==8)?4:dop);
old=clone final;
final._prev=old;
//AILog.Info(ii+"|"+AIMap.GetTileX(final.GetTile())+"/"+AIMap.GetTileY(final.GetTile()));
final._tile=lopt.GetTile();

final._direction=dop;
final._cost=lopt.GetCost();
final._length=ii++;
local popt=lopt.GetParent();
lopt=popt;
}
if (ofz>0){
for(local i=ofz-1;i>=0;i--){
dir[i]=((dir[i]==1)?2:(dir[i]==2)?1:(dir[i]==4)?8:(dir[i]==8)?4:dir[i]);
old=clone final;
final._prev=old;
final._tile=tile[i];
//AILog.Info(ii+"|"+AIMap.GetTileX(final.GetTile())+"/"+AIMap.GetTileY(final.GetTile()));
final._direction=dir[i];
final._cost=cost[i];
final._length=ii;
}}
z=0;
path= clone final;
local mp=-1;
while (path != null&&path!=false) {
		tile[z] = path.GetTile();
for (local i=0;i<z;i++){
if (tile[z]==tile[i]){
z=i;
mp=max(mp,i);
break;
}}
		par[z] = path.GetParent();
		len[z] = path.GetLength();
		cost[z] = path.GetCost();
		dir[z]=path.GetDirection();
len[z]=((z==0)?0:AIMap.DistanceManhattan(tile[z-1],tile[z]));
path=par[z];
z++;
}
if (mp>-1){
local llt=par[mp];
for(local i=mp;i>=0;i--){
final._prev=llt;
final._tile=tile[i];
final._direction=dir[i];
final._cost=cost[i];
final._length=len[i];
llt=clone final;
}
local ffinal=clone final
while (ffinal != null&&ffinal!=false) {
//AISign.BuildSign(ffinal.GetTile(),"XXX   XXX");
//AILog.Error(AIMap.GetTileX(ffinal.GetTile())+"/"+AIMap.GetTileY(ffinal.GetTile()));
local pfinal=ffinal.GetParent();
ffinal=pfinal;
}
}
if (rek==0)(::german)?AILog.Info("      Pfad erfolgreich um " +(max(1,plen-z)).tostring()+" Feld"+((plen-z>1)?"er":"")+" verkuerzt"):AILog.Info("      Path successfully shortened by " +(max(1,plen-z)).tostring()+" tile"+((plen-z>1)?"s":""));
return final;
}


/*
 * Build the path specified by path as a road of type roadType. If there any
 * construction errors the method will re-try to a limited extent. If this also
 * fails the method will return non-zero. If the returned value is greater than
 * zero it indicates the tile just before which construction failed. If it is 
 * less than zero it indicates that construction strictly cannot be completed.
 */
function PathWrapper::BuildPath(path, roadType) {	
	AIRoad.SetCurrentRoadType(roadType);
local ilo=0;
	local prevTile = null;
	local stopList = AIList();
(::german)?AILog.Info("      Baue eine Strasse..."):AILog.Info("      Building a road...")

	while (path != null&&path!=false) {
		local par = path.GetParent();
		local tile = path.GetTile();

		if (par != null) {
			local ptile = par.GetTile();
//AILog.Info(AIMap.GetTileX(tile)+"/"+AIMap.GetTileY(tile)+"-"+AIMap.GetTileX(ptile)+"/"+AIMap.GetTileY(ptile));
			local distance = AIMap.DistanceManhattan(tile, ptile);
			FinanceManager.EnsureFundsAvailable(PathZilla.FLOAT);

			// Check if we need to demolish the tile (e.g. is a town house is in the way)
			if(!AITile.IsWaterTile(ptile)&&!AITile.IsBuildable(ptile) &&!AIRoad.IsRoadStationTile(ptile)&&!AIRoad.IsRoadDepotTile(ptile)&&!AIRoad.IsDriveThroughRoadStationTile(ptile)&&!(AIRoad.IsRoadTile(ptile) ||AIBridge.IsBridgeTile(ptile) ||AITunnel.IsTunnelTile(ptile))) {
				AITile.DemolishTile(ptile);
				FinanceManager.EnsureFundsAvailable(PathZilla.FLOAT);
			}
			
			local success = false;
			local attempts = 0;

			// Try to build the next path segment
			while(!success && attempts++ < PathZilla.MAX_CONSTR_ATTEMPTS) {
				if(distance == 1) {
success=(((AITunnel.IsTunnelTile(tile)&&AITunnel.GetOtherTunnelEnd(tile)==ptile)||(AIBridge.IsBridgeTile(tile)&&AIBridge.GetOtherBridgeEnd(tile)==ptile)));
if (!success){
if(prevTile==null)prevTile=tile+tile-ptile;
local ppar=par.GetParent();
{
local ppTile=((ppar!=null)?ppar.GetTile():ptile+ptile-tile);
local pppar=((ppar!=null)?ppar.GetParent():ppar);
{
local pppTile=((pppar!=null)?pppar.GetTile():ppTile+ppTile-ptile);
{
local ppppar=((pppar!=null)?pppar.GetParent():pppar);
{
local ppppTile=((ppppar!=null)?ppppar.GetTile():pppTile+pppTile-ppTile);
local pppppar=((ppppar!=null)?ppppar.GetParent():ppppar);
local pppppTile=((pppppar!=null)?pppppar.GetTile():ppppTile+ppppTile-pppTile);
local ppppppar=((pppppar!=null)?pppppar.GetParent():pppppar);
local ppppppTile=((ppppppar!=null)?ppppppar.GetTile():pppppTile+pppppTile-ppppTile);
local pppppppar=((ppppppar!=null)?ppppppar.GetParent():ppppppar);
local pppppppTile=((pppppppar!=null)?pppppppar.GetTile():ppppppTile+ppppppTile-pppppTile);
local ppppppppar=((pppppppar!=null)?pppppppar.GetParent():pppppppar);
local ppppppppTile=((ppppppppar!=null)?ppppppppar.GetTile():pppppppTile+pppppppTile-ppppppTile);
/*
ilo++;
if (AIMap.DistanceManhattan(tile,pppppTile)==1||AIMap.DistanceManhattan(tile,ppppTile)==1||AIMap.DistanceManhattan(tile,pppTile)==1){
AISign.BuildSign(prevTile,"prevTile"+ilo);
AISign.BuildSign(tile,"tile"+ilo);
AISign.BuildSign(ptile,"ptile"+ilo);
AISign.BuildSign(ppTile,"ppTile"+ilo);
AISign.BuildSign(pppTile,"pppTile"+ilo);
AISign.BuildSign(ppppTile,"ppppTile"+ilo);
AISign.BuildSign(pppppTile,"pppppTile"+ilo);
AISign.BuildSign(ppppppTile,"ppppppTile"+ilo);
AISign.BuildSign(pppppppTile,"pppppppTile"+ilo);
}
*/
if ((AIMap.DistanceManhattan(tile,pppppppTile)==1&&pppppppar!=null&&AIMap.DistanceManhattan(pppppppTile,ppppppppTile)==1&&AIMap.DistanceManhattan(prevTile,tile)==1)&&((AIRoad.IsRoadTile(pppppppTile)||(AITile.IsBuildable(pppppppTile)||(AITown.GetDistanceManhattanToTile(AITile.GetClosestTown(pppppppTile),pppppppTile)*80<max(1500,AITown.GetPopulation(AITile.GetClosestTown(pppppppTile)))))))){
if (!AIRoad.IsRoadTile(tile)&&AITile.GetOwner(tile)!=AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)&&!AITile.IsWaterTile(tile))AITile.DemolishTile(tile);
if (!AIRoad.IsRoadTile(pppppppTile)&&AITile.GetOwner(pppppppTile)!=AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)&&!AITile.IsWaterTile(pppppppTile))AITile.DemolishTile(pppppppTile);
if (!AIRoad.IsRoadTile(ppppppppTile)&&AITile.GetOwner(ppppppppTile)!=AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)&&!AITile.IsWaterTile(ppppppppTile))AITile.DemolishTile(ppppppppTile);
success = AIRoad.AreRoadTilesConnected(tile,pppppppTile)||(AIRoad.CanBuildConnectedRoadPartsHere(prevTile,tile,pppppppTile)>0&&AIRoad.CanBuildConnectedRoadPartsHere(tile,pppppppTile,ppppppppTile)>0&&this.RoadManager.SafelyBuildRoad(tile, pppppppTile,30));
if (success){
ptile=pppppppTile;
par=pppppppar;
}}else if ((AIMap.DistanceManhattan(tile,ppppppTile)==1&&pppppppar!=null&&AIMap.DistanceManhattan(ppppppTile,pppppppTile)==1&&AIMap.DistanceManhattan(prevTile,tile)==1)&&((AIRoad.IsRoadTile(ppppppTile)||(AITile.IsBuildable(ppppppTile)||(AITown.GetDistanceManhattanToTile(AITile.GetClosestTown(ppppppTile),ppppppTile)*80<max(1500,AITown.GetPopulation(AITile.GetClosestTown(ppppppTile)))))))){
if (!AIRoad.IsRoadTile(tile)&&AITile.GetOwner(tile)!=AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)&&!AITile.IsWaterTile(tile))AITile.DemolishTile(tile);
if (!AIRoad.IsRoadTile(ppppppTile)&&AITile.GetOwner(ppppppTile)!=AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)&&!AITile.IsWaterTile(ppppppTile))AITile.DemolishTile(ppppppTile);
if (!AIRoad.IsRoadTile(pppppppTile)&&AITile.GetOwner(pppppppTile)!=AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)&&!AITile.IsWaterTile(pppppppTile))AITile.DemolishTile(pppppppTile);
success = AIRoad.AreRoadTilesConnected(tile,ppppppTile)||(AIRoad.CanBuildConnectedRoadPartsHere(prevTile,tile,ppppppTile)>0&&AIRoad.CanBuildConnectedRoadPartsHere(tile,ppppppTile,pppppppTile)>0&&this.RoadManager.SafelyBuildRoad(tile, ppppppTile,30));
if (success){
ptile=ppppppTile;
par=ppppppar;
}}else if ((AIMap.DistanceManhattan(tile,pppppTile)==1&&ppppppar!=null&&AIMap.DistanceManhattan(pppppTile,ppppppTile)==1&&AIMap.DistanceManhattan(prevTile,tile)==1)&&((AIRoad.IsRoadTile(pppppTile)||(AITile.IsBuildable(pppppTile)||(AITown.GetDistanceManhattanToTile(AITile.GetClosestTown(pppppTile),pppppTile)*80<max(1500,AITown.GetPopulation(AITile.GetClosestTown(pppppTile)))))))){
if (!AIRoad.IsRoadTile(tile)&&AITile.GetOwner(tile)!=AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)&&!AITile.IsWaterTile(tile))AITile.DemolishTile(tile);
if (!AIRoad.IsRoadTile(pppppTile)&&AITile.GetOwner(pppppTile)!=AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)&&!AITile.IsWaterTile(pppppTile))AITile.DemolishTile(pppppTile);
if (!AIRoad.IsRoadTile(ppppppTile)&&AITile.GetOwner(ppppppTile)!=AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)&&!AITile.IsWaterTile(ppppppTile))AITile.DemolishTile(ppppppTile);
success = AIRoad.AreRoadTilesConnected(tile,pppppTile)||(AIRoad.CanBuildConnectedRoadPartsHere(prevTile,tile,pppppTile)>0&&AIRoad.CanBuildConnectedRoadPartsHere(tile,pppppTile,ppppppTile)>0&&this.RoadManager.SafelyBuildRoad(tile, pppppTile,30));
if (success){
ptile=pppppTile;
par=pppppar;
}}else if ((AIMap.DistanceManhattan(tile,ppppTile)==1&&ppppar!=null&&pppppar!=null&&AIMap.DistanceManhattan(ppppTile,pppppTile)==1&&AIMap.DistanceManhattan(prevTile,tile)==1)&&((AIRoad.IsRoadTile(ppppTile)||(AITile.IsBuildable(ppppTile)||(AITown.GetDistanceManhattanToTile(AITile.GetClosestTown(ppppTile),ppppTile)*80<max(1500,AITown.GetPopulation(AITile.GetClosestTown(ppppTile)))))))){
if (!AIRoad.IsRoadTile(tile)&&AITile.GetOwner(tile)!=AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)&&!AITile.IsWaterTile(tile))AITile.DemolishTile(tile);
if (!AIRoad.IsRoadTile(ppppTile)&&AITile.GetOwner(ppppTile)!=AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)&&!AITile.IsWaterTile(ppppTile))AITile.DemolishTile(ppppTile);
if (!AIRoad.IsRoadTile(pppppTile)&&AITile.GetOwner(pppppTile)!=AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)&&!AITile.IsWaterTile(pppppTile))AITile.DemolishTile(pppppTile);
success = AIRoad.AreRoadTilesConnected(tile,ppppTile)||(AIRoad.CanBuildConnectedRoadPartsHere(prevTile,tile,ppppTile)>0&&AIRoad.CanBuildConnectedRoadPartsHere(tile,ppppTile,pppppTile)>0&&this.RoadManager.SafelyBuildRoad(tile, ppppTile,30));
if (success){
ptile=ppppTile;
par=ppppar;
}}else if ((AIMap.DistanceManhattan(pppTile,ppppTile)==1&&AIMap.DistanceManhattan(pppTile,tile)==1&&AIMap.DistanceManhattan(prevTile,tile)==1)&&((AIRoad.IsRoadTile(pppTile)||(AITile.IsBuildable(pppTile)||(AITown.GetDistanceManhattanToTile(AITile.GetClosestTown(pppTile),pppTile)*80<max(1500,AITown.GetPopulation(AITile.GetClosestTown(pppTile)))))))){
if (!AIRoad.IsRoadTile(tile)&&AITile.GetOwner(tile)!=AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)&&!AITile.IsWaterTile(tile))AITile.DemolishTile(tile);
if (!AIRoad.IsRoadTile(pppTile)&&AITile.GetOwner(pppTile)!=AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)&&!AITile.IsWaterTile(pppTile))AITile.DemolishTile(pppTile);
if (!AIRoad.IsRoadTile(ppppTile)&&AITile.GetOwner(ppppTile)!=AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)&&!AITile.IsWaterTile(ppppTile))AITile.DemolishTile(ppppTile);
success = AIRoad.AreRoadTilesConnected(tile, pppTile)||(AIRoad.CanBuildConnectedRoadPartsHere(prevTile,tile,pppTile)>0&&AIRoad.CanBuildConnectedRoadPartsHere(tile,pppTile,ppppTile)>0&&this.RoadManager.SafelyBuildRoad(tile, pppTile,30));
if (success){
ptile=pppTile;
par=pppar;
}else if ((AIMap.DistanceManhattan(ppppTile,pppppTile)==1&&tile+tile-prevTile=ppppTile+ppppTile-pppppTile&&AIMap.DistanceManhattan(prevTile,tile)==1)&&((AIRoad.IsRoadTile(ppppTile)||(AITile.IsBuildable(ppppTile)||(AITown.GetDistanceManhattanToTile(AITile.GetClosestTown(ppppTile),ppppTile)*80<max(1500,AITown.GetPopulation(AITile.GetClosestTown(ppppTile)))))))){
local abk=tile+tile-prevTile;
if (AIRoad.IsRoadTile(abk)||(AITile.IsBuildable(abk)||(AITown.GetDistanceManhattanToTile(AITile.GetClosestTown(abk),abk)*80<max(1500,AITown.GetPopulation(AITile.GetClosestTown(abk)))))){
if (!AIRoad.IsRoadTile(abk)&&AITile.GetOwner(abk)!=AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)&&!AITile.IsWaterTile(abk))AITile.DemolishTile(abk);
if (!AIRoad.IsRoadTile(tile)&&AITile.GetOwner(tile)!=AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)&&!AITile.IsWaterTile(tile))AITile.DemolishTile(tile);
if (!AIRoad.IsRoadTile(ppppTile)&&AITile.GetOwner(ppppTile)!=AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)&&!AITile.IsWaterTile(ppppTile))AITile.DemolishTile(ppppTile);
if (!AIRoad.IsRoadTile(pppppTile)&&AITile.GetOwner(pppppTile)!=AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)&&!AITile.IsWaterTile(pppppTile))AITile.DemolishTile(pppppTile);
success = AIRoad.AreRoadTilesConnected(tile, ppppTile)||(AIRoad.CanBuildConnectedRoadPartsHere(prevTile,tile,ppppTile)>0&&AIRoad.CanBuildConnectedRoadPartsHere(abk,ppppTile,pppppTile)>0&&this.RoadManager.SafelyBuildRoad(tile, abk,ppppTile,30));
local remove=((!AIRoad.IsRoadTile(abk))&&AITile.GetOwner(abk)!=AICompany.ResolveCompanyID(AICompany.COMPANY_SELF));
success = AIRoad.AreRoadTilesConnected(tile, abk)||this.RoadManager.SafelyBuildRoad(tile, abk,30);
if (success){
success = AIRoad.AreRoadTilesConnected(abk,ppppTile)||this.RoadManager.SafelyBuildRoad(abk,ppppTile,30);
if (success){
success = AIRoad.AreRoadTilesConnected(ppppTile, pppppTile)||this.RoadManager.SafelyBuildRoad(ppppTile, pppppTile,30);
if (success){
tile=ppppTile;
ptile=pppppTile;
par=pppppar;
}else{
if(remove){
AIRoad.RemoveRoad(ppppTile,abk);
AIRoad.RemoveRoad(abk,tile);
AITile.DemolishTile(abk);
}}}else{
if(remove){
AIRoad.RemoveRoad(abk,tile);
AITile.DemolishTile(abk);
}}
}}
}}else{
if (AIMap.DistanceManhattan(tile, prevTile) == 1&&AIMap.DistanceManhattan(tile, ptile) == 1 &&AIMap.DistanceManhattan(ptile, ppTile) == 1&&AIMap.DistanceManhattan(ppTile, pppTile) == 1){
local abk=(((tile - prevTile) != (ptile - tile))?tile+tile-prevTile:((ppTile - pppTile) != (ptile - ppTile))?ppTile+ppTile-pppTile:-1);
if (AIMap.IsValidTile(abk)&&(AITile.IsBuildable(abk)||AIRoad.IsRoadTile(abk))&&(AITile.IsBuildable(tile)||AIRoad.IsRoadTile(tile))&&(AITile.IsBuildable(ppTile)||AIRoad.IsRoadTile(ppTile))&&((tile+tile-prevTile==ppTile+ppTile-pppTile&&(AIRoad.HasRoadType(abk,roadType)||!AIRoad.HasRoadType(ptile,roadType)))||(AIRoad.HasRoadType(abk,roadType)&&!AIRoad.HasRoadType(ptile,roadType)))){
{{{{
local remove=((!AIRoad.IsRoadTile(abk))&&AITile.GetOwner(abk)!=AICompany.ResolveCompanyID(AICompany.COMPANY_SELF));
success = AIRoad.AreRoadTilesConnected(tile, abk)||this.RoadManager.SafelyBuildRoad(tile, abk,30);
if (success){
success = AIRoad.AreRoadTilesConnected(abk,ppTile)||this.RoadManager.SafelyBuildRoad(abk,ppTile,30);
if (success){
success = AIRoad.AreRoadTilesConnected(ppTile, pppTile)||this.RoadManager.SafelyBuildRoad(ppTile, pppTile,30);
if (success){
tile=ppTile;
ptile=pppTile;
par=pppar;
}else{
if(remove){
AIRoad.RemoveRoad(ppTile,abk);
AIRoad.RemoveRoad(abk,tile);
AITile.DemolishTile(abk);
}}}else{
if(remove){
AIRoad.RemoveRoad(abk,tile);
AITile.DemolishTile(abk);
}}}}}}}
}else if (AIMap.DistanceManhattan(tile, prevTile) == 1&&AIMap.DistanceManhattan(tile, ptile) == 1 &&AIMap.DistanceManhattan(ptile, ppTile) == 1&&AIMap.DistanceManhattan(ppTile, pppTile) == 1&&AIMap.DistanceManhattan(pppTile, ppppTile) == 1){
local abk1=(((tile - prevTile) != (ptile - tile))?tile+tile-prevTile:-1);
local abk2=(((pppTile - ppppTile) != (ppTile - pppTile))?pppTile+pppTile-ppppTile:-1);
if ((AIMap.IsValidTile(abk2)&&(AITile.IsBuildable(abk2)||AIRoad.IsRoadTile(abk2))&&AIMap.IsValidTile(abk1)&&(AITile.IsBuildable(abk1)||AIRoad.IsRoadTile(abk1))&&(AITile.IsBuildable(tile)||AIRoad.IsRoadTile(tile))&&(AITile.IsBuildable(pppTile)||AIRoad.IsRoadTile(pppTile)))&&((AIMap.DistanceManhattan(tile+tile-prevTile,pppTile+pppTile-ppppTile)==1&&((AIRoad.HasRoadType(abk1,roadType)&&AIRoad.HasRoadType(abk2,roadType))||(!AIRoad.HasRoadType(ptile,roadType)&&!AIRoad.HasRoadType(ppTile,roadType))))||(AIRoad.HasRoadType(abk1,roadType)&&(AIRoad.HasRoadType(abk2,roadType))&&(!AIRoad.HasRoadType(ppTile,roadType)&&!AIRoad.HasRoadType(ptile,roadType))))){
{{{{
local remove1=((!AIRoad.IsRoadTile(abk1))&&AITile.GetOwner(abk1)!=AICompany.ResolveCompanyID(AICompany.COMPANY_SELF));
local remove2=((!AIRoad.IsRoadTile(abk2))&&AITile.GetOwner(abk2)!=AICompany.ResolveCompanyID(AICompany.COMPANY_SELF));
success = AIRoad.AreRoadTilesConnected(tile, abk1)||this.RoadManager.SafelyBuildRoad(tile, abk1,30);
if (success){
success = AIRoad.AreRoadTilesConnected(abk1,abk2)||this.RoadManager.SafelyBuildRoad(abk1,abk2,30);
if (success){
success = AIRoad.AreRoadTilesConnected(abk2,pppTile)||this.RoadManager.SafelyBuildRoad(abk2,pppTile,30);
if (success){
success = AIRoad.AreRoadTilesConnected(pppTile, ppppTile)||this.RoadManager.SafelyBuildRoad(pppTile, ppppTile,30);
if (success){
tile=pppTile;
ptile=ppppTile;
par=ppppar;
}else{
if(remove2){
AIRoad.RemoveRoad(pppTile,abk2);
AIRoad.RemoveRoad(abk2,abk1);
AITile.DemolishTile(abk2);
}
if (remove1){
AIRoad.RemoveRoad(abk1,tile);
AITile.DemolishTile(abk1);
}}}else{
if(remove2){
AIRoad.RemoveRoad(abk2,abk1);
AITile.DemolishTile(abk2);
}
if (remove1){
AIRoad.RemoveRoad(abk1,tile);
AITile.DemolishTile(abk1);
}}}else{
if(remove1){
AIRoad.RemoveRoad(abk1,tile);
AITile.DemolishTile(abk);
}}}}}}}}

}}
}}}}}
success=(success||(AIRoad.IsRoadTile(ptile)&&AIRoad.IsRoadTile(tile)&&AIRoad.AreRoadTilesConnected(tile,ptile)&&AIRoad.HasRoadType(tile,roadType)&&AIRoad.HasRoadType(ptile,roadType)));
if (!success){
local ppar=par.GetParent();
local ppTile=((ppar!=null)?ppar.GetTile():ptile+ptile-tile);
local pppar=((ppar!=null)?ppar.GetParent():ppar);
local pppTile=((pppar!=null)?pppar.GetTile():ppTile+ppTile-ptile);
local ppppar=((pppar!=null)?pppar.GetParent():pppar);
local pppppar=((ppppar!=null)?ppppar.GetParent():ppppar);
if (pppppar!=null&&AIMap.DistanceManhattan(tile, prevTile) == 1&&AIMap.DistanceManhattan(tile, ptile) == 1 &&AIMap.DistanceManhattan(ptile, ppTile) == 1&&AIMap.DistanceManhattan(ppTile, pppTile) == 1&&AITile.GetMaxHeight(tile-prevTile)==AITile.GetMaxHeight(tile)&&((tile - prevTile) != (ptile - tile))&&((ppTile - pppTile) != (ptile - ppTile))&&(ppTile-pppTile==tile-prevTile)&&AIMap.DistanceManhattan(ptile, prevTile) == 2&&AIMap.DistanceManhattan(ptile, pppTile) == 2){
local abbb=tile+tile-prevTile;
if (AIMap.IsValidTile(abbb)&&(AITile.IsBuildable(abbb)||(AIRoad.IsRoadTile(abbb)&&AIRoad.HasRoadType(abbb,roadType)))){
local remove1=(!AIRoad.IsRoadTile(abbb)&&AITile.GetOwner(abbb)!=AICompany.ResolveCompanyID(AICompany.COMPANY_SELF));
local remove2=(!AIRoad.IsRoadTile(pptile)&&AITile.GetOwner(pptile)!=AICompany.ResolveCompanyID(AICompany.COMPANY_SELF));
success = AIRoad.AreRoadTilesConnected(tile, abbb)||this.RoadManager.SafelyBuildRoad(tile, abbb,30);
if (success){
success = AIRoad.AreRoadTilesConnected(abbb,ppTile)||this.RoadManager.SafelyBuildRoad(abbb,ppTile,30);
if (success){
success = AIRoad.AreRoadTilesConnected(ppTile, pppTile)||this.RoadManager.SafelyBuildRoad(ppTile, pppTile,30);
if (success){
tile=ppTile;
ptile=pppTile;
par=pppar;
}else{
if(remove2){
AIRoad.RemoveRoad(pppTile,pptile);
AIRoad.RemoveRoad(ppTile,abbb);
AITile.DemolishTile(pptile);
}
if(remove1){
AIRoad.RemoveRoad(ppTile,abbb);
AIRoad.RemoveRoad(abbb,tile);
AITile.DemolishTile(abbb);
}
}
}}
}}
if (!success){
success = AIRoad.BuildRoad(tile, ptile);
}
}}
				} else {
					// Build a bridge or tunnel.
					if(!AIBridge.IsBridgeTile(tile) && !AITunnel.IsTunnelTile(tile)&&!AIRoad.IsDriveThroughRoadStationTile(tile)&&!AIRoad.IsRoadStationTile(tile)&&!AIRoad.IsRoadDepotTile(tile)) {
						// If it was a road tile, demolish it first. Do this to work around expended roadbits.
						if(AIRoad.IsRoadTile(tile)) AITile.DemolishTile(tile);
						
						if(AITunnel.GetOtherTunnelEnd(tile) == ptile) {
							success = AITunnel.BuildTunnel(AIVehicle.VT_ROAD, tile);
						} else {
							if (AIMap.DistanceManhattan(tile, ptile)==2){
								local mtile=tile+(ptile-tile)/2;
								if (AITile.IsBuildable(mtile)&&!AIRail.IsRailTile(mtile)){
								success=AIRoad.BuildRoad(tile, mtile);
								if (success){
									success=AIRoad.BuildRoad(mtile, ptile);
									if (!success)AIRoad.RemoveRoad(mtile, tile);
									success=AIRoad.BuildRoad(ptile, pptile);
									if (!success){
									AIRoad.RemoveRoad(ptile, mtile);
									AIRoad.RemoveRoad(mtile, tile);
									}
								}}
							if (!success){
							local bridgeType = LandManager.ChooseBridgeType(tile, ptile);
							success = AIBridge.BuildBridge(AIVehicle.VT_ROAD, bridgeType, tile, ptile);
							}	
							}else{
							local bridgeType = LandManager.ChooseBridgeType(tile, ptile);
							success = AIBridge.BuildBridge(AIVehicle.VT_ROAD, bridgeType, tile, ptile);
						}}
					} else if (AIBridge.IsBridgeTile(tile)){
							if (AIBridge.GetOtherBridgeEnd(tile)==ptile) success = true;
					} else if (AITunnel.IsTunnelTile(tile)){
							if (AITunnel.GetOtherTunnelEnd(tile)==ptile) success = true;
					}		
				}
				
				// If something went wrong, try to fix it
				if(!success) {
//AILog.Info(AIMap.GetTileX(tile)+"/"+AIMap.GetTileY(tile)+"-"+AIMap.GetTileX(ptile)+"/"+AIMap.GetTileY(ptile));

					switch(AIError.GetLastError()) {
						case AIError.ERR_AREA_NOT_CLEAR:
							// Something must have been built since we check the tile. Clear it.
if (!AITile.IsWaterTile(tile)&&!AIBridge.IsBridgeTile(tile)&&!AITunnel.IsTunnelTile(tile)&&!AIRoad.IsRoadStationTile(tile)&&!AIRoad.IsRoadDepotTile(tile)&&!AIRoad.IsDriveThroughRoadStationTile(tile)){
							if(!AITile.DemolishTile(tile)) {
								if(AIError.GetLastError() == AIError.ERR_LOCAL_AUTHORITY_REFUSES) {
									// Try to influence the local authority 
									TownManager.HandleRating(TownManager.FindNearestTown(tile));
								} else {
									// Otherwise just give up
									attempts = PathZilla.MAX_CONSTR_ATTEMPTS + 1;
}							
	}
							}
						break;
						case AIError.ERR_NOT_ENOUGH_CASH:
							if(!FinanceManager.CanAfford(PathZilla.FLOAT)) {
								// We cant afford to borrow any more money, so give up!
(::german)?AILog.Error("      Kein Geld mehr fuer diesen Strassenabschnitt!"):AILog.Error("      Cannot afford path segment!");
								SaveMoney();
								PathZilla.Sleep(80);
								::pz.HandleEvents();
							} else {
								// Otherwise, borrow some more money
								FinanceManager.Borrow();
							}
						break;
						case AIError.ERR_VEHICLE_IN_THE_WAY:
							// Theres a vehicle in the way...
							if(attempts == 2) {
								// If we've already tried once, try to clear 
								// any of our own vehicles out of the way

								// First, find those vehicles that are blocking
								// the tile to be built on
								local blockers = AIVehicleList();
//								blockers.Valuate(AIVehicle.GetVehicleType)
//								blockers.KeepValue(AIVehicle.VT_ROAD);
								blockers.Valuate(function (v, ptile) {
									return AITile.GetDistanceManhattanToTile(ptile, AIVehicle.GetLocation(v));
								}, ptile);
								blockers.KeepBelowValue(2);

								// Then find those vehicles that lie just outside
								// the tile to be built on
								local outliers = AIVehicleList();
//								outliers.Valuate(AIVehicle.GetVehicleType)
//								outliers.KeepValue(AIVehicle.VT_ROAD);
								outliers.Valuate(function (v, ptile) {
									return AITile.GetDistanceManhattanToTile(ptile, AIVehicle.GetLocation(v));
								}, ptile);
								outliers.KeepBelowValue(3);
								
								// Stop the outliers from moving into the tile
								foreach(v, _ in outliers) {
									if(AIVehicle.GetState(v) != AIVehicle.VS_STOPPED&&AITile.GetDistanceManhattanToTile(ptile, AIVehicle.GetLocation(v))==2) AIVehicle.StartStopVehicle(v);
									stopList.AddItem(v, 0);
								}

								// Move the blockers out of the way
								foreach(v, _ in blockers) AIVehicle.ReverseVehicle(v);
							} else if(attempts == PathZilla.MAX_CONSTR_ATTEMPTS) {
								// If we STILL can't build due to traffic, remember the spot
								if (ptile!=null) ::trafficBlackSpots.AddItem(ptile, 0);
							}

							// Just try waiting a bit
							PathZilla.Sleep(80);
						break;
						// Just don't worry about the rest of these cases!
						case AIError.ERR_ALREADY_BUILT:
							success = true;
						break;
						case AIError.ERR_UNKNOWN:
							success = false;
						break;
					}
if (distance==1&&AITunnel.GetOtherTunnelEnd(tile)==ptile&&!success){
success = AITunnel.BuildTunnel(AIVehicle.VT_ROAD, tile)
if (!success) success = this.RoadManager.SafelyBuildRoad(tile,ptile,30);
}
				}
			}

			// Check that we DID succeed
			if(!success) {
				// Restart any stopped vehicles
				foreach(v, _ in stopList) {
					if(AIVehicle.GetState(v) == AIVehicle.VS_STOPPED) AIVehicle.StartStopVehicle(v);
				}

(::german)?AILog.Error("    Strassenbau abgebrochen!"):AILog.Error("    Could not complete road!")
				return (prevTile != null) ? prevTile : tile;
			}
		}
		
		prevTile = tile;
		path = par;
	}

	// Restart any stopped vehicles
	foreach(v, _ in stopList) { 
		if(AIVehicle.GetState(v) == AIVehicle.VS_STOPPED) AIVehicle.StartStopVehicle(v);
	}

(::german)?AILog.Info("    Strasse fertiggestellt..."):AILog.Info("    Done building road.")

	return 0;
}

function PathWrapper::GetFirstTile(path) {
	local cpath = clone path;
	local tile = null;
	while (cpath != null) {
		if(cpath.GetParent() != null) tile = cpath.GetTile(); 
		cpath = cpath.GetParent();
	}
	return tile;
}
