require("money.nut");
require("util.nut");
require("air.nut");
require("rail.nut");
class SnakeAI extends AIController 
{
	maxVehicles=2500;
    ut = Util();
	air = AirManager();
	rail = RailManager();
	loaded = false;
	function CanBuildMoreRoadVehicles();
    function SetCompanyName();
    function AttemptToConnectIndustry();
    function BuildRoadStop(location, type, station);
    function BuildRoadDepotNear(location);
	function TweakVehicleNumber();
	function UpgradeVehicles();
	function DeleteVehicle(vehicle);
	function FindPosNear();
	function DeleteEmptyStations();
	function ConnectIndustryCargoTown(industry, cargo);
	function ConnectIndToInd(in1, ind2, cargo);
	function ExpandOversuppliedStations();
	function ShrinkUndersuppliedStations();
    function Start();
}

function SnakeAI::CanBuildMoreRoadVehicles(margin){
	local vehicles = AIVehicleList();
	vehicles.Valuate(AIVehicle.GetVehicleType);
	vehicles.KeepValue(AIVehicle.VT_ROAD);
	local number = vehicles.Count();
	//AILog.Info("#"+number+"/"+maxVehicles);
	return vehicles.Count() + margin < maxVehicles;
}

function SnakeAI::SetCompanyName()
{
	local names = ["Cobra", "Python", "Viper", "mamba", "moccasin", "coral"]
	AICompany.SetPresidentName(names[0]);
    if (!AICompany.SetName("Snake")) {
		AICompany.SetPresidentName(names[1]);
        local i = 2;
        while (!AICompany.SetName("Snake #" + i)) {
			AICompany.SetPresidentName(names[i % names.length]);
            i = i + 1;
        }
    }
}

function SnakeAI::ConnectToNearestRoad(pre, location, maxIterations){
	foreach(sign, value in AISignList()){
		if (AITile.GetDistanceManhattanToTile(AISign.GetLocation(sign), location)<30){
			AISign.RemoveSign(sign);
		}
	}


	local roads=[ ];
	local oldnewroads = [[pre, location]]
	if (!ut.IsClearableTile(location) && !AIRoad.IsRoadTile(location)){
		return;
	}
	if (AIBridge.IsBridgeTile(location) || AITunnel.IsTunnelTile(location)){
		return;
	}
	
	local done=false;
	for (local i=0; i<maxIterations && !done; i++){
		local newroads = [];
		foreach (road in oldnewroads){
			foreach (tile in Util.GetDirectlyAdjacentTiles(road[1])){
				local found=false;
				foreach(road2 in roads){
					if (road2[1]==tile){
						found=true;
						break;
					}
				}
				if (found){
				
				}
				else if (tile!=location && !AIRoad.IsRoadStationTile(tile) && AIRoad.IsRoadTile(tile) &&
					    AIRoad.CanBuildConnectedRoadPartsHere(road[1], road[0], tile)){
					//AISign.BuildSign(tile, "final");
					newroads.push([road[1],tile]);
					done=true;
					break;
				}
				else if (ut.IsClearableTile(tile) && AIRoad.CanBuildConnectedRoadPartsHere(road[1], road[0], tile)){
					//AISign.BuildSign(tile, "not final");
					newroads.push([road[1],tile]);
				}
				else {
				}
			}
			
			if (done) {
				break;
			}
		}
		foreach (road in oldnewroads){
			roads.push(road);
		}
		oldnewroads=[];
		foreach (road in newroads){
			oldnewroads.push(road);
		}
	}
	
	if (done) {
		local lastRoad = oldnewroads[oldnewroads.len() - 1];
		ut.TryBuildRoad(lastRoad[0], lastRoad[1]);
		while (lastRoad[0] != roads[0][1]){
			local firstTile = null;
			foreach (road in roads){
				if (road[1]==lastRoad[0]){
					ut.TryBuildRoad(road[0], road[1]);
					lastRoad = road;
					break;
				}
			}
		}
		return true;
	}
	else {
		return false;
	}
}

function SnakeAI::TryCloneRoadVehicle(vehicle, depot){
	local vehicle2 = AIVehicle.CloneVehicle(depot, vehicle, true);
	return HandleBuildVehicleError(vehicle2);
}

function SnakeAI::TryBuildRoadVehicle(engine, depot){
	local vehicle2 = AIVehicle.BuildVehicle(depot, engine);
	return HandleBuildVehicleError(vehicle2);
}

function SnakeAI::HandleBuildVehicleError(vehicle2){
	if (vehicle2==null || !AIVehicle.IsValidVehicle(vehicle2)) {
		if (AIError.GetLastError() == AIVehicle.ERR_VEHICLE_TOO_MANY){
			local vehicles = AIVehicleList();
			vehicles.Valuate(AIVehicle.GetVehicleType);
			vehicles.KeepValue(AIVehicle.VT_ROAD);
			maxVehicles = vehicles.Count();
			AILog.Warning("too many vehicles, max: "+maxVehicles);
			return null;
		}
		else {
			AILog.Warning("Failed building vehicle: "+AIError.GetLastErrorString());
			return null;
		}
	}
	else {
		return vehicle2
	}
}

function SnakeAI::BuildRoadStop(location, type, station, name) {
	if (AIRoad.IsRoadStationTile(location) && AITile.GetOwner(location)==AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)){
		local adjacentTiles = Util.GetDirectlyAdjacentTiles(location);
		foreach (l2 in adjacentTiles){
			if (AIRoad.IsRoadStationTile(l2)){
				local l3 = l2;
				while (AIRoad.IsRoadStationTile(l3) && AITile.GetSlope(l3)==AITile.SLOPE_FLAT){
					local l4 = l3 + (l2 - location);
					if (AIRoad.IsRoadStationTile(l4)){
						//do nothing
					}
					else if (ut.IsClearableTile(l4)){
						local ftile = l4 + AIRoad.GetRoadStationFrontTile(location) - location;
						local btile = l4 + AIRoad.GetDriveThroughBackTile(location) - location;
						
						if ((AITile.GetSlope(l4)==AITile.SLOPE_FLAT || AITile.LevelTiles(l3, l4)) &&
							(AITile.GetSlope(ftile)==AITile.SLOPE_FLAT || AITile.LevelTiles(l4, ftile)) &&
							(AITile.GetSlope(btile)==AITile.SLOPE_FLAT || AITile.LevelTiles(l4, btile))){
							AIRoad.BuildDriveThroughRoadStation(l4, ftile,
								type, station);
						
							if (!ut.TryBuildRoad(l4, ftile)){
								AILog.Warning("Failed building road: "+AIError.GetLastErrorString());
							}
							if (!ut.TryBuildRoad(l4, btile)){
								AILog.Warning("Failed building road: "+AIError.GetLastErrorString());
							}
							
							ConnectToNearestRoad(l4, AIRoad.GetDriveThroughBackTile(l4), 9);
							ConnectToNearestRoad(l4, AIRoad.GetRoadStationFrontTile(l4), 9);
							return location;
						}
						else {
							AILog.Warning("Tile leveling failed: "+AIError.GetLastErrorString());
							AILog.Info("slope l4: "+(AITile.GetSlope(l4)==AITile.SLOPE_FLAT)+
							           " slope ftile: "+(AITile.GetSlope(ftile)==AITile.SLOPE_FLAT)+
									   " slope btile: "+(AITile.GetSlope(btile)==AITile.SLOPE_FLAT));
							break;
						}
					}
					else {
						break;
					}
					
					l3 = l4;
				}
			}
		}
		
		foreach (l2 in adjacentTiles){
			if (ut.IsClearableTile(l2) && AITile.GetSlope(l2)==AITile.SLOPE_FLAT){
				if (AIRoad.BuildDriveThroughRoadStation(l2, l2 + AIRoad.GetRoadStationFrontTile(location) - location
					, type, station)){
					local ftile = AIRoad.GetRoadStationFrontTile(l2);
					local btile = AIRoad.GetDriveThroughBackTile(l2);
					if (AITile.GetSlope(ftile)!=AITile.SLOPE_FLAT){
						if (!AITile.LevelTiles(l2, ftile)){
							AILog.Warning("leveling failed: "+AIError.GetLastErrorString());
						}
					}
					if (AITile.GetSlope(btile)!=AITile.SLOPE_FLAT){
						if (!AITile.LevelTiles(l2, btile)){
							AILog.Warning("leveling failed: "+AIError.GetLastErrorString());
						}
					}
					
					ut.TryBuildRoad(l2, ftile);
					ut.TryBuildRoad(l2, btile);
					
					ConnectToNearestRoad(l2, btile, 9);
					ConnectToNearestRoad(l2, ftile, 9);
					return location;
				}
				else {
					AILog.Warning("failed to build road station: "+AIError.GetLastErrorString());
				}
			}
		}
		foreach (location in adjacentTiles){
			if (Util.IsStraightRoadTile(location) && !AIRoad.IsRoadStationTile(location)) {
				local tries=0;
				while (tries<10){
					tries+=5;
					if (BuildRoadStop(location, type, station, name)){
						tries=200;
					}
					else if (AIError.GetLastError()==AIError.ERR_VEHICLE_IN_THE_WAY){
						tries-=4;
						AILog.Info("vehicle in the way, trying again...");
						this.Sleep(15);
					}
					else {
						AILog.Warning("failed to build station, "+AIError.GetLastErrorString());
					}
				};
				break;
			}
			
		}
		return location;
	}
	else {
		local adjacentTiles = Util.GetDirectlyAdjacentTiles(location);
		foreach (tile in adjacentTiles){
			if (AIRoad.AreRoadTilesConnected(location, tile)){
				local station = AIRoad.BuildDriveThroughRoadStation(location,tile,type,station);
				if (!station || !AIRoad.IsRoadStationTile(location) ||
					!AITile.GetOwner(location)==AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)){
					AILog.Warning("failed to build station: "+AIError.GetLastErrorString());
					return null;
				}
				if (name!="" && name!=null){
					local i=0;
					while (!AIBaseStation.SetName(AIStation.GetStationID(location), Util.CapLength(name,15)+" "+i)){
						i++;
					}
				}
				local backTile = AIRoad.GetDriveThroughBackTile(location);
				if (!ut.IsClearableTile(backTile)){
					backTile = AIRoad.GetRoadStationFrontTile(location);
				}
				if (ut.IsClearableTile(backTile) && AITile.GetSlope(backTile) == AITile.SLOPE_FLAT){
					
					ut.TryBuildRoad(location, backTile);
					ConnectToNearestRoad(location, backTile, 8);
				}
				else {
				}
				return location;
			}
		}
		return null;
	}
}

function SnakeAI::BuildRoadDepotNear(location) {
    local finishedTiles = [];
    local deadEnds = [location];
    local newDeadEnds = []
	local numIterations = 0;
    while (deadEnds.len() != 0 && numIterations < 50) {
		numIterations ++;
		if (numIterations == 10){
			AILog.Info("hmm, finding a depo seems to be taking longer than expected");
		}
		if (numIterations % 10 == 0){
			this.Sleep(1);
		}
		
        foreach (deadEnd in deadEnds){
            local surround = Util.GetDirectlyAdjacentTiles(deadEnd)
            foreach(tile in surround){
				if (AIRoad.IsRoadDepotTile(tile)
					&& AITile.GetOwner(tile) == AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)){
					return tile;
				}
				else if (AIRoad.IsRoadDepotTile(tile)) {
					AILog.Info("tile is a depo, but owned by "+AITile.GetOwner(tile)+
					" (AM "+AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)+")");
				}
				else if (AIRoad.AreRoadTilesConnected(deadEnd, tile) && AIBridge.IsBridgeTile(tile)) {
					local edge = AIBridge.GetOtherBridgeEnd(tile);
					if (ut.ArrayFind(finishedTiles,edge) || ut.ArrayFind(deadEnds,edge)){
                        //AISign.BuildSign(edge, "unnew dead end (bridge)");
                    }
                    else {
                        newDeadEnds.push(edge);
                        //AISign.BuildSign(edge, "new dead end (bridge)");
                    
                    }
				}
				else if (AIRoad.AreRoadTilesConnected(deadEnd, tile) && AITunnel.IsTunnelTile(tile)){
					local edge = AITunnel.GetOtherTunnelEnd(tile);
					if (ut.ArrayFind(finishedTiles,edge) || ut.ArrayFind(deadEnds,edge)){
                        //AISign.BuildSign(edge, "unnew dead end (tunnel)");
                    }
                    else {
                        newDeadEnds.push(edge);
                        //AISign.BuildSign(edge, "new dead end (tunnel)");
                    
                    }
				}
                else if (AIRoad.AreRoadTilesConnected(deadEnd, tile)){
                    
                    if (ut.ArrayFind(finishedTiles,tile) || ut.ArrayFind(deadEnds,tile)){
                        //AISign.BuildSign(tile, "unnew dead end");
                    }
                    else {
                        newDeadEnds.push(tile);
                        //AISign.BuildSign(tile, "new dead end ("+numIterations+")");
                    
                    }
                }
                else if (
						numIterations >= 5 &&
                         ut.IsClearableTile(tile) &&
                        !AIRoad.IsRoadStationTile(deadEnd) &&
						!AIRoad.IsRoadDepotTile(tile) &&
						!AIBridge.IsBridgeTile(deadEnd) &&
                         AITile.GetSlope(deadEnd)==AITile.SLOPE_FLAT &&
                         AITile.GetSlope(tile)==AITile.SLOPE_FLAT)
                {
					
                    if (AIRoad.BuildRoadDepot(tile, deadEnd)){
						ut.TryBuildRoad(deadEnd, tile);
						return tile;
					}
					else {
						AILog.Warning("failed building depo: "+AIError.GetLastErrorString());
					}
                }
                else {
                    if (numIterations >= 5){
						if (!ut.IsClearableTile(tile)){
							//AISign.BuildSign(tile, "not clearable ("+numIterations+")");
						}
						else if (AIRoad.IsRoadStationTile(deadEnd)){
							//AISign.BuildSign(tile, "no junction possible ("+numIterations+")");
						}
						else if (AITile.GetSlope(tile)!=AITile.SLOPE_FLAT ||
							AITile.GetSlope(deadEnd)!=AITile.SLOPE_FLAT){
							//AISign.BuildSign(tile, "not flat ("+numIterations+")");
						}
						else {
							AISign.BuildSign(tile, "invalid ("+numIterations+")");
						}
					}
                }
            }
        }
        foreach(deadEnd in deadEnds){
            //AISign.BuildSign(deadEnd, "dead end");
            finishedTiles.push(deadEnd);
        }
        deadEnds=newDeadEnds;
        newDeadEnds=[];
    }
	
	AILog.Info("Depo build failed, "+finishedTiles.len()+" tiles tried");
	return 0;
    
}

function SnakeAI::IsIndustryServicedByMe(industry){
	local radius = max(AIStation.GetCoverageRadius(AIStation.STATION_BUS_STOP),
	                   AIStation.GetCoverageRadius(AIStation.STATION_TRUCK_STOP));
	local cargos = AICargoList_IndustryAccepting(industry);
	cargos.AddList(AICargoList_IndustryProducing(industry));
	
	local tiles = AITileList_IndustryAccepting(industry, radius);
	tiles.AddList(AITileList_IndustryProducing(industry, radius));
	tiles.Valuate(AIRoad.IsRoadStationTile);
	tiles.KeepValue(1);
	tiles.Valuate(AITile.GetOwner);
	tiles.KeepValue(AICompany.ResolveCompanyID(AICompany.COMPANY_SELF));
	
	if (tiles.IsEmpty() || cargos.IsEmpty()){
		return false;
	}
	
	foreach (tile, _ in tiles){
		foreach (cargo, _ in cargos){
			local vehicles = AIVehicleList_Station(AIStation.GetStationID(tile));
			vehicles.Valuate(AIVehicle.GetCapacity, cargo);
			vehicles.KeepAboveValue(5);
			if (!vehicles.IsEmpty()){
				//AISign.BuildSign(AIIndustry.GetLocation(industry), "true");
				return true;
			}
		}
	}
	
	return false;
}

function SnakeAI::ConnectIndustryCargo(industry1, cargo) {
    //local sign = 0;
    local industryList = AIIndustryList();
    industryList.Valuate(AIIndustry.IsCargoAccepted, cargo);
    industryList.KeepValue(AIIndustry.CAS_ACCEPTED);
    local in1loc = AIIndustry.GetLocation(industry1);
	industryList.RemoveItem(industry1);
	industryList.Valuate(AIIndustry.GetDistanceManhattanToTile, in1loc);
    industryList.KeepBelowValue(100);
	industryList.KeepAboveValue(20);
	industryList.Valuate(IsIndustryServicedByMe);
	industryList.Sort(AIList.SORT_BY_VALUE, false);
	if (IsIndustryServicedByMe(industryList.Begin())){
		
	}
	else {
		industryList.Valuate(AIIndustry.GetDistanceManhattanToTile, in1loc);
		industryList.Sort(AIList.SORT_BY_VALUE, true);
    }
	
    if (industryList.IsEmpty()){
        //AILog.Info("Industry failed: "+AIIndustry.GetName(industry1) + "(" + AICargo.GetCargoLabel(cargo)+") No nearby industry accepting");
        return false;
    }
	local stopType = AICargo.HasCargoClass(cargo, AICargo.CC_PASSENGERS)?AIStation.STATION_BUS_STOP:AIStation.STATION_TRUCK_STOP;
	local industry2 = industryList.Begin();
	local in2loc = FindPosNear(AITileList_IndustryAccepting(industry2, AIStation.GetCoverageRadius(stopType)),
							    in1loc,
								stopType)
								
	if (in2loc[0] != null && AIMap.IsValidTile(in2loc[0]) && CanBuildMoreRoadVehicles(5)){
		return ConnectIndToInd(industry1, in2loc, AIIndustry.GetName(industry2), cargo);
	}
	else {
		return false;
	}
}

function SnakeAI::ConnectIndustryCargoTown(industry1, cargo) {
	if (AICargo.GetTownEffect(cargo) == AICargo.TE_NONE){
		return false;
	}
	local townList = AITownList();
	local in1loc = AIIndustry.GetLocation(industry1);
	townList.Valuate(AITown.GetDistanceManhattanToTile, in1loc);
	townList.KeepBelowValue(100);
	townList.KeepAboveValue(20);
	townList.Sort(AIList.SORT_BY_VALUE, true);
	
	if (townList.IsEmpty()){
		return false;
	}
	
	for (local town = townList.Begin(); !townList.IsEnd(); town = townList.Next()){
	
		local stopType = AICargo.HasCargoClass(cargo, AICargo.CC_PASSENGERS)?AIStation.STATION_BUS_STOP:AIStation.STATION_TRUCK_STOP;
		local tileList = Util.GetTileListRadius(AITown.GetLocation(town), 20)
		tileList.Valuate(AITile.GetCargoAcceptance, cargo, 1, 1, AIStation.GetCoverageRadius(stopType));
		tileList.KeepAboveValue(16);
		if (tileList.IsEmpty()){
			continue;
		}
		local in2loc = FindPosNear(tileList, in1loc, stopType);
		if (in2loc[0] != 0 && AIMap.IsValidTile(in2loc[0]) && CanBuildMoreRoadVehicles(5)){
			return ConnectIndToInd(industry1, in2loc, AITown.GetName(town), cargo)
		}
		else {
			continue;
		}
	}
	return false;
}

function SnakeAI::FindPosNear(ts, near, stopType) {
	local tileset = AITileList();
	tileset.AddList(ts);
    tileset = ut.FilterBuildableTiles(tileset, true);
	tileset.Valuate(AITile.GetOwner);
	for (local i=AICompany.COMPANY_FIRST; i<AICompany.COMPANY_LAST; i++){
		if (!AICompany.IsMine(i)){
			tileset.RemoveValue(i);
		}
	}
    tileset.Valuate(AITile.GetDistanceManhattanToTile, near);
    tileset.Sort(AIList.SORT_BY_VALUE, true);
	local idealLoc = tileset.Begin();
	
	tileset = AITileList();
	tileset.AddList(ts);
	tileset.Valuate(AITile.GetOwner);
	tileset.KeepValue(AICompany.ResolveCompanyID(AICompany.COMPANY_SELF));
	tileset.Valuate(AITile.GetDistanceManhattanToTile, idealLoc);
	tileset.KeepBelowValue(8);
	tileset.Valuate(AIRoad.IsRoadStationTile);
	tileset.KeepValue(1);
	if (tileset.IsEmpty()){
		return [idealLoc, AIStation.STATION_JOIN_ADJACENT];
	}
	tileset.Valuate(AIStation.GetStationID);
	foreach (tile, station in tileset){
		if (AIStation.HasStationType(station, stopType)){
			return [tile, station];
		}
	}
	return [idealLoc, AIStation.STATION_JOIN_ADJACENT];
}

function SnakeAI::ConnectIndToInd(industry1, in2loc, in2name, cargo) {
	local stopType = AICargo.HasCargoClass(cargo, AICargo.CC_PASSENGERS) ? AIStation.STATION_BUS_STOP : AIStation.STATION_TRUCK_STOP;
    local in1loc = AIIndustry.GetLocation(industry1);
    
    in1loc = FindPosNear(AITileList_IndustryProducing(industry1, AIStation.GetCoverageRadius(stopType)), in2loc[0], stopType);
	local in1station = in1loc[1];
	in1loc = in1loc[0];
	local in2station = in2loc[1];
	in2loc = in2loc[0];
    if (in1loc != null && AIMap.IsValidTile(in1loc)){
        //sign = AISign.BuildSign(in1loc, "loc1");
    }
    else {
        AILog.Info("Industry failed: "+AIIndustry.GetName(industry1)+", no build location");
        return false;
    }
    if (in2loc != null && AIMap.IsValidTile(in2loc)){
        //AISign.BuildSign(in1loc, "loc2");
    }
    else {
        //if (AISign.IsValidSign(sign)){
            //AILog.Info("no accepting location");
            //AISign.RemoveSign(sign);
        //}
        return false;
    }
    
    if (in1loc != null && AIMap.IsValidTile(in1loc) && in2loc != null && AIMap.IsValidTile(in2loc)){
    
		AILog.Info("from "+AIIndustry.GetName(industry1)+" to "+in2name);
        if (ut.BuildRoute(in1loc,in2loc)){
			
            local station1 = BuildRoadStop(in1loc,
				(stopType == AIStation.STATION_BUS_STOP ? AIRoad.ROADVEHTYPE_BUS : AIRoad.ROADVEHTYPE_TRUCK)
			    , in1station, AIIndustry.GetName(industry1));
			if (!AIRoad.IsRoadStationTile(in1loc)){
				return false;
			}
			local engine = ut.FindRoadVehicle(cargo);
			if (AIEngine.IsValidEngine(engine) && AIEngine.IsBuildable(engine)){
				local station2 = BuildRoadStop(in2loc,
							 (stopType == AIStation.STATION_BUS_STOP ? AIRoad.ROADVEHTYPE_BUS : AIRoad.ROADVEHTYPE_TRUCK)
							, in2station, in2name);
				if (!AIRoad.IsRoadStationTile(in2loc)){
					return false;
				}
				local depotTile = BuildRoadDepotNear(in1loc);
				if (depotTile != null && AIMap.IsValidTile(depotTile) && Money.ReserveMoney(AIEngine.GetPrice(engine))){
					local vehicle = TryBuildRoadVehicle(engine, depotTile);
					if (vehicle != null && AIVehicle.IsValidVehicle(vehicle)){
						AIVehicle.RefitVehicle(vehicle, cargo);
						local name = Util.CapLength(AICargo.GetCargoLabel(cargo), 4)+" "+
								     Util.CapLength(AITown.GetName(AITile.GetClosestTown(in1loc)),5)+" "+
									 Util.CapLength(AIIndustryType.GetName(AIIndustry.GetIndustryType(industry1)),5)+" to "+
								     Util.CapLength(in2name,10)
						if (!AIVehicle.SetName(vehicle, name)){
							AILog.Warning("name change to \""+name+"\" failed: "+AIError.GetLastErrorString());
						}
						AIOrder.AppendOrder(vehicle, in1loc, AIOrder.OF_FULL_LOAD);
						AIVehicle.StartStopVehicle(vehicle);
						AIOrder.AppendOrder(vehicle, in2loc, AIOrder.OF_UNLOAD);
						
						
						local speed = AIEngine.GetMaxSpeed(engine) * 30 / 27; //KM/h actually comes out to be approximately tiles/month
						local numVehicles = AIIndustry.GetLastMonthProduction(industry1, cargo) //cargo / month
						                    * AITile.GetDistanceManhattanToTile(in1loc, in2loc) //tiles
											/ speed                                             //tiles / month
											/ AIVehicle.GetCapacity(vehicle, cargo);             //cargo / vehicle
						//correct by unit analysis
						AILog.Info("building "+numVehicles+" "+AIEngine.GetName(engine)+"s (max speed: "+AIEngine.GetMaxSpeed(engine)+"...");
						//I starts at 1 since one vehicle allready built
						for (local i=1; i<numVehicles; i++){
							if (!Money.ReserveMoney(AIEngine.GetPrice(engine))){
								break;
							}
							vehicle = TryCloneRoadVehicle(vehicle, depotTile);
							if (vehicle==null){
								return true; //if at least one vehicle exists, it's a sucsess
							}
							AIVehicle.StartStopVehicle(vehicle);
							if (i % 4 == 3 && i>4){
								BuildRoadStop(in1loc,
											  (stopType == AIStation.STATION_BUS_STOP ? AIRoad.ROADVEHTYPE_BUS : AIRoad.ROADVEHTYPE_TRUCK),
											  in1station, null);
								BuildRoadStop(in2loc,
											  (stopType == AIStation.STATION_BUS_STOP ? AIRoad.ROADVEHTYPE_BUS : AIRoad.ROADVEHTYPE_TRUCK),
											  in2station, null);
							}
							this.Sleep(10);
						}
						return true;
					}
					else {
						return false;
					}
				}
				else {
					return false;
				}
			}
			else {
				return false;
			}
        }
        else {
            return false;
        }
    }
}

function SnakeAI::AttemptToConnectIndustry() {
	if (!CanBuildMoreRoadVehicles(10)){ //margin of 10
		return false;
	}
	local cargoList = AICargoList();
    local industries = AIIndustryList();
    industries.Valuate(Util.IndustryTotalCargo);
	industries.KeepAboveValue(10);
    industries.Sort(AIList.SORT_BY_VALUE, false);
    local industry = industries.Begin();
    local done=false;
    while (!industries.IsEnd() && !done) {
        foreach (cargo, prod in cargoList){
			if (ut.IsCargoPickedUp(industry, cargo) &&
			        AIIndustry.GetLastMonthTransportedPercentage(industry, cargo) == 0 &&
			        Money.ReserveMoney(50000)) {
				if (ConnectIndustryCargo(industry, cargo)){
					done = true;
					break;
				}
				else if (ConnectIndustryCargoTown(industry, cargo)){
					
				}
				else {
					AILog.Info("Industry failed: "+AIIndustry.GetName(industry) + "(" + AICargo.GetCargoLabel(cargo)+") No nearby industry accepting");
				}
			}
		}
        industry = industries.Next();
        this.Sleep(1);
    }
	return done;
    
}

function SnakeAI::DeleteNegativeIncomeVehicles(){
	local busses=AIVehicleList();
	busses.Valuate(AIVehicle.GetVehicleType);
	busses.KeepValue(AIVehicle.VT_ROAD);
	busses.Valuate(AIVehicle.GetProfitThisYear);
	busses.KeepBelowValue(-150);
	busses.Valuate(AIVehicle.GetProfitLastYear);
	busses.KeepBelowValue(-150);
	
	local planes = AIVehicleList();
	planes.Valuate(AIVehicle.GetVehicleType);
	planes.KeepValue(AIVehicle.VT_AIR);
	planes.Valuate(AIVehicle.GetProfitThisYear);
	planes.KeepBelowValue(-1500);
	planes.Valuate(AIVehicle.GetProfitLastYear);
	planes.KeepBelowValue(-1500);
	
	busses.AddList(planes);
	if (!busses.IsEmpty()){
		AILog.Info("selling "+busses.Count()+" vehicles for negative profit");
		foreach (bus, n in busses){
			if (Util.CapLength(AIVehicle.GetName(bus), 6) != "delete"){
				DeleteVehicle(bus);
			}
			else if (!AIOrder.IsGotoDepotOrder(bus, AIOrder.ORDER_CURRENT)){
				AIVehicle.SendVehicleToDepot(bus);
			}
		}
	}
	
	busses=AIVehicleList();
	busses.Valuate(AIVehicle.IsStoppedInDepot);
	busses.KeepValue(1);
	//busses.Valuate(AIVehicle.GetProfitLastYear);
	//busses.KeepBelowValue(-250);
	local num = 0;
	foreach (bus, n in busses){
		if (Util.CapLength(AIVehicle.GetName(bus),6) == "delete"){
			AILog.Info("Selling \""+AIVehicle.GetName(bus)+"\"");
			AIVehicle.SellVehicle(bus);
			num++;
		}
		else {
			//AILog.Info("not selling "+Util.CapLength(AIVehicle.GetName(bus),6));
		}
	}
	
	if (num >= 1){
		DeleteEmptyStations();
	}
}

function SnakeAI::DeleteVehicle(bus){
	if (!AIOrder.IsGotoDepotOrder(bus, AIOrder.ORDER_CURRENT)){
		AIVehicle.SendVehicleToDepot(bus);
	}
	if (AIVehicle.GetState(bus) == AIVehicle.VS_STOPPED && !AIVehicle.IsStoppedInDepot(bus)){
	    AIVehicle.StartStopVehicle(bus);
	}
	AILog.Info("Sending \""+AIVehicle.GetName(bus)+"\" to depo");
	local i=0;
	while (!AIVehicle.SetName(bus, "delete"+i)){
		i++;
	};
	AIVehicle.ReverseVehicle(bus);
}

function SnakeAI::DeleteEmptyStations() {
	local stations = AIStationList(AIStation.STATION_TRUCK_STOP);
	stations.AddList(AIStationList(AIStation.STATION_BUS_STOP));
	
	foreach (station, _ in stations){
		local vehicles = AIVehicleList_Station(station);
		if (vehicles.Count() == 0){
			AILog.Info("attempting to remove station "+AIStation.GetName(station));
			local tiles = Util.GetTileListRadius(AIStation.GetLocation(station), 20);
			local numBefore = tiles.Count();
			tiles.Valuate(AIRoad.IsRoadStationTile);
			tiles.KeepValue(1);
			local numStation = tiles.Count();
			tiles.Valuate(AIStation.GetStationID);
			tiles.KeepValue(station);
			local numAfter = tiles.Count();
			
			local num = 0;
			
			foreach (tile, _ in tiles){
				if (AIRoad.RemoveRoadStation(tile)){
					num++;
				}
			}
			if (num == 0){
				AILog.Warning("failed to remove station (total "+numBefore+" stations: "+numStation+" num after: "+numAfter+")");
			}
		}
	}
}

function SnakeAI::UpgradeVehicles() {
	if (!CanBuildMoreRoadVehicles(2)){
		return false;
	}
	local cargos = AICargoList();
	local idealEngines = [];
	foreach (cargo, what in cargos){
		idealEngines.push(ut.FindRoadVehicle(cargo));
	}

	local vehicles = AIVehicleList();
	vehicles.Valuate(AIVehicle.GetVehicleType);
	vehicles.KeepValue(AIVehicle.VT_ROAD);
	
	foreach (vehicle, type in vehicles){
		if (ut.CapLength(AIVehicle.GetName(vehicle),6) == "delete"){
			continue;
		}
		local needToBeUpgraded = true;
		foreach (engine in idealEngines){
			if (AIVehicle.GetEngineType(vehicle) == engine){
				needToBeUpgraded = false;
				break;
			}
		}
		needToBeUpgraded = needToBeUpgraded || AIVehicle.GetAgeLeft(vehicle)<AIBase.RandRange(150);
		
		if (needToBeUpgraded){
			AILog.Info("attempting to upgrade "+AIVehicle.GetName(vehicle)+" (age left: "+AIVehicle.GetAgeLeft(vehicle));
			local cargo = null;
			foreach (c, what in cargos){
				if (AIVehicle.GetCapacity(vehicle, c)){
					cargo = c;
					break;
				}
			}
			if (cargo != null){
				local engine = ut.FindRoadVehicle(cargo);
				local depot = BuildRoadDepotNear(AIVehicle.GetLocation(vehicle));
				if (depot && Money.ReserveMoney(AIEngine.GetPrice(engine))){
					local newvehicle = TryBuildRoadVehicle(engine, depot);
					local name = AIVehicle.GetName(vehicle);
					if (newvehicle != null && AIVehicle.IsValidVehicle(newvehicle)){
						AIVehicle.RefitVehicle(newvehicle, cargo);
						AIVehicle.StartStopVehicle(newvehicle);
						AIOrder.ShareOrders(newvehicle, vehicle);
						DeleteVehicle(vehicle);
						AIVehicle.SetName(newvehicle, name);
					}
				}
				else {
					break;
				}
			}
		}
	}
}

function SnakeAI::ExpandOversuppliedStations(){
	if (!CanBuildMoreRoadVehicles(5)){
		return false;
	}
	local stations = AIStationList(AIStation.STATION_TRUCK_STOP);
	stations.AddList(AIStationList(AIStation.STATION_BUS_STOP));
	local cargo = AICargoList();
	cargo.Valuate(ut.FindRoadVehicle);
	
	stations.Valuate(Util.StationWaitingCargo);
	stations.KeepAboveValue(40);
	stations.Sort(AIList.SORT_BY_VALUE, false);
	if (stations.IsEmpty()){
		return;
	}
	else {
		foreach (most, x in stations){ //x is unused
			foreach (gr, engine in cargo){
				local amount = AIStation.GetCargoWaiting(most, gr);
				if (amount <=40){
					continue;
				}
				local capacity = AIEngine.GetCapacity(engine);
				local date = AIDate.GetCurrentDate();
				local vehicleList = AIVehicleList_Station(most);
				vehicleList.Valuate(ut.CarriesCargo, gr);
				vehicleList.KeepValue(1);
				vehicleList.Valuate(AIVehicle.GetAge);
				vehicleList.Sort(AIList.SORT_BY_VALUE, true);
				local vehicleNumber=vehicleList.Count()
				if (amount > capacity * (vehicleNumber)){
					if (vehicleList.IsEmpty()){
						//AILog.Info("no vehicle available");
					}
					else {
						local vehicle = vehicleList.Begin();
						if (AIVehicle.GetProfitThisYear(vehicle)-500<=0 && AIVehicle.GetProfitLastYear(vehicle)-1000<=0){
							//AILog.Info("not enough profit");
						}
						else if (AIVehicle.GetAge(vehicle)>50+(10*vehicleList.Count()) && AIVehicle.GetAgeLeft(vehicle) > 0){
							if (Money.ReserveMoney(AIEngine.GetPrice(AIVehicle.GetEngineType(vehicle)))){
								//AILog.Info("station "+AIBaseStation.GetName(most)+" has "+amount+" "+AICargo.GetCargoLabel(gr)+
								//		   " waiting (capacity "+capacity+", "+(amount/capacity)+"/"+vehicleNumber+")");
								AILog.Info("cloning "+AIVehicle.GetName(vehicle)+" (profit this year: "+AIVehicle.GetProfitThisYear(vehicle)+
								           ", last year: "+AIVehicle.GetProfitLastYear(vehicle)+", age "+AIVehicle.GetAge(vehicle)+")");
								local depot = BuildRoadDepotNear(AIVehicle.GetLocation(vehicle));
								if (depot != null && AIMap.IsValidTile(depot)){
									local vclone = TryCloneRoadVehicle(vehicle, depot);
									if (vclone == null){
										return;
									}
									AIVehicle.StartStopVehicle(vclone);
									
									if (vehicleNumber % 4 == 3){
										for (local i=0; i<AIOrder.GetOrderCount(vclone);i++){
											if (AIOrder.IsValidVehicleOrder(vclone, i) && AIOrder.IsGotoStationOrder(vclone, i)){
												this.Sleep(50);
												BuildRoadStop(AIOrder.GetOrderDestination(vclone i),
													AICargo.HasCargoClass(gr, AICargo.CC_PASSENGERS)?AIRoad.ROADVEHTYPE_BUS : AIRoad.ROADVEHTYPE_TRUCK,
													AIStation.STATION_JOIN_ADJACENT, "");
											}
										}
										AILog.Info("station expanded");
									}
								}
								else {
									AILog.Info("failed cloning vehicle "+AIVehicle.GetName(vehicle));
								}
							}
							else {
								//AILog.Info("not enough money");
								return; //no use trying anything else
							}
						}
						else {
							//AILog.Info("too new ("+AIVehicle.GetAge(vehicle)+"/40 days)");
						}
					}
					//AILog.Info("fix finished");
				}
				else {
					//AILog.Info("not enough for intervention (" + (amount / capacity)+ "/4)");
				}
			}
		}
	}
}

function SnakeAI::ShrinkUndersuppliedStations(){
	local stations = AIStationList(AIStation.STATION_TRUCK_STOP);
	stations.AddList(AIStationList(AIStation.STATION_BUS_STOP));
	stations.Valuate(Util.StationWaitingCargo);
	stations.KeepBelowValue(30);
	//stations.KeepAboveValue(1);
	stations.Valuate(Util.hasAnyCargoRaiting);
	stations.KeepValue(1);
	
	local cargos = AICargoList();
	
	foreach (station, cargoWaiting in stations){
		foreach (cargo, nobodyCares in cargos){
			if (!AIStation.HasCargoRating(station, cargo)){
				continue;
			}
			local vehicles = AIVehicleList_Station(station);
			vehicles.Valuate(AIVehicle.GetCapacity, cargo);
			vehicles.KeepAboveValue(2);
			vehicles.Valuate(AIVehicle.GetCurrentSpeed);
			vehicles.KeepBelowValue(5);
			vehicles.Valuate(AIVehicle.GetState);
			vehicles.KeepValue(AIVehicle.VS_AT_STATION);
			vehicles.Valuate(AIVehicle.GetAge);
			vehicles.KeepAboveValue(150);
			vehicles.Valuate(AIVehicle.GetProfitThisYear);
			vehicles.Sort(AIList.SORT_BY_VALUE, true);
			
			if (vehicles.Count() >= 3){
				local vehicle = vehicles.Begin();
				AILog.Info("Station "+AIStation.GetName(station)+" has too many vehicles");
				AILog.Info("Deleting "+AIVehicle.GetName(vehicle));
				DeleteVehicle(vehicle);
			}
		}
	}
}

function SnakeAI::TweakVehicleNumber() {
	ShrinkUndersuppliedStations();
	ExpandOversuppliedStations();
}

function SnakeAI::DeleteRoad(tile){
	if (!AIRoad.IsRoadTile(tile)){
		return 0;
	}
	/*if (AIBridge.IsBridgeTile(tile) || AITunnel.IsTunnelTile(tile) || AIRoad.IsRoadDepotTile(tile) ||
	    (AIRoad.IsRoadTile(tile) && ut.GetRoadConnections(tile)==0)){
		AITile.DemolishTile(tile);
	}*/
	local tries = 0;
	do {
		if (AITile.DemolishTile(tile)){
			return 1;
			break;
		}
		else {
			if (AIError.GetLastError() != AIError.ERR_VEHICLE_IN_THE_WAY || tries % 20 == 0){
				AILog.Warning("Deleting road failed: "+AIError.GetLastErrorString());
			}
			else {
				this.Sleep(10);
			}
			tries++;
		}
	
	} while (AIError.GetLastError() == AIError.ERR_VEHICLE_IN_THE_WAY && tries < 70);
	
	if (tries >= 2){
		AILog.Info("Removed road after "+tries+" tries");
	}
	/*local s = 0;
	foreach (tile2 in Util.GetDirectlyAdjacentTiles(tile)){
		if (AIRoad.AreRoadTilesConnected(tile, tile2)){
			local tries = 0;
			do {
				if (AIRoad.RemoveRoad(tile, tile2)){
					
					s++;
					break;
				}
				else {
					AILog.Warning("Deleting road failed: "+AIError.GetLastErrorString());
					if (AIError.GetLastError() != AIError.ERR_VEHICLE_IN_THE_WAY){
						
						AILog.Warning("Deleting road failed: "+AIError.GetLastErrorString());
					}
					else {
						this.Sleep(10);
					}
					tries++;
				}
			
			} while (AIError.GetLastError() == AIError.ERR_VEHICLE_IN_THE_WAY && tries < 60);
			
			if (tries >= 2){
				AILog.Info("Removed road after "+tries+" tries");
			}
		}
	}*/
	return tries < 59 ? 1 : 0;
}

function SnakeAI::ReplaceLevelCrossing(tile, vehicle) {
	local vehicles = AIVehicleList_SharedOrders(vehicle);
	foreach (key, value in vehicles){
		AIVehicle.SendVehicleToDepot(key);
	}
	//
	this.Sleep(50);
	AILog.Info("Road tile: "+AIRoad.IsRoadTile(tile)+" level crossing "+AIRail.IsLevelCrossingTile(tile));
	local suc = DeleteRoad(tile);
	AILog.Info("Sucesfully removed "+suc+" road tiles");
	
	local num = 0;
	while (removeDeadEnds(tile, 15)>0 && num < 100){
		num++;
		// This loop repeats the side effect of the condition
	}
	
	if (AIVehicle.IsValidVehicle(vehicle)){
		local tile1 = AIOrder.GetOrderDestination(vehicle, 0);
		local tile2 = AIOrder.GetOrderDestination(vehicle, 1);
		AILog.Info("building route between "+tile1+" and "+tile2);
		AILog.Info("building route between "+AIStation.GetName(AIStation.GetStationID(AIOrder.GetOrderDestination(vehicle, 0)))+
				   " and "+AIStation.GetName(AIStation.GetStationID(AIOrder.GetOrderDestination(vehicle, 1))));
		ut.BuildRoute(tile1,
					  tile2);
	}
	
	foreach (key, value in vehicles){
		if (AIVehicle.IsStoppedInDepot(key)){
			AIVehicle.StartStopVehicle(key);
		}
	}
}

function SnakeAI::isOwnedOrLevel(x, c) {
	return AITile.GetOwner(x) == c || AIRail.IsLevelCrossingTile(x)
}

function SnakeAI::removeDeadEnds(tile, radius) {
	local i=0;
	
	local tiles = ut.GetTileListRadius(tile, radius);
	tiles.Valuate(AIRoad.IsRoadTile);
	tiles.KeepValue(1);
	tiles.Valuate(ut.GetRoadConnections);
	tiles.KeepBelowValue(2);
	tiles.Valuate(AIRoad.IsRoadStationTile);
	tiles.KeepValue(0);
	//tiles.Valuate(AIRoad.IsRoadDepotTile);
	//tiles.KeepValue(0);
	tiles.Valuate(isOwnedOrLevel, AICompany.ResolveCompanyID(AICompany.COMPANY_SELF));
	tiles.KeepValue(1);
	
	tiles.Valuate(AITile.GetDistanceManhattanToTile,tile);
	tiles.Sort(AIList.SORT_BY_VALUE, true);
	
	foreach (tile, value in tiles){
		i+=DeleteRoad(tile);
	}
	
	return i;
}


function SnakeAI::Start()
{
	if (!loaded){
		AILog.Warning("Snake AI starting... ("+Util.CapLength(null, 4)+")");
		SetCompanyName();
		Money.Payback();
	}
	
	local lastFixRouteTime = AIDate.GetCurrentDate();

    while (true) {
		//handle events
		while (AIEventController.IsEventWaiting()) {
			local e = AIEventController.GetNextEvent();
			local ec;
			switch (e.GetEventType()) {
				case AIEvent.ET_VEHICLE_CRASHED:
					ec = AIEventVehicleCrashed.Convert(e);
					local v  = ec.GetVehicleID();
					AILog.Info("We have a crashed vehicle (" + AIVehicle.GetName(v) + ")");
					if (!AIVehicle.IsValidVehicle(v)){
						AILog.Warning("Vehicle is invalid");
					}
					/* Handle the crashed vehicle */
					
					if (AIVehicle.GetVehicleType(v)==AIVehicle.VT_ROAD){
						local bus2 = null;
						local depo=BuildRoadDepotNear(AIOrder.GetOrderDestination(v, 0));
						if (depo != null){
							bus2=AIVehicle.CloneVehicle(depo,v,true);
							if (bus2 != null){
								AIVehicle.StartStopVehicle(bus2);
								AILog.Info("Replaced vehicle: "+AIVehicle.GetName(bus2));
							}
							else {
								AILog.Warning("Error building Vehicle: "+AIError.GetLastErrorString());
							}
							
						}
						else {
							AILog.Warning("Replacement failed, no depo found");
						}
						
						if (Money.ReserveMoney(20000)){
							ReplaceLevelCrossing(ec.GetCrashSite(), bus2!=null?bus2:v);
							//this.Sleep(50);
							//ReplaceLevelCrossing(ec.GetCrashSite(), bus2==null?bus2:v);
						}
					}
					else if (AIVehicle.GetVehicleType(v)==AIVehicle.VT_AIR) {
						air.ReplacePlane(v);
					}
					else {
						AILog.Info("was not RV");
					}
					break;
				case AIEvent. ET_ENGINE_PREVIEW:
					local ec = AIEventEnginePreview.Convert(e);
					ec.AcceptPreview();
					break;
				case AIEvent.ET_VEHICLE_LOST:
					if (Money.ReserveMoney(20000) && AIDate.GetCurrentDate() - 100 > lastFixRouteTime){
						lastFixRouteTime = AIDate.GetCurrentDate();
						local vehicle = AIEventVehicleLost.Convert(e).GetVehicleID();
						
						if (AIVehicle.GetAge(vehicle) > 150){
							local tile1 = AIVehicle.GetLocation(vehicle);
							local tile2 = AIOrder.GetOrderDestination(vehicle, 1);
							AILog.Info("building route between "+tile1+" and "+tile2);
							ut.BuildRoute(tile1,
										  tile2);
							this.Sleep(100);
										  
							removeDeadEnds(AIVehicle.GetLocation(vehicle), 10);
						}
					}
					else {
						AILog.Warning("Not fixing route, since date too new or not enough money");
					}
			}
		}
		ut.BuildUnbuiltRoads();
		if (Money.ReserveMoney(60000) && !loaded){
			/*if (rail.ExpandRailNetwork()){
			
			}
			else*/ if (AIBase.Chance(2, 3)){
				if (!air.BuildAirRoute()){
					AttemptToConnectIndustry()
				}
			}
			else {
				if (!AttemptToConnectIndustry()){
					air.BuildAirRoute();
				}
			}
		}
		if (Money.ReserveMoney(50000)){
			UpgradeVehicles();
			air.UpgradePlanes();
		}
		if (Money.ReserveMoney(15000)){
			TweakVehicleNumber();
		}
		
		
		Money.Payback();
		
		DeleteNegativeIncomeVehicles();

        this.Sleep(100);
		
		loaded = false;
    }
}

function SnakeAI::Save()
{
    local table = {
	    base = { },
	    road = {
	 		maxVehicles = maxVehicles
	    },
	    air = air.Save(),
	    util = ut.Save(),
    };
    //TODO: Add your save data to the table.
    return table;
 }
 
function SnakeAI::Load(version, data)
{
    if ("base" in data){
       ; //do nothing
    }
    else {
		AILog.Warning("invalid save: missing field \"base\"");
    }
   
    if ("road" in data){
		if ("maxVehicles" in data.road){
			maxVehicles = data.road.maxVehicles;
		}
		else {
			AILog.Warning("invalid save: missing fied \"road.maxVehicles\"");
		}
    }
	
	if ("air" in data){
		air.Load(data.air);
	}
	else {
		AILog.Warning("invalid save: missing field \"air\"");
	}
	
	if ("util" in data){
		ut.Load(data.util);
	}
	else {
		AILog.Warning("invalid save: missing field \"util\"");
	}
    AILog.Info(" Loaded");
    loaded = true;
   //TODO: Add your loading routines.
}