/* 
 *	Copyright 2013 Varen De Meersman
 *  This file is part of MailAI.
 *
 *  MailAI is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  MailAI is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with MailAI.  If not, see <http://www.gnu.org/licenses/>.
 */
class BankManager {
	savings = 0;
	
	constructor() {}
}

/**
*	Increase the savings by one loaninterval,
*	providing false will put savings back to 0
*/
function BankManager::IncreaseSavings(bool) {
	if(!bool) {savings = 0;}
	else {savings += AICompany.GetLoanInterval();}
}

function BankManager::CanBuildRoute(town_array, engine) {
	if(engine == null) {MailAI.Info(0,"No engine in CanBuildRoute"); return false;}
	if(town_array == null) {throw("null in canbuildroute");}
	local t1_tile = AITown.GetLocation(town_array[0]);
	local t2_tile = AITown.GetLocation(town_array[1]);
	local distance = AIMap.DistanceManhattan(t1_tile,t2_tile);	
	local available_money = AICompany.GetBankBalance(AICompany.COMPANY_SELF)
		+ (AICompany.GetMaxLoanAmount() - AICompany.GetLoanAmount());
	local cost = 1000 + BankManager.FarmEstimate(t1_tile,t2_tile);	
	if(AIEngine.GetVehicleType(engine) == AIVehicle.VT_RAIL) {
		cost += 2*AIEngine.GetPrice(engine);
		cost += 10*AIEngine.GetPrice(EngineMaster.GetWagon(engine,null));
		cost += 2*AIRail.GetBuildCost(AIRail.GetCurrentRailType(),AIRail.BT_DEPOT);
		cost += 2*distance*AIRail.GetBuildCost(AIRail.GetCurrentRailType(),AIRail.BT_TRACK);
		cost += 12*AIRail.GetBuildCost(AIRail.GetCurrentRailType(),AIRail.BT_STATION);
		//likely feeder lines so add a little road costs as well
		cost += 6*AIRoad.GetBuildCost(AIRoad.GetCurrentRoadType(),AIRoad.BT_TRUCK_STOP);
		cost += 2*AIRoad.GetBuildCost(AIRoad.GetCurrentRoadType(),AIRoad.BT_DEPOT);	
		cost += 30*AIRoad.GetBuildCost(AIRoad.GetCurrentRoadType(),AIRoad.BT_ROAD);
	} else if(AIEngine.GetVehicleType(engine) == AIVehicle.VT_ROAD) {
		cost += 2*AIEngine.GetPrice(engine);
		cost += 2*AIRoad.GetBuildCost(AIRoad.GetCurrentRoadType(),AIRoad.BT_TRUCK_STOP);
		cost += 2*AIRoad.GetBuildCost(AIRoad.GetCurrentRoadType(),AIRoad.BT_DEPOT);
		cost += 2*distance*AIRoad.GetBuildCost(AIRoad.GetCurrentRoadType(),AIRoad.BT_ROAD);
		cost += savings;//savings are for train lines only
	} else {throw("Engine Type not supported in BankManager.RouteEstimate");}
	//MailAI.Info(0,"Est. costs : "+cost);
	if (AICompany.GetBankBalance(AICompany.COMPANY_SELF) > cost) {return true;}	
	local counter = AICompany.GetLoanInterval();
	while(cost > counter) {counter += AICompany.GetLoanInterval();}
	if ((AICompany.GetMaxLoanAmount() - AICompany.GetLoanAmount()) > counter+(2*AICompany.GetLoanInterval())) {
		//keep two loaninterval to prevent getting into too much trouble when still running out of money
		if(AICompany.SetLoanAmount(AICompany.GetLoanAmount() + counter)) {
			return true;	
		} else {
			MailAI.Info(0,"Couldt loan."); return false;	
		}
	}
	return false;
}

/**
*	Checks to see if there's enough money available (might need to loan) 
*	to buy a new engine of provided model.
*	@return true if enough money to buy
*/
function BankManager::EnoughMoneyForEngine(engine) {
	if(engine==null){return false;}
	local cost = AIEngine.GetPrice(engine);
	if (AICompany.GetBankBalance(AICompany.COMPANY_SELF) > cost) {
		return true;
	}	
	local counter = AICompany.GetLoanInterval();
	while(cost > counter) {
		counter += AICompany.GetLoanInterval();
	}
	if ((AICompany.GetMaxLoanAmount() - AICompany.GetLoanAmount()) > counter) {
		return AICompany.SetLoanAmount(AICompany.GetLoanAmount() + counter);
	}
	MailAI.Info(1,"Not enough money for new engine!");
	return false;
}

/**
*	Makes sure there's enough money to buy the vehicle.
*	@return the new vehicle or null if something went wrong
*/
function BankManager::BuildVehicle(depot,model) {
	if(model==null || depot==null){return null;}
	local new_veh = AIVehicle.BuildVehicle(depot, model);
	local err = AIError.GetLastErrorString(), loan = true, money = 0;
	while(err == "ERR_NOT_ENOUGH_CASH" && !AIVehicle.IsValidVehicle(new_veh) && loan) {	
		loan = BankManager.LoanOne();
		new_veh = AIVehicle.BuildVehicle(depot, model);
		err = AIError.GetLastErrorString();
	}
	if(err == "ERR_NOT_ENOUGH_CASH" && !AIVehicle.IsValidVehicle(new_veh)) {
		MailAI.Info(2,"Reached max loan while buying vehicle, waiting...");
		money = AICompany.GetBankBalance(AICompany.COMPANY_SELF);
	}
	while(err == "ERR_NOT_ENOUGH_CASH" && !AIVehicle.IsValidVehicle(new_veh)) {
		new_veh = AIVehicle.BuildVehicle(depot, model);
		err = AIError.GetLastErrorString();
		MailAI.Sleep(500);
		if(AICompany.GetBankBalance(AICompany.COMPANY_SELF) < abs(money/2))
		{MailAI.Info(2,"Bleeding money... chairman asked to resign...");break;}
	}
	if(new_veh == null || !AIVehicle.IsValidVehicle(new_veh)) {
		MailAI.Info(3,"Error building vehicle or not valid: "+err);
		return null;
	}
	return new_veh;
}

/**
*	Tries to pay back loan if any.
*/
function BankManager::ManageLoan() {	
	local loan = AICompany.GetLoanAmount();
	if(loan == 0){return;}
	local balance = AICompany.GetBankBalance(AICompany.COMPANY_SELF);
	local min_balance =  2 * AICompany.GetLoanInterval(); 
	local pay_back = 0;
	while(balance - pay_back > min_balance){pay_back += AICompany.GetLoanInterval();}
	if (pay_back > loan){pay_back = loan;}
	if (pay_back > 0){AICompany.SetLoanAmount(loan - pay_back);}	
}

/**
*	Sometimes we run out of money when building, this loans one interval
*	@return true if we were able to loan
*/
function BankManager::LoanOne() {
	if(!AICompany.SetLoanAmount(AICompany.GetLoanAmount() + AICompany.GetLoanInterval())) {
		if(AIError.GetLastErrorString() != "ERR_PRECONDITION_FAILED") {
			MailAI.Info(2,"Couldn't loan! "+AIError.GetLastErrorString());
		}
		return false;
	}
	return true;
}

/**
*	Adds 400 per farm tile it will likely traverse on average.
* 	400 is a guestimate cost in the beginning of a game
*/
function BankManager::FarmEstimate(t1_tile, t2_tile) {
	local tl = AITileList();
	tl.AddRectangle(t1_tile, t2_tile);
	tl.Valuate(AITile.IsFarmTile); tl.KeepValue(1);
	if(tl.IsEmpty()) {return 0;}
	local yy = AIMap.GetTileY(t1_tile) - AIMap.GetTileY(t2_tile);
	if(yy == 0){yy++;}
	local xx = AIMap.GetTileX(t1_tile) - AIMap.GetTileX(t2_tile);
	if(xx == 0){xx++;}
	local cost = 0;
	if(yy < xx) {
		cost = 400 * abs(tl.Count()/yy);
	} else {
		cost = 400 * abs(tl.Count()/xx);	
	}
	//MailAI.Info(0,"Farmtile cost: "+cost);
	return cost;		
}