/* 
 *	Copyright 2013 Varen De Meersman
 *  This file is part of MailAI.
 *
 *  MailAI is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  MailAI is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with MailAI.  If not, see <http://www.gnu.org/licenses/>.
 */
 
 class EventManager {
	constructor() {}
}

/**
*	Function that checks for news on new vehicles.
*/
function EventManager::Check() {
	 while (AIEventController.IsEventWaiting()) {
	  	local e = AIEventController.GetNextEvent();
	  	switch (e.GetEventType()) {
	    	case AIEvent.ET_VEHICLE_WAITING_IN_DEPOT: {
	      		local event = AIEventVehicleWaitingInDepot.Convert(e);
	      		local veh  = event.GetVehicleID();
	      		MailAI.Info(2,"Checking vehicle that's in a depot.");
				if(AIVehicle.GetAge(veh) >= MailAI.days_to_make_profit) {
					local profit_last = AIVehicle.GetProfitLastYear(veh), profit_this = AIVehicle.GetProfitThisYear(veh);		
					local no_prof = MailAI.min_truck_profit;
					if(AIVehicle.GetVehicleType(veh) == AIVehicle.VT_RAIL) {
						no_prof = AIVehicle.GetNumWagons(veh) * MailAI.min_train_profit;
					}		
					if((profit_last < no_prof || profit_last == 0) && (profit_this < no_prof || profit_this == 0)) {
						if(AIVehicle.GetNumWagons(veh) > MailAI.min_wagons+1) {
							EngineMaster.SellWagon(veh);
							EngineMaster.SellWagon(veh);
							EngineMaster.RenewVehicleCheaper(veh);
						} else {
							EngineMaster.SellVehicle(veh);
						}
						continue;
					}
					if(AIVehicle.GetVehicleType(veh) == AIVehicle.VT_RAIL) {		
						local prof_needed = AIVehicle.GetNumWagons(veh) * MailAI.low_train_profit;	
						if(profit_last < prof_needed && profit_this < prof_needed
						&& AIVehicle.GetNumWagons(veh) > MailAI.min_wagons) {
							EngineMaster.SellWagon(veh);
							EngineMaster.RenewVehicleCheaper(veh);
							continue;
						}
					}	
				}	
				local dist = AITile.GetDistanceManhattanToTile(AIOrder.GetOrderDestination(veh,0),AIOrder.GetOrderDestination(veh,1));	
				local engine = EngineMaster.GetEngine(dist,null,AIVehicle.GetVehicleType(veh)==AIVehicle.VT_ROAD,
														AIVehicle.GetVehicleType(veh)==AIVehicle.VT_RAIL,true);
				if(engine==null){MailAI.Info(1,"No good engines available to replace vehicle! "+veh);return;}
				if(!BankManager.EnoughMoneyForEngine(engine)) {return;}//msg allready in bankmanager
				if(AIVehicle.GetAge(veh) > AIVehicle.GetMaxAge(veh) - MailAI.days_before_max_age) {
					EngineMaster.RenewVehicle(veh);
					continue;
				}					
				if(AIEngine.GetReliability(AIVehicle.GetEngineType(veh)) < MailAI.min_max_reliability) {	
					EngineMaster.RenewVehicle(veh);
					continue;				
				}
			}
	    	case AIEvent.ET_VEHICLE_CRASHED: {
	      		local event = AIEventVehicleCrashed.Convert(e);
	      		local veh  = event.GetVehicleID();
	      		MailAI.Info(2,"Oh NO! We have a crashed vehicle!");
	      		/* Handle the crashed vehicle */
	      		break;
	      	}
	      	case AIEvent.ET_ENGINE_AVAILABLE: {
				local event = AIEventEngineAvailable.Convert(e);
	      		local id = event.GetEngineID();
	      		//MailAI.Info(0,"New engine available: " + AIEngine.GetName(id));
	      		break;
	      	}
	      	case AIEvent.ET_ENGINE_PREVIEW: {
				local event = AIEventEnginePreview.Convert(e);
				//if (event.AcceptPreview()) MailAI.Info(0,"New engine accepted for preview: " + event.GetName());
				break;
	      	}
			case AIEvent.ET_COMPANY_NEW: {
				local event = AIEventCompanyNew.Convert(e);
				local company = event.GetCompanyID();
				MailAI.Info(1,"Welcome to the party, " + AICompany.GetName(company)+"!");
				break;
			}
	   	}
	 }
 }
 
 function EventManager::BuildRailTrack(start,dir) {
 	if(start==null){throw("buildrailtrack on null tile");return null;}
	local it_count = 0, track_dir = dir;
	switch (dir) {case "NE": case "SW": track_dir = AIRail.RAILTRACK_NE_SW; break;
	   			  case "SE": case "NW": track_dir = AIRail.RAILTRACK_NW_SE; break;}
	while(!AIRail.BuildRailTrack(start,track_dir) && it_count < 33
	&& AIError.GetLastErrorString() != "ERR_ALREADY_BUILT" && AIError.GetLastErrorString() != "ERR_NONE") {
		EventManager.HandleTrackErrors(start);		
		it_count++;
	}
	if(it_count >= 33) {
		MailAI.Info(0,"Couldn't build railtrack: "+AIError.GetLastErrorString());
		MailAI.BuildSign(start,"!");
		return false;
	}
	return true;	
 }
 
 function EventManager::BuildRail(t1,t2,t3) {
	local it_count = 0; 
	while(!AIRail.BuildRail(t1,t2,t3) && it_count < 33
	&& AIError.GetLastErrorString() != "ERR_ALREADY_BUILT") {
		EventManager.HandleTrackErrors(t2);		
		it_count++;
	}
	if(it_count >= 33) {
		MailAI.Info(0,"Couldn't build rail: "+AIError.GetLastErrorString());
		MailAI.BuildSign(t2,"=");
		return false;
	}
	return true;	
 }

function EventManager::ConvertRailType(t1,t2,int) {
	local it_count = 0;
	while(!AIRail.ConvertRailType(t1,t2,int) && it_count < 33
	&& AIError.GetLastErrorString() != "ERR_UNSUITABLE_TRACK") {
		EventManager.HandleTrackErrors(t2);		
		it_count++;
	}
	if(it_count >= 33) {
		MailAI.Info(0,"Couldn't convert track: "+AIError.GetLastErrorString());
		MailAI.BuildSign(t1,"T");
		return false;
	}
	return true;
}
 
 function EventManager::BuildSignal(t1,t2,t3) {
 	if(t1==null||t2==null){throw("null in buildsignal");}
 	local it_count = 0; 
	while(!AIRail.BuildSignal(t1,t2,t3) && it_count < 33
	&& AIError.GetLastErrorString() != "ERR_ALREADY_BUILT"
	&& AIError.GetLastErrorString() != "ERR_PRECONDITION_FAILED") {
		EventManager.HandleTrackErrors(t2);		
		it_count++;
	}
	if(it_count >= 33) {
		MailAI.Info(0,"Couldn't build signal: "+AIError.GetLastErrorString());
		MailAI.BuildSign(t1,"S");
		return false;
	}
	return true;	
 }
 
 /** 
 *	Demolishing tiles might not be a good idea for track building errors.
 */
 function EventManager::HandleTrackErrors(tile) {
 	if(AIError.GetLastErrorString() == "ERR_NONE") {return true;}
	if(AIError.GetLastErrorString() == "ERR_NOT_ENOUGH_CASH") {
		return BankManager.LoanOne();
	} 	
	if(AIError.GetLastErrorString() == "ERR_VEHICLE_IN_THE_WAY") {
		MailAI.Sleep(50);
		return true;
	} 
	if(AIError.GetLastErrorString() == "ERR_AREA_NOT_CLEAR") {
		if(AICompany.IsMine(AITile.GetOwner(tile))){
			if(AIRoad.IsRoadTile(tile) && AIRail.IsRailTile(tile)) {
				local tr = AIRail.GetRailTracks(tile);
				if(Tile.DemolishTile(tile))
				{return AIRail.BuildRailTrack(tile,tr);}
			}
			return false;
		}
 		if(!Tile.DemolishTile(tile)) {
 			MailAI.Info(0,"Couldn't demolish: "+AIError.GetLastErrorString());
 			return false;
 		}
 		return true;
 	}
	return false; 	
 }
 	
 /**
 *	Will handle common errors local auth refuses, not enough cash and vehicle in the way.
 *	This function should be iterated so it keeps looping until solved or max it reached.
 *	@param tile1 If the tried action has an AITile that might be causing a problem.
 *	@param tile2 If this is not null, tile1 must be not null as well.
 *	@return True if the error has been succesfully dealt with FOR THIS ITERATION.
 */
 function EventManager::HandleCommonErrors(tile1,tile2) {
 	if(AIError.GetLastErrorString() == "ERR_NONE"
 	|| AIError.GetLastErrorString() == "ERR_ALREADY_BUILT") {return true;}
 	if(AIError.GetLastErrorString() == "ERR_AREA_NOT_CLEAR") {
 		if(!Tile.DemolishTile(tile1)) {
 			MailAI.Info(0,"Couldn't demolish: "+AIError.GetLastErrorString());
 			return false;
 		}
 		if(tile2 != null) {
 			if(!Tile.DemolishTile(tile2)) {
 				MailAI.Info(0,"Couldn't demolish: "+AIError.GetLastErrorString());
 				return false;
 			}
 		}
 		return true;
 	}
	if(AIError.GetLastErrorString() == "ERR_LOCAL_AUTHORITY_REFUSES") {
		return EventManager.HandleRating(AITile.GetClosestTown(tile1));
	}
	if(AIError.GetLastErrorString() == "ERR_NOT_ENOUGH_CASH") {
		return BankManager.LoanOne();
	} 	
	if(AIError.GetLastErrorString() == "ERR_VEHICLE_IN_THE_WAY") {
		MailAI.Sleep(50);
		return true;
	} 
	if(AIError.GetLastErrorString() == "ERR_FLAT_LAND_REQUIRED") {
		return Tile.MakeTileFlat(tile1);
	}
	if(AIError.GetLastErrorString() == "ERR_LAND_SLOPED_WRONG") {
		return Tile.MakeTileFlat(tile1);
	}
	if(AIError.GetLastErrorString() == "ERR_OWNED_BY_ANOTHER_COMPANY") {
		MailAI.Info(2,"Another company has built in our way!");
		MailAI.BuildSign(tile1,"I wanted to built here...");
		return false;
	}
	if(AIError.GetLastErrorString() == "ERR_BRIDGE_HEADS_NOT_ON_SAME_HEIGHT"
	|| AIError.GetLastErrorString() == "ERR_TUNNEL_CANNOT_BUILD_ON_WATER"
	|| AIError.GetLastErrorString() == "ERR_UNKNOWN"
	|| AIError.GetLastErrorString() == "ERR_PRECONDITION_FAILED") {
		return false;
	}
	MailAI.Info(2,"Error not handled in handlecommonerrors! "+AIError.GetLastErrorString());
	return false;
 }
 
/**
*	Taken from PathZilla v.6 and adapted to work for this MailAI.
*	Will build trees to try get the rating back to at least GOOD.
*	@param town The town that's being a problem.
*	@return True if rating is GOOD or higher.
*/
function EventManager::HandleRating(town) {
	MailAI.Info(1,"Improving relations with "+AITown.GetName(town)+"'s city council.");
	if(AITown.GetRating(town, AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)) >= AITown.TOWN_RATING_GOOD) {
		return true;
	}
	local townLocation = AITown.GetLocation(town);
	// Get a list of tiles to search in
	local searchRadius = min(AIMap.DistanceFromEdge(townLocation) - 1, 20);
	local offset = AIMap.GetTileIndex(searchRadius, searchRadius);
	// After that, find places we can build trees
	local tileList = AITileList();
	tileList.AddRectangle(townLocation - offset, townLocation + offset);
	foreach(tile, _ in tileList) {
		local suitable = (!AITile.IsWithinTownInfluence(tile, town) && AITile.IsBuildable(tile) && !AITile.HasTreeOnTile(tile));
		tileList.SetValue(tile, (suitable) ? 1 : 0);
	}
	tileList.RemoveValue(0);
	foreach(tile, _ in tileList) {
		local r = AITile.GetDistanceManhattanToTile(tile, townLocation) + AIBase.RandRange(6) - 3;
		tileList.SetValue(tile, r);
	}
	tileList.Sort(AIList.SORT_BY_VALUE, true);	
	// For the places that are available, build a "green belt" around the town
	if(!tileList.IsEmpty()) {
		local expenditure = 0;
		local tile = tileList.Begin();	
		while(AITown.GetRating(town, AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)) < AITown.TOWN_RATING_GOOD
				 && expenditure < 9000 && !tileList.IsEnd()) {
			local acc = AIAccounting();
			AITile.PlantTree(tile);
			expenditure += acc.GetCosts();
			tile = tileList.Next();
		}
	}
	if(AITown.GetRating(town, AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)) < AITown.TOWN_RATING_GOOD){
		//MailAI.Info(1,"Couldn't get rating back up. :(");
		return false;
	} else {
		//MailAI.Info(0,"Rating improved.");
		return true;
	}
}