/* 
 *	Copyright 2013 Varen De Meersman
 *  This file is part of MailAI.
 *
 *  MailAI is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  MailAI is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with MailAI.  If not, see <http://www.gnu.org/licenses/>.
 */
class RoutePlanner {
	build_man = null;
	bank_man = null;
	use_trucks = true;
	use_trains = true;	
	candidate_road_routes = [];
	candidate_rail_routes = [];
	existing_routes = [];

	constructor() { 
		build_man = BuildManager();
		bank_man = BankManager();
		this.GetExistingRoutes();//Check for existing routes, necessary after loading
	}
}

function RoutePlanner::GetExistingRoutes() {
	//if(this.existing_routes.len() > 0){return this.existing_routes;}
	local route_count = 0, veh_list = AIVehicleList();
	if(veh_list.IsEmpty()) {return null;}
	foreach(veh, _ in veh_list) {
		local stations = AITileList(), sorted_towns = [];
		for (local i = 0; i < AIOrder.GetOrderCount(veh); i++) {
			stations.AddTile(AIOrder.GetOrderDestination(veh,i));
			sorted_towns = MailAI.Append(sorted_towns,AITile.GetClosestTown(AIOrder.GetOrderDestination(veh,i)));
		}
		local exists = false;
		foreach(route in existing_routes) {
			local station_check = route[1];//second element always stations list!	
			if(station_check.Count() != stations.Count()) {continue;}
			station_check.RemoveList(stations);
			if(station_check.Count()==0) {exists = true; break;}	
		}
		if(!exists) {
			route_count++;
			this.existing_routes.append([sorted_towns,stations,AIVehicle.GetVehicleType(veh)]);
		}
	}
	MailAI.Info(1,"Counted "+veh_list.Count()+" vehicles for "+route_count+" routes.");
	return this.existing_routes;
}

function RoutePlanner::GetCandidateRoadRoutes() {
	if(!MailAI.AllowTrucks()){return;}
	if(this.candidate_road_routes.len() > 0){return this.candidate_road_routes;}
	this.candidate_road_routes = [];
	local counter = 0 , town_list = AITownList();
	town_list.Valuate(AITown.GetPopulation); town_list.Sort(AIList.SORT_BY_VALUE, false);
	town_list.KeepAboveValue(MailAI.min_road_population);
	MailAI.Info(1,town_list.Count()+" towns with enough population for Mail Service.");
	local compareList = AIList(); compareList.AddList(town_list);
	foreach(town, _ in town_list) {	
		compareList.RemoveItem(town);//don't compare with self
		foreach(comp, _ in compareList)	{	
			local sorted_towns = [town]; sorted_towns = MailAI.Append(sorted_towns,comp);
			local distance = AITile.GetDistanceManhattanToTile (AITown.GetLocation(town),AITown.GetLocation(comp));
			if(MailAI.AllowTrains) {
				if(distance > MailAI.max_short_route_distance || distance <= MailAI.min_distance){
					continue;}
			} else {
				if(distance > MailAI.max_medium_route_distance || distance <= MailAI.min_distance){
					continue;}
			}
			counter++;
			this.candidate_road_routes.append([sorted_towns,distance,AITown.GetPopulation(comp)]);			
		}					
	}
	MailAI.Info(1,"Found "+counter+" possible road routes. (broad search)");	
	return this.candidate_road_routes;
}

function RoutePlanner::GetCandidateRailRoutes() {
	if(!MailAI.AllowTrains()){return;}
	if(this.candidate_rail_routes.len() > 0){return this.candidate_rail_routes;}
	this.candidate_rail_routes = [];
	local counter = 0 , town_list = AITownList();
	town_list.Valuate(AITown.GetPopulation); town_list.Sort(AIList.SORT_BY_VALUE, false);
	town_list.KeepAboveValue(MailAI.min_rail_population);
	MailAI.Info(1,town_list.Count()+" cities with enough population for a Sorting Centre.");
	local compareList = AIList(); compareList.AddList(town_list);
	foreach(town, _ in town_list) {	
		compareList.RemoveItem(town);//don't compare with self
		foreach(comp, _ in compareList)	{	
			local sorted_towns = [town]; sorted_towns = MailAI.Append(sorted_towns,comp);
			local distance = AITile.GetDistanceManhattanToTile (AITown.GetLocation(town),AITown.GetLocation(comp));
			if(distance > MailAI.max_long_route_distance || distance <= MailAI.max_short_route_distance){continue;}
			counter++;
			this.candidate_rail_routes.append([sorted_towns,distance,AITown.GetPopulation(comp)]);			
		}					
	}
	MailAI.Info(1,"Found "+counter+" possible rail routes (broad search).");	
	return this.candidate_rail_routes;
}

function RoutePlanner::BuildNewRoute() {
	local next_route = null, engine = null, route_object = null;
	this.use_trucks = MailAI.AllowTrucks(); this.use_trains = MailAI.AllowTrains();
//Long train line (and feeder trucks if allowed)
	if(this.use_trains) {
		local counter = 0;
		while(counter < 13) {
			counter++;
			next_route = this.GetNextRailRoute(MailAI.max_medium_route_distance,
				MailAI.max_long_route_distance,MailAI.min_long_rail_population);
			if(next_route != null) {
				if(bank_man.CanBuildRoute(next_route[0],next_route[1])) {	
					route_object = this.build_man.BuildLine(next_route);
					if(route_object != null) {
						bank_man.IncreaseSavings(false);
						this.existing_routes.append(route_object.GetRouteArray());
						MailAI.AnnounceRoute(route_object);
						if(this.use_trucks) {this.BuildFeederRoutes(route_object);}
						return route_object;	
					}
				} else {break;}		
			} else {break;}
		}
//medium length train line (and feeder trucks if allowed)
		counter = 0;
		while(counter < 13) {
			counter++;
			next_route = this.GetNextRailRoute(MailAI.max_short_route_distance,
				MailAI.max_medium_route_distance,MailAI.min_rail_population);
			if(next_route != null) {
				if(bank_man.CanBuildRoute(next_route[0],next_route[1])) {
					route_object = this.build_man.BuildLine(next_route);
					if(route_object != null) {
						bank_man.IncreaseSavings(false);
						this.existing_routes.append(route_object.GetRouteArray());
						MailAI.AnnounceRoute(route_object);
						if(this.use_trucks) {this.BuildFeederRoutes(route_object);}
						return route_object;
					}
				} else {break;}	
			} else {break;}	
		}
	}
//medium length truck line if trains not allowed
	if(!this.use_trains) { next_route = 
		this.GetNextRoadRoute(MailAI.min_distance,MailAI.max_medium_route_distance,MailAI.min_road_population);
	} else { next_route = 
		this.GetNextRoadRoute(MailAI.min_distance,MailAI.max_short_route_distance,MailAI.min_road_population);
	}
	if(next_route != null) {
		if(bank_man.CanBuildRoute(next_route[0],next_route[1])) {
			if(this.use_trains){bank_man.IncreaseSavings(true);}
			route_object = this.build_man.BuildLine(next_route);
			if(route_object != null) {
				this.existing_routes.append(route_object.GetRouteArray());
				MailAI.AnnounceRoute(route_object);
				return route_object;
			}
		}
	}		
	return null;		
}

function RoutePlanner::BuildFeederRoutes(route_object) {
	local sorted = route_object.GetSortedTowns();
	foreach(town in sorted) {
		if(AITown.GetPopulation(town) < MailAI.min_feeder_population) {continue;}
		MailAI.Info(0,"Building feeder line in "+AITown.GetName(town));
		local route_object2 = null;
		local results = this.build_man.BuildFourTruckStops(town,route_object);	
		if(results==null){continue;}
		if(results[0].Count() > 1) {
			route_object2 = Route([town],results[0],results[1],20,results[2],null);
		} else if(results[0].Count() == 1) {//if only one station built use it for a normal PtP
			route_object2 = this.BuildForcedTruckLine(results[0].Begin(),results[1]);	
		}
		if(route_object2 != null) {
			this.existing_routes.append(route_object2.GetRouteArray());
			MailAI.AnnounceRoute(route_object2);
		}
	}
}

/**
*	We need a feeder truck route to this (unused) truck station attached to train station.
*/
function RoutePlanner::BuildForcedTruckLine(station_tile, depot_tile) {
	if(this.candidate_road_routes.len() == 0){this.GetCandidateRoadRoutes();}
	local start_town = AITile.GetClosestTown(station_tile), end_town;
	local route_object = null, it_count = 0;
	while(route_object == null && it_count < 10) {	
		local distance = MailAI.max_short_route_distance;
		foreach(opt in this.candidate_road_routes) {
			//MailAI.Info(0,opt[0][0]+" vs "+start_town);
			if(opt[0][0]==start_town) {
				if(opt[1] < distance) {
					distance = opt[1];
					end_town=opt[0][1];
					//MailAI.Info(0,"Distance shortened : "+distance);
				}	
			} else if(opt[0][1]==start_town) {
				if(opt[1] < distance) {
					distance = opt[1];
					end_town=opt[0][0];
					//MailAI.Info(0,"Distance shortened : "+distance);
				}			
			}
		}
		if(end_town==null){MailAI.Info(0,"No feeder towns close by");return null;}
		local sorted_towns = [start_town];sorted_towns = MailAI.Append(sorted_towns,end_town);
		//we know it can only be in candidate_road_routes
		this.candidate_road_routes = this.RemoveRoute(this.candidate_road_routes,sorted_towns);
		route_object = build_man.BuildForcedTruckLine(station_tile,end_town,depot_tile);
		it_count++;
	}
	return route_object;	
}

function RoutePlanner::GetNextRailRoute(min_distance, max_distance, min_population) {
	if(this.candidate_rail_routes.len() == 0){this.GetCandidateRailRoutes();}
	local town1 = null, town2 = null, pop = min_population, distance = 0;
	//take highest lowest population out of the array keeping distance under maximum provided
	foreach(fut in this.candidate_rail_routes) {//sorted towns, distance, comp
		if(fut[1] <= max_distance && fut[2] >= pop && fut[1] >= min_distance) {
			town1 = fut[0][0]; town2 = fut[0][1]; distance = fut[1]; pop = fut[2];
		}
	}
	if (town1 == null) {this.GetCandidateRailRoutes();return null;}
	local sorted_towns = [town1]; sorted_towns = MailAI.Append(sorted_towns,town2);
	this.candidate_rail_routes = this.RemoveRoute(this.candidate_rail_routes,sorted_towns);
	if(this.ContainsRoute(this.existing_routes,sorted_towns)){return null;}//no existing
	local height_difference = abs(AITile.GetMaxHeight(AITown.GetLocation(town1))-AITile.GetMaxHeight(AITown.GetLocation(town2)));
	local distance = AITile.GetDistanceManhattanToTile(AITown.GetLocation(town1),AITown.GetLocation(town2));
	local engine = EngineMaster.GetEngine(distance, height_difference, false, true, false);
	if(engine == null) {return null;}
	return [sorted_towns,engine,pop];
}

function RoutePlanner::GetNextRoadRoute(min_distance, max_distance, min_population) {
	if(this.candidate_road_routes.len() == 0){this.GetCandidateRoadRoutes();}
	local town1 = null, town2 = null, pop = min_population, distance = 0;
	//take highest lowest population out of the array keeping distance under maximum provided
	foreach(fut in this.candidate_road_routes) {//sorted towns, distance, comp
		if(fut[1] <= max_distance && fut[2] >= pop && fut[1] >= min_distance) {
			town1 = fut[0][0]; town2 = fut[0][1]; distance = fut[1]; pop = fut[2];
		}
	}
	if (town1 == null) {this.GetCandidateRoadRoutes();return null;}
	local sorted_towns = [town1]; sorted_towns = MailAI.Append(sorted_towns,town2);
	this.candidate_road_routes = this.RemoveRoute(this.candidate_road_routes,sorted_towns);
	if(this.ContainsRoute(this.existing_routes,sorted_towns)){return null;}//no existing
	local height_difference = abs(AITile.GetMaxHeight(AITown.GetLocation(town1))-AITile.GetMaxHeight(AITown.GetLocation(town2)));
	local distance = AITile.GetDistanceManhattanToTile(AITown.GetLocation(town1),AITown.GetLocation(town2));
	local engine = EngineMaster.GetEngine(distance, height_difference, true, false, false);
	if(engine == null) {return null;}
	return [sorted_towns,engine,pop];
}

/**
*	Tries to updata all rails on the map to elec rail.
*	@note Elec rail is defined by '1' lacking better coding in AIRail.
*	@return True or False
*/
function RoutePlanner::UpdateRailRoutesToElec() {
	local nul_tile = AIMap.GetTileIndex(1,1); 
	local tl = AITileList();
	tl.AddRectangle(nul_tile, AIMap.GetTileIndex(AIMap.GetMapSizeX()-2,AIMap.GetMapSizeY()-2));
	//MailAI.Info(0,"count of tiles is "+tl.Count());
	local bool;
	foreach(t, _ in tl) {
		bool = EventManager.ConvertRailType(t,t,1);
	}
	return bool;
}

function RoutePlanner::CheckUnfinishedRoutes() {
	//REMEMBER that this will be called without a routeplanner instance so no arrays etc are available!!
	//Could move this method to somewhere else maybe to make the difference clear....
}

/**
*	This goes over the list of vehicles and compares every one to the provided parameters in main.nut
*	deciding wheter to service, sell or replace vehicles. It stops vehicles that need to be send to the
*	depot because we can then send the stopped vehicles in a second loop to the depot. Skipping the stopping
*	part of this code would lead the sendtodepot line to be executed more then once while the vehicle is
*	on it's way to the depot and would cancel the order which offcourse we don't want.
*/
function RoutePlanner::CheckVehicles() {
	local veh_list = AIVehicleList(); if(veh_list.IsEmpty()){return null;}
	foreach(veh, _ in veh_list) {	
//with long loading times the reliability can drop too much so force to service and skip order
		if(AIVehicle.GetReliability(veh) < MailAI.service_reliability
		&& AIVehicle.GetState(veh) == AIVehicle.VS_AT_STATION) {	
			local cur = AIOrder.GetOrderDestination(veh,AIOrder.ORDER_CURRENT);
			local ord = AIOrder.GetOrderDestination(veh,0);
			if(cur == ord){AIOrder.SkipToOrder(veh,1);}
			else {AIOrder.SkipToOrder(veh,0);}
			AIVehicle.SendVehicleToDepotForServicing(veh);
			continue;
		}	
//vehicle not making profit	
		local profit_last = AIVehicle.GetProfitLastYear(veh), profit_this = AIVehicle.GetProfitThisYear(veh);
		if(AIVehicle.GetAge(veh) >= MailAI.days_to_make_profit) {		
			local no_prof = MailAI.min_truck_profit;
			if(AIVehicle.GetVehicleType(veh) == AIVehicle.VT_RAIL) {
				no_prof = AIVehicle.GetNumWagons(veh) * MailAI.min_train_profit;
			}		
			if((profit_last < no_prof || profit_last == 0) && (profit_this < no_prof || profit_this == 0)) {			
				if(!AIOrder.IsGotoDepotOrder(veh,AIOrder.ORDER_CURRENT)){
					MailAI.Info(1,"Vehicle ("+AIVehicle.GetUnitNumber(veh)+") is not making profit. Selling vehicle.");
					AIVehicle.SendVehicleToDepot(veh);
					continue;
				}			
			}
//below here we will replace a vehicle so check if that's possible first
			local dist = AITile.GetDistanceManhattanToTile(AIOrder.GetOrderDestination(veh,0),AIOrder.GetOrderDestination(veh,1));	
			local engine = EngineMaster.GetEngine(dist,null,AIVehicle.GetVehicleType(veh)==AIVehicle.VT_ROAD,
												AIVehicle.GetVehicleType(veh)==AIVehicle.VT_RAIL,true);
			if(engine==null || !BankManager.EnoughMoneyForEngine(engine)) {return;}
//train making too little profit
			if(AIVehicle.GetVehicleType(veh) == AIVehicle.VT_RAIL) {		
				local full_stat, cur_ord = AIOrder.ResolveOrderPosition(veh,AIOrder.ORDER_CURRENT);
				local order_flags = AIOrder.GetOrderFlags(veh,0);
				//currently all trains have exactly one full load and one non full load
				if(order_flags == AIOrder.OF_FULL_LOAD) {
					full_stat = AIOrder.GetOrderDestination(veh,0);
				} else {
					full_stat = AIOrder.GetOrderDestination(veh,1);
				}
				local rating = AIStation.GetCargoRating(AIStation.GetStationID(full_stat),AICargo.CC_MAIL);
				local prof_needed = AIVehicle.GetNumWagons(veh) * MailAI.low_train_profit;
				if(profit_last < prof_needed && profit_this < prof_needed && rating > MailAI.high_station_rating
				&& AIVehicle.GetNumWagons(veh) > MailAI.min_wagons) {	
					if(!AIOrder.IsGotoDepotOrder(veh,AIOrder.ORDER_CURRENT)){
						MailAI.Info(1,"Train ("+AIVehicle.GetUnitNumber(veh)+") is making low profit. Resizing.");
						AIVehicle.SendVehicleToDepot(veh);
						continue;
					}
				}	
			}	
		}	
//Engine too old
		if(AIVehicle.GetAge(veh) > AIVehicle.GetMaxAge(veh) - MailAI.days_before_max_age) {
			if(!AIOrder.IsGotoDepotOrder(veh,AIOrder.ORDER_CURRENT)){
				MailAI.Info(1,"Vehicle ("+AIVehicle.GetUnitNumber(veh)+") is getting old. Renewing.");
				AIVehicle.SendVehicleToDepot(veh);
				continue;
			}			
		}
//Engine unreliable			
		if(AIEngine.GetReliability(AIVehicle.GetEngineType(veh)) < MailAI.min_max_reliability) {	
			if(!AIOrder.IsGotoDepotOrder(veh,AIOrder.ORDER_CURRENT)){
				MailAI.Info(1,"Vehicle ("+AIVehicle.GetUnitNumber(veh)+") getting unreliable. Renewing.");
				AIVehicle.SendVehicleToDepot(veh);
				continue;
			}				
		}
	}
}

function RoutePlanner::RemoveRoute(routes_array,contains_array) {
	local result_array = [];
	foreach(route in routes_array) {
		local check_array = route[0];
		if(check_array.len() != contains_array.len()) {
			result_array.append(route);
			continue;
		}
   		for (local i = 0; i < check_array.len(); i++) {
       		if (check_array[i] != contains_array[i]) {
       			result_array.append(route);
       			break;//only breaks for loop, not the foreach
       			
       		}
    	}
	}	
	return result_array;
}

/**
*	Checks if the FIRST element of the first array is identical to the second array
*/
function RoutePlanner::ContainsRoute(routes_array, contains_array) {
	foreach(route in routes_array) {
		local check_array = route[0];
		if(check_array.len() != contains_array.len()) {continue;}
		local found = true;
   		for (local i = 0; i < check_array.len(); i++) {
       		if (check_array[i] != contains_array[i]) {
       			found=false;
       			break;//only breaks for loop, not the foreach
       		}
    	}
    	if(found){return true;}
	}	
	return false;	
}

/**
*	Checks if all the items of the SECOND element of the first array (should be a list)
*	are in the list provided as a second parameter. The second list can contain more as well.
*/
function RoutePlanner::ContainsStations(routes_array, contains_list) {
	foreach(route in routes_array) {
		local check_list = route[1];
		if(check_list.Count() != contains_list.Count()) {continue;}
		local found = true;
   		foreach(item, _ in check_list) {
       		if(!contains_list.HasItem(item)) {
       			found=false;
       			break;
       		}
    	}
    	return found;
	}	
	return false;	
}
