/* 
 *	Copyright 2013 Varen De Meersman
 *  This file is part of MailAI.
 *
 *  MailAI is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  MailAI is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with MailAI.  If not, see <http://www.gnu.org/licenses/>.
 */  
import("Pathfinder.Road", "RoadPathFinder", 4);
import("Pathfinder.Rail", "RailPathFinder", 1);
require("BankManager.nut");require("EngineMaster.nut");
require("Valuator.nut");require("RoutePlanner.nut");require("StationManager.nut");
require("Route.nut");require("Tile.nut");require("BuildManager.nut");
require("TownManager.nut");require("EventManager.nut");require("RailBuilder.nut");

class MyRoadPF extends RoadPathFinder {}
class MyRailPF extends RailPathFinder {}
	function MyRailPF::_Estimate(cur_tile, cur_direction, goal_tiles, self) {
	   local est = ::RailPathFinder._Estimate(cur_tile, cur_direction, goal_tiles, self);
	   return abs(est * 1.3);
	}
	
/*--------------------------------------------------------------------
|                                                                     |
|    MailAI                                                          |
|                                                                     |
 --------------------------------------------------------------------*/
 class MailAI extends AIController {	
 	inauguration = null;
 	last_station_check = null;
 	last_route_check = null;
 	
 	/**Route Planning**/
 	max_long_route_distance = 300;
 	max_medium_route_distance = 150;//max truck distance if trains not allowed
 	max_short_route_distance = 70;//max truck distance and min train distance
 	min_distance = 25;//minimal distance between towns for a normal route
						//a truck feeder route has no min distance	
 	min_road_population = 500;
 	min_rail_population = 1000;
 	min_long_rail_population = 1500;
 	min_feeder_population = 1500;
	
	/**Engine Related**/
 	min_train_route_capacity = 60;//will make sure all trains have at least this capacity
 	early_mail = 30;//try to deliver mail in 35 days first
	normal_mail = 60;//if no engine fast enough try 45 days
	try_reliability = 77;//always try to stay above this value first
	min_reliability = 73;//absolute minimum
	service_reliability = 67;//if engine drops below while loading, send for service
	min_max_reliability = 70;//if engine max rel. has dropped below this, mark vehicle for upgrade
	days_to_make_profit = 700;//after this check vehicle if it's making profit
	min_truck_profit = -100;
	min_train_profit = 100;//profit per wagon
	low_train_profit = 400;//used to make a train smaller, this is profit per wagon
	high_station_rating = 60;//used to make a train smaller if high station rating and low profit
	min_wagons = 3;//minimum number of wagons a train is allowed to have, counts engine as well??
	days_before_max_age = 365;//how many days before reaching max age to replace vehicle
	
 	/**Variables in TownManager**/
 	min_train_coverage = 12;//Value below 8 means no acceptance; the more the better.
 	min_truck_coverage = 9;	
	
	/**Variables in Valuator**/
	min_train_cargo_waiting = 150;//Used to check if station needs extra service
	max_train_cargo_rating = 70;//Used to check if station needs extra service
	min_truck_cargo_waiting = 70;//Used to check if station needs extra service
	max_truck_cargo_rating = 70;//Used to check if station needs extra service

	/**Values used by pathfinders**/
	max_train_bridge_length = 31;//default in railpathfinder uses only 6
	max_truck_bridge_length = 16;
	bridge_cost = 100;//100 would be like a normal flat tile, default uses 150, encourages bridge buildin
	max_train_go_around = 8;
	max_truck_go_around = 4;
	
	constructor() {}
 }
 
 function MailAI::Start() {
 	if(this.GetMailID()==null) {
		MailAI.Info(2,"No Mail found to transport! Postman unemployed!");
		MailAI.Stop();
	}	
 	if(!this.AllowTrucks()) {
 		MailAI.Info(1,"No trucks are allowed.");
 	} else {
 		AIRoad.SetCurrentRoadType(AIRoad.ROADTYPE_ROAD);	
 	}
 	if(!this.AllowTrains()) {
 		MailAI.Info(1,"No trains are allowed.");
 		AIRail.SetCurrentRailType(1);//to prevent the check for electrified run in the loop
 	} else {
 		if(AIGameSettings.IsValid("forbid_90_deg") && AIGameSettings.GetValue("forbid_90_deg") != 0){
	 		MailAI.Info(1,"Forbid 90 degree turns is on! This may cause some problems for trains in this AI!");
		}
		local types = AIRailTypeList();
		if (types.Count()>1){
			MailAI.Info(0,"Rail type set at 1, electrified");
			AIRail.SetCurrentRailType(1);
		} else {
			MailAI.Info(0,"Rail type set at 0, non-electrified");
			AIRail.SetCurrentRailType(0);	
		}
 	}
	if(!AICompany.SetName("Mail AI")){
		local i = 2;
		while(!AICompany.SetName("Mail AI #" + i)){
			i = i + 1;
			if(i > 255) {break;}
		}
	}
	AICompany.SetPresidentName("Postman, The");
	this.inauguration = AIDate.GetYear(AIDate.GetCurrentDate()); 
	MailAI.Info(1,"---------------------------------");
	MailAI.Info(2,"The MailAI always rings twice!");
	MailAI.Info(1,"---------------------------------");	
	/* Initialize the class that will control everything */
	local route_planner = RoutePlanner(); 		
	/* Start the infinte loop */
	while (true) {
		BankManager.ManageLoan();
		EventManager.Check();
		if(AIRail.GetCurrentRailType()!=1) {
			local types = AIRailTypeList();
			if(types.Count()>1) {
				MailAI.Info(2,"Electrified rail became available!");
				if(route_planner.UpdateRailRoutesToElec()) {
					AIRail.SetCurrentRailType(1);
				}
			}
		}	    
	    local month = AIDate.GetMonth(AIDate.GetCurrentDate());
	   	if(month == 5 || month == 10){route_planner.CheckVehicles();}	
		this.Sleep(100); 			
		route_planner.BuildNewRoute(); //can return null even when a route has been build
	}
 }

 /**
 *	Checks to see if the AI setting and the game setting allow trucks.
 *	Also sees if there are not too many trucks in the game.
 *	@return True if allowed otherwise false.
 */
 function MailAI::AllowTrucks() {
	if(!AIController.GetSetting("use_trucks")) {return false;}
	if(AIGameSettings.IsDisabledVehicleType(AIVehicle.VT_ROAD)) {return false;}
	local veh_list = AIVehicleList();
	veh_list.Valuate(AIVehicle.GetVehicleType);	veh_list.KeepValue(AIVehicle.VT_ROAD);
	if(AIGameSettings.IsValid("vehicle.max_roadveh") &&
		(veh_list.Count() + 4) > AIGameSettings.GetValue("vehicle.max_roadveh")) {return false;}
	return true;
 }
 
 /**
 *	Checks to see if the AI setting and the game setting allow trains.
 *	Also sees if there are not too many trains in the game.
 *	@return True if allowed otherwise false.
 */
 function MailAI::AllowTrains() {
	if(!AIController.GetSetting("use_trains")) {return false;}
	if(AIGameSettings.IsDisabledVehicleType(AIVehicle.VT_RAIL)) {return false;}
	local veh_list = AIVehicleList();
	veh_list.Valuate(AIVehicle.GetVehicleType);	veh_list.KeepValue(AIVehicle.VT_RAIL);
	if(AIGameSettings.IsValid("vehicle.max_trains") &&
		(veh_list.Count() + 2) > AIGameSettings.GetValue("vehicle.max_trains")) {return false;}
	return true;
 }
 
 /**
 *	Checks to see if there are armoured goods and if the AI setting allows it.
 *	@return True if available and allowed otherwise false.
 */
 function MailAI::AllowValuables() {
 	if(MailAI.getCargoID(AICargo.CC_ARMOURED)!=null && AIController.GetSetting("transport_valuables")) {
 		return true;
 	} else {
 		return false;
 	}
 }
 
 /**
 *	Will return the first 'name' in the AICargoList
 *	@param name the AICargo.NAME to be found
 *	@return the first one of the list or null
 */
 function MailAI::GetCargoID(name){
 	local cargo_list = AICargoList();
	cargo_list.Valuate(AICargo.HasCargoClass, name);
	cargo_list.KeepValue(1);
	if(!cargo_list.IsEmpty()){return cargo_list.Begin();}
	else {return null;}	
 }
 
/**
*	Convenience method.
*	@return the first AICargo.CC_MAIL
*/
 function MailAI::GetMailID(){
 	return MailAI.GetCargoID(AICargo.CC_MAIL);
 }
 
 /**
 *	Will show text, warnings and errors in the debug window
 *	when show_debug_lines it also shows level 0 messages
 *	@param level 0 debug, 1 text, 2 warning, 3 error
 *	@param text The text to display.
 */
 function MailAI::Info(level,text) {
 	if(AIController.GetSetting("show_debug") || level >= 1 ) {
 		switch(level) {
 			case 0: AILog.Info(text);break;
 			case 1: AILog.Info(text);break;
			case 2: AILog.Warning(text);break;
			case 3: AILog.Error(text);break;
 		}
 	}
 }
 
 function MailAI::AnnounceRoute(route) {
 	if(route == null) {return null;}
	local speed = AIEngine.GetMaxSpeed(route.GetEngine())/1.6;
	local towns = route.GetSortedTowns();
	if(route.GetTravelDistance()==20) {
		MailAI.Info(2,"Local Delivery System created for "+AITown.GetName(towns[0])+".");
		MailAI.Info(1,"Using "+AIEngine.GetName(route.GetEngine())+ " at max. speed: "+speed);
		MailAI.Info(1,"----------------------------------------------");
		return;
	}	
	local str = "Route created connecting ";
	foreach(town in towns){
		str = str + AITown.GetName(town) + ", ";
	}
	local days_travel = abs(route.GetTravelDistance() / (0.056 * speed));
	MailAI.Info(1,"----------------------------------------------");
	MailAI.Info(2,""+str);
	MailAI.Info(1,"Using "+AIEngine.GetName(route.GetEngine())+ " at max. speed: "+speed);
	MailAI.Info(1,"Distance travelled: "+route.GetTravelDistance()+" in "+days_travel+" days.");
	MailAI.Info(1,"----------------------------------------------");	
 }
 
 /**
 *	If show_debug_lines it will build a sign
 *	@param tile The location to build the sign
 *	@param text The text to be shown on the sign
 */
 function MailAI::BuildSign(tile,text) {
 	if(AIController.GetSetting("show_debug") && tile !=null) {
 		AISign.BuildSign(tile,""+text);
 	}
 }

function MailAI::Stop(){
	MailAI.Info(3,"Stopping MailAI...");
}
 
 /**
 *	It saves a 'table' = {name = value, name = value, ...}
 *	@return A table that will be kept saved to be loaded.
 */
 function MailAI::Save() {
 	local arraytje = [];
	arraytje.append(["el1","el2"]);
	local table = {inauguration_value = this.inauguration, testarr = arraytje};
	return table;
 }
 
 /**
 *	This function is ran before the start() function on load.
 *	@param version Provided by loader
 *	@param data Provided by loader
 */
 function MailAI::Load(version, data) {
	if (data.rawin("inauguration_value")) { //rawin(key) returns true if the slot ‘key’ exists.
		this.inauguration = data.rawget("inauguration_value");//rawget(key) tries to get a value from the slot ‘key’ 
	}
	MailAI.Info(2,"Gladly serving your mail since "+this.inauguration+"!");
	//if (data.rawin("testarr")) AILog.Warning("this is testarray "+data.rawget("testarr"));
	RoutePlanner.CheckUnfinishedRoutes();//load komt voor start dus geen instantie van route_planner!
 }
 
function MailAI::Append(array,number) {
	if(number==null){MailAI.Info(0,"Appending null");return array;}
	local temp = [];
	foreach(n in array) {
		if(n < number) {//append all the values smaller
			temp.append(n);	
		}
		if(n == number) {
			MailAI.Info(0,"Appending same number!");
		}
	}
	temp.append(number);//put new value in
	foreach(n in array) {
		if(n > number) {//append all the values bigger
			temp.append(n);	
		}
	}
	return temp;
}