/**
 *    This file is part of Rondje.
 *
 *    Rondje is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    Rondje is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with Rondje.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright 2008-2009 Marnix Bakker, Willem de Neve, Michiel Konstapel and Otto Visser.
 * Suggestions/comments and bugs can go to: rondjeomdekerk@konstapel.nl
 */

require("RoadFinder/graph/AStar.nut");
require("RoadFinder/RoadFinder.nut");
require("RoadFinder/ConnectedChecker.nut");

const RETRIES = 10;

class BuildException {
	
	msg = null;
	
	constructor(msg) {
		this.msg = msg;
	}
	
	function _tostring() {
		return msg;
	}
}

class PathfindingException extends BuildException {
	constructor(from, to) {
		BuildException.constructor("no path from " + from + " to " + to);
	}
}

class NoRoomForStationException extends BuildException {
	constructor(destination) {
		Debug("no room for station at " + destination.GetName());
		BuildException.constructor("no room for station at " + destination.GetName());
	}
}

class NoRoomForDepotException extends BuildException {
	constructor(destination) {
		Debug("no room for depot at " + destination.GetName());
		BuildException.constructor("no room for depot at " + destination.GetName());
	}
}

class NotEnoughMoneyException extends BuildException {
	constructor() {
		BuildException.constructor("not enough money");
	}
}

class PissedOffAuthorityException extends BuildException {
	constructor(destination) {
		BuildException.constructor("pissed off the local authority in " + destination.GetName());
	}
}

class RoadBuildingException extends BuildException {
	constructor(msg) {
		BuildException.constructor("couldn't build road: " + msg);
	}
}

/**
 * Build a route between two Destinations that are already connected by a road network.
 * @return [pickupStation, pickupDepot, dropoffStation, dropoffDepot]
 * @throw when a complete route could not be built
 */
function BuildRoute(route) {
	MaxLoan();
	local network = route.network;
	local pickup = route.from;
	local dropoff = route.to;
	
	local pickupStation = null;
	local pickupDepot = null;
	local dropoffStation = null;
	local dropoffDepot = null;

	Debug("Building pickup station at " + pickup.GetName());
	pickupStation = BuildStation(pickup, dropoff, ListToArray(pickup.GetPickupArea()), network);

	Debug("Building dropoff station at " + dropoff.GetName());
	dropoffStation = BuildStation(dropoff, pickup, ListToArray(dropoff.GetDropoffArea()), network);

	Debug("Building pickup depot at " + pickup.GetName());
	pickupDepot = BuildDepot(pickup, pickupStation);

	Debug("Building dropoff depot at " + pickup.GetName());
	dropoffDepot = BuildDepot(dropoff, dropoffStation);

	pickup.SetMyStation(pickupStation);
	dropoff.SetMyStation(dropoffStation);

	route.SetBuilt();
	return [pickupStation, pickupDepot, dropoffStation, dropoffDepot];
}

/**
 * Build a station.
 * @param at Destination to build at
 * @param to Destination to connect to
 * @param network Network to connect to
 * @param area Array of tiles to build in
 */
function BuildStation(at, to, area, network) {
	if (area.len() == 0) {
		throw NoRoomForStationException(at);
	}
	
	// find the tile in the network near "at" that is closest to "to"
	// but not in "area", because we need a two-tile route at least to figure out
	// our station front tile
	local tiles = AITileList();
	SafeAddRectangle(tiles, at.GetLocation(), 10);
	tiles.KeepList(network.tiles);
	tiles.RemoveList(ArrayToList(area));
	if (tiles.Count() == 0) {
		throw NoRoomForStationException(at);
	}
	
	local connectTo = FindClosestTile(tiles, to.GetLocation());
	//Sign(connectTo, "connect " + at.GetName());
	
	local result;
	local station = at.GetMyStation();
	if (station) {
		// reuse the station, but ensure it is connected!
		Debug("We already have a station at " + at.GetName());
		local path = FindPath([station.GetLocation()], [connectTo]);
		if (path == null)
			throw PathfindingException(station.GetName(), "tile " + connectTo);
		BuildRoad(path);
		result = station;
	} else {
		// build a road and a new station
		local path = FindPath([connectTo], area);
		if (path == null) {
			throw PathfindingException(at.GetName(), "tile " + connectTo);
		}
		BuildRoad(path);
		result = Build(at, area, path, true, 0);
	}
	
	return result;
}

/**
 * Find the tile in AITileList tiles that is closest to the target tile.
 */
function FindClosestTile(tiles, target) {
	local closest = tiles.Begin();
	for (local tile = tiles.Next(); tiles.HasNext(); tile = tiles.Next()) {
		if (AIMap.DistanceManhattan(tile, target) < AIMap.DistanceManhattan(closest, target)) closest = tile;
	}
	
	return closest;
}

/**
 * Build a depot.
 * @param destination Destination to build at
 * @param station Station to connect to
 */
function BuildDepot(destination, station) {
	local depot = destination.GetMyDepot();
	if (depot) return depot;

	local stationTile = station.GetLocation();
		
	// for passenger routes, we're building in the town, so try and clear a spot for a depot
	//if (destination.GetCargo() == PASSENGERS) ClearForDepot(stationTile);
	
	// find buildable tiles around the station
	local area = AITileList();
	SafeAddRectangle(area, stationTile, DEPOT_RANGE);
	area.Valuate(AITile.IsBuildable);
	area.KeepValue(1);
	
	if (area.Count() == 0) { // no room: make room
		ClearForDepot(stationTile);
		area = AITileList();
		SafeAddRectangle(area, stationTile, DEPOT_RANGE);
		area.Valuate(AITile.IsBuildable);
		area.KeepValue(1);
	}
	
	if (area.Count() == 0)	// failed to make room
		throw PathfindingException("depot at " + destination.GetName(), station.GetName());

	local path = FindPath([stationTile], ListToArray(area));
	if (path == null) {	// try it the destructive way
		ClearForDepot(stationTile);
		path = FindPath([stationTile], ListToArray(area));
	}
	if (path == null) // still no luck
		throw PathfindingException("destructive depot at " + destination.GetName(), station.GetName());
	
	BuildRoad(path);
	return Build(destination, ListToArray(area), path, false, 0);
}

function ClearForDepot(stationTile) {
	local depotTiles = AITileList();
		
	// try to clear the front and back tiles
	depotTiles.AddTile(stationTile + AIMap.GetTileIndex( 1,  0));
	depotTiles.AddTile(stationTile + AIMap.GetTileIndex(-1,  0));
	depotTiles.AddTile(stationTile + AIMap.GetTileIndex( 0,  1));
	depotTiles.AddTile(stationTile + AIMap.GetTileIndex( 0, -1));
	RemoveDriveThroughSideTiles(depotTiles, stationTile);
	
	for (local tile = depotTiles.Begin(); depotTiles.HasNext(); tile = depotTiles.Next()) {
		// don't destroy road tiles
		if (!AIRoad.IsRoadTile(tile)) {
			// stop if we clear a spot
			if (AITile.DemolishTile(tile)) break;
		}
	}
}

/**
 * Remove the two tiles to the side of a drivethrough station
 * from an AITileList.
 */
function RemoveDriveThroughSideTiles(tiles, stationTile) {
	if (!AIRoad.IsDriveThroughRoadStationTile(stationTile)) return;
	if (AIMap.GetTileX(stationTile) == AIMap.GetTileX(AIRoad.GetRoadStationFrontTile(stationTile))) {
		// oriented along the X axis -> remove the X +- 1 tiles
		tiles.RemoveTile(stationTile - AIMap.GetTileIndex(1, 0));
		tiles.RemoveTile(stationTile - AIMap.GetTileIndex(-1, 0));
	} else {
		// oriented along the Y axis -> remove the Y +- 1 tiles
		tiles.RemoveTile(stationTile - AIMap.GetTileIndex(0, 1));
		tiles.RemoveTile(stationTile - AIMap.GetTileIndex(0, -1));
	}
	
	return tiles;
}

/**
 * Build a station or depot.
 * @param destination Destination to build at
 * @param area array of tiles to build in
 * @param path path to connect to
 * @param start whether to connect to the start (true) or end (false) of the path
 * @param station build a station (true) or depot (false);
 * @param retries number of retries we had due to ERR_NONE errors
 */
function Build(destination, area, path, station, retries) {
	local tiles = StartOfPath(path);
	local tile = tiles[0];
	local exit = tiles[1];
	local building = null;
	
	if (station) {
		local type = destination.GetStationType() == AIStation.STATION_TRUCK_STOP ? AIRoad.ROADVEHTYPE_TRUCK : AIRoad.ROADVEHTYPE_BUS;
		if (AIRoad.BuildDriveThroughRoadStation(tile, exit, type, AIStation.STATION_NEW)) {
			// station names and signs are limited to 30 characters
			local name = destination.GetName();
			if (name.len() > 30) name = name.slice(0, 27) + "...";
			local stationID = AIStation.GetStationID(tile);
			AIStation.SetName(stationID, name);
			return Station(stationID, tile, destination);
		}
	} else {
		if (AIRoad.BuildRoadDepot(tile, exit)) {
			Debug("Built depot at " + destination.GetName());
			return Depot(tile, destination);
		}
	}
	
	local error = AIError.GetLastError();
	Debug(AIError.GetLastErrorString());
	local removeTile = false;
	switch (error) {
		case AIError.ERR_VEHICLE_IN_THE_WAY:
			// give it the option to get out of the way
		case AIError.ERR_NONE:
			PrintError();
			// shouldn't happen, but it does
			// just retrying seems to work
			// however, we don't want to end up in an infinite loop due to some freak occurence
			retries++;
			if (retries == RETRIES) {
				retries = 0;
				removeTile = true;
			}
			
			break;

		case AIError.ERR_NOT_ENOUGH_CASH:
			// we'll retry later
			throw NotEnoughMoneyException();

		case AIError.ERR_LOCAL_AUTHORITY_REFUSES:
			// we managed to get a very poor or worse town rating
			// give them trees to hug and/or leave town alone for now
			throw PissedOffAuthorityException(destination);
						
		case AIError.ERR_AREA_NOT_CLEAR:
			// if it is not a road, demolish and retry
			// this also prevents destroying our other stations
			if (!AIRoad.IsRoadTile(path.GetTile()) && AITile.DemolishTile(path.GetTile())) {
				// try again in the same spot
			} else {
				// try elsewhere
				removeTile = true;
			}
			
			break;
		
		default:
			removeTile = true;
			break;
	}
	
	//Sign(tile, AIError.GetLastErrorString());
	if (removeTile) {
		Debug("Area size left: " + area.len());
		// abandon this spot
		area.remove(IndexOf(area, tile));
		if (area.len() == 0)
			throw (station ? NoRoomForStationException(destination) : NoRoomForDepotException(destination));
		// connect to the previous tile
		path = FindPath([tile], area);
		if (path == null)
			throw (station ? NoRoomForStationException(destination) : NoRoomForDepotException(destination));
		BuildRoad(path);
	}
	
	return Build(destination, area, path, station, retries);
}

/**
 * Return the two nodes at a start of a path (building and building exit).
 */
function StartOfPath(path) {
	return [path.GetTile(), path.GetParent().GetTile()];
}

/**
 * Return the two nodes at a end of a path (building and building exit).
 */
function EndOfPath(path) {
	// traverse the path to find the other end
	local p = path.GetParent();
	while (p.GetParent().GetParent() != null) {
		p = p.GetParent();
	}
	
	return [p.GetParent().GetTile(), p.GetTile()];
}

/**
 * Verify that two MapObjects are in fact connected by roads.
 */
function Connected(a, b) {
	local max_cost = 3*AIMap.DistanceManhattan(a.GetLocation(), b.GetLocation());
	local pathfinder = ConnectedChecker(max_cost);
	pathfinder.InitializePath([a.GetLocation()], [b.GetLocation()]);
	
	local path = false;
	while (path == false) {
		path = pathfinder.FindPath(VEHICLE_MANAGEMENT_INTERVAL);
		ManageVehicles();
	}
	
	return path != null;
}

/**
 * Return a RoadPathFinder path, or null if no path was found.
 */
function FindPath(startTiles, endTiles) {
	try {
		//local pathfinder = RoadPathFinder();
		local pathfinder = RoadFinder();
		
		// since we connect to existing road networks manually,
		// don't worry too much about prebuilt roads
		pathfinder.cost.tile = 1;
		pathfinder.cost.max_cost = 15*AIMap.DistanceManhattan(startTiles[0], endTiles[0]);
		pathfinder.cost.no_existing_road = 3;
		pathfinder.cost.turn  = 1;
		pathfinder.cost.slope =	1;
		pathfinder.cost.coast =	20;
		pathfinder.cost.bridge_per_tile = 100;
		pathfinder.cost.tunnel_per_tile = 100;
		pathfinder.InitializePath(startTiles, endTiles);
		
		Debug("Pathfinding...");
		local path = false;
		local tick = GetTick();
		while (path == false) {
			path = pathfinder.FindPath(VEHICLE_MANAGEMENT_INTERVAL);
			ManageVehicles();
			if ((GetTick() - tick) > MAX_PATHFINDING_TICKS) {
				Warning("Pathfinding took more than " + MAX_PATHFINDING_TICKS + " ticks, aborting");
				path = null;
				break;
			}
		}
		
		return path;
	} catch (e) {
		AILog.Error("Path finder: " + e);
		return null;
	}
}

/**
 * Build a road for a pathfinder path.
 */
function BuildRoad(path) {
	Debug("Building road...");
	while (path != null) {
		local par = path.GetParent();
		if (par != null) {
			local count = 0;
			while (!BuildSegment(path, par)) {
				// see if it was an actual error
				if (AIError.GetLastError() == AIError.ERR_ALREADY_BUILT) break;
				
				PrintError();
				count++;
				
				// retry N times
				if (count > RETRIES) {
					throw RoadBuildingException(AIError.GetLastErrorString());
				}
				 
				switch (AIError.GetLastError()) {
					case AIError.ERR_NONE:
						// shouldn't happen, but it does
						// just retrying seems to work
						//Sign(path, "E_N");
						break;
						
					case AIError.ERR_AREA_NOT_CLEAR:
						// demolish and retry
						AITile.DemolishTile(path.GetTile());
						break;
					
					case AIError.ERR_VEHICLE_IN_THE_WAY:
						// wait and retry, backing off longer and longer with each retry
						Sleep(count);
						break;
					
					default:
						// all other errors mean we can't build this path
						throw RoadBuildingException(AIError.GetLastErrorString());
				}
			}
			
			count = 0;
		}
		
		path = par;
	}
}

/**
 * Build a segment of a path and return whether this was successful.
 */
function BuildSegment(path, par) {
	if (AIMap.DistanceManhattan(path.GetTile(), par.GetTile()) == 1 ) {
		// verify the connection after building to catch, for example,
		// trying to build into the side of a drive-through station
		return AIRoad.BuildRoad(path.GetTile(), par.GetTile()) && AIRoad.AreRoadTilesConnected(path.GetTile(), par.GetTile());
	} else {
		/* Build a bridge or tunnel. */
		if (!AIBridge.IsBridgeTile(path.GetTile()) && !AITunnel.IsTunnelTile(path.GetTile())) {
			/* If it was a road tile, demolish it first. Do this to work around expended roadbits. */
			if (AIRoad.IsRoadTile(path.GetTile())) AITile.DemolishTile(path.GetTile());
			if (AITunnel.GetOtherTunnelEnd(path.GetTile()) == par.GetTile()) {
				return AITunnel.BuildTunnel(AIVehicle.VT_ROAD, path.GetTile());
			} else {
				local bridges = AIBridgeList_Length(AIMap.DistanceManhattan(path.GetTile(), par.GetTile()) + 1);
				bridges.Valuate(AIBridge.GetMaxSpeed);
				bridges.Sort(AIAbstractList.SORT_BY_VALUE, false);
				return AIBridge.BuildBridge(AIVehicle.VT_ROAD, bridges.Begin(), path.GetTile(), par.GetTile());
			}
		}
	}
}

function BuildHQ(area) {
	Debug("Building company HQ...");
	for (local tile = area.Begin(); area.HasNext(); tile = area.Next()) {
		if (AICompany.BuildCompanyHQ(tile)) {
			Sign(tile, NAME + " HQ");
			return;
		} else {
			PrintError();
			//Sign(tile, AIError.GetLastErrorString());
		}
	}
		
	Debug("No possible HQ location found");
}

