/**
 *    This file is part of Rondje.
 *
 *    Rondje is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    Rondje is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with Rondje.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright 2008-2009 Marnix Bakker, Willem de Neve, Michiel Konstapel and Otto Visser.
 * Suggestions/comments and bugs can go to: rondjeomdekerk@konstapel.nl
 */

require("util.nut");
require("world.nut");
require("RoadFinder/ConnectedChecker.nut");

const NAME = "Rondje Om De Kerk";
const DAYS_PER_YEAR = 365;
const MIN_DISTANCE = 10;
const MAX_DISTANCE = 256;
const NEW_ROUTE_MONEY_REQUIRED = 10000;	// TODO still tweaking this (not changing this before the challenge)
const MAX_DROPOFF_VEHICLES = 20;		// 10-20 is about right, don't go too high
const VEHICLE_MANAGEMENT_INTERVAL = 15;	// ticks
const MAP_INTERVAL = 1000;				// turns
const DEPOT_RANGE = 15;		// range in which to build a depot for a station

const NUM_ROUTES_TO_STOP_MAPPING = 500;		// when more than this many routes are found, stop mapping
const NUM_ROUTES_TO_STOP_EXPLORING = 200;	// when more than this many routes have a confirmed profit, stop trying unknown routes

// when we have more money than this, send a vehicle instead of using the pathfinder
const MAX_PATHFINDER_MONEY = 200000;

const MONTHLY_STATION_MAINTENANCE = 50;	// station upkeep per month
// TODO: make the safety margin dynamic; ie: dependent on amount of real estate and vehicles
const SAFETY_MARGIN = 2500;	// extra money to keep around for bankrupcy checks

const MAX_PATHFINDING_TICKS = 500;

enum Districts {
	CENTER
	NORTH
	SOUTH
	WEST
	EAST
};

enum RouteBuildingResult {
	ROUTE_TOO_SHORT
	ROUTE_TOO_LONG
	ROUTE_NO_PROFIT
	ROUTE_TOO_SOON
	ROUTE_NOT_ENOUGH_CASH
	ROUTE_LONGER_THAN_EXPECTED
	ROUTE_BUILDING_FAILED
	ROUTE_BUILT
};

enum VehicleBuildingResult {
	NOT_ENOUGH_CASH
	NOT_ENOUGH_TIME
	ROUTE_JAMMED
	DESTINATION_TOO_CROWDED
	PICKUP_OCCUPIED
	CARGO_NOT_ACCEPTED
	NO_VEHICLES_BUILT
	VEHICLES_BUILT
};
			
class Rondje extends AIController {

	TESTING_vehiclesBuilt = 0;
	TESTING_vehiclesSent = 0;
	TESTING_vehiclesSold = 0;
	
	world = null;
	turn = 0;
	//startEndgame = 0;
	//endgameCheck = 0;		// variable which will replace startEndgame
	//endgameDate = 0;
	lastUpdate = 0;			// tick at which we last did a ManageVehicles
	//endgameTiles = null;	// tiles on which we can build stations during endgame
	//cheapTilesCount = 0;
	averageRouteValue = 0;
	
	vehiclesAtStartEndgame = 0; // for determining the percentage of vehicles left before endgame.
	
	routeList = [];	// the unsorted list of routes generated during mapping
	routeIndex = null;	// a sorted AIList that indexes into routeList
	currentRoute = 0;	// index into routeIndex
	
	explore = true;		// try out new routes
	map = true;			// periodically map the world
	forceRemap = true;	// a flag indicating we should map the world again on the next turn
	sort = false;		// a flag indicating we should create a new routeIndex
	report = false;		// a flag indicating we should print out routes as we process them
	analyze = false;	// re-analyze the route list
	turnsWithoutBuilding = 0;	// number of times we went through the route processing loop without building a new route
	buildNewRouteNextTurn = true;	// allow or prohibit building a new route on the next turn
	HQArea = null;		// area options for HQ
	prebuildRoutes = null;
		
	function constructor() {
		world = World();
		prebuildRoutes = AIList();

		InitEngines();
		local cargoList = AICargoList();
		for (local cargo = cargoList.Begin(); cargoList.HasNext(); cargo = cargoList.Next()) {
			GetBestEngine(cargo);
		}
		
		::PASSENGERS <- GetPassengerCargoID();
		::INSTANCE <- this;
		::BUS_STATION_RADIUS <- AIStation.GetCoverageRadius(AIStation.STATION_BUS_STOP);
		::TRUCK_STATION_RADIUS <- AIStation.GetCoverageRadius(AIStation.STATION_TRUCK_STOP);
		::MAX_VEHICLES <- AIGameSettings.IsValid("max_roadveh") ?  AIGameSettings.GetValue("max_roadveh") : 500;
		// FindEndgameTiles();
		
		// Find biggest town for HQ
		local towns = AITownList();
		towns.Valuate(AITown.GetPopulation);
		towns.Sort(AIAbstractList.SORT_BY_VALUE, false);
		local town = towns.Begin();
		
		// Find empty 2x2 square as close to town centre as possible
		local maxRange = Sqrt(AITown.GetPopulation(town)/100) + 5; //TODO check value correctness
		HQArea = AITileList();
		
		HQArea.AddRectangle(AITown.GetLocation(town) - AIMap.GetTileIndex(maxRange, maxRange), AITown.GetLocation(town) + AIMap.GetTileIndex(maxRange, maxRange));
		HQArea.Valuate(AITile.IsBuildableRectangle, 2, 2);
		HQArea.KeepValue(1);
		HQArea.Valuate(AIMap.DistanceManhattan, AITown.GetLocation(town));
		HQArea.Sort(AIList.SORT_BY_VALUE, true);
		
		AIRoad.SetCurrentRoadType(AIRoad.ROADTYPE_ROAD);
	}		

	function Start() {
		NameCompany();

		::START_DATE <- AIDate.GetCurrentDate();
		::TICKS_PER_DAY <- TicksPerDay();
		Debug("Max vehicles: " + MAX_VEHICLES);

		local year = AIDate.GetYear(START_DATE);

		// connected map scanning
		local townlist = AITownList();
		townlist.Valuate(AITown.GetPopulation);
		townlist.Sort(AIList.SORT_BY_VALUE, false);
		local townlist2 = AITownList();
		local maxTowns = 10;
		for (local startTown = townlist.Begin(); townlist.HasNext() && maxTowns--; startTown = townlist.Next()) {
			for (local endTown = townlist2.Begin(); townlist2.HasNext(); endTown = townlist2.Next()) {
				if (startTown == endTown || AIMap.DistanceManhattan(AITown.GetLocation(startTown), AITown.GetLocation(endTown)) > 150)
					continue;

				local max_cost = 3*AIMap.DistanceManhattan(AITown.GetLocation(startTown), AITown.GetLocation(endTown));
				local pathfinder = ConnectedChecker(max_cost);
				//pathfinder.cost.max_cost = 2*(AIMap.GetMapSizeX() + AIMap.GetMapSizeY());
				pathfinder.InitializePath([AITown.GetLocation(startTown)], [AITown.GetLocation(endTown)]);

				local path = false;
				while (path == false) {
					path = pathfinder.FindPath(-1);
				}
				if (path != null) {
					Debug("W00tNess!!! there is pre-connected towns: " + AITown.GetName(startTown) + " <--> " + AITown.GetName(endTown));
					// maxTowns = 10;	// either we started later or there's prebuild stuff: do more searching
					prebuildRoutes.AddItem(startTown, AITown.GetPopulation(startTown));
					prebuildRoutes.AddItem(endTown, AITown.GetPopulation(endTown));
				}
			}
		}

		BuildHQ(HQArea);

		// endgameDate = AIDate.GetDate(year + 10, 1, 1);//may need to become + 9, 12, 31 or + 9, 12, 30... uncertain about that.
		// endgameCheck = AIDate.GetDate(year + 8, 2, 1);//first check in february 2006.
		// startEndgame = AIDate.GetDate(year + 9, 12, 1);//always start if there is less than a month to go.

		// local eday = AIDate.GetDayOfMonth(endgameCheck);
		// local emonth = AIDate.GetMonth(endgameCheck);
		// local eyear = AIDate.GetYear(endgameCheck);

		// Debug("End of game at: " + AIDate.GetDayOfMonth(endgameDate)  + "-"+AIDate.GetMonth(endgameDate)  + "-"+AIDate.GetYear(endgameDate));
		// Debug("First endgamecheck will be at date: " + eday + "-" + emonth + "-" + eyear + ".");

		// Debug("Total number of stations that can be prebuild: " + prebuildRoutes.Count());
		for (local station = prebuildRoutes.Begin(); prebuildRoutes.HasNext(); station = prebuildRoutes.Next()) {
			buildPassengerStation(station);
		}

		ManageLoan();

		Debug("Started, scanning...");
		while (true) {
			local bigTick = GetTick();
			TESTING_vehiclesSent = 0;
			TESTING_vehiclesSold = 0;
			TESTING_vehiclesBuilt = 0;
			try {
				/* if (AIDate.GetCurrentDate() >= endgameCheck) {
				   local nextCheck = CheckEndgame(endgameDate);
				// Debug("endgame check returned: " + nextCheck);
				switch (nextCheck) {
				case 0:
				startEndgame = AIDate.GetCurrentDate();
				vehiclesAtStartEndgame = AIVehicleList().Count(); // for determining the percentage of vehicles left before endgame.
				break;
				case 1:
				endgameCheck += AIDate.GetDate(0, 1, 2);
				break;
				case 2:
				endgameCheck += AIDate.GetDate(0, 1, 3);
				break;
				case 3:
				endgameCheck += AIDate.GetDate(0, 1, 5);
				break;
				case 4:
				endgameCheck += AIDate.GetDate(0, 1, 11);
				break;
				case 5:
				endgameCheck += AIDate.GetDate(0, 1, 21);
				break;
				case 6:
				endgameCheck += AIDate.GetDate(0, 2, 1);
				break;
				case 7:
				endgameCheck += AIDate.GetDate(0, 2, 15);
				break;
				case 8:
				endgameCheck += AIDate.GetDate(0, 3, 1);
				break;
				}
				eday = AIDate.GetDayOfMonth(endgameCheck);
				emonth = AIDate.GetMonth(endgameCheck);
				eyear = AIDate.GetYear(endgameCheck);
				Debug("next endgamecheck will be at date: " + eday + "-" + emonth + "-" + eyear + ".");
				} */

				// endgame disabled; uncomment to enable
				// if (AIDate.GetCurrentDate() < startEndgame) {
				local tick;

				CheckEvents();
				ManageVehicles();
				ManageLoan();

				if (map && (forceRemap || turn % MAP_INTERVAL == 0)) {
					MapWorld();
				}

				if (sort) {
					sort = false;

					tick = GetTick();
					routeIndex = AIList();
					foreach (i, route in routeList) {
						if (!explore && !route.IsValueConfirmed()) continue;
						routeIndex.AddItem(i, route.GetRouteValue().tointeger());
					}

					routeIndex.Sort(AIList.SORT_BY_VALUE, false);
					report = true;

					tick = GetTick() - tick;
					if (tick > 20) Warning("Sorting took " + tick + " ticks");
				}

				if (analyze) {
					tick = GetTick();
					AnalyzeRouteList();
					tick = GetTick() - tick;
					if (tick > 20) Warning("Analyzing routes took " + tick + " ticks");
				}

				tick = GetTick();
				local t;
				local manageVehiclesTime = 0;
				local buildVehiclesTime = 0;
				local buildRoutesTime = 0;
				local processedPickups = {};

				local buildNewRoutes = buildNewRouteNextTurn && explore;
				local builtRoute = false;
				buildNewRouteNextTurn = true;

				MaxLoan();
				for (local i = routeIndex.Begin(); routeIndex.HasNext(); i = routeIndex.Next()) {
					/*if (AIDate.GetCurrentDate() > endgameCheck) {
					// see if it's endgame time
					break;
					}*/

					if (GetAvailableMoney() <= GetMinimumSafeMoney()) {
						// max out our loan to provide the cash for station maintenance
						// and stop spending so we don't get declared bankrupt
						FullyMaxLoan();
						break;
					}

					local route = routeList[i];

					// routes can disappear after mapping, since industries can close down
					if (!route.IsValid()) {
						continue;
					}

					// don't consider routes that don't make a profit
					if (route.GetRouteValue() <= 0) {
						// everything else will be even worse
						break;
					}

					if (route.IsBuilt()) {
						// if we have sent vehicles from A to B, don't bother considering A to C, D, E...
						if (route.GetPickup() in processedPickups) {
							continue;
						}

						t = GetTick();

						local result = AddVehicles(route);
						if (result == VehicleBuildingResult.NOT_ENOUGH_CASH) {
							// no money to spare
							// don't try to build a new route next turn
							buildNewRouteNextTurn = false;

							// don't try to build vehicles on routes that are worse
							break;
						} else if (result == VehicleBuildingResult.VEHICLES_BUILT) {
							// still room left on current routes
							// don't try to build a new route next turn
							buildNewRouteNextTurn = false;

							processedPickups[route.GetPickup()] <- true;
						} else {
							// don't care, next!
						}
						buildVehiclesTime += GetTick() - t;
					}
					else if (buildNewRoutes && route.IsBuildable()) {
						t = GetTick();
						local result = ConsiderBuildingRoute(route);
						if (result == RouteBuildingResult.ROUTE_NOT_ENOUGH_CASH) {
							// stop building routes this time around, because we don't want to spend our money
							// on worse routes, but keep building vehicles or we might run out of money
							buildNewRoutes = false;
						} else if (result == RouteBuildingResult.ROUTE_BUILT || result == RouteBuildingResult.ROUTE_LONGER_THAN_EXPECTED) {
							// only try to build one route per loop, because building one route may connect others
							// (especially towns) and we don't want unnecessary pathfinding
							// and we need to keep pumping out vehicles
							buildNewRoutes = false;
							analyze = true;

							if (result == RouteBuildingResult.ROUTE_BUILT) {
								builtRoute = true;
								processedPickups[route.GetPickup()] <- true;
							}
						} else if (result == RouteBuildingResult.ROUTE_NO_PROFIT) {
							// the rest is going to be even worse, so don't bother
							break;
						} else {
							// too short, too long, no profit, couldn't build: continue with the next route
						}
						buildRoutesTime += GetTick() - t;
					}

					t = GetTick();
					ManageVehicles();
					manageVehiclesTime += GetTick() - t;
				}

				tick = GetTick() - tick;
				if (tick > 20) {
					Warning("Processing routes took " + tick + " ticks, of which " +
							manageVehiclesTime + " for managing vehicles " +
							"(" + TESTING_vehiclesSent + " sent, " + TESTING_vehiclesSold + " sold), " +
							buildVehiclesTime + " for building " + TESTING_vehiclesBuilt + " vehicles, " +
							buildRoutesTime + " for building routes");
				}

				// keep track of whether we've built anything
				// if not, we may need to search for new routes
				if (buildNewRoutes && !builtRoute) {
					turnsWithoutBuilding++;
				} else {
					turnsWithoutBuilding = 0;
				}

				// TODO: a better critirion? (not changing this before the game)
				// map and search for new routes if we have money left and didn't build a route on the last N passes
				if (map && turnsWithoutBuilding > 20 && GetAvailableMoney() > 2*NEW_ROUTE_MONEY_REQUIRED) {
					// we need more routes!
					turnsWithoutBuilding = 0;
					forceRemap = true;
				}
				// } else {
				// 	BuildStations();
				// }

				ManageVehicles();
				ManageLoan();

				turn++;
				if (bigTick == GetTick()) {
					Sleep(1);
				}
			} catch (e) {
				AILog.Error("Exception caught:");
				AILog.Error(e);
			}

			bigTick = GetTick() - bigTick;
			if (bigTick > 50) Warning("Main loop took " + bigTick + " ticks");
		}
	}
	
	function getRoadTile(t) {
		local adjacent = AITileList();
		adjacent.AddTile(t - AIMap.GetTileIndex(1,0));
		adjacent.AddTile(t - AIMap.GetTileIndex(0,1));
		adjacent.AddTile(t - AIMap.GetTileIndex(-1,0));
		adjacent.AddTile(t - AIMap.GetTileIndex(0,-1));
		adjacent.Valuate(AIRoad.IsRoadTile);
		adjacent.KeepValue(1);
		adjacent.Valuate(AIRoad.IsRoadDepotTile);
		adjacent.KeepValue(0);
		adjacent.Valuate(AIRoad.IsRoadStationTile);
		adjacent.KeepValue(0);
		if (adjacent.Count())
			return adjacent.Begin();
		else
			return null;
	}
	
	function buildPassengerStation(town) {
		Debug("Building station in " + AITown.GetName(town));
		// Find empty square as close to town centre as possible
		local spotFound = false;
		local curRange = 1;
		local maxRange = Sqrt(AITown.GetPopulation(town)/100) + 2;
		local area = AITileList();
		
		while (curRange < maxRange) {
			//AILog.Info("looking with range " + curRange);
			area.AddRectangle(AITown.GetLocation(town) - AIMap.GetTileIndex(curRange, curRange), AITown.GetLocation(town) + AIMap.GetTileIndex(curRange, curRange));
			area.Valuate(AIRoad.IsRoadTile);
			area.KeepValue(1);
			area.Valuate(AIRoad.IsDriveThroughRoadStationTile);
			area.KeepValue(0);
			area.Valuate(AIRoad.GetNeighbourRoadCount);
			area.KeepValue(2);	// entrance and exit; allow 1 as well?
			if (area.Count()) {
				//AILog.Info("Found " + area.Count() + " possible options");
				for (local t = area.Begin(); area.HasNext(); t = area.Next()) {
					local opening = getRoadTile(t);
					if(opening) {
						// AILog.Info("tile " + t + " has opening " + opening);
						if (/*AIRoad.BuildRoad(opening, t) &&*/ AIRoad.BuildDriveThroughRoadStation(t, opening, AIRoad.ROADVEHTYPE_BUS, AIStation.STATION_JOIN_ADJACENT)) {
							//AISign.BuildSign(t, ("bus " + AITown.GetName(town)));
							return t;
						}
						if(AIError.GetLastError() == AIRoad.ERR_ROAD_CANNOT_BUILD_ON_TOWN_ROAD)
							return buildNonInlinePassengerStation(town);
						AILog.Info("Error building passenger station " + AIError.GetLastErrorString());
						//AISign.BuildSign(t, AIError.GetLastErrorString());

					} //else {
						// AILog.Info("tile" + t + " has no opening");
					//}
				}
				curRange++;
			} else {
				curRange++;
			}
		}
		AILog.Info("No possible station location found");
		return null;
	}
	
	function buildNonInlinePassengerStation(town) {
		Debug("Building non-inline station as it's not allowed to build on town roads...");
		// Find empty square as close to town centre as possible
		local spotFound = false;
		local curRange = 1;
		local maxRange = Sqrt(AITown.GetPopulation(town)/100) + 4; // search larger area
		local area = AITileList();

		while (curRange < maxRange) {
			//Debug("looking with range " + curRange);
			area.AddRectangle(AITown.GetLocation(town) - AIMap.GetTileIndex(curRange, curRange), AITown.GetLocation(town) + AIMap.GetTileIndex(curRange, curRange));
			area.Valuate(AITile.IsBuildable);
			area.KeepValue(1);
			if (area.Count()) {
				//Debug("Found " + area.Count() + " possible options");
				for (local t = area.Begin(); area.HasNext(); t = area.Next()) {
					local opening = getRoadTile(t);
					if(opening) {
						// Debug("tile " + t + " has opening " + opening);
						if (AIRoad.BuildRoad(opening, t) && AIRoad.BuildRoadStation(t, opening, AIRoad.ROADVEHTYPE_BUS, AIStation.STATION_JOIN_ADJACENT)) {
							//AISign.BuildSign(t, ("bus " + AITown.GetName(town)));
							return t;
						}
						Error("Error building non-inline passenger station " + AIError.GetLastErrorString());
						//AISign.BuildSign(t, "" + AIRoad.GetNeighbourRoadCount(t));
						// Can this actually happen or is this bad copy paste work??
						if(AIError.GetLastError() == AIRoad.ERR_ROAD_CANNOT_BUILD_ON_TOWN_ROAD)
							return buildNonInlinePassengerStation(town);

					} //else {
						// Debug("tile" + t + " has no opening");
					//}
				}
				curRange++;
			} else {
				curRange++;
			}
		}
		Debug("No possible station location found");
		return null;
	}

	function MapWorld() {
		Debug("Mapping the world...");
		forceRemap = false;
		sort = true;
		analyze = true;
		
		local tick = GetTick();
		world.Map();
		routeList = world.GetRouteList();
		
		tick = GetTick() - tick;
		if (tick > 20) Warning("Mapping took " + tick + " ticks");
	}
	
	function TileValuator(tile) {
		local value = 247;
		
		if (AITile.HasTreeOnTile(tile))
			value += 66;	// can't determine how many trees there are, so assume max
		if (AITile.IsFarmTile(tile))
			value += 540;
		if (AITile.IsRockTile(tile))
			value += 203;
		if (AITile.IsRoughTile(tile))
			value += 23;
		if (AITile.IsSnowTile(tile))
			value += 23;
		if (AITile.IsDesertTile(tile))
			value += 23;
		if (AITile.IsCoastTile(tile))
			value += 123;
		if (AITile.GetSlope(tile))
			value += 281;	// NOTE: needs finding correct orientation, or loop with 4 orientations if fails
		
		if (value == 247)
			INSTANCE.cheapTilesCount++;
		
		return value;
	}
	
	/**
	 * Process game events.
	 */
	function CheckEvents() {
		while (AIEventController.IsEventWaiting()) {
  			local e = AIEventController.GetNextEvent();
  			local converted;
  			switch (e.GetEventType()) {
  				case AIEvent.AI_ET_VEHICLE_UNPROFITABLE:
  					converted = AIEventVehicleUnprofitable.Convert(e);
  					Warning("Vehicle unprofitable: " + AIVehicle.GetName(converted.GetVehicleID()));
  					local vehicle = Vehicle.GetVehicle(converted.GetVehicleID());
  					if (!vehicle) break;
  					
  					// VerifyRoutes() can take a VERY long time, so don't do this for potentially
  					// many vehicles! Just send them to a depot.
  					vehicle.GoToDepot();
  					
  					//local route = vehicle.GetRoute();
  					//if (route && route.IsConnected()) {
  					//	// seems to be OK? maybe just a long route or a jam
  					//} else {
  					//	// route is broken, VerifyRoutes to send others to depot
  					//	VerifyRoutes();
  					//}
  					break;
  					
				case AIEvent.AI_ET_VEHICLE_LOST:
					converted = AIEventVehicleLost.Convert(e);
					Warning("Vehicle lost: " + AIVehicle.GetName(converted.GetVehicleID()) + ", sending to depot");
					local vehicle = Vehicle.GetVehicle(converted.GetVehicleID());
  					if (vehicle) vehicle.GoToDepot();
  					break;

				case AIEvent.AI_ET_COMPANY_IN_TROUBLE:
					converted = AIEventCompanyInTrouble.Convert(e);
					if (converted.GetCompanyID() == AICompany.COMPANY_SELF) {
						Warning("!!!   We're in money trouble   !!!");
						// TODO: feed this to getminimumsafemoney function, so we
						// can be extra extra careful for a while.
					}

					break;
  					
				case AIEvent.AI_ET_COMPANY_BANKRUPT:
					// we may have been using their tunnels/bridges, which are now gone
					// if we're not too far into the game, verify
					// else, just let it slide
					Warning("A company went bankrupt; checking for side effects...");
					//if (startEndgame - AIDate.GetCurrentDate() < 2*365) {
					VerifyRoutes();
					//} else {
					//	Warning("Too late to verify routes");
					//}
					
					break;
				
				case AIEvent.AI_ET_ENGINE_AVAILABLE:
					InitEngines();
					break;
				
      			default:
      				// Debug("Unhandled event:" + e);
  			}
		}
		
		ManageLoan();
	}
	
	function AnalyzeRouteList() {
		local count = 0;
		local profitable = 0;
		local built = 0;
		local confirmed = 0;
		local confirmedProfitable = 0;
		local serviced = 0;
		local totalRouteValue = 0;
		
		for (local i = routeIndex.Begin(); routeIndex.HasNext(); i = routeIndex.Next()) {
			local route = routeList[i];
			count++;
			
			//if (report) {
				if (count < 26) Debug(count + ": " + route);
				if (count == 26) Debug("... and " + (routeList.len() - count + 1) + " more");
			//}
			
			if (route.GetRouteValue() > 0) profitable++;
			if (route.IsValueConfirmed()) confirmed++;
			if (route.GetRouteValue() > 0 && route.IsValueConfirmed()) confirmedProfitable++;
			if (route.IsBuilt()) built++;
			if (route.GetNumberOfVehicles() > 0) serviced++;
			
			if (route.GetRouteValue() > 0) {
				// ignore unprofitable routes
				totalRouteValue += route.GetRouteValue();
			}
		}
		
		averageRouteValue = (profitable == 0) ? 0 : totalRouteValue / profitable;
		
		Debug("Total/profitable/built/confirmed/conf.profitable/serviced number of routes:  " +
			routeList.len() + "  /  " + profitable + "  /  " + built + "  /  " + confirmed + "  /  " + confirmedProfitable + "  /  " + serviced);
		Debug("Average profitable route value: " + averageRouteValue + " G/d");
		
		if (count > NUM_ROUTES_TO_STOP_MAPPING) {
			Warning(count + " routes found, no longer mapping");
			map = false;
		}
		
		if (confirmedProfitable > NUM_ROUTES_TO_STOP_EXPLORING) {
			Warning(confirmed + " confirmed profitable routes, no longer exploring");
			explore = false;
		}
		
		report = false;
		analyze = false;
	}
	
	/**
	 * Called during pauses in mapping and path finding, so we can send vehicles to depots and build new vehicles.
	 */
	function ManageVehicles() {
		if (!AIVehicleList().IsEmpty()) {
			if (GetTick() > lastUpdate + VEHICLE_MANAGEMENT_INTERVAL) {
				lastUpdate = GetTick();
				SendVehiclesToDepot();
				SellVehiclesInDepot();
			}
		}
	}
	
	function VerifyRoutes() {
		Debug("Verifying routes...");
		local tick = GetTick();
		
		MapWorld();
		local routeNames = {};
		foreach (route in routeList) {
			routeNames[route.GetName()] <- true;
		}
		
		local vehicleIDs = AIVehicleList();
		for (local vehicleID = vehicleIDs.Begin(); vehicleIDs.HasNext(); vehicleID = vehicleIDs.Next()) {
			try {
				local vehicle = Vehicle.GetVehicle(vehicleID);
				if (vehicle) {	// might have arrived in depot already, so check for NULL
					if (vehicle.GetRoute().GetName() in routeNames) {
						// OK
					} else {
						Warning(vehicle.GetName() + " has a broken route, sending to depot");
						vehicle.GoToDepot();
					}
				}
			} catch (e) {}
		}
		
		Debug("Verifying routes took " + (GetTick() - tick) + " ticks");
	}
	
	function SendVehiclesToDepot() {
		// process vehicles currently at their dropoff station
		// Not working anymore :-(
	/*	local vehicleIDs = AIVehicleList();
		vehicleIDs.Valuate(AIVehicle.GetState);
		vehicleIDs.KeepValue(AIVehicle.VS_AT_STATION);
		vehicleIDs.Valuate(AIOrder.ResolveOrderPosition, AIOrder.ORDER_CURRENT);
		vehicleIDs.KeepValue(1);	// pickup -> dropoff -> depot
		
		for (local vehicleID = vehicleIDs.Begin(); vehicleIDs.HasNext(); vehicleID = vehicleIDs.Next()) {
			local vehicle = Vehicle.GetVehicle(vehicleID);
			if (!vehicle.IsHeadingForDepot() && !vehicle.IsHinUndWieder() && vehicle.IsUnloading() && !vehicle.IsFull()) {
				if (vehicle.GoToDepot()) {
					TESTING_vehiclesSent++;
				}
			}
		}*/
		
		// process vehicles heading for their dropoff depot
		local vehicleIDs = AIVehicleList();
		vehicleIDs.Valuate(AIOrder.ResolveOrderPosition, AIOrder.ORDER_CURRENT);
		vehicleIDs.KeepValue(2);	// pickup -> dropoff -> depot
		for (local vehicleID = vehicleIDs.Begin(); vehicleIDs.HasNext(); vehicleID = vehicleIDs.Next()) {
			local vehicle = Vehicle.GetVehicle(vehicleID);
			if (!vehicle.IsHeadingForDepot() && !vehicle.IsHinUndWieder() && vehicle.GoToDepot()) {
				TESTING_vehiclesSent++;
			}
		}
	}
	
	function SellVehiclesInDepot() {
		local vehicleIDs = AIVehicleList();
		vehicleIDs.Valuate(AIVehicle.GetState);
		vehicleIDs.KeepValue(AIVehicle.VS_IN_DEPOT);
		if (vehicleIDs.Count() > 0) {
			for (local vehicleID = vehicleIDs.Begin(); vehicleIDs.HasNext(); vehicleID = vehicleIDs.Next()) {
				local vehicle = Vehicle.GetVehicle(vehicleID);
				local route = vehicle.GetRoute();
				local oldValue = route.GetRouteValue();
				vehicle.Sell();
				TESTING_vehiclesSold++;
				local newValue = route.GetRouteValue();
				
				local ratio = newValue.tofloat()/oldValue;
				if (ratio > 2 || ratio < 0.5) {
					Debug(route.GetName() + " changed value from " + oldValue + " to " + newValue + ", going to re-sort routes");
					sort = true;
				}
			}
		}
	}
	
	/**
	 * See if we should build more vehicles for this route.
	 */
	function AddVehicles(route) {
		// see if we can pay for it
		local engine = GetBestEngine(route.GetCargo());
		if (AIEngine.GetPrice(engine) > GetAvailableMoney()) return VehicleBuildingResult.NOT_ENOUGH_CASH;
		
		// see if the dropoff station isn't too crowded
		local numDropoffVehicles = AIVehicleList_Station(route.GetDropoffStation().stationID).Count();
		if (numDropoffVehicles >= MAX_DROPOFF_VEHICLES)
			return VehicleBuildingResult.DESTINATION_TOO_CROWDED;
		
		// don't build more vehicles on jammed routes
		if (route.IsJammedOrBroken())
			return VehicleBuildingResult.ROUTE_JAMMED;

		// see if the vehicle still has time to make the trip before the end of the game
		// local daysRemaining = startEndgame - AIDate.GetCurrentDate();
		// if (route.GetTripTime() > daysRemaining)
		//	return VehicleBuildingResult.NOT_ENOUGH_TIME;
		
		// see if the station still accepts this cargo
		if(!route.GetDropoffStation().AcceptsCargo(route.GetCargo()))
			return VehicleBuildingResult.CARGO_NOT_ACCEPTED;
		
		// see if the pickup station is occupied
		if (route.GetPickupStation().IsOccupied())
			return VehicleBuildingResult.PICKUP_OCCUPIED;
		
		// calculate the number of vehicles to add
		local capacity = AIEngine.GetCapacity(engine); 
		local numNeeded = max(1, route.GetCargoWaiting() / capacity);
		local toBuild = min(MAX_DROPOFF_VEHICLES - numDropoffVehicles, numNeeded);
		
		local added;
		for (added = 0; added < toBuild; added++) {
			local vehicle = route.AddVehicle();
			if (!vehicle) break;
			
			// add a non-teleporting vehicle if:
			// - we have lots of money
			// - not too many vehicles,
			// - room to spare at the dropoff
			// but only on our worse-than-average routes (keep teleporting on the high profit ones)
			if (added == 0 &&
				GetAvailableMoney() > 500000 &&
				!route.IsFreight() &&
				AIVehicleList().Count() < MAX_VEHICLES - 50 &&
				numDropoffVehicles < 0.75*MAX_DROPOFF_VEHICLES &&
				route.GetRouteValue() < averageRouteValue) {
				Debug("Added non-teleporter to " + route.GetName());
				vehicle.SetHinUndWieder(true);
			}
			
			// it takes approximately 3 days to load a vehicle and get it out of the station,
			// so don't penalize vehicles for waiting in line
			// (which would lower the route value estimate when they arrive)
			vehicle.SetAgeOffset(added * 3);
		}
		
		TESTING_vehiclesBuilt += added;
		//if (added > 0) Debug("Added " + added + " vehicles to " + route);
		return added > 0 ? VehicleBuildingResult.VEHICLES_BUILT : VehicleBuildingResult.NO_VEHICLES_BUILT;
	}
	
	/**
	 * If a route looks interesting, build stations, depots and vehicles for it.
	 * @return true if it was built, false otherwise
	 */
	function ConsiderBuildingRoute(route) {
		if (route.GetDistance() < MIN_DISTANCE) {
			return RouteBuildingResult.ROUTE_TOO_SHORT;
		} else if (route.GetDistance() > MAX_DISTANCE) {
			return RouteBuildingResult.ROUTE_TOO_LONG;
		} else if (route.GetRouteValue() < 0) {
			return RouteBuildingResult.ROUTE_NO_PROFIT;
		} else if (GetAvailableMoney() < NEW_ROUTE_MONEY_REQUIRED + AIEngine.GetPrice(GetBestEngine(route.GetCargo()))) {
			return RouteBuildingResult.ROUTE_NOT_ENOUGH_CASH;
		} else if (IsFirstYear() && IsSuburb(route)) {
			return RouteBuildingResult.ROUTE_TOO_SOON;
		} else {				
			try {
				local tick = GetTick();
				// if we have lots of routes and lots of money, don't waste time with the pathfinder
				if (GetAvailableMoney() < MAX_PATHFINDER_MONEY) {
					// either we're short on routes or short on money (or both)
					// so invest some ticks in pathfinding
					Debug("Re-estimating route: " + route.GetName());
					local estimatedDistance = route.GetDistance();
					route.CalculateImprovedEstimate();
					Debug("Re-estimating route took " + (GetTick() - tick) + " ticks");
				
					if (route.GetDistance().tofloat()/estimatedDistance >= 1.5) {
						// much longer - don't build the route so that it is revalued the next time around
						Warning(route.GetName() + " is now estimated at " + route.GetDistance() + " tiles, instead of " + estimatedDistance);
						Warning(route.GetName() + " is now valued at " + route.GetRouteValue() + " G/d");
						sort = true;
						return RouteBuildingResult.ROUTE_LONGER_THAN_EXPECTED;
					}
				} else {
					// set the route value to 0 so we won't send any more vehicles until our scout vehicle arrives
					// and confirms the route makes a profit
					Debug("Sending scout vehicle on " + route);
					route.SetRouteValue(0);
				}
				
				Debug("Building route: " + route);
				tick = GetTick();
				BuildRoute(route);
				local vehicle = route.AddVehicle();
				
				// for subsidized routes, send the first vehicle off as soon as it has a single unit of cargo
				if (vehicle && route.IsSubsidized()) {
					local ticks = 0;
					// don't wait longer than a week
					while (vehicle.IsEmpty() && ticks < 7*TICKS_PER_DAY) {
						ticks++;
						Sleep(1);
					}
					
					if (!vehicle.IsEmpty()) {
						// it's almost empty, so it won't make a profit
						// this flag prevents it from lowering the route value
						vehicle.isSubsidyRunner = true;
						
						// send it off
						AIVehicle.SkipToVehicleOrder(vehicle.vehicleID, 1);
					}
				}
				
				// make it prove itself: lower estimated route value
				// it'll be replaced by the first real data point
				route.SetRouteValue(route.GetRouteValueEstimate()/2);
				
				Debug("Building route took " + (GetTick() - tick) + " ticks");
				return RouteBuildingResult.ROUTE_BUILT;
			} catch (e) {
				Error("Could not build route " + route + ": " + e);
				if (typeof(e) == "instance" && (e instanceof NoRoomForStationException || e instanceof NoRoomForDepotException || e instanceof PathfindingException || e instanceof PissedOffAuthorityException)) {
					Debug("Marking as unbuildable");
					route.SetBuildable(false);
				}
				
				return RouteBuildingResult.ROUTE_BUILDING_FAILED;
			}
		}
	}
	
	function IsFirstYear() {
		return AIDate.GetCurrentDate() - START_DATE < DAYS_PER_YEAR;
	}
	
	function IsSuburb(route) {
		// a suburb is any district other than the center
		return route.GetPickup() instanceof District && route.GetPickup().GetDistrict() != Districts.CENTER;
	}
	
	/**
	 * Build stations until we run out of money, or tiles.
	 */
	function BuildStations() {
		// don't re-valuate the list, that'll destroy the sort order!
		// since each build takes a tick anyway, just test for trees inside the loop
		// endgameTiles.Valuate(AITile.IsBuildable);	// remove everything that has been built past 9-ish years
		// endgameTiles.KeepValue(1);
		// endgameTiles.Valuate(AITile.HasTreeOnTile);	// remove new trees
		// endgameTiles.KeepValue(0);

		Debug("There are " + endgameTiles.Count() + " buildable tiles");
		
		// at this point, we don't need to worry about bankrupcy
		FullyMaxLoan();
		
		foreach (id, vehicle in Vehicle.vehicles) {
			if (vehicle) vehicle.SetHinUndWieder(false);
		}
		
		local count = 0;
		local tile = endgameTiles.Begin();
		while (endgameTiles.HasNext()) {
			//local tickteller = AIController.GetTick();// for testing
			
			 /*/sell vehicles when we need money (anders verkoop je toch gewoon de boot?)
			while (GetAvailableMoney() < 2500) {
				local vehicles = AIVehicleList();
				vehicles.Valuate(AIVehicle.IsStoppedInDepot);
				vehicles.KeepValue(1);
				
				// sell the most valuable ones first, since their value will drop the fastest.
				vehicles.Valuate(AIVehicle.GetCurrentValue);
				vehicles.Sort(AIList.SORT_BY_VALUE, false);
				if (vehicles.IsEmpty()) {
					// nothing to sell, wait and see if more vehicles arrive in a depot
					SendVehiclesToDepot();
					Sleep(1);
					break;
				} else {
					local vehicle = vehicles.Begin();
					Debug("Selling vehicle " + AIVehicle.GetName(vehicle) + " for " + AIVehicle.GetCurrentValue(vehicle));
					AIVehicle.SellVehicle(vehicle);
				}
				
				//this part is commented out because we will now sell the vehicels as soon as we can.
			}*/
			
			/*// another sellvehicle setup, compatible with letting the vehicles run for as long as possible...
			while (GetAvailableMoney() < 2500) {
				local vehicles = AIVehicleList();
				if (vehicles.IsEmpty()) break;
				vehicles.Valuate(AIVehicle.IsStoppedInDepot);
				vehicles.KeepValue(1);
				
				if (vehicles.IsEmpty()) {
					SendVehiclesToDepot();
				} else {
					SellVehiclesInDepot();
				}
			}*/
			
			
			// skip tiles on which trees have grown, or stuff has been built
			if (AITile.HasTreeOnTile(tile) || !AITile.IsBuildable(tile) || AITile.IsFarmTile(tile)) {
				tile = endgameTiles.Next();
				continue;
			}
			
			if (AIRoad.BuildDriveThroughRoadStation(tile, tile - 1, AIRoad.ROADVEHTYPE_TRUCK, AIStation.STATION_NEW)) {
				count++;
			}
			
			if (AIError.GetLastError() == AIError.ERR_NOT_ENOUGH_CASH) {
				// if we failed from lack of money, stay on this tile
			} else {
				tile = endgameTiles.Next();
			}
			
			if (count % VEHICLE_MANAGEMENT_INTERVAL == 0) {
				SendVehiclesToDepot();
				SellVehiclesInDepot();//added instead of selling the vehicles only when we need money
				//beensellingvehicles = true;
			}
			
			//local tookticks = AIController.GetTick() - tickteller;
			//Debug("total ticks are at: " + AIController.GetTick());
			//Debug("this loop took " + tookticks + " ticks,");
			//if (beensellingvehicles){
			//	Debug("And we have not been selling vehicles!");
			//} else {
			//	Debug("But we were selling vehicles.");
			//}
			//beensellingvehicles = false;
			
			/*if (AIDate.GetCurrentDate() >= endgameDate) {
				// game over!
				break;
			}*/
		}
		
		// for determining the percentage of vehicles left before endgame.
		Debug("Number of vehicles at start endgame: " + vehiclesAtStartEndgame + ".");
		Debug("Number of vehicles at end of game: " + AIVehicleList().Count() + ".");
		Debug("Percentage of vehicles left at end of game: " + ((AIVehicleList().Count() / (vehiclesAtStartEndgame + 1))*100) + ".");
	}
	
	function ClearSigns() {
		for (local i=0; i<AISign.GetMaxSignID(); i++) {
		   if (AISign.IsValidSign(i)) {
		     AISign.RemoveSign(i);
		   }
		}
	}
	
	/**
	 * Create a sorted list of tiles on which we'll build stations during the endgame.
	 */
	function FindEndgameTiles() {
		endgameTiles = AITileList();
		endgameTiles.AddRectangle(AIMap.GetTileIndex(3, 3), AIMap.GetTileIndex(AIMap.GetMapSizeX() - 3, AIMap.GetMapSizeY() - 3));
		endgameTiles.Valuate(AITile.IsWaterTile);	// can't build there
		endgameTiles.KeepValue(0);
		endgameTiles.Valuate(AITile.IsBuildable);
		endgameTiles.KeepValue(1);
		Debug("Valuating " + endgameTiles.Count() + " tiles...");
		endgameTiles.Valuate(TileValuator);
		endgameTiles.Sort(AIList.SORT_BY_VALUE, true);
		Debug("Interesting tiles left: " + endgameTiles.Count());
		Debug("Cheap tile count: " + cheapTilesCount);
	}
	
	function CheckEndgame(endDate) {
		// endgame station building disabled
		return 8;
		
		Debug("Checking wether it is time to start endgame...");
		local date = AIDate.GetCurrentDate();
		local daysuntilend = endDate - date;
		Debug("days until end: "  + daysuntilend);
		local maxcostperstation = 247;
		//add a loop which calculates the cost for each month. Calcualtion goes backward.
		local costuntilend = 0;
		local upkeep = 0;	// start with 0, because we don't mind going negative on teh 1st of january 2008.
		local odd = 1; //used to calculate the difference between months with 30 or 31 days. Works fine until August (calculated back from december).
		for(local days = daysuntilend; days > 0; days = days - 30 - odd) {
			//add the costs for building and upkeep for this moth.
			costuntilend += (30 + odd) * TICKS_PER_DAY * (maxcostperstation + upkeep);
			//if the month we are considering has already started, we will need to subtract the days which have already passed.
			if (days < 30 + odd) {
				local surplusdays = 30 + odd - days;
				costuntilend -= surplusdays * TICKS_PER_DAY * (maxcostperstation + upkeep);
			}
			odd = 1 - odd;
			upkeep += MONTHLY_STATION_MAINTENANCE;
		}
		Debug("Cost for building stations and maintenance (full): " + costuntilend);
		
		local ourVehicles = AIVehicleList();
		local vehicleCount = ourVehicles.Count();
		local stationsNotBuilt = 2 * vehicleCount * maxcostperstation;//the building costs for the stations which can't be built because we're selling vehicles.
		//the factor 2 is beacuse both the order to go to the depot and the selling cost a tick.
		Debug("Cost until end, minus not built due to selling vehicles: " + costuntilend);
		
		//calculate cost for maintenance of existing stations (will be subtracted).
		local numberOfStations = AIStationList(AIStation.STATION_ANY).Count();
		local monthsUntilEnd = (daysuntilend/30.5).tointeger();
		local stationMaintenance = monthsUntilEnd * numberOfStations * MONTHLY_STATION_MAINTENANCE;
		Debug("stationmaintenance: " + stationMaintenance);
		
		//calculate vehicle running cost until enddate.
		local runningCosts = 0;
		local vehicleMoney = 0;
		local vehicleIncome = 0;
		local maintenancediscount = 0;
		for(local vehicleID = ourVehicles.Begin(); ourVehicles.HasNext(); vehicleID = ourVehicles.Next()) {
			local vehicle = Vehicle.GetVehicle(vehicleID);
			if (vehicle && vehicle.GetRoute() && !vehicle.IsHinUndWieder()) {
				local route = vehicle.GetRoute();
				local triptimeleft = route.GetTripTime() - AIVehicle.GetAge(vehicleID);
				if (triptimeleft < 1) triptimeleft = 1;
				if (route.IsJammedOrBroken()) triptimeleft += 20;
				vehicleIncome += route.GetRouteValue() * route.GetTripTime();
				runningCosts += triptimeleft * (AIVehicle.GetRunningCost(vehicleID)/DAYS_PER_YEAR);
				vehicleMoney += AIVehicle.GetCurrentValue(vehicleID) - ((AIVehicle.GetCurrentValue(vehicleID) * 0.16) * (triptimeleft / DAYS_PER_YEAR));
				//last thing we do is to subtract the stationmaintenance we overcalculated earlier due to ticks lost for ordering and selling vehicles.
				local discountmonths = ((daysuntilend - triptimeleft) / 30.5).tointeger();
				maintenancediscount += discountmonths * MONTHLY_STATION_MAINTENANCE * 2;
			} else if (vehicle && vehicle.GetRoute() && vehicle.IsHinUndWieder()) {
				local route = vehicle.GetRoute();
				vehicleIncome += (route.GetRouteValue() * route.GetTripTime());
				local hinundwiederfactor = 1;
				local destination = AIOrder.ResolveOrderPosition(vehicleID, AIOrder.ORDER_CURRENT);
				if (destination == 1) hinundwiederfactor = 0.5;
				else hinundwiederfactor = 1.5; 
				local estimatedTripTimeLeft = route.GetTripTime() * hinundwiederfactor;//we may need a factor here...
				runningCosts += (estimatedTripTimeLeft * AIVehicle.GetRunningCost(vehicleID))/DAYS_PER_YEAR;
				vehicleMoney += AIVehicle.GetCurrentValue(vehicleID) - ((AIVehicle.GetCurrentValue(vehicleID) * 0.16) * (estimatedTripTimeLeft / DAYS_PER_YEAR));
				//last thing we do is to subtract the stationmaintenance we overcalculated earlier due to ticks lost for ordering and selling vehicles.
				local discountmonths = ((daysuntilend - estimatedTripTimeLeft) / 30.5).tointeger();
				maintenancediscount += discountmonths * MONTHLY_STATION_MAINTENANCE * 2;
			}
		}
		
		Debug("Running costs until end: " + runningCosts);
		Debug("money stored in vehicles minus devaluation: " + vehicleMoney);
		Debug("Income until end: " + vehicleIncome);
		Debug("Discount on maintenance due to stations not built: " + maintenancediscount);
		
		local totalBalance = vehicleIncome + GetAvailableMoney() + vehicleMoney;
		costuntilend += runningCosts + stationMaintenance - maintenancediscount - stationsNotBuilt;
		Debug("Total balance: " + totalBalance + "this has to be bigger than cost until end; " + costuntilend);
		
		local diff = costuntilend - totalBalance;
		Debug("The difference is: " + diff);
		
		if(diff <= 0) {
			Debug("Yup, it is time :)");
			return 0;	//no more datechange, thus start endgame.
		} else {
			Debug("Not time yet...");
			if (diff <  250000)
				return 1;	//1 day until next check
			if (diff <  500000)
				return 2;	//2 days until next check
			if (diff <  750000)
				return 3;	//4 days until next check
			if (diff <  1000000)
				return 4;	//10 days until next check
			if (diff <  2000000)
				return 5;	//20 days until next check
			if (diff < 3000000)
				return 6;	// 1 month until next check
			if (diff < 5000000)
				return 7;	// 1.5 months until next check
			return 8;	// 2 months until next check
		}
	}
	
	function TicksPerDay() {
		//measure the number of ticks per day.
		local dateOne = AIDate.GetCurrentDate();
		while (dateOne == AIDate.GetCurrentDate()) {
			Sleep(1);
		}
		
		dateOne = AIDate.GetCurrentDate();
		local tickOne = AIController.GetTick();
		while (dateOne == AIDate.GetCurrentDate()) {
			Sleep(1);
		}
		
		local tickTwo = AIController.GetTick();
		local ticksPerDay = tickTwo - tickOne;
		return ticksPerDay;
		Debug("Ticks per day: " + ticksPerDay);
	}
	
	function NameCompany() {
		if (!AICompany.SetName(NAME)) {
			local i = 2;
			while (!AICompany.SetName(NAME + " #" + i)) i++;
		}
	}
	
	function Save() {
		// things to save:
		// vehicles, stations, depots
		// startEndgame, endgameCheck, endgameDate
		// TODO
		return {};
	}
	
	function Load(version, data) {
		Debug("TODO: implement load and save");
	}
}

function ManageVehicles() {
	::INSTANCE.ManageVehicles();
}

