/**
 *    This file is part of Rondje.
 *
 *    Rondje is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    Rondje is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with Rondje.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright 2008-2009 Marnix Bakker, Willem de Neve, Michiel Konstapel and Otto Visser.
 * Suggestions/comments and bugs can go to: rondjeomdekerk@konstapel.nl
 */

require("util.nut");

const ALPHA = 0.3;			// update rate of the exponential average
const JAM_DELAY = 1.2;		// multiplier for vehicle age that determines whether a route is delayed/jammed
const BROKEN_DELAY = 3;		// multiplier for vehicle age that determines whether a route might be broken
//const MINIMUM_SPEED = 34;	// see http://wiki.openttd.org/index.php/Game_Mechanics#Vehicle_speeds 
const MINIMUM_SPEED = 20;	// 20 is slower than a vehicle would go on any terrain, including the slowest bridges
const INFINITE = 0;			// "infinite" route distance
const SUBSIDY_MULTIPLIER = 2;
const JAM_CHECK_INTERVAL_DAYS = 7;		// after a check, remember jammed status for this many days

/**
 * Maintains persistent information about routes.
 */
class RouteDatabase {
	
	routes = null;
	
	constructor() {
		routes = {};
	}
	
	/**
	 * Add route to the "database" if it is not already in.
	 * @return whether the route was added (new).
	 */
	function Add(route) {
		local key = GetKey(route);
		if (key in routes) return false;
		
		Debug("Adding: " + key);
		routes[key] <- RouteInfo(route);
		return true;
	}
	
	function GetKey(route) {
		return route.GetName();
	}
	
	function Update(route, tripTime, value) {
		local routeInfo = routes[GetKey(route)];
		if (routeInfo.isEstimate) {
			// replace with real data
			Debug("Replacing estimate for " + GetKey(route) + ": was " + routeInfo.value + ", is now: " + value + "G/d");
			routeInfo.tripTime = tripTime;
			routeInfo.value = value;
			routeInfo.isEstimate = false;
		} else {
			// average with existing value
			routeInfo.tripTime = (1-ALPHA)*routeInfo.tripTime + ALPHA*tripTime;
			routeInfo.value = (1-ALPHA)*routeInfo.value + ALPHA*value;
			//Debug("Route updated: " + GetKey(route) + ": " + routeInfo.value + " G/d, trip time " + routeInfo.tripTime + " days");
		} 
	}
	
	function SetValue(route, value) {
		routes[GetKey(route)].value = value;
	}
	
	function GetValue(route) {
		return routes[GetKey(route)].value;
	}
	
	function GetTripTime(route) {
		return routes[GetKey(route)].tripTime;
	}
	
	function SetTripTime(route, tripTime) {
		routes[GetKey(route)].tripTime = tripTime;
	}
	
	function IsBuildable(route) {
		return routes[GetKey(route)].buildable;
	}
	
	function SetBuildable(route, buildable) {
		routes[GetKey(route)].buildable = buildable;
	}
	
	function IsEstimate(route) {
		return routes[GetKey(route)].isEstimate;
	}
	
	function GetDistance(route) {
		return routes[GetKey(route)].distance;
	}
	
	function SetDistance(route, distance) {
		routes[GetKey(route)].distance = distance;
	}
	
	function IsBuilt(route) {
		return routes[GetKey(route)].built;
	}
	
	function SetBuilt(route) {
		routes[GetKey(route)].built = true;
	}
		
}

class RouteInfo {
	
	value = 0;
	tripTime = 0;
	distance = 0;
	buildable = true;
	isEstimate = true;
	built = false;
	
	constructor(route) {
		this.value = 0;
		this.tripTime = 0;
		this.distance = 0;
		this.buildable = true;
		this.isEstimate = true;
		this.built = false;
	}
}

class Route {
	
	/**
	 * Maintains persistent info.
	 */
	static database = RouteDatabase();
	
	from = null;
	to = null;
	network = null;
	lastJamCheck = 0;
	jammed = false;
	broken = false;
	name = null;
	
	constructor(from, to, network) {
		this.from = from;
		this.to = to;
		this.network = network;
		this.lastJamCheck = 0;
		this.jammed = false;
		this.broken = false;
		this.name = from.GetCargoLabel() + " from " + from.GetName() + " to " + to.GetName();
		
		if (Route.database.Add(this)) {
			Route.database.SetDistance(this, GetDrivingDistanceEstimate());
			Route.database.SetValue(this, GetRouteValueEstimate());
			Route.database.SetTripTime(this, GetTripTimeEstimate());
		}
	}
	
	function _tostring() {
		return GetName() + ": " +
			GetDistance() + " tiles, " +
			GetTripTime().tointeger() + " days, " +
			GetRouteValue().tointeger() + " G/day, " +
			(IsBuilt() ? GetNumberOfVehicles() + " vehicles" : "not built");
	}
	
	function GetName() {
		return name;
	}
	
	/**
	 * Use the pathfinder to measure the actual distance and stores the result in the route database.
	 * @return the distance in tiles. 
	 */
	function CalculateImprovedEstimate() {
		local fromTile = FindDistanceEstimateRoadTile(from, to);
		local toTile = FindDistanceEstimateRoadTile(to, from);
		
		local pathfinder = ConnectedChecker(2*AIMap.DistanceManhattan(fromTile, toTile));
		pathfinder.InitializePath([fromTile], [toTile]);
		
		local distance;
		try {
			local tick = GetTick();
			local path = false;
			while (path == false) {
				path = pathfinder.FindPath(VEHICLE_MANAGEMENT_INTERVAL);
				ManageVehicles();
				if (GetTick() - tick > MAX_PATHFINDING_TICKS) {
					Warning("Pathfinding is taking too long!");
					path = null;
					break;
				}
			}
			
			if (path == null) {
				Warning("No path found!");
				distance = INFINITE;
			} else {
				distance = path.GetCost() + 2;	// add station tiles
				Debug("Actual distance: " + distance + " tiles, estimate was " + GetDrivingDistanceEstimate());
			}
		} catch (e) {
			Warning("Error during pathfinding!");
			distance = INFINITE;
		}
		
		Route.database.SetDistance(this, distance);
		Route.database.SetTripTime(this, GetTripTimeEstimate());
		Route.database.SetValue(this, distance == INFINITE ? -1 : GetRouteValueEstimate());
	}
	
	/**
	 * Find the road tile at "from" to use for as a pathfinding endpoint.
	 */
	function FindDistanceEstimateRoadTile(from, to) {
		// if we already have a station here *in the network*, use that as the endpoint
		local fromStation = from.GetMyStation();
		if (fromStation && network.Contains(fromStation.GetStationID()))
			return fromStation.GetLocation();
		
		local tiles = AITileList();
		tiles.AddList(from.GetStationArea());
		tiles.KeepList(network.tiles);
		if (tiles.Count() == 0) {
			// look further out
			SafeAddRectangle(tiles, from.GetLocation(), 10);
			tiles.KeepList(network.tiles);
			if (tiles.Count() == 0) {
				// still nothing
				throw NoRoomForStationException(from);
			}
		}
		
		return FindClosestTile(tiles, to.GetLocation());
	}
	
	function GetPickup() {
		return from;
	}
	
	function GetDropoff() {
		return to;
	}
	
	function GetPickupStation() {
		return from.GetMyStation();
	}
	
	function GetDropoffStation() {
		return to.GetMyStation();
	}
	
	/**
	 * Returns the estimated distance for this route.
	 */
	function GetDistance() {
		return Route.database.GetDistance(this);
	}
	
	/**
	 * Returns true if this is not a passenger route.
	 */
	function IsFreight() {
		return GetCargo() != PASSENGERS;
	}
	
	/**
	 * The type of cargo that this route provides.
	 */
	function GetCargo() {
		return from.GetCargo();
	}
	
	/**
	 * The amount of cargo waiting at the source station.
	 */
	function GetCargoWaiting() {
		return AIStation.GetCargoWaiting(from.GetMyStation().GetStationID(), from.GetCargo());
	}
	
	/**
	 * The cargo rating at the pickup station.
	 */
	function GetCargoRating() {
		return AIStation.GetCargoRating(from.GetMyStation().GetStationID(), from.GetCargo());
	}
	
	/**
	 * Check whether there is an unawarded subsidy for this route.
	 */
	function IsSubsidized() {
		local subsidies = AISubsidyList();
		subsidies.Valuate(AISubsidy.IsAwarded);
		subsidies.KeepValue(0);
		subsidies.Valuate(AISubsidy.GetCargoType);
		subsidies.KeepValue(GetCargo());
		
		for (local subsidy = subsidies.Begin(); subsidies.HasNext(); subsidy = subsidies.Next()) {
			// apparently this sometimes returns false even in all-passenger test games
			// probably the "IsValid" or "IsAwarded" checks?
			// local source = AISubsidy.SourceIsTown(subsidy) ? from.town.townID : from.industryID;
			local source = IsFreight() ? from.industryID : from.town.townID;
			local destination = IsFreight() ? to.industryID : to.town.townID;
			if (AISubsidy.GetSourceIndex(subsidy) == source && AISubsidy.GetDestinationIndex(subsidy) == destination) {
				return true;
			}
		}
		
		return false;
	}
	
	/**
	 * True if both destinations exist.
	 */
	function IsValid() {
		return from != null && to != null && from.Exists() && to.Exists();
	}
	
	/**
	 * True if the route has been built (stations, depots and connecting roads).
	 */
	function IsBuilt() {
		return database.IsBuilt(this);
	}
	
	/**
	 * Mark the road as built.
	 */
	function SetBuilt() {
		database.SetBuilt(this);
	}
	
	/**
	 * True if the route is built, the buildings are connected and the dropoff station accepts the route's cargo.
	 */
	function IsConnected() {
		return IsBuilt() &&
			Connected(from.GetMyDepot(), from.GetMyStation()) &&
			Connected(from.GetMyStation(), to.GetMyStation()) &&
			Connected(to.GetMyStation(), to.GetMyDepot()) &&
			to.GetMyStation().AcceptsCargo(GetCargo());
	}
	
	/**
	 * A route is jammed if there are vehicles on it that are older than the expected trip time
	 * and that are moving too slow. It is broken if there are vehicles on it that are much older.
	 */
	function IsJammedOrBroken() {
		if (AIDate.GetCurrentDate() - lastJamCheck < JAM_CHECK_INTERVAL_DAYS) {
			return jammed || broken;
		} else {
			local maxJammedAge = (JAM_DELAY*GetTripTime()).tointeger();
			local maxBrokenAge = (BROKEN_DELAY*GetTripTime()).tointeger();
			local vehicles = GetVehicleList();
			
			// only look at vehicles that haven't yet made their first run
			vehicles.Valuate(GetVehicleProfit);
			vehicles.KeepBelowValue(0);
			
			// for jams, look for oldish vehicles that are moving too slowly
			local jammedVehicles = AIList();
			jammedVehicles.AddList(vehicles);
			jammedVehicles.Valuate(AIVehicle.GetCurrentSpeed);
			jammedVehicles.KeepAboveValue(0);
			jammedVehicles.KeepBelowValue(MINIMUM_SPEED);
			jammedVehicles.Valuate(AIVehicle.GetAge);
			jammedVehicles.KeepAboveValue(maxJammedAge);
			jammed = jammedVehicles.Count() > 0;
			
			// for broken routes, look for vehicles that are very old
			local brokenVehicles = AIList();
			brokenVehicles.AddList(vehicles);
	 		brokenVehicles.Valuate(AIVehicle.GetAge);
			brokenVehicles.KeepAboveValue(maxBrokenAge);
			broken = brokenVehicles.Count() > 0;
			
			lastJamCheck = AIDate.GetCurrentDate();
			if (jammed) Warning(GetName() + " is jammed");
			if (broken) Warning(GetName() + " is broken");
			return jammed || broken;
		}
	}
	
	/**
	 * Returns the number of vehicles on this route.
	 */
	function GetNumberOfVehicles() {
		return GetVehicleList().Count();
	}
	
	/**
	 * Return an AIVehicleList of vehicles on this route.
	 */
	function GetVehicleList() {
		if (from.GetMyStation() == null || to.GetMyStation() == null) return AIList();
		
		// count vehicles going from the pickup to dropoff station
		local vehicles = AIVehicleList_Station(from.GetMyStation().stationID);
		local dropoffLocation = to.GetMyStation().GetLocation();
		vehicles.Valuate(AIOrder.GetOrderDestination, 1);
		vehicles.KeepValue(dropoffLocation);
		return vehicles;
	}
	
	/**
	 * Add a vehicle to this route by building one from the depot
	 * at the given destination.
	 */
	function AddVehicle() {
		local depot = Depot.GetDepot(from);
		
		local vehicles = GetVehicleList();
		
		// don't clone hin und wieder vehicles, which don't have NO_LOAD orders at the dropoff
		vehicles.Valuate(AIOrder.GetOrderFlags, 1);
		vehicles.KeepValue(AIOrder.AIOF_NO_LOAD | AIOrder.AIOF_NON_STOP_INTERMEDIATE);
		local prototype = vehicles.IsEmpty() ? null : vehicles.Begin();
		local prototypeEngine = prototype ? AIVehicle.GetEngineType(prototype) : null;
		
		local vehicleID;
		if (vehicles.IsEmpty() || prototypeEngine != GetBestEngine(from.GetCargo())) {
			// no suitable prototype, so build a vehicle from scratch
			vehicleID = depot.BuildVehicle(from.GetCargo());
			if (!AIVehicle.IsValidVehicle(vehicleID)) return null;
			
			AIOrder.AppendOrder(vehicleID, from.GetMyStation().GetLocation(), AIOrder.AIOF_FULL_LOAD | AIOrder.AIOF_NON_STOP_INTERMEDIATE);
			AIOrder.AppendOrder(vehicleID, to.GetMyStation().GetLocation(), AIOrder.AIOF_NO_LOAD | AIOrder.AIOF_NON_STOP_INTERMEDIATE);
			AIOrder.AppendOrder(vehicleID, to.GetMyDepot().GetLocation(), AIOrder.AIOF_NON_STOP_INTERMEDIATE);
		} else {
			// clone the prototype to save 3 ticks on adding orders
			vehicleID = depot.CloneVehicle(prototype);
			if (!AIVehicle.IsValidVehicle(vehicleID)) return null;
		}
		
		AIVehicle.StartStopVehicle(vehicleID);
		return Vehicle(vehicleID, this);	// NB: constructor adds it to the id->vehicle table
	}
	
	/**
	 * Returns an estimate for the expected driving distance.
	 */
	function GetDrivingDistanceEstimate() {
		// the actual path won't be a straight line from pickup to dropoff
		return (1.5 * GetPaymentDistance()).tointeger();
	}
	
	/**
	 * Returns the Manhattan distance between the two destinations, or, if built, stations.
	 */
	function GetPaymentDistance() {
		if(IsBuilt()) {
			return AIMap.DistanceManhattan(from.GetMyStation().GetLocation(), to.GetMyStation().GetLocation());
		} else {
			local distance = AIMap.DistanceManhattan(from.GetLocation(), to.GetLocation());
			
			// for passengers, assume the actual route will be shorter
			// due to pickup/dropoff placement
			if (!IsFreight()) distance -= DROPOFF_DISTANCE;
			
			return distance;
		}
	}
	
	/**
	 * Approximate time it'll take a vehicle to make one trip on this route, from pickup to dropoff.
	 */
	function GetTripTimeEstimate() {
		local engine = GetBestEngine(from.GetCargo());
		
		// convert listed speed in km/h to tiles per day
		// vehicles won't drive at full speed the whole route (not even close),
		// so estimate them at half their top speed
		local tilesPerDay = (5.6/100) * AIEngine.GetMaxSpeed(engine)/2;
		local loadingTime = 4;	// 4 days to load or unload
		
		// for passengers, add some city size dependent fudge to account for manoeuvring and traffic
		local trafficDelay = IsFreight() ? 0 : from.town.GetRadius();
		return loadingTime + trafficDelay + GetDistance()/tilesPerDay;
	}
	
	function GetTripTime() {
		return Route.database.GetTripTime(this);
	}
		
	/**
	 * Income for a single haul, divided by the trip time, minus daily running costs.
	 */
	function GetRouteValueEstimate() {
		local cargoID = from.GetCargo();
		local engine = GetBestEngine(cargoID);
		local tripTime = GetTripTimeEstimate();
		local distance = GetPaymentDistance();
		local unitIncome = AICargo.GetCargoIncome(cargoID, distance, tripTime.tointeger());
		local units = AIEngine.GetCapacity(GetBestEngine(cargoID));
		return (unitIncome * units / tripTime) - (AIEngine.GetRunningCost(engine)/DAYS_PER_YEAR);
	}
	
	function IsValueConfirmed() {
		return !Route.database.IsEstimate(this);
	}
	
	function SetRouteValue(value) {
		Route.database.SetValue(this, value);
	}

	function GetRouteValue() {
		local value = Route.database.GetValue(this);
		
		// take the subsidy multiplier into account for estimates
		// (not for confirmed routes, because extra profit is reported by the vehicle)
		if (!IsValueConfirmed() && IsSubsidized()) value *= SUBSIDY_MULTIPLIER;
		return value; 
	}
		
	function Update(tripTime, value) {
		database.Update(this, tripTime, value);
	}
	
	function SetBuildable(buildable) {
		database.SetBuildable(this, buildable);
	}
	
	function IsBuildable() {
		return database.IsBuildable(this) && GetDistance() != INFINITE;
	}
	
	/**
	 * Compare routes by estimated profitability.
	 */
	static function CompareRouteIncome(a, b) {
		if (a.GetRouteValue() > b.GetRouteValue()) return -1;
		if (a.GetRouteValue() < b.GetRouteValue()) return 1;
		
		if (a.from.GetProduction() > b.from.GetProduction()) return -1;
		if (a.from.GetProduction() < b.from.GetProduction()) return 1;
		return 0;
	}	
}
