/*
 * This file is part of SuperLib, which is an AI Library for OpenTTD
 * Copyright (C) 2010  Leif Linse
 *
 * SuperLib is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License
 *
 * SuperLib is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with SuperLib; If not, see <http://www.gnu.org/licenses/> or
 * write to the Free Software Foundation, Inc., 51 Franklin Street, 
 * Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

class _SuperLib_Money
{
	//////////////////////////////////////////////////////////////////////
	//                                                                  //
	//  Money, money                                                    //
	//                                                                  //
	//////////////////////////////////////////////////////////////////////

	static function GetMaxSpendingAmount();
	static function MakeSureToHaveAmount(amount);
	static function MakeMaximumPayback();

	static function GetCompanyProfitLastYear();


	//////////////////////////////////////////////////////////////////////
	//                                                                  //
	//  Temporary max loans                                             //
	//                                                                  //
	//////////////////////////////////////////////////////////////////////

	static function MaxLoan(); // Returns previous balance
	static function RestoreLoan(old_balance); // restores the old balance
};

function _SuperLib_Money::GetMaxSpendingAmount()
{
	local balance = AICompany.GetBankBalance(AICompany.COMPANY_SELF);
	local max_loan = AICompany.GetMaxLoanAmount();
	local loan_amount = AICompany.GetLoanAmount();

	return balance + (max_loan - loan_amount);
}

function _SuperLib_Money::MakeSureToHaveAmount(amount, call_number = 1)
{
	local balance = AICompany.GetBankBalance(AICompany.COMPANY_SELF);
	
	if(balance > amount)
	{
		return true; // Already enough money on the bank balance
	}

	local needed_loan_amount = amount - balance;

	local max_loan = AICompany.GetMaxLoanAmount();
	local current_loan_amount = AICompany.GetLoanAmount();

	if(max_loan - current_loan_amount < needed_loan_amount)
	{
		return false; // Not possible to loan the needed extra money from the bank
	}
	
	local num_loan_steps = needed_loan_amount / AICompany.GetLoanInterval();
	local extra_loan_amount = AICompany.GetLoanInterval() * num_loan_steps;
	if(extra_loan_amount < needed_loan_amount)
	{
		// In most cases the integer division will lead to a to small loan, 
		// so compensate for that. Though in rare cases this is not needed.
		num_loan_steps++; 
		extra_loan_amount += AICompany.GetLoanInterval();
	}

	AICompany.SetLoanAmount(current_loan_amount + extra_loan_amount);

	if(AICompany.GetBankBalance(AICompany.COMPANY_SELF) < amount)
	{
		if(call_number <= 1)
		{
			_SuperLib_Log.Error("Money::MakeSureToHaveAmount failed to aquire enough money even though the needed loan size is allowed.", _SuperLib_Log.LVL_INFO);
			_SuperLib_Log.Info("amount = " + amount + " balance = " + balance + " needed loan " + needed_loan_amount + " num loan steps = " + num_loan_steps + " extra loan = " + extra_loan_amount + " current loan = " + current_loan_amount, _SuperLib_Log.LVL_INFO);
			_SuperLib_Log.Info("Retrying .. ", _SuperLib_Log.LVL_INFO);
			return _SuperLib_Money.MakeSureToHaveAmount(amount, call_number + 1)
		}
		else
		{
			_SuperLib_Log.Error("Money::MakeSureToHaveAmount retried but failed again!", _SuperLib_Log.LVL_INFO);
			_SuperLib_Log.Info("amount = " + amount + " balance = " + balance + " needed loan " + needed_loan_amount + " num loan steps = " + num_loan_steps + " extra loan = " + extra_loan_amount + " current loan = " + current_loan_amount, _SuperLib_Log.LVL_INFO);
		}
	
		return false;
	}

	return true;
}

function _SuperLib_Money::MakeMaximumPayback()
{
	local balance = AICompany.GetBankBalance(AICompany.COMPANY_SELF);
	local current_loan_amount = AICompany.GetLoanAmount();

	if(balance < 0 || current_loan_amount == 0)
		return;

	local loan_interval = AICompany.GetLoanInterval();
	local num_payback_steps = balance / loan_interval

	local new_loan_amount = current_loan_amount - num_payback_steps * loan_interval;

	AICompany.SetLoanAmount(new_loan_amount);
}

function _SuperLib_Money::GetCompanyProfitLastYear()
{
	local all_vehicles = AIVehicleList();
	local company_profit = 0;

	foreach(vehicle_id in all_vehicles)
	{
		local veh_profit = AIVehicle.GetProfitLastYear(vehicle_id);

		// If the vehicle didn't make any money last year, but this year then use the numbers for this yea instead
		if(veh_profit <= 0)
		{
			local this_year = AIVehicle.GetProfitThisYear(vehicle_id);
			if(this_year > 0) veh_profit = this_year;
		}

		company_profit += veh_profit;
	}

	local all_stations = AIStationList(AIStation.STATION_ANY);
	foreach(station_id in all_stations)
	{
		local num = 0;
		num += AIStation.HasStationType(station_id, AIStation.STATION_BUS_STOP)? 1 : 0;
		num += AIStation.HasStationType(station_id, AIStation.STATION_TRUCK_STOP)? 1 : 0; // does truck + bus cost 500 each or just one of them?
		num += AIStation.HasStationType(station_id, AIStation.STATION_AIRPORT)? 1 : 0;
		num += AIStation.HasStationType(station_id, AIStation.STATION_TRAIN)? 1 : 0;
		num += AIStation.HasStationType(station_id, AIStation.STATION_DOCK)? 1 : 0;

		company_profit -= STATION_MAINTENANCE_COST * num;
	}

	return company_profit;
}

function _SuperLib_Money::MaxLoan()
{
	local old_balance = AICompany.GetBankBalance(AICompany.COMPANY_SELF);
	AICompany.SetLoanAmount(AICompany.GetMaxLoanAmount());

	return old_balance;
}
function _SuperLib_Money::RestoreLoan(old_balance)
{
	_SuperLib_Money.MakeMaximumPayback();
	_SuperLib_Money.MakeSureToHaveAmount(old_balance);
}
