/*
 *	Copyright  2008 George Weller
 *	
 *	This file is part of PathZilla.
 *	
 *	PathZilla is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, either version 2 of the License, or
 *	(at your option) any later version.
 *	
 *	PathZilla is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *	
 *	You should have received a copy of the GNU General Public License
 *	along with PathZilla.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * Schema.nut
 *
 * A unique entity that encapsulates the graph, road and cargo types for an 
 * independant road network. The roads are not necessairly distinct from those
 * used by other schemas, but any roads shared by schemas is strictly informal,
 * as dictated by the path finder.
 * 
 * Author:  George Weller (Zutty)
 * Created: 30/08/2008
 * Version: 1.0
 */

class Schema {
	// Serialization constants
	CLASS_NAME = "Schema";
	SRLZ_SCHEMA_ID = 0;
	SRLZ_SOURCE_NODE = 1;
	SRLZ_CARGOS = 2;
	SRLZ_TRANSPORT_TYPE = 3;
	SRLZ_SUB_TYPE = 4;
	SRLZ_PLAN_GRAPH = 5;
	SRLZ_ACTUAL_GRAPH = 6;
	SRLZ_INDUSTRIAL = 7;
	SRLZ_TARGETS = 8;
	
	// Member variables
	id = 0;
	sourceNode = null;
	cargos = null;
	transportType = null;
	cid = 0;
	subType = null;
	planGraph = null;
	actualGraph = null;
	industrial = null;
	targets = null;

	constructor(sourceNode, cargos, transportType, subType) {
		this.id = 0;
		this.sourceNode = sourceNode;
		this.cargos = cargos;
		this.transportType = transportType;
		this.cid=0;
		this.subType = subType;
		this.planGraph = null;
		this.actualGraph = null;
		this.targets = null;

		// Decide if this is an industrial service or not		
		local sampleCargo = cargos.Begin();
		local noEffect = (AICargo.GetTownEffect(sampleCargo) == AICargo.TE_NONE);
		if(!AICargo.IsFreight(sampleCargo)&&!noEffect) {
			industrial = false;
		} else {
			industrial = true;
		
// TODO - Heterogenous services
		}
	}
}

/*
 * Gets the schema id.
 */
function Schema::GetId() {
	return this.id;
}

function Schema::AddTarget(target) {
if (this.targets==null) this.targets = Map();
this.targets.Insert(target);
}

/*
 * Sets the schema id.
 */
function Schema::SetId(schemaId) {
	this.id = schemaId;
}

/*
 * Get the list of cargo IDs.
 */
function Schema::GetCargos() {
	return this.cargos;
}

function Schema::GetNextCargo() {
local cc=this.cargos.Count();
if (cc==1)return this.cargos.Begin();
if (cc==0)return -1;
local zz=0;
this.cid+=1+this.cid%23;
local i=this.cid%cc;
foreach (cargo,_ in this.cargos){
if (i>zz++)continue;
return cargo;
}
return -1;
}

/*
 * Get the transport type 
 */
function Schema::GetTransportType() {
	return this.transportType;
}

/*
 * Get the sub type 
 */
function Schema::GetSubType() {
	return this.subType;
}

/*
 * Get a graph showing which links we plan to build.
 */
function Schema::GetPlanGraph() {

	if(this.planGraph == null) this.Initialise();
	
	return this.planGraph;
}

/*
 * Get a graph showing which links we have already built.
 */
function Schema::GetActualGraph() {

	if(this.actualGraph == null) this.Initialise();

	return this.actualGraph;
}

/*
 * Check if the schema is industrial, i.e. that it services industries rather
 * than towns.
 */
function Schema::IsIndustrial() {
	return this.industrial;
}

function Schema::GetTargets() {

	if(this.targets==null) {
	this.InitialiseTargets();
}
local targets=[];
foreach (target in this.targets.GetData()){
if (target.GetId()>=1000000){
targets.append(Target(Target.TYPE_TOWN,target.GetId()))
}else{
targets.append(Target(Target.TYPE_INDUSTRY,target.GetId()))
}
}
return targets;
}

function Schema::GetTarget(id,loc=false) {

if (!loc){
if (this.targets==null)this.Initialise();
foreach (target in this.targets.GetData()){
if (target.GetId()==id){::etarget=target; return target;}
}
}
local townlist=AITownList();
for (local town=townlist.Begin();hasnext(townlist);town=townlist.Next()){
if (AITown.GetLocation(town)==id) {::etarget=Target(Target.TYPE_TOWN, town+1000000);return Target(Target.TYPE_TOWN, town+1000000);}
}
local indlist=AIIndustryList();
for (local industry=indlist.Begin();hasnext(indlist);industry=indlist.Next()){
if (AIIndustry.GetLocation(industry)==id) {::etarget=Target(Target.TYPE_INDUSTRY, industry);return Target(Target.TYPE_INDUSTRY, industry);}
}
return ::etarget;
}

/*
 * Create an array of targets from all towns (up to a maximum of MAX_TARGETS)  
 * on the map.
 */
function Schema::GetTownTargets(tat=999567) {

	// Prime a list of the closest MAX_TARGETS targets to the home town
	local allTowns = AITownList();
	local pass=0;
	local ind=3;	
foreach(cargo, _ in AICargoList()) {
	if (AICargo.GetTownEffect(cargo)==AICargo.TE_PASSENGERS)pass=cargo;	
	}
	foreach(cargo, _ in this.cargos) {
	if (AICargo.IsFreight(cargo)&&AICargo.GetTownEffect(cargo)!=AICargo.TE_MAIL)ind=1;	
	}
foreach(town, _ in AITownList()) {
	
	allTowns.SetValue(town,(AITown.GetPopulation(town)-(AITown.GetPopulation(town)*AITown.GetLastMonthTransportedPercentage(town,pass)/200)/((sqrt(AITown.GetDistanceManhattanToTile(town,((tat!=999567&&AIIndustry.IsValidIndustry(tat))?AIIndustry.GetLocation(tat):AITown.GetLocation(this.sourceNode))))).tointeger()+1))*(AITown.GetPopulation(town)%4+3));
}
	allTowns.Sort(AIAbstractList.SORT_BY_VALUE, false);
	allTowns.KeepTop((((PathZilla.GetSetting("targets")*ind)*22)/max(8,AICargoList().Count())).tointeger());
	// HACK: If using trams, only consider large towns
	if(this.GetSubType() == AIRoad.ROADTYPE_TRAM) {
		allTowns.Valuate(AITown.GetPopulation);
		allTowns.RemoveBelowValue(1200);
	}
local hList=AIList();
if(!AIIndustry.IsValidIndustry(tat)&&AITown.IsValidTown(this.sourceNode)){
hList.AddItem(this.sourceNode,0);
allTowns.AddList(hList);
}
	// Build a list of targets
	local targets = Map();
	foreach(town, _ in allTowns) {
	targets.Insert(Target(Target.TYPE_TOWN, town+1000000));
	}

	return targets;
}

/*
 * Create an array of targets from industries on the map that accept or produce 
 * the predefined cargo for this schema. 
 */
function Schema::GetIndustryTargets(tat=999567) {
	// Get a list of all industries that handle the appropriate cargo

	local pass=0;
	local allList = AIList();
	local indList = AIList();
	local accList= AIList();
	foreach(cargo, _ in this.cargos) {
	indList.Clear();
	accList.Clear();
		indList.AddList(AIIndustryList_CargoProducing(cargo));
		accList.AddList(AIIndustryList_CargoAccepting(cargo));
	
	
	// The source node is currently a town, which is no good!
	indList.Valuate(function (target, cargo,tile) {
			return ((AIIndustry.GetLastMonthProduction(target,cargo)-AIIndustry.GetLastMonthTransported(target,cargo))*500-(AIIndustry.GetDistanceManhattanToTile(target,tile)*AIIndustry.GetDistanceManhattanToTile(target,tile))*(AIIndustry.GetLastMonthProduction(target,cargo)%4));
		}, this.cargos.Begin(),((tat!=999567&&AIIndustry.IsValidIndustry(tat))?AIIndustry.GetLocation(tat):AITown.GetLocation(this.sourceNode)))
	indList.Sort(AIAbstractList.SORT_BY_VALUE, false);
	indList.KeepTop(((((PathZilla.GetSetting("targets")-((PathZilla.GetSetting("targets")/6)/this.cargos.Count()))*22)/(max(8,AICargoList().Count()))).tointeger()+1));
if (cargo==this.cargos.Begin()){
this.sourceNode = indList.Begin();
tat=this.sourceNode;
}
	accList.Valuate(function (target, tile) {
			return ((((AIIndustry.GetAmountOfStationsAround(target)==0)?3000+AIIndustry.GetDistanceManhattanToTile(target,tile)*AIIndustry.GetDistanceManhattanToTile(target,tile):AIIndustry.GetAmountOfStationsAround(target)*AIIndustry.GetAmountOfStationsAround(target)*500)+(AIIndustry.GetDistanceManhattanToTile(target,tile)*AIIndustry.GetDistanceManhattanToTile(target,tile)))*((AIIndustry.GetDistanceManhattanToTile(target,tile)+AIDate.GetCurrentDate())%444)*((AIIndustry.GetAmountOfStationsAround(target)+AICompany.GetMaxLoanAmount()+((AIGameSettings.GetValue("economy.inflation")==1)?0:AIDate.GetCurrentDate()))%3));
		},AIIndustry.GetLocation(this.sourceNode));
	accList.Sort(AIAbstractList.SORT_BY_VALUE, true);
	accList.KeepTop(((((PathZilla.GetSetting("targets")/6/this.cargos.Count()))*22)/(max(8,AICargoList().Count()))).tointeger()+1);
	allList.AddList(indList);
	allList.AddList(accList);
	indList.Valuate(AIIndustry.GetDistanceManhattanToTile, AITown.GetLocation(this.sourceNode));
	indList.Sort(AIAbstractList.SORT_BY_VALUE, true);
}	
// Build a list of targets
	local targets = Map();
	foreach(industry, _ in allList) {
	if (!AIIndustryType.IsBuiltOnWater(AIIndustry.GetIndustryType(industry)))targets.Insert(Target(Target.TYPE_INDUSTRY, industry));
}

	return targets;
}

/*
 * Create the list of targets that can be serviced in this schema.
 */

function Schema::InitialiseTargets(tat=999567) {
	// Start with either industries or towns

this.targets=null;
	if(this.industrial) {
	if(this.targets == null){		this.targets = this.GetIndustryTargets(tat);
} else {
local tlist=Map()
tlist=this.GetIndustryTargets(tat);
foreach (target in tlist.GetData()){
local insert=0;
foreach (t in this.targets.GetData()){
if (target.GetName()==t.GetName()){
insert=1;
break;
}
}
if (insert==0){
 this.targets.Insert(target);
}
}
}
		// Add towns if we need to route cargo through them
		if(this.id!=0&&(AICargo.GetTownEffect(this.cargos.Begin()) != AICargo.TE_NONE/*||Settings.RouteCargoThroughTowns()*/)) {
			this.targets.Extend(this.GetTownTargets(tat));
		
}
	} else {
	if(this.targets == null){		this.targets = this.GetTownTargets(tat);
}else {
local tlist=Map()
tlist=this.GetTownTargets(tat);
foreach (target in tlist.GetData()){
local insert=0;
foreach (t in this.targets.GetData()){
if (target.GetName()==t.GetName()){
insert=1;
break;
}
}
if (insert==0) this.targets.Insert(target);
}
}
	}
	}

/*
 * Create the plan and actual graphs based on a triangulation over a list of
 * targets, chosen based on the type of schema and global settings.
 */
function Schema::Initialise(tat=999567) {

	// Ensure the list of targets has been initialised
	if(this.targets == null) this.InitialiseTargets(tat);

	// Get the master graph for the whole map
local ind=5;
	foreach(cargo, _ in this.cargos) {
	if (AICargo.IsFreight(cargo)&&AICargo.GetTownEffect(cargo)!=AICargo.TE_MAIL)ind=1;	
	}
local tlist=Map();
local tlz=0
foreach (target in this.targets.GetData()){
if (target.IsValid()){
local insert=0;
foreach (t in tlist.GetData()){
if (target.GetName()==t.GetName()){
insert=1;
break;
}
}
if (insert==0){
 tlist.Insert(target);
tlz++;
}
//if (tlz>=PathZilla.GetSetting("targets")*ind) break;
}
}	
if (tlist.Begin()!=null&&(tlist.Begin()).IsValid())       this.sourceNode = ((tat!=999567&&AIIndustry.IsValidIndustry(tat))?tat:tlist.Begin().GetId());
	local masterGraph = Triangulation(tlist);
//if (this.planGraph!= null) masterGraph.Merge(this.planGraph);
	// For the plan graph use a combination of the shortest path from the home 
	// town and the minimum spanning tree.

	this.planGraph = ShortestPathTree(masterGraph, (this.sourceNode>=1000000)?AITown.GetLocation(this.sourceNode-1000000):AIIndustry.GetLocation(this.sourceNode));
	this.planGraph.Merge(MinimumSpanTree(masterGraph));
	
	// Create a blank graph to represent what has actually been built
	this.actualGraph = Graph();
}
/*
 * Saves data to a table.
 */
function Schema::Serialize() {
	local data = {};

	data[SRLZ_SCHEMA_ID] <- this.id;
	data[SRLZ_SOURCE_NODE] <- this.sourceNode;
	data[SRLZ_CARGOS] <- ListToArray(this.cargos);
	data[SRLZ_TRANSPORT_TYPE] <- this.cid;
	data[SRLZ_SUB_TYPE] <- this.subType;
	data[SRLZ_INDUSTRIAL] <- this.industrial;

	if(this.planGraph != null) data[SRLZ_PLAN_GRAPH] <- this.planGraph.Serialize();
	if(this.actualGraph != null) data[SRLZ_ACTUAL_GRAPH] <- this.actualGraph.Serialize();
	if(this.targets!=null){
	data[SRLZ_TARGETS] <- {};
	foreach(idx, target in this.targets) {
	data[SRLZ_TARGETS][idx] <- target.Serialize();
	}
	
}

return data;
}

/*
 * Loads data from a table.
 */
function Schema::Unserialize(data) {
	this.id = data[SRLZ_SCHEMA_ID];
	this.sourceNode = data[SRLZ_SOURCE_NODE];
	this.cargos = ArrayToList(data[SRLZ_CARGOS]);
	this.transportType = AITile.TRANSPORT_ROAD;
	this.cid = data[SRLZ_TRANSPORT_TYPE];
	this.subType = data[SRLZ_SUB_TYPE];
	this.industrial = data[SRLZ_INDUSTRIAL];
	
	if(SRLZ_PLAN_GRAPH in data) {
		this.planGraph = Graph();
		this.planGraph.Unserialize(data[SRLZ_PLAN_GRAPH]);
	}
	
	if(SRLZ_ACTUAL_GRAPH in data) {
		this.actualGraph = Graph();
		this.actualGraph.Unserialize(data[SRLZ_ACTUAL_GRAPH]);
	}
	if(SRLZ_TARGETS in data) {
		this.targets=Map();
		local targetsdata={};
		foreach(idx, targetData in data[SRLZ_TARGETS]) {
			targetsdata[idx] <- Target.instance();
			targetsdata[idx].Unserialize(targetData);
			this.targets.Insert(targetsdata[idx]);
		}
}
}
