 /*	Copyright  2008 George Weller
 *	
 *	This file is part of PathZilla.
 *	
 *	PathZilla is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, either version 2 of the License, or
 *	(at your option) any later version.
 *	
 *	PathZilla is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *	
 *	You should have received a copy of the GNU General Public License
 *	along with PathZilla.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Service.nut
 * 
 * A bus service between two towns
 * 
 * Author:  George Weller (Zutty)
 * Created: 18/06/2008
 * Version: 1.1
 */

class Service {
	// Serialization constants
	CLASS_NAME = "Service";
	SRLZ_SCHEMA_ID = 0;
	SRLZ_TARGET_IDS = 1;
	SRLZ_CARGO = 2;
	SRLZ_TRANSPORT_TYPE = 3;
	SRLZ_SUB_TYPE = 4;
	SRLZ_ENGINE = 5;
	SRLZ_PROFITABILITY = 6;
	SRLZ_GROUP = 7;
	SRLZ_DISTANCE = 8;
	SRLZ_RAW_INCOME = 9;
	SRLZ_COVERAGE_TARGET = 10;
	SRLZ_TRANSPORTED = 11;	
	SRLZ_LOCKED = 11;	

	schemaId = null;
	targetIds = null;
	cargo = 0;
	transportType = null;
	subType = null;
	engine = null;
	profitability = 0;
	vehicles = null;
	group = null;
	distance = 0;
	rawIncome = 0;
	coverageTarget = 0;
	transported = 1;
	locked=0;

	constructor(schemaId, targetIds, cargo, transportType, subType, engine, distance, rawIncome, coverageTarget) {
		this.schemaId = schemaId;
		this.targetIds = targetIds;
		this.cargo = cargo;
		this.transportType = transportType;
		this.subType = subType;
		this.engine = engine;
		this.distance = distance;
		this.rawIncome = rawIncome;
		this.coverageTarget = coverageTarget;
		this.transported = (this.transported!=1)?this.transported:(targetIds[0]<1000000)?(::pz.GetSchema(this.schemaId).GetTarget(targetIds[0])).GetTransportation(-1,cargo)*max(1,(AIIndustry.GetAmountOfStationsAround(targetIds[0])-RoadManager.GetStations((::pz.GetSchema(this.schemaId).GetTarget(targetIds[0])), this.cargo, this.subType).Count())):((::pz.GetSchema(this.schemaId).GetTarget(targetIds[0])).GetTransportation(-1,cargo)+20);
		this.locked = (this.transported!=0)?this.locked:AIDate.GetCurrentDate();
	}
}

function Service::Create() {

	// Create a group for the vehicles
	this.vehicles = AIList();
	this.group = AIGroup.CreateGroup(AIVehicle.VT_ROAD);
	
	// Name the group
	local schema = ::pz.GetSchema(this.schemaId);
	local last = this.targetIds.len() - 1;
	local fstr = chopstr(schema.GetTarget(this.targetIds[0]).GetName(), 7);
	local tstr = chopstr(schema.GetTarget(this.targetIds[last]).GetName(), 7);
	local strName = AICargo.GetCargoLabel(this.cargo) + " " + fstr + " to " + tstr;
	AIGroup.SetName(this.group, trnc(strName));
if (AIGroup.GetName(this.group)!=trnc(strName)){
for(local i=2;i<10;i++){
AIGroup.SetName(this.group, ((strName.len() > 29) ? strName.slice(0, 29) : strName)+i.tostring());
if (AIGroup.GetName(this.group)==((strName.len() > 29) ? strName.slice(0, 29) : strName)+i.tostring())break;
}
}
}


/*
 * Get the schema id
 */
function Service::GetSchemaId() {
	return this.schemaId;
}

function Service::Lock(time) {
this.locked=max(this.locked,AIDate.GetCurrentDate()+time);
	return this.locked;
}

function Service::IsLocked() {
	return this.locked>AIDate.GetCurrentDate();
}

/*
 * Get the ids of the targets this service visits.
 */
function Service::GetTargetIds() {
	return this.targetIds;
}

/*
 * Get the targets this service visits.
 */
function Service::GetTargets() {

	local schema = ::pz.GetSchema(this.schemaId);
	local targets = [];
	local nr=0;
	foreach(id in this.targetIds) {
		targets.append(schema.GetTarget(id));
	nr=1;
	}
	
	return targets;
}

function Service::ChangeTarget(targetId,i) {
this.targetIds[i]=targetId;
	return;
}

/*
 * Get the cargo this service carries.
 */
function Service::GetCargo() {
	return this.cargo;
}

/*
 * Get the transport type this service uses.
 */
function Service::GetTransportType() {
	return this.transportType;
}

/*
 * Get the sub-type this service uses.
 */
function Service::GetSubType() {
	return this.subType;
}

/*
 * Get the engine that this service uses
 */
function Service::GetEngine() {
	return this.engine;
}

/*
 * Set the engine that this service uses
 */
function Service::SetEngine(e) {
	return this.engine = e;
}

/*
 * Get the graph path this this service would run along.
 */
function Service::GetDistance() {
	return this.distance;

}

function Service::CloseService() {
	return this.distance = 1;
}

/*
 * Get the estimated income for the proposed service.
 */
function Service::GetRawIncome() {
	return this.rawIncome;
}

/*
 * Get the town coverage target percentage
 */
function Service::GetCoverageTarget() {
	return this.coverageTarget;
}

function Service::WasTransported() {
	return this.transported;
}
/*
 * Check if the service visits a target with specified Id.
 */
function Service::GoesTo(tgtId) {
	foreach(targetId in this.targetIds) {
if(targetId == tgtId) return true;
	}
	return false;
}

/*
 * Check if the service visits all in a list of targets
 */
function Service::GoesToAll(tgtIds) {
	foreach(tgtId in tgtIds) {
		if(!this.GoesTo(tgtId)) return false;
	}
	return true;
}

/*
 * Checks that all targets in this service are still valid.
 */
function Service::IsValid() {
	foreach(targetId in this.targetIds) {
		local target = ::pz.GetSchema(this.schemaId).GetTarget(targetId);
if(!AIIndustry.IsValidIndustry(targetId)&&!AITown.IsValidTown(targetId-1000000)) return false;

		if(target==0||target==null||target=={}||!target.IsValid()) return false;
	}
	return true;
}

/*
 * Checks that all targets in this service are still valid.
 */
function Service::IsTwoWayService() {

foreach(target in this.GetTargets()) {
	if (!target.IsTown()&&target.IsValid()){
		foreach (prodcargo , _ in AIIndustryType.GetProducedCargo(AIIndustry.GetIndustryType(target.GetId()))){
		foreach (acceptcargo , _ in AIIndustryType.GetAcceptedCargo(AIIndustry.GetIndustryType(target.GetId()))){
		if (this.GetCargo()==prodcargo&&prodcargo==acceptcargo) return true;
		}
		}
	return false;
	}
}
return true;
}


/*
 * Add a vehicle to the service
 */
function Service::AddVehicle(vehicleId) {
	this.vehicles.AddItem(vehicleId, 0);
	AIGroup.MoveVehicle(this.group, vehicleId);
}
function Service::RemoveVehicle(vehicleId) {
	this.vehicles.RemoveItem(vehicleId);

}

/*
 * Get the vehicles that are currently operating this service.
 */
function Service::GetVehicles() {
	return ((this.group!=null&&AIGroup.IsValidGroup(this.group))?AIVehicleList_Group(this.group):AIList());
}

function Service::VehiclesWaiting(station,sendDepot=false){
	local vlist = AIVehicleList_Station(station);
local cargo=this.GetCargo();
local z=0;
local cz=0;
for(local j = vlist.Begin(); hasnext(vlist); j = vlist.Next()) {
if (AIVehicle.IsValidVehicle(j)){
if ((this.IsTwoWayService()||AIVehicle.GetCargoLoad(j,AIEngine.GetCargoType(AIVehicle.GetEngineType(j)))<AIEngine.GetCapacity(AIVehicle.GetEngineType(j))/2)&&(AIVehicle.GetCurrentSpeed(j)<25&&AIEngine.GetMaxSpeed(AIVehicle.GetEngineType(j))>35)){
if (AIVehicle.GetCapacity(j,cargo)>0)cz++;
if (sendDepot&&AIStation.GetCargoWaiting(station,cargo)<AIEngine.GetCapacity(AIVehicle.GetEngineType(j))/2&&AIEngine.GetCargoType(AIVehicle.GetEngineType(j))==cargo) {
if(AIMap.DistanceManhattan(AIVehicle.GetLocation(j),AIStation.GetLocation(station))<4){
		AIVehicle.SendVehicleToDepotForServicing(j);
		AIVehicle.ReverseVehicle(j);
}
}
if (AIVehicle.GetAge(j)>360&&(((AIEngine.GetCargoType(AIVehicle.GetEngineType(j))==cargo||AIMap.DistanceManhattan(AIVehicle.GetLocation(j),AIStation.GetLocation(station))<8)&&AIVehicle.GetState(j)!=AIVehicle.VS_BROKEN)))z=z+min(((5/(AIMap.DistanceManhattan(AIVehicle.GetLocation(j),AIStation.GetLocation(station))+1))*1000).tointeger(),1000);
}
}}

return (cz>2)?(z/1000).tointeger():0;

}
/*
 * Check if the service turned an overall profit last year
 */
function Service::IsProfitable(poor=false) {
if (this.GetVehicles() == null||(poor==false && AIDate.GetMonth(AIDate.GetCurrentDate())<5)) return 0;
	local vlist = this.GetVehicles();
local z=0;
for(local j = vlist.Begin(); hasnext(vlist); j = vlist.Next()) {
	if (AIVehicle.IsValidVehicle(j)&&(AIVehicle.GetAge(j)>((GetDistance()+70)*200)/AIEngine.GetMaxSpeed(AIVehicle.GetEngineType(j)))&& AIVehicle.GetProfitLastYear(j)<0&&AIVehicle.GetProfitThisYear(j)<0){
if (this.distance<400||AIVehicle.GetCurrentSpeed(j)<30){
z++;
}
}
}
	
	return z;
}


function Service::WithProfit(poor=false) {

if (this.GetVehicles() == null) return 0
	local vlist = this.GetVehicles();
local z=0;
for(local j = vlist.Begin(); hasnext(vlist); j = vlist.Next()) {
	if (AIVehicle.IsValidVehicle(j)){
	if(AIVehicle.GetAge(j)>((GetDistance()+70)*200)/AIEngine.GetMaxSpeed(AIVehicle.GetEngineType(j))){
	if (AIVehicle.GetProfitLastYear(j)<0&&AIVehicle.GetProfitThisYear(j)<0){
if (this.distance<400||AIVehicle.GetCurrentSpeed(j)<30){
z++;
}}
}
}
}
	
	return this.GetVehicles().Count()-z;
}


/*
 * Get the number of vehicles that are currently operating this service.
 */
function Service::GetActualFleetSize() {

	return (this.GetVehicles() != null) ? this.GetVehicles().Count() : 0;
}

/*
 * Get a string representation of this service.
 */
function Service::_tostring() {
	local strType = "";
	if(transportType == AITile.TRANSPORT_ROAD) {
(::german)?strType = (subType == AIRoad.ROADTYPE_ROAD) ? "" : "(Tram)":strType = (subType == AIRoad.ROADTYPE_ROAD) ? "road" : "tram";
	} else if(transportType == AITile.TRANSPORT_AIR) {
		strType = "air";
	}

	local schema = ::pz.GetSchema(this.schemaId);
	local last = this.targetIds.len() - 1;
local t1=((this.targetIds[0]>=1000000&&AITown.IsValidTown(this.targetIds[0]-1000000))?AITown.GetName(targetIds[0]-1000000):(AIIndustry.IsValidIndustry(this.targetIds[0])?AIIndustry.GetName(this.targetIds[0]):"???"));
local t2=((this.targetIds[last]>=1000000&&AITown.IsValidTown(this.targetIds[last]-1000000))?AITown.GetName(targetIds[last]-1000000):(AIIndustry.IsValidIndustry(this.targetIds[last])?AIIndustry.GetName(this.targetIds[last]):"???"));

	local strTgts = t1 + ((::german)?" nach ":" to ") + t2;

	local str = "";
	if(this.targetIds.len() == 2) {
(::german)?str = AICargo.GetCargoLabel(this.cargo) + " von " + strTgts + strType:str = AICargo.GetCargoLabel(this.cargo) + " from " + strTgts + " by " + strType;
	}
	return str;
}

/*
 * Saves data to a table.
 */
function Service::Serialize() {
	local data = {};
	
	data[SRLZ_SCHEMA_ID] <- this.schemaId;
	data[SRLZ_TARGET_IDS] <- this.targetIds;
	data[SRLZ_CARGO] <- this.cargo;
	data[SRLZ_TRANSPORT_TYPE] <- this.transportType;
	data[SRLZ_SUB_TYPE] <- this.subType;
	data[SRLZ_ENGINE] <- this.engine;
	data[SRLZ_PROFITABILITY] <- this.profitability;
	data[SRLZ_GROUP] <- this.group;
	data[SRLZ_DISTANCE] <- this.distance;
	data[SRLZ_RAW_INCOME] <- this.rawIncome;
	data[SRLZ_COVERAGE_TARGET] <- this.coverageTarget;
	data[SRLZ_TRANSPORTED] <- this.transported;
	data[SRLZ_LOCKED] <- this.locked;
	
	return data;
}

/*
 * Loads data from a table.
 */
function Service::Unserialize(data) {
	this.schemaId = data[SRLZ_SCHEMA_ID];
	this.targetIds = data[SRLZ_TARGET_IDS];
	this.cargo = data[SRLZ_CARGO];
	this.transportType = data[SRLZ_TRANSPORT_TYPE];
	this.subType = data[SRLZ_SUB_TYPE];
	this.engine = data[SRLZ_ENGINE];
	this.profitability = data[SRLZ_PROFITABILITY];
	this.group = data[SRLZ_GROUP];
	this.distance = data[SRLZ_DISTANCE];
	this.rawIncome = data[SRLZ_RAW_INCOME];
	this.coverageTarget = data[SRLZ_COVERAGE_TARGET];
	this.transported = data[SRLZ_TRANSPORTED];
	this.transported = data[SRLZ_LOCKED];
	this.vehicles = AIList();
}

/*
 * Compare this service to another. This function returns 0 (i.e. equal) for 
 * services that go to/from the same towns, and otherwise orders services by
 * profitability. 
 */
function Service::_cmp(svc) {
	local same = this.cargo == svc.cargo;
	same = same && this.transportType == svc.transportType;
	same = same && this.subType == svc.subType;
	if(same) {
		foreach(targetId in this.GetTargetIds()) {
			same = same && svc.GoesTo(targetId);
		}
	}
	if(same) return 0; 
	if(this.rawIncome < svc.rawIncome) return -1;
	return 1;
}
