/*
 * This file is part of SynTrans, which is an AI for OpenTTD
 *
 * SynTrans is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License
 *
 * SynTrans is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with SynTrans; If not, see <http://www.gnu.org/licenses/> or
 * write to the Free Software Foundation, Inc., 51 Franklin Street, 
 * Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */
import("util.superlib","SuperLib",36);
import("pathfinder.rail", "RailPathFinder", 1);
import("pathfinder.road", "RoadPathFinder", 4);

require("network.nut");
require("town.nut");
require("pathfinder.nut");
require("rpf.nut");
require("convoy.nut");

require("valuators.nut");
require("utils.nut");

Result <- SuperLib.Result;
Log <- SuperLib.Log;
Helper <- SuperLib.Helper;
ScoreList <- SuperLib.ScoreList;
Money <- SuperLib.Money;
Tile <- SuperLib.Tile;
Direction <- SuperLib.Direction;
Engine <- SuperLib.Engine;
Vehicle <- SuperLib.Vehicle;
Station <- SuperLib.Station;
Airport <- SuperLib.Airport;
Industry <- SuperLib.Industry;
//Town <- SuperLib.Town;
Order <- SuperLib.Order;
OrderList <- SuperLib.OrderList;
Road <- SuperLib.Road;
RoadBuilder <- SuperLib.RoadBuilder;

aiInstance <- null;

class SynTrans extends AIController {
	networks = null;
	airnet = null;
	enableAir = null;
	enableRoad = null;
	saved = null;
	center = null;
	crashes = null;
	fast = null;
	blacklist = null; // For moneymaker routes
	engineBlacklist = null; // For moneymaker routes
	constructor() {
		networks = [];
		crashes = []
		enableAir = false;
		enableRoad = false;
		saved = false;
		fast = false;
		blacklist = [];
		engineBlacklist = [];
		//airnet = AirNet();
	}
	function Start();
	function Save();
	function Load(version,data);
	function BuildMoneymaker();
}

function SynTrans::Start()
{
	local oldMoney = 0;
	AIController.Sleep(5);
	aiInstance = this;
	AIRoad.SetCurrentRoadType(AIRoad.ROADTYPE_ROAD);
	AIRail.SetCurrentRailType(AIRailTypeList().Begin());
	Log.Info("SynTrans Started",Log.LVL_SUB_DECISIONS);
	if(AIGameSettings.IsDisabledVehicleType(AIVehicle.VT_AIR) && AIGameSettings.IsDisabledVehicleType(AIVehicle.VT_ROAD)) {
		Log.Error("This is a road and air only AI!  I cannot play unless both road and air is enabled!");
		Log.Error("If you do not know how to fix this issue go to Advanced Settings -> Competitors -> Computer Players -> Enable Road or Air");
		Log.Error("Once enabled, go ahead and click the 'Restart AI' button on this dialog");
		Log.Error("Until then, I will crash because I cannot play!");
		FATAL_NoSupportedVehicles();
	}
	if(saved == true) {
		Log.Info("Loading networks...",Log.LVL_SUB_DECISIONS);
		local stationlist = null;
		// Load RoadNetwork
		if(!AIGameSettings.IsDisabledVehicleType(AIVehicle.VT_ROAD)) {
			local depots = AIDepotList(AITile.TRANSPORT_ROAD);
			local newNet = Network(AIVehicle.VT_ROAD);
			stationlist = AIStationList(AIStation.STATION_BUS_STOP);
			foreach(station, _ in stationlist) {
				// TODO update for multiple stations per town
				local town = AIStation.GetNearestTown(station);
				local found = false;
				foreach(stown in newNet.towns) {
					// Make sure the town is not already on the list
					// If it is, put the station in it and continue;
					if(stown.id == town) {
						Log.Info("Adding station of same town to " + AITown.GetName(town), Log.LVL_SUB_DECISIONS);
						stown.stations.append(station);
						found = true;
						break;
					}
				}
				if(found) continue;
				Log.Info("Processing town " + AITown.GetName(town), Log.LVL_SUB_DECISIONS);
				//newNet.AddTown(town);
				local theTown = Town(town);
				theTown.stations.append(station);
				// Find the depot for the station... the nearest one is most appropriate
				/*local dtilex = AIMap.GetTileX(depot);
				Log.Info("Depot Tile X: " + dtilex);
				local dtiley = AIMap.GetTileY(depot);
				Log.Info("Depot Tile Y: " + dtiley);
				local stilex = AIMap.GetTileX(AIStation.GetLocation(station));
				Log.Info("Station Tile X: " + stilex);
				local stiley = AIMap.GetTileY(AIStation.GetLocation(station));
				Log.Info("Station Tile Y: " + stiley);
				for(local i = 10;i < 1000;i += 5) {
					if(stilex - i < dtilex && stilex + i > dtilex && stiley - i < dtilex && stiley + i > dtiley) {
						theTown.depot = depot;
						continue;
					}
				}*/
				// Just get the closest depot
				depots.Valuate(AIMap.DistanceManhattan, AIStation.GetLocation(station));
				theTown.depot = depots.Begin();
				if(!theTown.depot) {
					Log.Error("Could not find a depot for " + AITown.GetName(theTown.id) + "... this could result in fatality later");
				}
				newNet.towns.append(theTown);
			}
			newNet.defaultEngine = newNet.ChooseVehicle();
			Log.Info("Adding network", Log.LVL_SUB_DECISIONS);
			networks.append(newNet);
			enableRoad = true;
		}
		// Load AirNetwork
		if(!AIGameSettings.IsDisabledVehicleType(AIVehicle.VT_AIR)) {
			airnet = Network(AIVehicle.VT_AIR);
			stationlist = AIStationList(AIStation.STATION_AIRPORT);
			foreach(station, _ in stationlist)
			{
				local town = AIStation.GetNearestTown(station);
				Log.Info("Processing town " + AITown.GetName(town));
				//airnet.AddTown(town);
				local theTown = Town(town);
				theTown.stations.append(station);
				theTown.depot = Airport.GetHangarTile(station);
				airnet.towns.append(theTown);
			}
			airnet.defaultEngine = airnet.ChooseVehicle();
			enableAir = true;
		}
	}
	else {
		// Name the company
		if(!AICompany.SetName("Synaptic Transport Inc.")) {
			if(!AICompany.SetName("Synchronized Transport Inc.")) {
				if(!AICompany.SetName("Another Synaptic Transport Inc.")) {
					local i = 1;
					while(!AICompany.SetName("Synaptic Impersonator #" + i)) {
						i = i + 1;
					}
				}
			}
		}
		local built = false;
		local tries = 0;
		Log.Info("Building HQ...",Log.LVL_SUB_DECISIONS);
		// Build the HQ
		local townlist = AITownList();
		townlist.Valuate(AITown.GetPopulation);
		townlist.Sort(AIList.SORT_BY_VALUE, false);
		foreach(hqtown, _ in townlist) {
			Log.Info("Attempting to build HQ in " + AITown.GetName(hqtown),Log.LVL_SUB_DECISIONS);
			local skip = false;
			local tilelist = Tile.GetTownTiles(hqtown);
			tilelist.Valuate(AITile.GetClosestTown);
			tilelist.KeepValue(hqtown);
			// Scan for any company HQs and go in a different city if one is there
			for(local i = 0;i < 16;i++) {
				if(AICompany.ResolveCompanyID(i) == AICompany.COMPANY_INVALID) continue;
				local tile = AICompany.GetCompanyHQ(i);
				if(tile != AIMap.TILE_INVALID) {
					if(AITile.GetClosestTown(tile) == hqtown) {
						Log.Info("Found another player's HQ in town " + AITown.GetName(hqtown));
						skip = true;
						break;
					}
				}
			}
			if(skip) continue;
			tilelist.Valuate(AITile.IsBuildableRectangle,2,2);
			tilelist.KeepValue(1);
			tilelist.Valuate(AIMap.DistanceManhattan,AITown.GetLocation(hqtown));
			tilelist.Sort(AIList.SORT_BY_VALUE, true);
			if(!AICompany.BuildCompanyHQ(tilelist.Begin())) {
				Log.Warning("First attempt to build HQ failed.");
			}
			else {
				built = true;
				center = hqtown;
			}
			while(!built && tries <= 5) {
				if(!AICompany.BuildCompanyHQ(tilelist.Next())) {
					Log.Warning("Could not build HQ!");
					tries++;
				}
				else {
					built = true;
					center = hqtown;
					break;
				}
			}
			if(built) {
				Log.Info("Successfully Built HQ :)", Log.LVL_SUB_DECISIONS);
				break;
			}
		}
		if(!center) {
			Log.Warning("Completely failed to build HQ :(");
		}
		if(!AIGameSettings.IsDisabledVehicleType(AIVehicle.VT_RAIL)) {
			// Start building some moneymaker routes
			if(!BuildMoneymaker()) BuildMoneymaker();
			if(!BuildMoneymaker()) BuildMoneymaker();
			if(!BuildMoneymaker()) BuildMoneymaker();
		}
		// Make initial network
		// Make (but dont start) Airnet
		airnet = Network(AIVehicle.VT_AIR);
		networks.append(Network(AIVehicle.VT_ROAD, center));
		// Additionaly, start with 2 more networks to develop more smoothly
		networks.append(Network(AIVehicle.VT_ROAD));
		networks.append(Network(AIVehicle.VT_ROAD));
		if(!AIGameSettings.IsDisabledVehicleType(AIVehicle.VT_AIR)) enableAir = true;
		else enableRoad = true;
		
	}
	local townlist;
	local networkStart;
	local event;
	while(true)
	{
		// Manage events...
		while(AIEventController.IsEventWaiting()) {
			event = AIEventController.GetNextEvent();
			switch(event.GetEventType()) {
				case AIEvent.ET_ENGINE_PREVIEW:
					local preview = AIEventEnginePreview.Convert(event);
					preview.AcceptPreview();
					Log.Info("Accepted preview for vehicle " + preview.GetName());
					break;
				case AIEvent.ET_VEHICLE_CRASHED:
					Log.Info("Caught vehicle crash report!", Log.LVL_SUB_DECISIONS);
					local crash = AIEventVehicleCrashed.Convert(event);
					// If it was a rail crossing and another player was not considerate enough to fix it or just plain want
					// to destroy our vehicles, we will fix the problem for them :)
					if(crash.GetCrashReason() == AIEventVehicleCrashed.CRASH_RV_LEVEL_CROSSING) {
						Log.Info("Trying to fix road vehicle crash with train...", Log.LVL_SUB_DECISIONS);
						local site = crash.GetCrashSite();
						foreach(crash in crashes) if(site == crash) continue;
						local adjTiles = Tile.GetNeighbours4MainDir(site);
						local adjTiles2 = Tile.GetNeighbours4MainDir(site);
						adjTiles2.Valuate(AIBridge.IsBridgeTile);
						adjTiles2.RemoveValue(0);
						foreach(bridge, _ in adjTiles2) {
							AIBridge.RemoveBridge(bridge);
							AIRail.SetCurrentRailType(AIRailTypeList().Begin());
							local raildir = Direction.TurnDirClockwise90Deg(Direction.GetDirectionToAdjacentTile(site, bridge), 1);
							local railResult = null;
							if(raildir == Direction.DIR_NE || raildir == Direction.DIR_SW) railResult = AIRail.BuildRailTrack(bridge, AIRail.RAILTRACK_NE_SW);
							if(raildir == Direction.DIR_NW || raildir == Direction.DIR_SE) railResult = AIRail.BuildRailTrack(bridge, AIRail.RAILTRACK_NW_SE);
							if(!railResult) Log.Warning("Failed to build rail track: " + AIError.GetLastErrorString());
						}
						adjTiles.Valuate(AITile.HasTransportType, AITile.TRANSPORT_ROAD);
						adjTiles.RemoveValue(0);
						Log.Info("Beginning Conversion...");
//						Helper.SetSign(site, "site", true);
//						Helper.SetSign(adjTiles.Begin(), "adj", true);
						local result = Road.ConvertRailCrossingToBridge(site, adjTiles.Begin());
						local i = 0;
						crashes.append(site);
						while(!result.succeeded && i < 8) {
							if(result.permanently) {
								Log.Warning("Could not convert rail crossing to bridge!");
								break;
							}
							else {
								Log.Info("Trying again...", Log.LVL_SUB_DECISIONS);
								result = Road.ConvertRailCrossingToBridge(site, adjTiles.Begin());
								AIController.Sleep(250);
							}
							i++;
						}
					}
			}
		}
		/*// Give money to the networks
		local income = AICompany.GetBankBalance(AICompany.COMPANY_SELF) - oldMoney;
		Log.Info("My income is " + income, Log.LVL_SUB_DECISIONS);
		airnet.money += income * 0.7;
		income = income * 0.7;
		foreach(network in networks) {
			network.money += income / networks.len();
		}
		oldMoney = AICompany.GetBankBalance(AICompany.COMPANY_SELF);*/
		// finally, is the airnet ready to start?
		//if(enableAir) airnet.ManageNetwork();
		if(enableRoad) {
			foreach(network in networks) {
				network.ManageNetwork();
				if(enableAir) airnet.ManageNetwork();
			}
		}
		else airnet.ManageNetwork();
		// Should we build a new network?
		if(!AIGameSettings.IsDisabledVehicleType(AIVehicle.VT_ROAD)) {
		if(AIController.GetSetting("max_networks") > networks.len()) {
				if(AICompany.GetBankBalance(AICompany.COMPANY_SELF) > 250000) {
					Log.Info("==========BUILDING NEW NETWORK!==========",Log.LVL_SUB_DECISIONS);
					townlist = AITownList();
					// take out the cities we already own
					foreach(network in networks) {
						foreach(town in network.towns) {
							townlist.RemoveItem(town.id);
						}
					}
					// now find the biggest city
					townlist.Valuate(AITown.GetPopulation)
					townlist.Sort(AIList.SORT_BY_VALUE, false);
					networkStart = townlist.Begin();
					local newNet = Network(AIVehicle.VT_ROAD,networkStart);
					networks.append(newNet);
				}
			}
		}
		if(AICompany.GetQuarterlyCompanyValue(AICompany.COMPANY_SELF, AICompany.CURRENT_QUARTER) > 250000 && !enableAir && !AIGameSettings.IsDisabledVehicleType(AIVehicle.VT_AIR)) {
			Log.Info("==========STARTING AIRCRAFT SYSTEM==========", Log.LVL_SUB_DECISIONS);
			enableAir = true;
		}
		if(AICompany.GetQuarterlyCompanyValue(AICompany.COMPANY_SELF, AICompany.CURRENT_QUARTER) > 500000 && !enableRoad && !AIGameSettings.IsDisabledVehicleType(AIVehicle.VT_ROAD)) {
			Log.Info("==========STARTING ROAD NETWORKS==========", Log.LVL_SUB_DECISIONS);
			enableRoad = true;
			fast = true;
		}
		if(AICompany.GetBankBalance(AICompany.COMPANY_SELF) > 500000 && !fast) fast = true;
		Log.Info("Waiting...",Log.LVL_SUB_DECISIONS);
		AIController.Sleep(10);
	}
}

function SynTrans::BuildMoneymaker() {
	WaitForMoney(50000, -1);
	Log.Info("Creating moneymaker route...", Log.LVL_SUB_DECISIONS);

	local cargos = AICargoList();
	cargos.Valuate(AICargo.HasCargoClass, AICargo.CC_BULK);
	cargos.KeepValue(1);
	// Go for the most expensive cargo
	//cargos.Valuate(AICargo.GetCargoIncome, 50, 30);
	// Go for a random cargo
	cargos.Valuate(AIBase.RandItem);
	cargos.Sort(AIList.SORT_BY_VALUE, false);
	if(cargos.IsEmpty()) {
		Log.Warning("No cargos to work with!");
		return null;
	}
	local cargo = cargos.Begin();
	Log.Info("Route cargo: " + AICargo.GetCargoLabel(cargo));
	
	local industries = AIIndustryList_CargoProducing(cargo);
	industries.Valuate(AIIndustry.GetAmountOfStationsAround);
	industries.KeepValue(0);
	industries.RemoveList(Helper.SquirrelListToAIList(blacklist));
	industries.Valuate(AIIndustry.GetLastMonthProduction, cargo);
	industries.Sort(AIList.SORT_BY_VALUE, false);
	if(industries.IsEmpty()) {
		Log.Warning("Could not find good industry!");
		return false;
	}
	local from = industries.Begin();
	
	industries = AIIndustryList_CargoAccepting(cargo);
	//industries.Valuate(AIIndustry.GetAmountOfStationsAround);
	//industries.KeepValue(0);
	//industries.RemoveList(Helper.SquirrelListToAIList(blacklist));
	industries.Valuate(AIIndustry.GetDistanceManhattanToTile, AIIndustry.GetLocation(from));
	industries.KeepBetweenValue(75, 150);
	industries.Sort(AIList.SORT_BY_VALUE, false);
	if(industries.IsEmpty()) {
		Log.Warning("Could not find good industry near " + Tile.GetTileString(AIIndustry.GetLocation(from)));
		blacklist.append(from);
		return false;
	}
	local to = industries.Begin();
	
	Log.Info("Route industries: " + AIIndustry.GetName(from) + " => " + AIIndustry.GetName(to), Log.LVL_SUB_DECISIONS);
	
	
	local tiles = Tile.MakeTileRectAroundTile(AIIndustry.GetLocation(from), 15);
	tiles.Valuate(AITile.IsBuildableRectangle, 7, 2);
	tiles.KeepValue(1);
	tiles.Valuate(AITile.GetDistanceManhattanToTile, AIIndustry.GetLocation(from));
	tiles.RemoveAboveValue(4);
	tiles.Valuate(GetDistanceFromWater);
	//tiles.KeepAboveValue(Helper.Max(Station.STATION_RAILMAP.len(), Station.STATION_RAILMAP[0].len()) + 2);
	tiles.Sort(AIList.SORT_BY_VALUE, false);
	if(tiles.IsEmpty()) {
		Log.Error("Cannot not build station near industry!");
		blacklist.append(from);
		return false;
	}
	local tile = tiles.Begin();
	// Build station
	Tile.FlatternRect(tile, 5, 1);
	if(!AIRail.BuildRailStation(tile, AIRail.RAILTRACK_NE_SW, 1, 5, AIStation.STATION_NEW)) {
		Log.Error("Could not build station: " + AIError.GetLastErrorString());
		blacklist.append(from);
		return false;
	}
	local fromid = AIStation.GetStationID(tile);
	
	tiles = Tile.MakeTileRectAroundTile(AIIndustry.GetLocation(to), 15);
	tiles.Valuate(AITile.IsBuildableRectangle, 7, 2);
	tiles.KeepValue(1);
	tiles.Valuate(AITile.GetDistanceManhattanToTile, AIIndustry.GetLocation(to));
	tiles.RemoveAboveValue(4);
	tiles.Valuate(GetDistanceFromWater);
	tiles.Sort(AIList.SORT_BY_VALUE, false);
	if(tiles.IsEmpty()) {
		Log.Error("Cannot not build station near industry!");
		blacklist.append(from);
		return false;
	}
	tile = tiles.Begin();
	Tile.FlatternRect(tile, 5, 1);
	if(!AIRail.BuildRailStation(tile, AIRail.RAILTRACK_NE_SW, 1, 5, AIStation.STATION_NEW)) {
		Log.Error("Could not build station!");
		blacklist.append(to);
		return false;
	}
	local toid = AIStation.GetStationID(tile);
	
	// Connect;
	Log.Info("Connecting industries...", Log.LVL_SUB_DECISIONS);
	local pf = RRPathfinder([Direction.GetTileInDirection(AIStation.GetLocation(fromid), Direction.DIR_SW, 5),Direction.GetTileInDirection(AIStation.GetLocation(fromid), Direction.DIR_SW, 4)], [Direction.GetTileInDirection(AIStation.GetLocation(toid), Direction.DIR_SW, 5),Direction.GetTileInDirection(AIStation.GetLocation(toid), Direction.DIR_SW, 4)], false, [], 2.0);
	Log.Info("Finding Path...");
	local ret = pf.FindPath();
	if(!ret) {
		Log.Warning("Failed to pathfind rail!");
		blacklist.append(from);
		return false;
	}
	
	Log.Info("Building Path...");
	ret = pf.BuildPath();
	if(ret != "success") {
		Log.Warning("Failed to build rail!");
		return false;
	}
	Log.Info("Success!", Log.LVL_SUB_DECISIONS);
	
	// Find a good spot to build depot
	local tiles = Tile.MakeTileRectAroundTile(AIStation.GetLocation(fromid), 12);
	tiles.Valuate(IsAlongRail);
	tiles.KeepValue(1);
	tiles.Valuate(AITile.IsBuildable);
	tiles.KeepValue(1);
	tiles.Valuate(AIMap.DistanceManhattan, AIStation.GetLocation(fromid));
	tiles.Sort(AIList.SORT_BY_VALUE, false);
	local tile = tiles.Begin();
	Log.Info("Building Depot for " + AIStation.GetName(fromid), Log.LVL_SUB_DECISIONS);
	local built = false;
	local front = null;
	while(!built) {
	// Find the tile where the railroad is at
		local ts = Tile.GetNeighbours4MainDir(tile);
		ts.Valuate(AIRail.IsRailTile);
		ts.KeepValue(1);
		// Now build
		front = ts.Begin();
		if(!AIRail.BuildRailDepot(tile, front)) {
			tile = tiles.Next();
			if(!tiles.IsEnd()) {}
			else {
				Log.Warning("Could not build depot!");
				break;
			}
		}
		else built = true;
	}
	// Connect the depot to the railway, and we are done!
	local dir = Direction.GetDirectionToAdjacentTile(tile, front);
	// Remove signal
	if(AIRail.GetSignalType(front, Direction.GetAdjacentTileInDirection(front, Direction.TurnDirClockwise90Deg(dir, 1))) != AIRail.SIGNALTYPE_NONE) {
		AIRail.RemoveSignal(front, Direction.GetAdjacentTileInDirection(front, Direction.TurnDirClockwise90Deg(dir, 1)));
	}
	if(AIRail.GetSignalType(front, Direction.GetAdjacentTileInDirection(front, Direction.TurnDirAntiClockwise90Deg(dir, 1))) != AIRail.SIGNALTYPE_NONE) {
		AIRail.RemoveSignal(front, Direction.GetAdjacentTileInDirection(front, Direction.TurnDirAntiClockwise90Deg(dir, 1)));
	}
	// Connect to railway
	AIRail.BuildRailTrack(front, ConvertDirectionToRail(Direction.TurnDirClockwise45Deg(Direction.OppositeDir(dir), 1)));
	AIRail.BuildRailTrack(front,ConvertDirectionToRail(Direction.TurnDirAntiClockwise45Deg(Direction.OppositeDir(dir), 1)));
	
	local depot = tile;
	
	// Find a good spot to build
	local tiles = Tile.MakeTileRectAroundTile(AIStation.GetLocation(toid), 12);
	tiles.Valuate(IsAlongRail);
	tiles.KeepValue(1);
	tiles.Valuate(AITile.IsBuildable);
	tiles.KeepValue(1);
	tiles.Valuate(AIMap.DistanceManhattan, AIStation.GetLocation(toid));
	tiles.Sort(AIList.SORT_BY_VALUE, false);
	local tile = tiles.Begin();
	Log.Info("Building Depot for " + AIStation.GetName(toid), Log.LVL_SUB_DECISIONS);
	local built = false;
	local front = null;
	while(!built) {
	// Find the tile where the railroad is at
		local ts = Tile.GetNeighbours4MainDir(tile);
		ts.Valuate(AIRail.IsRailTile);
		ts.KeepValue(1);
		// Now build
		front = ts.Begin();
		if(!AIRail.BuildRailDepot(tile, front)) {
			tile = tiles.Next();
			if(!tiles.IsEnd()) {}
			else {
				Log.Warning("Could not build depot!");
				break;
			}
		}
		else built = true;
	}
	// Connect the depot to the railway, and we are done!
	local dir = Direction.GetDirectionToAdjacentTile(tile, front);
	// Remove signal
	if(AIRail.GetSignalType(front, Direction.GetAdjacentTileInDirection(front, Direction.TurnDirClockwise90Deg(dir, 1))) != AIRail.SIGNALTYPE_NONE) {
		AIRail.RemoveSignal(front, Direction.GetAdjacentTileInDirection(front, Direction.TurnDirClockwise90Deg(dir, 1)));
	}
	if(AIRail.GetSignalType(front, Direction.GetAdjacentTileInDirection(front, Direction.TurnDirAntiClockwise90Deg(dir, 1))) != AIRail.SIGNALTYPE_NONE) {
		AIRail.RemoveSignal(front, Direction.GetAdjacentTileInDirection(front, Direction.TurnDirAntiClockwise90Deg(dir, 1)));
	}
	// Connect to railway
	AIRail.BuildRailTrack(front, ConvertDirectionToRail(Direction.TurnDirClockwise45Deg(Direction.OppositeDir(dir), 1)));
	AIRail.BuildRailTrack(front,ConvertDirectionToRail(Direction.TurnDirAntiClockwise45Deg(Direction.OppositeDir(dir), 1)));
	local redo = true;
	local train = null;
	while(redo) {
		redo = false;
		// Briefly choose engine
		local engines = AIEngineList(AIVehicle.VT_RAIL);
		engines.Valuate(AIEngine.IsWagon);
		engines.KeepValue(0); // No wagons
		engines.Valuate(AIEngine.CanRunOnRail, AIRailTypeList().Begin());
		engines.KeepValue(1); // Must be able to run on our rail type
		engines.RemoveList(Helper.SquirrelListToAIList(engineBlacklist));
	
		engines.Valuate(AIEngine.GetMaxSpeed);
		engines.Sort(AIList.SORT_BY_VALUE, false);
		if(engines.IsEmpty()) {
			Log.Warning("No good engines!");
			return false;
		}
		local engine = engines.Begin();
	
		// Briefly choose wagon
		local wagons = AIEngineList(AIVehicle.VT_RAIL);
		wagons.Valuate(AIEngine.IsWagon);
		wagons.KeepValue(1); // Only wagons
		wagons.Valuate(AIEngine.CanRefitCargo, cargo);
		wagons.KeepValue(1);
		wagons.Valuate(AIEngine.GetCapacity);
		wagons.Sort(AIList.SORT_BY_VALUE, false);
		if(wagons.IsEmpty()) {
			Log.Warning("No good wagons for cargo: " + AICargo.GetCargoLabel(cargo));
			return false;
		}
		local wid = wagons.Begin();
		
		WaitForMoney(AIEngine.GetPrice(engine) + AIEngine.GetPrice(wid) * 9 + 1000, -1);

		// Build train
		train = AIVehicle.BuildVehicle(depot, engine);
		if(!AIVehicle.IsValidVehicle(train)) {
			Log.Warning("Could not build train engine: " + AIError.GetLastErrorString());
			return false;
		}
		for(local i = 0;i < 8;i++) {
			local wagon = AIVehicle.BuildVehicle(depot, wid);
			//local wagon = AIVehicle.BuildVehicle(depot, wid);
			if(!AIVehicle.IsValidVehicle(wagon)) {
				Log.Warning("Could not build wagon!");
				return false;
			}
			if(!AIVehicle.RefitVehicle(wagon, cargo)) {
				Log.Warning("Could not refit wagon to " + AICargo.GetCargoLabel(cargo));
				return false;
			}
			if(!AIVehicle.MoveWagon(wagon, 0, train, 0)) {
				Log.Warning("Could not move wagon to trian engine!");
				Log.Warning("Blacklisting engine: " + AIEngine.GetName(engine));
				engineBlacklist.append(engine);
				// No return since we keep trying until an engine works
				redo = true;
			}
		}
	}
	// Add orders
	AIOrder.AppendOrder(train, AIStation.GetLocation(fromid), AIOrder.OF_FULL_LOAD_ANY | AIOrder.OF_NON_STOP_INTERMEDIATE);
	AIOrder.AppendOrder(train, AIStation.GetLocation(toid), AIOrder.OF_NON_STOP_INTERMEDIATE);
	
	if(!AIVehicle.StartStopVehicle(train)) {
		Log.Warning("Could not start train!");
		return false;
	}
	return true;
}

function SynTrans::Save()
{
	local syntrans = {};
	return syntrans;
}

function SynTrans::Load(version,data)
{
	saved = true;
}
