/*
 * This file is part of SynTrans, which is an AI for OpenTTD
 *
 * SynTrans is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License
 *
 * SynTrans is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with SynTrans; If not, see <http://www.gnu.org/licenses/> or
 * write to the Free Software Foundation, Inc., 51 Franklin Street, 
 * Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

class Network
{
	lastVehicleManage = null;
	lastVehicleBuild = null;
	lastNewTown = null;
	lastUpdateEngine = null;
	lastStartUpgrade = null;
	towns = null;
	blacklist = null;
	vehicles = null;
	defaultEngine = null;
	center = null;
	size = null;
	type = null;
	replace = null;
	noReplace = null;
	serviceStation = null;
	routeDistance = null;
	constructor(vtype,epicenter = null) {
		this.towns         = [];
		this.vehicles      = [];
		this.blacklist     = AIList();
		this.defaultEngine = null;
		this.center        = epicenter;
		this.size          = 1;
		this.type          = vtype;
	}
	function ManageNetwork();
	function ManageVehicles();
	function CreateLink();
	function BuyVehicles(depot,engine,number,order1,order2, essential = false);
	function SellVehicle(vehicle);
	function FindTown(chosenCity = null, desprate = false);
	function FindCloseConnection(town);
	function AddTown(theTown);
	function LeaveTown(index);
	function Connect(point1,point2);
	function PerformFullUpgrade();
	function ExpandStation(station = null);
	function ChooseVehicle(cheap = false);
	/*function BuildServiceStation();
	function SendToServiceStation(v);
	function ReturnAllToNetwork();*/ // Does this to ALL vehicles at the service station 
	function StopInDepot(v, station); // Intended for aircraft only!
}

// Function from TutorialAI's Road Tutorial
function TownDistValuator(town1, town2, targetDistance)
{
	local res = Helper.Abs(AIMap.DistanceManhattan(AITown.GetLocation(town1), AITown.GetLocation(town2)) - targetDistance);
	return res;
}

function Network::ManageNetwork()
{
	Log.Info("=========================================",Log.LVL_SUB_DECISIONS);
	Log.Info("==========MANAGING NEXT NETWORK==========",Log.LVL_SUB_DECISIONS);
	Log.Info("=========================================",Log.LVL_SUB_DECISIONS);
	if(type == AIVehicle.VT_AIR) Log.Info("This is an airport network",Log.LVL_SUB_DECISIONS);
	// Verify there is no shortage on the vehicle limit
	if(type == AIVehicle.VT_ROAD) {
		if(Vehicle.GetVehiclesLeft(AIVehicle.VT_ROAD) < 5) {
			Log.Warning("Road vehicle Limit Reached!  Please increase the vehicle limit in the advanced settings.  Until then, this AI will do nothing regarding road vehicles.");
			return;
		}
	}
	if(type == AIVehicle.VT_AIR && AIVehicleList().Count() < 5) {
		if(Vehicle.GetVehiclesLeft(AIVehicle.VT_AIR) < 2) {
			Log.Warning("Aircraft vehicle Limit Reached!  Please increase the vehicle limit in the advanced settings.  Until then, this AI will do nothing regarding aircraft.");
			return;
		}
	}
	// Upon an empty network
	if(towns.len() == 0) {
		// Calculate the route distance from the setting
		if(type == AIVehicle.VT_ROAD) routeDistance = 25 + AIController.GetSetting("route_distance") * 25;
		else routeDistance = 100 + AIController.GetSetting("route_distance") * 200;
		Log.Info("Set route distance to " + routeDistance, Log.LVL_SUB_DECISIONS);
		
		
		Log.Info("Building First City",Log.LVL_SUB_DECISIONS);
		if(type == AIVehicle.VT_AIR) defaultEngine = ChooseVehicle(true);
		else defaultEngine = ChooseVehicle();
		if(defaultEngine < 0) {
			Log.Warning("Could not find a road vehicle to use!  Cannot play this time...");
			return;
		}
		lastUpdateEngine = AIController.GetTick();
		Log.Info("Default engine is " + AIEngine.GetName(defaultEngine));
		if(type == AIVehicle.VT_AIR) {
				while(!Money.MakeSureToHaveAmount((AIAirport.GetPrice(Airport.GetLastAvailableAirportType(defaultEngine)) * 2) + (AIEngine.GetPrice(defaultEngine) * 2) + 1000)) {
					Log.Info("Waiting to have enough money to start... Need " + (AIAirport.GetPrice(Airport.GetLastAvailableAirportType(defaultEngine)) * 2) + (AIEngine.GetPrice(defaultEngine) * 2) + 1000 + " total");
					AIController.Sleep(100);
				}
		}
		local town1;
		local townlist = AITownList();
		// Remove towns occupied by other networks
		foreach(network in aiInstance.networks) {
			foreach(town in network.towns) {
				townlist.RemoveItem(town.id);
			}
		}
		townlist.Valuate(AITown.GetPopulation);
		if(type == AIVehicle.VT_AIR) townlist.RemoveAboveValue(10000);
		townlist.Sort(AIList.SORT_BY_VALUE,false);
		townlist.RemoveList(blacklist);
		if(center == null) {
			Log.Info("No city specified, finding instead");
			town1 = townlist.Begin();
		}
		else {
			town1 = center;
			townlist.Begin();
		}
		if(town1 == null) Log.Error("Cannot create initial city");
		center = AITown.GetLocation(town1);
		local result = AddTown(town1);
		if(!result) {
			Log.Warning("Failed to build the inital city.  Leaving and finding a new one next time we have the chance on this network.");
			blacklist.AddItem(town1,0);
			center = null;
			return;
		}
		Log.Info("STATIONS LEN: " + towns[0].stations.len());
		if(type != AIVehicle.VT_AIR) ExpandStation(towns[0].stations[0]);
		Log.Info("First City built");
		local town2 = FindTown(towns[0],true);
		Log.Info("Finished starting second town");
		if(!town2) {
			Log.Info("Failed to build station in second town.  Will retry next time");
			LeaveTown(0);
			return;
		}
		//AddTown(town2);
		//Connect(town1,town2);
	}
	if(AICompany.GetBankBalance(AICompany.COMPANY_SELF) > 2000 + (AIEngine.GetPrice(defaultEngine) * 2) && lastNewTown < AIController.GetTick() - 1500) FindTown();
	if(replace) PerformFullUpgrade();
	else {
		if(lastNewTown < AIController.GetTick() - 4000) {
			Log.Info("Finding a new town because we havent added one in awhile...");
			FindTown(null, true);
		}
		ManageVehicles();
	}
}

function Network::ManageVehicles() {
	Log.Info("Managing Vehicles...");
	// Check for new defaultEngine every once and awhile...
	if(lastUpdateEngine < AIController.GetTick() - 10000) {
		Log.Warning("Checking for a new default engine...",Log.LVL_SUB_DECISIONS);
		local newEngine = ChooseVehicle();
		if(!newEngine || newEngine == defaultEngine) Log.Warning("Keeping the old vehicle");
		else if(newEngine != defaultEngine) {
			Log.Info("Switching to " + AIEngine.GetName(newEngine),Log.LVL_SUB_DECISIONS);
			local oldEngine = defaultEngine;
			defaultEngine = newEngine;
			noReplace = 0;
			local upgrade = true;
			lastStartUpgrade = AIController.GetTick();
			// While we are at it check if it is a good idea to mass upgrade vehicles
			if(AICompany.GetBankBalance(AICompany.COMPANY_SELF) > (AIEngine.GetPrice(defaultEngine) * vehicles.len()) - (AIEngine.GetPrice(oldEngine) * vehicles.len()) + 100000) PerformFullUpgrade();
			else upgrade = false;
			// If this is the aircraft network, might as well upgrade stations as well if possible
			AIController.Sleep(1000);
			if(upgrade && type == AIVehicle.VT_AIR && AICompany.GetBankBalance(AICompany.COMPANY_SELF) > 2000 * towns.len()) ExpandStation();
			return;
		}
		else Log.Info("Keeping the old vehicle");
		lastUpdateEngine = AIController.GetTick();
	}
	local frontTiles;
	local i = 0;
	for(local i = 0;i < vehicles.len();i++) {
		// Get rid of the sold vehicles in the depot
		if(AIVehicle.IsStoppedInDepot(vehicles[i])) {
			AIVehicle.SellVehicle(vehicles[i]);
			vehicles.remove(i);
			continue;
		}
		// Sell non-profitable vehicles (over 2 years of age)
		if(AIVehicle.GetProfitLastYear(vehicles[i]) < 0 && AIVehicle.GetAge(vehicles[i]) > 730) {
			Log.Info("Selling " + AIVehicle.GetName(vehicles[i]) + " due to lack of income from vehicle");
			AIVehicle.SendVehicleToDepot(vehicles[i]);
		}
	}
	local usedTowns = AIList();
	foreach(town in towns) {
		// check, if more than 30 people are waiting at this station, and suspended is on, turn it off
		/*if(type == AIVehicle.VT_ROAD && town.suspended) {
			if(AIStation.GetCargoWaiting(town.stations[0],Helper.GetPAXCargo()) > 30) {
				Log.Info("Town " + AITown.GetName(town.id) + " is no longer overloaded, un-suspending");
				town.suspended = false;
			}
		}*/
		if(type == AIVehicle.VT_AIR && town.suspended) {
			if(Airport.GetNumAircraftsInAirportQueue(town.stations[0]) <= 3 && Airport.GetNumNonStopedAircraftInAirportDepot(town.stations[0]) <= 4) {
				Log.Info("Town " + AITown.GetName(town.id) + " is no longer overloaded, un-suspending", Log.LVL_SUB_DECISIONS);
				town.suspended = false;
			}
		}
		// if town is already suspended, than why bother spending the time to figure out if it needs more vehicles?
		if(town.suspended) continue;
		local skip = false;
		// check to see if the town has already been added vehicles to
		foreach(used, _ in usedTowns) {
			if(town.id == used) {
				Log.Warning("Skipping used town " + AITown.GetName());
				skip = true;
			}
		}
		if(skip) continue;
		Log.Info("Checking to improve town " + AITown.GetName(town.id),Log.LVL_SUB_DECISIONS);
		// Grow the station
		if(town.lastUsageCheck < AIController.GetTick() - 2500) {
			Log.Info("Performing town usage check...", Log.LVL_SUB_DECISIONS);
			foreach(station in town.stations) {
			if(type == AIVehicle.VT_ROAD) {
				local numVehicles = 0;
				local numTiles = 0;
				local stationTiles = AITileList_StationType(station, AIStation.STATION_BUS_STOP);
				// check the front tile of every singe station peice.
					foreach(stationTile, _ in stationTiles) /* numVehicles += Vehicle.GetVehiclesAtTile(stationTile).Count(); */ numVehicles += GetVehiclesAtTileForStation(stationTile, station).Count();
					foreach(stationTile, _ in Station.GetRoadFrontTiles(station)) /* numVehicles += Vehicle.GetVehiclesAtTile(stationTile).Count(); */ numVehicles += GetVehiclesAtTileForStation(stationTile, station).Count();
					numTiles = AITileList_StationType(station, AIStation.STATION_BUS_STOP).Count();
					numTiles += Station.GetRoadFrontTiles(station).Count();
					Log.Info("Number of station and front tiles: " + numTiles);
					Log.Info("Number of vehicles:                " + numVehicles);
					if(numTiles * 1.5 < numVehicles && stationTiles.Count() < 5) ExpandStation(station); // This is an easy way to fix the problem without limiting the station's prosperity :)
					// Control suspended attribute, to ensure not to many vehicles get sent to this station. like blacklist
					/*if(Station.GetRoadFrontTiles(town.stations[0]).Count() == 5  && AIStation.GetCargoWaiting(town.stations[0],Helper.GetPAXCargo())<5) {
						Log.Info("Station is suspended.  Suspending the sending of vehicles to it");
						town.suspended = true;
					}*/
				}
				else {
					if(AIAirport.IsValidAirportType(AIAirport.GetAirportType(AIStation.GetLocation(station)))) {
						if(Airport.GetNumAircraftsInAirportQueue(station) > 3 || Airport.GetNumNonStopedAircraftInAirportDepot(town.stations[0]) > 4) {
							// Find a new station for some of the aircraft
							Log.Info("Airport " + AIStation.GetName(station) + " is overloaded.  Suspending...", Log.LVL_SUB_DECISIONS);
							town.suspended = true;
						}
					}
				}
			}
			town.lastUsageCheck = AIController.GetTick();
		}
		//frontTiles = Station.GetRoadFrontTiles(town.stations[0]);
		// Add more vehicles to the station
		foreach(station in town.stations) {
			if(AIStation.GetCargoWaiting(station,Helper.GetPAXCargo()) > AIEngine.GetCapacity(defaultEngine) || AIStation.GetCargoRating(station, Helper.GetPAXCargo()) <  40) {
				// First off, always ensure to have a reasonable amount of cash :)
				if(!Money.MakeSureToHaveAmount(AIEngine.GetPrice(defaultEngine) * (AIStation.GetCargoWaiting(station,Helper.GetPAXCargo()) / AIEngine.GetCapacity(defaultEngine))) && type != AIVehicle.VT_AIR) {
					if(Money.GetMaxSpendingAmount() + 10000 < AIEngine.GetPrice(defaultEngine) * (AIStation.GetCargoWaiting(station,Helper.GetPAXCargo()) / AIEngine.GetCapacity(defaultEngine))) {
						Log.Info("Not enough cash to create needed vehicles for now");
						continue;
					}
				}
				// On the other hand, for aircraft, the AI should only build a maximum of 1 extra each month
				Log.Info("Buying more vehicles for " + AITown.GetName(town.id) + "'s station, " + AIStation.GetName(station));
				// Variables to help us
				local list;
				// Get each town that goes to this station
				local same = false;
				// OR find stations that are nearby this station
				local townprep = [];
				// Add items from our network (while we are at it scan previous destinations for suspended stations)
				foreach(otherTown in towns) {
					if(!otherTown.suspended) townprep.append(otherTown);
				}
				AIController.Sleep(1);
				Log.Info("Number of towns added: " + townprep.len());
				// Transfer vars to new townlist
				/*local townlist = AIList();
				for(local i = 0;i < townprep.len();i++) {
					/*if(AIStation.GetCargoWaiting(townprep[i].stations[0],Helper.GetPAXCargo()) < 10) {
						// This town is not suitable to have vehicles added to it
						Log.Info("Removing town " + AITown.GetName(townprep[i].id) + " because there is not enough cargo...");
						townprep.remove(i);
						i--;
						continue;
					}
					else townlist.AddItem(townprep[i].id,0);
				}
				Log.Info("COUNT BEFORE: " + townlist.Count());
				// Valuate to nearby towns (be a bit less restrictive of distance away
				if(type == AIVehicle.VT_ROAD) {
					townlist.Valuate(TownDistValuator,town.id,50);
					townlist.Sort(AIList.SORT_BY_VALUE, true);
					townlist.RemoveAboveValue(60);
				}
				else {
					townlist.Valuate(TownDistValuator,town.id,300);
					townlist.Sort(AIList.SORT_BY_VALUE, true);
					townlist.RemoveAboveValue(100);
				}
				Log.Info("COUNT AFTER: " + townlist.Count());
				// ReAdd to townprep
				local match = false;
				Log.Info("LENGTH OF DEST: " + destinations.len());
				// NOTE: In the future we may be able to get rid of that later loop for finding suspended functions by checking in this loop.  Another way to speed up this long process.
				for(local i = 0;i < townprep.len();i++) {
					Log.Info("TOWN: " + AITown.GetName(townprep[i].id));
					foreach(changedTown, item in townlist) {
						if(townprep[i].id==changedTown) {
							match = true;
						}
						if(match) break;
					}
					Log.Info("MATCH: " + match);
					if(!match) {
						townprep.remove(i);
						i--;
					}
					match = false;
					AIController.Sleep(1);
				}*/
				// Make sure the towns are close
				// Finally, prevent the same station as the origin from being picked
				// All in the same loop :)
				for(local i = 0;i < townprep.len();i++) {
					if(TownDistValuator(town.id, townprep[i].id, routeDistance) > (type == AIVehicle.VT_ROAD ? 80 : 200) || town.id == townprep[i].id) {
						Log.Info("Removing town " + AITown.GetName(townprep[i].id));
						townprep.remove(i);
						i--;
					}
				}
				Log.Info("Number of towns: " + townprep.len());
				if(townprep.len() == 0) {
					if(!FindTown(town)) Log.Info("Could not find a good destination for " + AIStation.GetName(station));
					continue;
				}
				// Now find the station with the most passengers
				local bestTown = townprep[0];
				local bestWait = -1;
				foreach(destTown in townprep) {
					Log.Info("Checking station passengers " + AITown.GetName(destTown.id));
					local waiting = 0;
					foreach(destStation in destTown.stations) waiting += AIStation.GetCargoWaiting(destStation, Helper.GetPAXCargo())
					if(waiting > bestWait) {
						bestTown = destTown;
						bestWait = waiting;
					}
				}
				AIController.Sleep(1);
				Log.Info("Will build to " + AITown.GetName(bestTown.id));
				// Now find the best station in the town :)
				local bestStation = townprep[0];
				bestWait = -1;
				foreach(destStation in bestTown.stations) {
					if(AIStation.GetCargoWaiting(destStation, Helper.GetPAXCargo()) >= bestWait) {
						bestStation = destStation;
						bestWait = AIStation.GetCargoWaiting(destStation, Helper.GetPAXCargo());
					}
				}
				if(AIStation.GetCargoWaiting(bestStation,Helper.GetPAXCargo()) < AIEngine.GetCapacity(defaultEngine) - (type == AIVehicle.VT_ROAD ? 10 : 0)) {
					if(type == AIVehicle.VT_ROAD) {
						// Add a new station to the city :)
						bestStation = bestTown.BuildStation(AIStation.STATION_BUS_STOP);
						if(!bestStation) {
							local result = FindTown(town);
							if(!result) {
								Log.Warning("Could not find a town to build with " + AITown.GetName(town.id));
							}
							continue;
						}
						else ExpandStation(bestStation);
					}
					else {
						if(!FindTown(town)) {
							Log.Warning("Could not find a town to build with " + AITown.GetName(town.id));
							continue;
						}
					}
				}
				// Ensure there is a road between the two stations (if road network)
				if(type != AIVehicle.VT_AIR) {
					Log.Info("Checking roads...");
					/*local result = false;
					foreach(connection, _ in town.connections) {
						//Log.Info("A good station is " + AIStation.GetName(connection));
						if(connection == bestTown.id) result = true;
					}*/
					if(!town.connections.HasItem(bestTown.id)) {
						Log.Info(AITown.GetName(town.id) + " is not connected to " + AITown.GetName(bestTown.id) + ".  Connecting it now...");
						if(!Connect(AITown.GetLocation(town.id), AITown.GetLocation(bestTown.id), AIMap.DistanceManhattan(AITown.GetLocation(town.id), AITown.GetLocation(bestTown.id)) > 50 ? true : false)) {
							Log.Warning("Could not connect roads.  Cancelling...");
							continue;
						}
						// Add town to the list of both towns
						town.connections.AddItem(bestTown.id, 0);
						bestTown.connections.AddItem(town.id, 0);
					}
					else Log.Info("Stations at " + AITown.GetName(town.id) + " and " + AITown.GetName(bestTown.id) + " are already connected.");
				}
				if(town.lastNewVehicles > AIController.GetTick() - 500) {
					Log.Info("Not buying vehicles for the station because we already did recently.", Log.LVL_SUB_DECISIONS);
					continue;
				}
				Log.Info("Building the vehicles to " + AIStation.GetName(bestStation),Log.LVL_SUB_DECISIONS);
				// We now (finally) build the vehicles
				local number = AIStation.GetCargoWaiting(town.stations[0],Helper.GetPAXCargo()) / AIEngine.GetCapacity(defaultEngine);
				
				number = Helper.Clamp(number, 1, 5);
				if(type == AIVehicle.VT_AIR) number = 1;
				if(AIStation.GetCargoRating(town.stations[0],Helper.GetPAXCargo()) < 40 && type == AIVehicle.VT_ROAD) BuyVehicles(town.depot,defaultEngine,2,town.stations[0],bestStation);
				else if(type == AIVehicle.VT_AIR) BuyVehicles(town.depot, defaultEngine, number, station, bestStation, AIStation.GetCargoWaiting(station, Helper.GetPAXCargo()) > 100 && AIStation.GetCargoWaiting(bestStation, Helper.GetPAXCargo()) > 100 ? true : false);
				else BuyVehicles(town.depot, defaultEngine, number, station, bestStation);
				town.lastNewVehicles = AIController.GetTick();
				// Add the other city to used towns
	//			usedTowns.AddItem(bestStation,0);
				// Wait a little
				AIController.Sleep(2);
			}
		}
	}
	lastVehicleManage = AIController.GetTick();
}

function Network::CreateLink() {
	local newTown = FindTown();
	AddTown(newTown);
	local index = FindCloseConnection(town);
	Connect(AIStation.GetLocation(newTown.stations[0]),AIStation.GetLocation(towns[index].stations[0]));
}

function Network::BuyVehicles(depot,engine,number,order1,order2, essential = false) {
	if(!depot || !engine) {
		Log.Warning("Cannot build vehicles because depot or engine is not valid");
	}
	Log.Info("Building " + number + " vehicles of " + AIEngine.GetName(engine),Log.LVL_SUB_DECISIONS);
	//Log.Warning("Depot: " + Tile.GetTileString(depot));
	//Log.Warning("Engine: " + AIEngine.GetName(engine));
	WaitForMoney(AIEngine.GetPrice(engine) * number + 5000, essential ? 100000 : 2500);
	local newVehicle = AIVehicle.BuildVehicle(depot,engine);
	if(!AIVehicle.IsValidVehicle(newVehicle)) {
		Log.Warning("Could not build a vehicle: " + AIError.GetLastErrorString());
		return false;
	}
	vehicles.append(newVehicle);
	local orders = OrderList();
	orders.AddStop(order1,AIOrder.OF_NONE);
	orders.AddStop(order2,AIOrder.OF_NONE);
	orders.ApplyToVehicle(newVehicle);
	AIVehicle.StartStopVehicle(newVehicle);
	local nextVehicle;
	for(local i = 1;i<number;i++) {
		nextVehicle = AIVehicle.CloneVehicle(depot,newVehicle,false);
		if(!AIVehicle.IsValidVehicle(nextVehicle)) {
			Log.Warning("Could not build a vehicle");
			continue;
		}
		vehicles.append(nextVehicle);
		AIVehicle.StartStopVehicle(nextVehicle);
	}
}

function Network::SellVehicle(vehicle) {
	Log.Info("Selling " + AIVehicle.GetName(vehicle),Log.LVL_SUB_DECISIONS);
	AIVehicle.SellVehicle(vehicle);
}

function Network::FindTown(chosenCity = null,desprate = false) {
	// for the porpouses of ease of code reading, I have split the if
	// statement into to parts (yes I know I am lazy)
	if(!desprate) {
		if(type == AIVehicle.VT_ROAD && !Money.MakeSureToHaveAmount((AIEngine.GetPrice(defaultEngine) * 2))) {
			if(Money.GetMaxSpendingAmount() + 10000 < AIEngine.GetPrice(defaultEngine) * 2) {
				Log.Warning("Do not have enough money to create city connection.");
				return false;
			}
		}
		else if(type == AIVehicle.VT_AIR && !Money.MakeSureToHaveAmount((AIEngine.GetPrice(defaultEngine) * 2) + AIAirport.GetPrice(Airport.GetLastAvailableAirportType(defaultEngine)))) {
			if(Money.GetMaxSpendingAmount() + 15000 < AIEngine.GetPrice(defaultEngine) * 2) {
				Log.Warning("Do not have enough money to create city connection.");
				return false;
			}
		}
	}
	Log.Info("Finding a new town...",Log.LVL_SUB_DECISIONS);
	if(desprate) Log.Warning("Desprately finding a new town...");
	local i = 1;
	local qualify = [];
	while(chosenCity == null) {
		local chosenWait = -1;
		Log.Info("Length: " + towns.len());
		foreach(town in towns) {
			/*if(AIStation.GetCargoWaiting(town.stations[0],Helper.GetPAXCargo())>chosenWait) {
				chosenCity = town;
				chosenWait = AIStation.GetCargoWaiting(town.stations[0],Helper.GetPAXCargo());
			}*/
			if(town.connections.Count() <= i) {
				qualify.append(town);
				break;
			}
		}
		if(qualify.len() > 0) {
			chosenCity = qualify[AIBase.RandRange(qualify.len())];
			break;
		}
		i++;
	}
	//if(chosenCity.newStationTries > 3) desprate = true;
	Log.Info("Adding a new town for " + AITown.GetName(chosenCity.id),Log.LVL_SUB_DECISIONS);
	local townlist = AITownList();
	townlist.RemoveList(blacklist);
	// It helps more than it hurts if we remove anything from our own network first
	foreach(town in towns) {
		townlist.RemoveItem(town.id);
	}
	// Remove towns from other networks
	if(type != AIVehicle.VT_AIR) {
		foreach(network in aiInstance.networks) {
			foreach (town in network.towns) {
				townlist.RemoveItem(town.id);
			}
		}
	}
	townlist.Valuate(AITown.GetPopulation);
	if(type == AIVehicle.VT_ROAD) townlist.RemoveBelowValue(275); //cities must be at least 275 in population in order to be considered for use!
	if(type == AIVehicle.VT_AIR) townlist.RemoveBelowValue(800); // same idea for aircraft
	//Log.Warning("townlist.Begin(): " + AITown.GetLocation(townlist.Begin()));
	//Log.Warning("chosenCity.id: " + AITown.GetLocation(chosenCity.id));
	//Log.Warning("Return of the locations: " + Helper.Abs(AIMap.DistanceSquare(AITown.GetLocation(townlist.Begin()),AITown.GetLocation(chosenCity.id))));
	local value = TownDistValuator(townlist.Begin(),chosenCity.id, routeDistance);
	//Log.Warning("Return of TownDistValuator: " + value);
	townlist.Valuate(TownDistValuator, chosenCity.id, routeDistance);
	townlist.Sort(AIList.SORT_BY_VALUE, true);
	Log.Info("DUMP:  ");
	Log.Info(AITown.GetName(townlist.Begin()) + "  :  " + townlist.GetValue(townlist.Begin()));
	Log.Info(AITown.GetName(townlist.Next()) + "  :  " + townlist.GetValue(townlist.Next()));
	if(!desprate && type == AIVehicle.VT_AIR) townlist.RemoveAboveValue(200);
	else if(!desprate && type == AIVehicle.VT_ROAD) townlist.RemoveAboveValue(60);
	else if(desprate && type == AIVehicle.VT_ROAD) townlist.RemoveAboveValue(100);
	else townlist.RemoveAboveValue(500);
	if(townlist.IsEmpty()) {
		AILog.Info("Could not find a good enough town to connect to this city.");
		chosenCity.newStationTries++;
		return false;
	}
	local newTown = townlist.Begin();
	if(newTown==null) {
		Log.Warning("Could not connect to a new city in the network");
		return false;
	}
	// Ensure we have enough money for the station
	if(type == AIVehicle.VT_AIR) {
		if(Money.GetMaxSpendingAmount() < AIAirport.GetPrice(Airport.GetLastAvailableAirportType(defaultEngine))) WaitForMoney(AIAirport.GetPrice(Airport.GetLastAvailableAirportType(defaultEngine)));
	}
	// Now set up the city
	local builtTown = AddTown(newTown);
	if(!builtTown) {
		Log.Info("Because we could not build the town connecting the town is cancelled.");
		return false;
	}
	// expand the station
	if(type == AIVehicle.VT_ROAD) ExpandStation(builtTown.stations[0]);
	// If road then build a road between them
	if(type == AIVehicle.VT_ROAD) {
		if(!Connect(AITown.GetLocation(chosenCity.id),AITown.GetLocation(builtTown.id), true, true)) {
			Log.Info("Building road failed.  Leaving Town...");
			blacklist.AddItem(towns[towns.len()-1].id,0);
			LeaveTown(towns.len()-1);
			return false;
		}
		builtTown.connections.AddItem(chosenCity.id, 0);
		chosenCity.connections.AddItem(builtTown.id, 0);
	}
	// Add the chosen town to the connected towns list
	builtTown.connections.AddItem(chosenCity.stations[0],0);
	chosenCity.connections.AddItem(builtTown.stations[0],0);
	// And now some vehicles
	BuyVehicles(builtTown.depot,defaultEngine,2,builtTown.stations[0],chosenCity.stations[0], true);
	chosenCity.outerLayer = false;
	Log.Info("Finished building new link :D",Log.LVL_SUB_DECISIONS);
	lastNewTown = AIController.GetTick();
	chosenCity.newStationTries = 0;
	return true;
}

function Network::AddTown(theTown) {
	Log.Info("==========Adding town " + AITown.GetName(theTown) + "==========",Log.LVL_SUB_DECISIONS);
	local newTown;
	if(type == AIVehicle.VT_AIR) newTown = Town(theTown);
	else newTown = Town(theTown);
	local result;
	if(type == AIVehicle.VT_AIR) result = newTown.Begin(AIStation.STATION_AIRPORT);
	else if(type == AIVehicle.VT_ROAD && AIEngine.IsArticulated(defaultEngine)) result = newTown.Begin(AIStation.STATION_BUS_STOP,true);
	else result = newTown.Begin(AIStation.STATION_BUS_STOP);
	if(result) {
		towns.append(newTown);
		return newTown;
	}
	else {
		Log.Info("Could not build the town...");
		return null;
	}
}

function Network::LeaveTown(index) {
	Log.Info("Leaving town " + AITown.GetName(towns[index].id));
	towns[index].End();
	towns.remove(index);
	Log.Info("Finished leaving town");
}

function Network::Connect(point1,point2, fast = false, essential = false)
{
	// Briefly test for water
	local checkWater = AIMap.GetTileIndex((AIMap.GetTileX(point1) + AIMap.GetTileX(point2)) / 2, (AIMap.GetTileY(point1) + AIMap.GetTileY(point2)) / 2);
	if(AITile.IsWaterTile(checkWater)) {
		Log.Info("Large body of water suspected...");
		if(AITile.IsWaterTile(AIMap.GetTileIndex((AIMap.GetTileX(checkWater) + AIMap.GetTileX(point1)) / 2, (AIMap.GetTileY(checkWater) + AIMap.GetTileY(point1)) / 2)) || AITile.IsWaterTile(AIMap.GetTileIndex((AIMap.GetTileX(checkWater) + AIMap.GetTileX(point2)) / 2, (AIMap.GetTileY(checkWater) + AIMap.GetTileY(point2)) / 2))) {
			Log.Warning("Detected water between here and destination.  Chances are high its a big body of water.  Cancelling...");
			return false;
		}
	}
	Log.Info("Connecting points...",Log.LVL_SUB_DECISIONS);
	local result = null;
	local path = null;
	local pathbuilder = Pathfinder(point1,point2, 1.7);
	for(local i = 0;i < 5;i++) {
		result = pathbuilder.FindPath();
		if(!result) {
			Log.Info("Failed to find path...",Log.LVL_SUB_DECISIONS);
			return false;
		}
		path = pathbuilder.path;
		{
			local tm = AITestMode();
			local costs = AIAccounting();
			result = pathbuilder.BuildPath();
			if(!result) {
				Log.Info("Failed to build path...",Log.LVL_SUB_DECISIONS);
				return false;
			}
			if(!Money.MakeSureToHaveAmount(costs.GetCosts())) {
				// Should we wait until we have enough money?
				if(costs.GetCosts() - AICompany.GetBankBalance(AICompany.COMPANY_SELF) > 10000 && !essential) {
					Log.Warning("Cannot build road because of lack of money...");
					return false;
				}
				else if(!WaitForMoney(costs.GetCosts())) return false;
			}
		}
		pathbuilder.path = path;
		result = pathbuilder.BuildPath();
		Log.Info("Result: " + result, Log.LVL_SUB_DECISIONS);
		if(result == "fail") {
			Log.Info("Failed to build path...",Log.LVL_SUB_DECISIONS);
			return false;
		}
		else if(result == "success") return true;
	}
	/*local builder = RoadBuilder();
	builder.Init(point1, point2);
	builder.SetEstimateMultiplier(aiInstance.fast ? 2.0 : (fast ? 1.9 : 1.4));
	local res = builder.ConnectTiles();
	if(res == RoadBuilder.CONNECT_SUCCEEDED) return true;
	else return false;*/
}

function Network::PerformFullUpgrade() {
	Log.Info("----------PERFORMING FULL UPGRADE----------",Log.LVL_SUB_DECISIONS);
	//Log.Warning("Full upgrade not yet implemented");
	// First look through all the vehicles, and find the ones not upgraded. from there, upgrade them
	if(!replace) {
		Log.Info("Beginning to upgrade vehicles...",Log.LVL_SUB_DECISIONS);
		if(type == AIVehicle.VT_AIR) {
			local t = Airport.GetLastAvailableAirportType(defaultEngine);
			// Get a list of already upgraded stations
			local goodStations = [];
			foreach(town in towns) {
				local station = town.stations[0];
				if(AIAirport.GetAirportType(AIStation.GetLocation(station)) == t) goodStations.append(station); 
			}
			if(goodStations.len() == 0) {
				// for now, dont upgrade if no good stations
				Log.Warning("Will not upgrade vehicles because no suitable stations available!");
				return;
			}
			local opts = Helper.SquirrelListToAIList(goodStations);
			// Send each vehicle to the depot of its closest station
			foreach(vehicle in vehicles) {
				if(AIVehicle.GetEngineType(vehicle) != defaultEngine) {
					opts.Valuate(AIStation.GetDistanceManhattanToTile, AIVehicle.GetLocation(vehicle));
					opts.Sort(AIList.SORT_BY_VALUE, true);
					// Send to that station
					StopInDepot(vehicle, opts.Begin());
				}
			}
		}
		else {
			foreach(vehicle in vehicles) {
				if(AIVehicle.GetEngineType(vehicle) != defaultEngine) {
					// Just send it to depot using the API function for now
					// TODO: If you feel like it, specify in orders like we do with aircraft
					AIVehicle.SendVehicleToDepot(vehicle);
				}
			}
		}
		replace = true;
	}
	// When this function is recalled, upgrade any vehicles that have entered the depot.
	else {
	Log.Info("Continuing to upgrade vehicles...",Log.LVL_SUB_DECISIONS);
	local done = true;
	local replaced = false;
	foreach(vehicle in vehicles) {
		if(AIVehicle.GetEngineType(vehicle) == defaultEngine) continue;
		else {
			done = false;
			if(AIVehicle.IsStoppedInDepot(vehicle)) {
				WaitForMoney(AIEngine.GetPrice(defaultEngine)); // Should have plenty of money anyway
				// Copy orders and depot (after removing the go to depot order which seems to mess it up)
				if(type == AIVehicle.VT_AIR) AIOrder.RemoveOrder(vehicle, 3);
				local stationlist = Order.GetStationListFromOrders(vehicle);
				local origdepot = AIVehicle.GetLocation(vehicle);
				// Sell Vehicle
				AIVehicle.SellVehicle(vehicle);
				// Rebuild vehicle and send out
				BuyVehicles(origdepot,defaultEngine,1,stationlist.Begin(),stationlist.Next());
				//AIVehicle.StartStopVehicle(vehicle);
				replaced = true;
				//noReplace = 0;
			}
		}
	}
	if(!replaced) noReplace++;
		Log.Warning("NOREPLACE: " + noReplace);
		if(done || noReplace > 10 || lastStartUpgrade < AIController.GetTick()-10000) {
			replace = false;
			// make sure that vehicles are moving even if not upgraded
			foreach(vehicle in vehicles) {
			Log.Info("----------Finishing Upgrade Process----------");
			if(AIVehicle.IsStoppedInDepot(vehicle)) AIVehicle.StartStopVehicle(vehicle);
			}
		}
	}
}

function Network::ExpandStation(station = null) {
	// Dont bother if the station is MagicDTRS (it is currently unsupported)
	/*if(!station) {
		Log.Warning("NULL Station passed to expand station!");
		return;
	}*/
	if(station != null) {
		if(!AIStation.IsValidStation(station)) {
			Log.Warning("INVALID Station passed to expand station!");
			return;
		}
	}
	// This function is equivalent to updating the station in airport mode
	if(type == AIVehicle.VT_ROAD) {
		if(AIRoad.IsDriveThroughRoadStationTile(AIStation.GetLocation(station))) return;
		Log.Info("Trying to expand station " + AIStation.GetName(station),Log.LVL_SUB_DECISIONS);
		if(Result.IsFail(Road.GrowStationParallel(station,AIStation.STATION_BUS_STOP))) {
			Log.Info("Failed to grow station parallel, trying not paralell...");
			if(Result.IsFail(Road.GrowStation(station,AIStation.STATION_BUS_STOP))) Log.Warning("Failed to grow station :(");
		}
	}
	else {
		// if station is null, upgrade everything
		if(!station) {
			Log.Info("Upgrading aircraft stations...",Log.LVL_SUB_DECISIONS);
			// Get perferred station
			local stype = Airport.GetLastAvailableAirportType(defaultEngine);
			if(!stype) return;
			local airlist = [];
			airlist.append(stype);
			for(local i = 0;i < towns.len();i++) {
				if(AIAirport.GetAirportType(AIStation.GetLocation(towns[i].stations[0])) == stype) continue;
				Log.Info("Upgrading airport in " + AITown.GetName(towns[i].id), Log.LVL_SUB_DECISIONS);
				// First, try to send any vehicles that go to this station to take a break in the depot temporarily while the station is changed
				// Send any vehicles that have it inside its orders to the service station
				//local vehs = AIVehicleList_Station(towns[i].stations[0];
				// Should already be sent
				local result = Airport.UpgradeAirportInTown(towns[i].id,towns[i].stations[0], airlist, Helper.GetPAXCargo(), Helper.GetPAXCargo());
				if(Result.IsFail(result)) {
					Log.Warning("Failed to upgrade station :(");
					if(result == Result.REBUILD_FAILED) {
						Log.Warning("Even worse, rebuild failed.  It is easiest for now just to leave the town.");
						LeaveTown(i);
					}
				}
			}
		}
	}
}

function Network::ChooseVehicle(cheap = false)
{
	local engines = AIEngineList(type);
	// Take out zero price vehicles
	engines.Valuate(AIEngine.GetPrice);
	engines.RemoveValue(0);
	engines.Valuate(AIEngine.GetRunningCost);
	engines.RemoveValue(0);
	// Prevent trams (until I build support in this class for trams)
	if(type == AIVehicle.VT_ROAD) {
		engines.Valuate(AIEngine.GetRoadType);
		engines.KeepValue(AIRoad.ROADTYPE_ROAD);
		local testlist = engines;
		testlist.Valuate(AIEngine.IsArticulated);
		testlist.RemoveValue(1);
		if(!testlist.IsEmpty()) {
			engines.Valuate(AIEngine.IsArticulated);
			engines.RemoveValue(1);
		}
	}
	engines.Valuate(AIEngine.CanRefitCargo, Helper.GetPAXCargo());
	engines.KeepValue(1);
	/*if(!cheap) {
		engines.Valuate(AIEngine.GetCapacity);
		engines.Sort(AIList.SORT_BY_VALUE,false);
	}
	else {
		engines.Valuate(AIEngine.GetPrice);
		engines.Sort(AIList.SORT_BY_VALUE,true);
	}*/
	engines.Valuate(AIEngine.GetMaxSpeed);
	// Most go at least 40 mph (road only)
	if(type == AIVehicle.VT_ROAD) engines.RemoveBelowValue(40);
	engines.Valuate(AIEngine.GetCapacity);
	// Must hold at least 15 passengers (road) or 50 (air)
	if(type == AIVehicle.VT_ROAD) engines.RemoveBelowValue(15);
	else engines.RemoveBelowValue(50);

	
	if(engines.IsEmpty()) {
		Log.Warning("Could not find an engine to build...");
		return -1;
	}

	// Finally, get sort by price if going cheap
	if(cheap) {
		engines.Valuate(AIEngine.GetPrice);
		engines.Sort(AIList.SORT_BY_VALUE, true);
	} else {
		engines.Valuate(AIEngine.GetMaxSpeed);
		engines.Sort(AIList.SORT_BY_VALUE, false);
	}
	return engines.Begin();
}

/*function Network::BuildServiceStation() {
	Log.Info("Building service station for airport network!", Log.LVL_SUB_DECISIONS);
	// Get the center of the map
	local c = AIMap.GetTileIndex(AIMap.GetMapSizeX() / 2, AIMap.GetMapSizeY() / 2);
	local t = Airport.GetLastAvailableAirportType(defaultEngine);
	local tiles = Tile.MakeTileRectAroundTile(c, 50);
	tiles.Valuate(AITile.IsBuildableRectangle, AIAirport.GetAirportWidth(t), AIAirport.GetAirportHeight(t));
	tiles.KeepValue(1);
	if(tiles.IsEmpty()) {
		Log.Warning("Could not find a place to build service station!");
		return false;
	}
	//tiles.Valuate(Tile.CostToFlattern, AIAirport.GetAirportWidth(t), AIAirport.GetAirportHeight(t));
	//tiles.Sort(AIList.SORT_BY_VALUE, true);
	local tile = tiles.Begin();
	Tile.FlatternRect(tile, AIAirport.GetAirportWidth(t), AIAirport.GetAirportHeight(t));
	if(!AIAirport.BuildAirport(tile, t, AIStation.STATION_NEW)) {
		Log.Warning("Could not build service station: " + AIError.GetLastErrorString());
		return false;
	}
	// Get the airport
	local sid = AIStation.GetStationID(tile);
	while(!AIStation.SetName(sid, "AIR SERVICE")) {}
	serviceStation = sid;
	return true;
}

function Network::SendToServiceStation(v) {
	if(serviceStation == null) {
		if(!BuildServiceStation()) return;
	}
	if(AIOrder.AppendOrder(v, Airport.GetHangarTile(serviceStation), AIOrder.OF_STOP_IN_DEPOT)) {
		AIOrder.SkipToOrder(v, AIOrder.GetOrderCount(v) - 1);
	}
	else Log.Warning("Could not send vehicle to service station!");
}

function Network::ReturnAllToNetwork() {
	foreach(v, _ in AIVehicleList_Depot(Airport.GetHangarTile(serviceStation))) {
		if(AIOrder.RemoveOrder(v, AIOrder.GetOrderCount(v) - 1)) AIVehicle.StartStopVehicle(v);
		else Log.Warning("Could not remove order for vehicle from service station!");
	}
} 
*/

function Network::StopInDepot(v, station) {
	Log.Info("Stopping " + AIVehicle.GetName(v) + " in the depot at " + AIStation.GetName(station), Log.LVL_SUB_DECISIONS);
	if(AIOrder.AppendOrder(v, Airport.GetHangarTile(station), AIOrder.OF_STOP_IN_DEPOT)) {
		AIOrder.SkipToOrder(v, AIOrder.GetOrderCount(v) - 1);
	}
	else Log.Warning("Could not send vehicle to service station!");
}
