/*
 * This file is part of SynTrans, which is an AI for OpenTTD
 *
 * SynTrans is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License
 *
 * SynTrans is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with SynTrans; If not, see <http://www.gnu.org/licenses/> or
 * write to the Free Software Foundation, Inc., 51 Franklin Street, 
 * Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */


// This function builds a simple road from tile 1 to 2 if not already existing
function BuildRoadStrip(tile1, tile2) {
	if(!AIRoad.AreRoadTilesConnected(tile1, tile2)) {
		Log.Info("Building road: " + Tile.GetTileString(tile1) + " to " + Tile.GetTileString(tile2));
		if(!AIRoad.BuildRoad(tile1, tile2)) return false;
	}
	else Log.Info("Not building road because already connected");
	return true;
}

// This class is simply to increase the speed of the pathfinder, because that is part of what SynTrans is about
// Much of the code has been copied over from the origional Road Pathfinder, only with my modifications in overriden functions
class CustomPathFinder extends RoadPathFinder
{
	speed_factor = null;
	constructor()
	{
		this._max_cost = 10000000;
		this._cost_tile = 100;
		this._cost_no_existing_road = 100; // 40
		this._cost_turn = 100;
		this._cost_slope = 200;
		this._cost_bridge_per_tile = 150;
		this._cost_tunnel_per_tile = 120;
		this._cost_coast = 20;
		this._max_bridge_length = 10;
		this._max_tunnel_length = 20;
		this._pathfinder = this._aystar_class(this, this._Cost, this._Estimate, this._Neighbours, this._CheckDirection);
		speed_factor = 1.0;

		this.cost = this.Cost(this);
		this._running = false;
	}
	function _Estimate(self, cur_tile, cur_direction, goal_tiles)
	{
		local min_cost = self._max_cost;
		/* As estimate we multiply the lowest possible cost for a single tile with
		 * with the minimum number of tiles we need to traverse. */
		foreach (tile in goal_tiles) {
			min_cost = min(AIMap.DistanceManhattan(cur_tile, tile) * self._cost_tile, min_cost);
		}
		return min_cost * self.speed_factor;
	}
}

class Pathfinder
{
	pathfinder = null;
	path = null;
	start = null;
	end = null;
	constructor(start, end, speedfactor = 1.0) {
		pathfinder = CustomPathFinder();
		pathfinder.speed_factor = speedfactor;
		this.start = start;
		this.end = end;
	}
	function FindPath();
	function BuildPath();
}

function Pathfinder::FindPath()
{
	if(!start || !end) {
		Log.Warning("Not all arguments specified");
		return false;
	}
	if(!AIMap.IsValidTile(start) || !AIMap.IsValidTile(end)) {
		Log.Warning("Invalid tiles passed!");
		return false;
	}
	pathfinder.cost.max_bridge_length = 30;
	pathfinder.cost.max_tunnel_length = 30;
	pathfinder.InitializePath([start],[end]);
	path = false;
	//while(path == false) {
		path = pathfinder.FindPath(8000);
		//AIController.Sleep(1);
	//}
	if(!path) {
		Log.Warning("Path could not be found (most likely because it took to long) :(");
		return false;
	}
	return true;
}

// The following function was from Convoy

function Pathfinder::BuildPath()
{
	/*local retry_find   = false;	
	local plan_retries = 0;
	local pathfinder = RoadPathFinder();
	local path_len = 0;	
	do {
		if(retry_find) {
			FindPath();
			retry_find = false;
		}
		while (path != null) {
			path_len = path_len + 1;
			if (path.GetParent() != null) {
				local parnt = path.GetParent().GetTile();
				if (!AIRoad.AreRoadTilesConnected(path.GetTile(), parnt) && (!AIBridge.IsBridgeTile(path.GetTile()) || AIBridge.GetOtherBridgeEnd(path.GetTile()) != parnt)) {
					local retry_build = false;
					local build_retries = 0;	 
					do {
						retry_build = false;
						if (AIMap.DistanceManhattan(path.GetTile(), parnt) == 1 ) {
							//						   AILog.Info("PathFinder: tiles " + 
							//								AIMap.GetTileX(path.GetTile()) + " " + AIMap.GetTileY(path.GetTile()) +
							//								AIMap.GetTileX(parnt) + " " + AIMap.GetTileY(parnt));
							local built_road = AIRoad.BuildRoad(path.GetTile(), parnt);
							if (!built_road) {
								local err = AIError.GetLastError();
								switch (err) {
									/* ignore these errors
									case AIError.ERR_NONE: 	
									case AIError.ERR_ALREADY_BUILT: 
									/* decrement retry counter, so it does not limit the nr of retries 
									else after 35 tiles over existing road, the building stops 
									 									
									if (build_retries) build_retries = build_retries - 1;
									break;
									/* can't handle this locally, return unsuccessful
									case AIError.ERR_PRECONDITION_FAILED:
									case AIError.ERR_NEWGRF_SUPPLIED_ERROR:
									case AIError.ERR_NOT_ENOUGH_CASH:  
									case AIError.ERR_LOCAL_AUTHORITY_REFUSES: 
									Log.Warning("Build road failed(fatal)"+ AIError.GetLastErrorString());
									return false;
									break;
									case AIError.ERR_VEHICLE_IN_THE_WAY: 
									Log.Warning("Build road failed (retry): "+ AIError.GetLastErrorString());	
									AIController.Sleep(1);
									retry_build = true;
									break;
									case AIError.ERR_AREA_NOT_CLEAR: 	
									Log.Warning("Build road failed (demolish + retry): "+ AIError.GetLastErrorString());	
									AITile.DemolishTile(path.GetTile());
									retry_build = false;
									retry_find = true;
									break;
									case AIError.ERR_OWNED_BY_ANOTHER_COMPANY :
									case AIError.ERR_FLAT_LAND_REQUIRED: 	
									case AIError.ERR_LAND_SLOPED_WRONG: 	
									case AIError.ERR_SITE_UNSUITABLE: 	
									case AIError.ERR_TOO_CLOSE_TO_EDGE: 
									Log.Warning("Build road failed (replan): "+ AIError.GetLastErrorString());	
									retry_build = false;
									retry_find = true;
									break;
									case AIError.ERR_UNKNOWN:
									default:
									Log.Warning("Build road failed (replan): "+ AIError.GetLastErrorString());	
									retry_find = true;
									break;
								}
							} 
							build_retries++;
						}
						else {
							Log.Info("PathFinder: build bridge");
							/* Build a bridge or tunnel.
							if (!AIBridge.IsBridgeTile(path.GetTile()) && !AITunnel.IsTunnelTile(path.GetTile())) {
								/* If it was a road tile, demolish it first. Do this to work around expended roadbits.
								if (AIRoad.IsRoadTile(path.GetTile())) AITile.DemolishTile(path.GetTile());
								if (AITunnel.GetOtherTunnelEnd(path.GetTile()) == parnt) {
									if (!AITunnel.BuildTunnel(AIVehicle.VT_ROAD, path.GetTile())) {
										Log.Warning("Build tunnel failed");
									}
								}
								else {
									local bridge_list = AIBridgeList_Length(AIMap.DistanceManhattan(path.GetTile(), parnt) + 1);
									bridge_list.Valuate(AIBridge.GetMaxSpeed);
									bridge_list.Sort(AIList.SORT_BY_VALUE, false);
									if (!AIBridge.BuildBridge(AIVehicle.VT_ROAD, bridge_list.Begin(), path.GetTile(), parnt)) {
										Log.Warning("Build bridge failed");
									}
								}
							}
						}
					}
					while (retry_build && build_retries < 80 && !retry_find);
				}
			}
			path = path.GetParent();
		}
		plan_retries++;
	} while (retry_find && (plan_retries < 10));
	//return path_len;
	return true;*/
	local lastRoadDir = Direction.DIR_INVALID;
	local roadDir = Direction.DIR_INVALID;
	local lastRoadStart = null;
	local roadStart = null;
	while (path != null) {
		local fail = false;
		local par = path.GetParent();
		local retry = false;
		if (par != null) {
			local last_node = path.GetTile();
			if (AIMap.DistanceManhattan(path.GetTile(), par.GetTile()) == 1 ) {
				if(!roadStart) {
					// The road starts here
					roadStart = path.GetTile();
					roadDir = Direction.GetDirectionToAdjacentTile(path.GetTile(), par.GetTile());
				}
				else if(Direction.GetDirectionToAdjacentTile(path.GetTile(), par.GetTile()) != roadDir) {
					if(!BuildRoadStrip(roadStart, path.GetTile())) fail = true;
					lastRoadStart = roadStart;
					roadStart = path.GetTile();
					lastRoadDir = roadDir;
					roadDir = Direction.GetDirectionToAdjacentTile(path.GetTile(), par.GetTile());
				}
			} else {
				if(!BuildRoadStrip(roadStart, path.GetTile())) fail = true;
				lastRoadStart = roadStart;
				roadStart = par.GetTile();
				lastRoadDir = roadDir;
				roadDir = Direction.GetDirectionToAdjacentTile(par.GetTile(), par.GetParent().GetTile());
				/* Build a bridge or tunnel. */
				if (!AIBridge.IsBridgeTile(path.GetTile()) && !AITunnel.IsTunnelTile(path.GetTile())) {
					/* If it was a road tile, demolish it first. Do this to work around expended roadbits. */
					if (AIRoad.IsRoadTile(path.GetTile())) AITile.DemolishTile(path.GetTile());
					if (AITunnel.GetOtherTunnelEnd(path.GetTile()) == par.GetTile()) {
						if (!AITunnel.BuildTunnel(AIVehicle.VT_ROAD, path.GetTile())) fail = true;
					} else {
						local bridge_list = AIBridgeList_Length(AIMap.DistanceManhattan(path.GetTile(), par.GetTile()) + 1);
						bridge_list.Valuate(AIBridge.GetMaxSpeed);
						bridge_list.Sort(AIList.SORT_BY_VALUE, false);
						if (!AIBridge.BuildBridge(AIVehicle.VT_ROAD, bridge_list.Begin(), path.GetTile(), par.GetTile())) fail = true;
					}
				}
			}
		}
		else if(roadStart != null && roadStart != path.GetTile()) {
			if(!BuildRoadStrip(roadStart, path.GetTile())) fail = true;
			else roadStart = null;
		}
		// Check for errors
		if(fail) {
			local err = AIError.GetLastError();
			switch (err) {
				case AIError.ERR_NONE: 	
				case AIError.ERR_ALREADY_BUILT:
				// ignore and move on :)
				break;
				case AIError.ERR_PRECONDITION_FAILED:
				case AIError.ERR_NEWGRF_SUPPLIED_ERROR:
				case AIError.ERR_NOT_ENOUGH_CASH:  
				case AIError.ERR_LOCAL_AUTHORITY_REFUSES: 
				Log.Warning("Build road failed (fatal)" + AIError.GetLastErrorString());
				return "fail";
				break;
				case AIError.ERR_VEHICLE_IN_THE_WAY: 
				Log.Warning("Build road failed (retry): " + AIError.GetLastErrorString());	
				AIController.Sleep(1);
				roadStart = lastRoadStart;
				roadDir = lastRoadDir;
				retry = true;
				break;
				case AIError.ERR_AREA_NOT_CLEAR: 	
				case AIError.ERR_OWNED_BY_ANOTHER_COMPANY:
				case AIError.ERR_FLAT_LAND_REQUIRED: 	
				case AIError.ERR_LAND_SLOPED_WRONG: 	
				case AIError.ERR_SITE_UNSUITABLE: 	
				case AIError.ERR_TOO_CLOSE_TO_EDGE: 
				case AIError.ERR_UNKNOWN:
				default:
				Log.Warning("Build road failed (replan): "+ AIError.GetLastErrorString());	
				return "replan";
				break;
			}
		}
		if(!retry) path = par;
	}
	return "success";
}
