/*
 * This file is part of SynTrans, which is an AI for OpenTTD
 *
 * SynTrans is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License
 *
 * SynTrans is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with SynTrans; If not, see <http://www.gnu.org/licenses/> or
 * write to the Free Software Foundation, Inc., 51 Franklin Street, 
 * Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

class Town
{
	lastUsageCheck = null;
	lastNewVehicles = null;
	lastNewStation = null;
	id = null;
	outerLayer = null;
	stations = null;
	depot = null;
	suspended = null;
	newStationTries = null;
	connections = null;
	constructor(townid) {
		this.lastUsageCheck = 0;
		this.lastNewVehicles = 0;
		this.lastNewStation = 0;
		this.id = townid;
		this.outerLayer = true;
		this.stations = [];
		this.depot = null;
		this.newStationTries = 0;
		this.connections = AIList();
	}
	function Begin(stationType);
	function BuildStation();
	function End();
}

function Town::Begin(stationType, useDTRS = false) {
	if(depot != null) {
		Log.Warning("Town::Begin called when town has already begun...");
		// We still want to acknowledge we are initialized
		return true;
	}
	connections = AIList();
	local station = BuildStation(stationType, useDTRS);
	if(!station) return false;
	if(stationType == AIStation.STATION_BUS_STOP && !useDTRS) depot = Road.BuildDepotNextToRoad(AIStation.GetLocation(station), 50, 150);
	else if(stationType == AIStation.STATION_AIRPORT) depot = Airport.GetHangarTile(station);
	if(!depot) {
		Log.Warning("Could not build depot!");
		return false;
	}
	suspended = false;
	newStationTries = 0;
	return true;
}

function Town::BuildStation(stationType, useDTRS = false) {
	if(lastNewStation > AIController.GetTick() - 500 && lastNewStation != 0) {
		Log.Info("Not building a station because it happened recently...", Log.LVL_SUB_DECISIONS);
		return null;
	}
	Log.Info("Building a new station in " + AITown.GetName(id), Log.LVL_SUB_DECISIONS);
	local stationLocation;
	local thestation;
	if(stationType == AIStation.STATION_BUS_STOP) {
		if(useDTRS) {
			thestation = Road.BuildMagicDTRSInTown(id,AIRoad.ROADTYPE_ROAD,4,Helper.GetPAXCargo(),Helper.GetPAXCargo());
			if(thestation.result) {
				Log.Warning("DEPOT: " + thestation.depot_tile);
				stationLocation = AIStation.GetLocation(thestation.station_id);
				depot = thestation.depot_tile;
			}
		}
		else {
			local cfunctions = ConvoyFunc();
			local theTile = cfunctions.FindLineBusStopLocation(id, Helper.GetPAXCargo(), false);
			if(!theTile) {
				Log.Warning("Could not find a good place to build in " + AITown.GetName(id));
				return null;
			}
			local result = cfunctions.BuildBusStop(theTile);
			if(!result) return null;
			else stationLocation = theTile;
		}
	}
	else stationLocation = Airport.BuildAirportInTown(id,Airport.GetLastAvailableAirportType(Engine.GetEngine_PAXLink(0,AIVehicle.VT_AIR)),Helper.GetPAXCargo(),Helper.GetPAXCargo());
	if(!stationLocation) {
		Log.Warning("Could not build station in this city");
		return null;
	}
	local station;
	/*if(stationType != AIStation.STATION_BUS_STOP)*/ station = AIStation.GetStationID(stationLocation);
	//station = stationLocation;
	Log.Info("STATION NAME: " + AIStation.GetName(station));
	if(!AIStation.IsValidStation(station)) {
		Log.Info("Could not build the station");
		return null;
	}
	// Name the station
	if(stations.len() == 0) AIStation.SetName(station, AITown.GetName(id) + " Synaptic");
	else AIStation.SetName(station, AITown.GetName(id) + " Sub #" + stations.len());
	stations.append(station);
	return station;
}

function Town::End() {
	Log.Info("Leaving the town...");
	local demolishDepot = false;
	for(local i = 0;i < stations.len();i++) {
		if(AIStation.HasStationType(stations[i], AIStation.STATION_BUS_STOP)) demolishDepot = true;
		Station.DemolishStation(stations[i]);
	}
	if(demolishDepot) AIRoad.RemoveRoadDepot(this.depot);
	this.id = null;
	Log.Info("Finished ending town...");
}
