/**
 *    This file is part of Rondje.
 *
 *    Rondje is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    Rondje is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with Rondje.  If not, see <http://www.gnu.org/licenses/>.
 * 
 *    This file is based on the RoadFinder class (which in turn is based on 
 *    Truebrains work), but is a slimmed down version optimized for one goal:
 *    check whether two tiles are connected. This pathfinder can be run from
 *    the constructor as it does no do-commands at all.
 */

require("graph/AStar.nut");

/**
 * A Road Pathfinder that is optimized for connectedness checking
 */
class ConnectedChecker {
	_pathfinder = null;            ///< A reference to the used AyStar object.
	_running = null;
	_max_cost = null;

	constructor(max_cost) {
		this._pathfinder = AStar(this._Cost, this._Estimate, this._Neighbours, this, this, this);
		this._running = false;
		this._max_cost = max_cost;
	}

	/**
	 * Initialize a path search between sources and goals.
	 * @param sources The source tiles.
	 * @param goals The target tiles.
	 * @see AyStar::InitializePath()
	 */
	function InitializePath(sources, goals) {
		local nsources = [];

		foreach (node in sources) {
			nsources.push([node, 0xFF]);
		}
		this._pathfinder.InitializePath(nsources, goals);
	}
	
	/**
	 * Try to find the path as indicated with InitializePath with the lowest cost.
	 * @param iterations After how many iterations it should abort for a moment.
	 *  This value should either be -1 for infinite, or > 0. Any other value
	 *  aborts immediatly and will never find a path.
	 * @return A route if one was found, or false if the amount of iterations was
	 *  reached, or null if no path was found.
	 *  You can call this function over and over as long as it returns false,
	 *  which is an indication it is not yet done looking for a route.
	 * @see AStar::FindPath()
	 */
	function FindPath(iterations) {
		local ret = this._pathfinder.FindPath(iterations);
		this._running = (ret == false) ? true : false;
		return ret;
	}

	function reset() {
		this._pathfinder.reset();
		this._running = false;
	}

	function _Cost(path, new_tile, new_direction, self) {
		/* path == null means this is the first node of a path, so the cost is 0. */
		if (path == null)
			return 0;
	
		local prev_tile = path.GetTile();
	
		if (AIRoad.AreRoadTilesConnected(prev_tile, new_tile)) {
			return path.GetCost() + 1;
		}
	
		return self._max_cost;
	}

	function _Estimate(cur_tile, cur_direction, goal_tiles, self) {
		local min_cost = self._max_cost;
		/* As estimate we multiply the cost for a single new road tile with
		 * with the minimum number of tiles we need to traverse. */
		foreach (tile in goal_tiles) {
			min_cost = min(AIMap.DistanceManhattan(cur_tile, tile), min_cost);
		}
		return min_cost;
	}
	
	function _Neighbours(path, cur_node, self) {
		/* If we go over max_cost, the path isn't valid. */
		if (path.GetCost() >= self._max_cost)
			return [];
		
		local tiles = [];
	
		/* Check if the current tile is part of a bridge or tunnel. */
		if ((AIBridge.IsBridgeTile(cur_node) || AITunnel.IsTunnelTile(cur_node)) && AITile.HasTransportType(cur_node, AITile.TRANSPORT_ROAD)) {
			local other_end = AIBridge.IsBridgeTile(cur_node) ? AIBridge.GetOtherBridgeEnd(cur_node) : AITunnel.GetOtherTunnelEnd(cur_node);
			local next_tile = cur_node + (cur_node - other_end) / AIMap.DistanceManhattan(cur_node, other_end);
			if (AIRoad.AreRoadTilesConnected(cur_node, next_tile) || AIRoad.IsRoadTile(next_tile)) {
				tiles.push([next_tile, self._GetDirection(cur_node, next_tile, false)]);
			}
			/* The other end of the bridge / tunnel is a neighbour. */
			tiles.push([other_end, self._GetDirection(next_tile, cur_node, true) << 4]);
		} else if (path.GetParent() != null && AIMap.DistanceManhattan(cur_node, path.GetParent().GetTile()) > 1) {
			local other_end = path.GetParent().GetTile();
			local next_tile = cur_node + (cur_node - other_end) / AIMap.DistanceManhattan(cur_node, other_end);
			if (AIRoad.AreRoadTilesConnected(cur_node, next_tile)) {
				tiles.push([next_tile, self._GetDirection(cur_node, next_tile, false)]);
			}
		} else {
			local offsets = [AIMap.GetTileIndex(0, 1), AIMap.GetTileIndex(0, -1), AIMap.GetTileIndex(1, 0), AIMap.GetTileIndex(-1, 0)];
			/* Check all tiles adjacent to the current tile. */
			foreach (offset in offsets) {
				local next_tile = cur_node + offset;
				/* We add them to the to the neighbours-list if one of the following applies:
				 * 1) There already is a connections between the current tile and the next tile.
				 * 2) We can build a road to the next tile.
				 * 3) The next tile is the entrance of a tunnel / bridge in the correct direction. */
				if (AIRoad.AreRoadTilesConnected(cur_node, next_tile)) {
					tiles.push([next_tile, self._GetDirection(cur_node, next_tile, false)]);
				} else if (self._CheckTunnelBridge(cur_node, next_tile)) {
					tiles.push([next_tile, self._GetDirection(cur_node, next_tile, false)]);
				}
			}
		}
		return tiles;
	}
	
	function _GetDirection(from, to, is_bridge) {
		if (!is_bridge && AITile.GetSlope(to) == AITile.SLOPE_FLAT) return 0xFF;
		if (from - to == 1) return 1;
		if (from - to == -1) return 2;
		if (from - to == AIMap.GetMapSizeX()) return 4;
		if (from - to == -AIMap.GetMapSizeX()) return 8;
	}
	
	function _CheckTunnelBridge(current_tile, new_tile)
	{
		if (!AIBridge.IsBridgeTile(new_tile) && !AITunnel.IsTunnelTile(new_tile)) return false;
		local dir = new_tile - current_tile;
		local other_end = AIBridge.IsBridgeTile(new_tile) ? AIBridge.GetOtherBridgeEnd(new_tile) : AITunnel.GetOtherTunnelEnd(new_tile);
		local dir2 = other_end - new_tile;
		if ((dir < 0 && dir2 > 0) || (dir > 0 && dir2 < 0)) return false;
		dir = abs(dir);
		dir2 = abs(dir2);
		if ((dir >= AIMap.GetMapSizeX() && dir2 < AIMap.GetMapSizeX()) ||
		    (dir < AIMap.GetMapSizeX() && dir2 >= AIMap.GetMapSizeX())) return false;
	
		return true;
	}
}
