/**
 *    This file is part of Rondje.
 *
 *    Rondje is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    Rondje is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with Rondje.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright 2008-2009 Marnix Bakker, Willem de Neve, Michiel Konstapel and Otto Visser.
 * Suggestions/comments and bugs can go to: rondjeomdekerk@konstapel.nl
 */

/**
 * Return the entry of an item in an array, or -1 if not found.
 */
function IndexOf(list, item) {
	foreach (index, entry in list) {
		if (entry == item) return index;
	}
	
	return -1;
}

/**
 * Calculates an integer square root.
 */
function Sqrt(i) {
	if (i == 0)
		return 0;   // Avoid divide by zero
	local n = (i / 2) + 1;       // Initial estimate, never low
	local n1 = (n + (i / n)) / 2;
	while (n1 < n) {
		n = n1;
		n1 = (n + (i / n)) / 2;
	}
	return n;
}

/**
 * Return the minimum of three numbers.
 */
function Min3(a, b, c) {
	if (a < b && a < c) return a;
	if (b < c) return b;
	return c;
}

/**
 * Find the cargo ID for passengers.
 * Otto: newgrf can have tourist (TOUR) which qualify as passengers but townfolk won't enter the touristbus...
 * hence this rewrite; you can check for PASS as string, but this is discouraged on the wiki
 */
function GetPassengerCargoID() {
	local list = AICargoList();
	local candidate = -1;
	for (local i = list.Begin(); list.HasNext(); i = list.Next()) {
		if (AICargo.HasCargoClass(i, AICargo.CC_PASSENGERS))
		candidate = i;
	}
	if(candidate != -1)
		return candidate;
	
	throw "no passenger cargo in this game!";
}

/**
 * (Re-)initialize the list of engines.
 */
function InitEngines() {
	::bestEngines <- {};
}
		
/**
 * Find the best (largest) engine for a given cargo ID.
 */
function GetBestEngine(cargoID) {
	if (!(cargoID in bestEngines)) {
		local engines = AIEngineList(AIVehicle.VT_ROAD);
		engines.Valuate(AIEngine.CanRefitCargo, cargoID)
		engines.KeepValue(1);
		engines.Valuate(AIEngine.GetRoadType);
		engines.KeepValue(AIRoad.ROADTYPE_ROAD);
		engines.Valuate(AIEngine.GetCapacity);
		engines.KeepTop(1);
		bestEngines[cargoID] <- engines.Begin();
	}
	
	return bestEngines[cargoID];
}

/**
 * Return the sum of a vehicle's profit over the last year and the current year.
 */
function GetVehicleProfit(vehicleID) {
	return AIVehicle.GetProfitThisYear(vehicleID) + AIVehicle.GetProfitLastYear(vehicleID);
}

function MaxLoan() {
	// since we can't (accidentally) spend money that's not in our bank balance,
	// keep enough available in the form of loanable money to avoid bankrupcy
	local safetyIntervals = Ceiling(GetMinimumSafeMoney().tofloat()/AICompany.GetLoanInterval());
	local me = AICompany.ResolveCompanyID(AICompany.COMPANY_SELF);
	if (AICompany.GetBankBalance(me) < 125000)
		AICompany.SetLoanAmount(AICompany.GetMaxLoanAmount() - safetyIntervals * AICompany.GetLoanInterval());
}

/**
 * Max out our loan, including the amount kept in reserve for paying station maintenance.
 * Make sure you don't spend it!
 */
function FullyMaxLoan() {
	AICompany.SetLoanAmount(AICompany.GetMaxLoanAmount());
}

function ManageLoan() {
	// repay loan
	// if we dip below our MinimumSafeMoney, we'll extend the loan up to its maximum,
	// but the AI must take care not to spend this extra cash!
	local me = AICompany.ResolveCompanyID(AICompany.COMPANY_SELF);
	local balance = AICompany.GetBankBalance(me) - GetMinimumSafeMoney();
	local balanceMod = (balance / AICompany.GetLoanInterval()) * AICompany.GetLoanInterval();
	local loan = AICompany.GetLoanAmount();

	if (balance < 0) {
		// not good... we lost money?
		FullyMaxLoan();
		balance = AICompany.GetBankBalance(me) - GetMinimumSafeMoney();
		balanceMod = (balance / AICompany.GetLoanInterval()) * AICompany.GetLoanInterval();
		loan = AICompany.GetLoanAmount();
	}

	if (loan > 0 && balanceMod > 0) {
		if (balanceMod >= loan) {
			AICompany.SetLoanAmount(0);
		} else {
			AICompany.SetLoanAmount(loan - balanceMod);
		}
	}
}

function GetAvailableMoney() {
	// how much we have, plus how much we can borrow
	local me = AICompany.ResolveCompanyID(AICompany.COMPANY_SELF);
	return AICompany.GetBankBalance(me) + AICompany.GetMaxLoanAmount() - AICompany.GetLoanAmount();
}

/**
 * If this is the month before a bankrupcy check (March, June, September, December),
 * keep enough cash around to pay for station maintenance.
 * In other months, feel free to spend what we have.
 */
function GetMinimumSafeMoney() {
	local date = AIDate.GetCurrentDate();
	local month = AIDate.GetMonth(date);
	if (month % 3 == 0) {
		local safe = SAFETY_MARGIN;
		safe += (AIStationList(AIStation.STATION_ANY).Count() * MONTHLY_STATION_MAINTENANCE);
		// this is of course just a guesstimate, as the amount of vehicles constantly fluctuates
		safe += (AIVehicleList().Count() * MONTHLY_STATION_MAINTENANCE);	// TODO get cost from a vehicle(engine)
		return safe;
	} else {
		return 0;
	}
}

/**
 * Add a rectangular area to an AITileList containing tiles that are within /radius/
 * tiles from the center tile, taking the edges of the map into account.
 */  
function SafeAddRectangle(list, tile, radius) {
	local x1 = max(0, AIMap.GetTileX(tile) - radius);
	local y1 = max(0, AIMap.GetTileY(tile) - radius);
	
	local x2 = min(AIMap.GetMapSizeX() - 2, AIMap.GetTileX(tile) + radius);
	local y2 = min(AIMap.GetMapSizeY() - 2, AIMap.GetTileY(tile) + radius);
	
	list.AddRectangle(AIMap.GetTileIndex(x1, y1),AIMap.GetTileIndex(x2, y2)); 
}

/**
 * Filter an AITileList for AITile.IsBuildable tiles.
 */
function KeepBuildableArea(area) {
	area.Valuate(AITile.IsBuildable);
	area.KeepValue(1);
	return area;
}

/**
 * Create an array from an AIList.
 */
function ListToArray(l) {
	local a = [];
	for (local item = l.Begin(); l.HasNext(); item = l.Next()) a.append(item);
	return a;
}

/**
 * Create an AIList from an array.
 */
function ArrayToList(a) {
	local l = AIList();
	foreach (item in a) l.AddItem(item, 0);
	return l;
}

/**
 * Create a string of all elements of an array, separated by a comma.
 */
function ArrayToString(a) {
	local s = "";
	foreach (index, item in a) {
		if (index > 0) s += ", ";
		s += item;
	}
	
	return s;
}

/**
 * Return the closest integer equal to or greater than x.
 */
function Ceiling(x) {
	if (x.tointeger().tofloat() == x) return x.tointeger();
	return x.tointeger() + 1;
}

/**
 * Remove a trailing \0 character from the end of a string.
 * These occur after town names containing accented characters.
 */
function RemoveTrailingNull(s) {
	if (s == null) return null;
	
	local lastChar = s.slice(-1)[0];
	return (lastChar == 0) ? s.slice(0, -1) : s;
}

function Debug(s) {
	AILog.Info(GetDate() + ": " + s);
}

function Warning(s) {
	AILog.Warning(GetDate() + ": " + s);
}

function Error(s) {
	AILog.Error(GetDate() + ": " + s);
}

function GetTick() {
	return ::INSTANCE.GetTick();
}

function GetDate() {
	local date = AIDate.GetCurrentDate();
	return AIDate.GetYear(date) + "/" + AIDate.GetMonth(date) + "/" + AIDate.GetDayOfMonth(date);
}

function PrintError() {
	AILog.Error(AIError.GetLastErrorString());
}
