/**
 *    This file is part of Rondje.
 *
 *    Rondje is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    Rondje is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with Rondje.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright 2008-2009 Marnix Bakker, Willem de Neve, Michiel Konstapel and Otto Visser.
 * Suggestions/comments and bugs can go to: rondjeomdekerk@konstapel.nl
 */

require("RoadFinder/graph/queue/BinHeap.nut");
require("util.nut");
require("route.nut");
require("builder.nut");

// NOTE: these values are designed to create areas which don't overlap
// which is important so that we don't confuse pickup and dropoff stations 
const DISTRICT_DISTANCE = 5;
const DISTRICT_RADIUS = 2;
const DROPOFF_DISTANCE = 8;	// further out reduces congestion, but closer in means we can build it for smaller towns
const DROPOFF_RADIUS = 2;

enum Dropoffs {
	NORTH_WEST
	NORTH_EAST
	SOUTH_WEST
	SOUTH_EAST
};

class World {

	numConnections = 0;
	producers = null;
	consumers = null;
	towns = null;
	routeList = null;
	networks = null;
	
	constructor() {
		numConnections = 0;
		producers = [];
		consumers = [];
		networks = [];
		routeList = [];
	}
	
	/**
	 * Construct a model of the world: producers, consumers and routes.
	 */
	function Map() {
		// keep old data available until we're done mapping
		local newRouteList = [];
		local newNetworks = [];
		
		local tick = GetTick();
		producers = Industry.FindServicedProducers();
		consumers = Industry.FindServicedConsumers();
		towns = Town.FindServicedTowns();
		Debug("Finding serviced destinations took " + (GetTick() - tick) + " ticks");
		
		// follow the road networks from all stations
		tick = GetTick();
		foreach (producer in producers) Trace(newNetworks, producer);
		ManageVehicles();
		foreach (consumer in consumers) Trace(newNetworks, consumer);
		ManageVehicles();
		foreach (town in towns) Trace(newNetworks, town);
		ManageVehicles();
		Debug("Tracing road networks took " + (GetTick() - tick) + " ticks");
		
		tick = INSTANCE.GetTick();
		local connections = {};
		foreach (network in newNetworks) {
			foreach (destination1 in network.destinations) {
				foreach (destination2 in network.destinations) {
					local distance = AIMap.DistanceManhattan(destination1.GetLocation(),destination2.GetLocation());
					if (distance > MIN_DISTANCE && distance < MAX_DISTANCE) { 
						connections[Key(destination1, destination2)] <- network;
					}
				}
			}
		}
		Debug("Creating connection graph took " + (GetTick() - tick) + " ticks");
		
		if (connections.len() == numConnections) {
			// no change, no need to continue
			return;
		}
		
		numConnections = connections.len();
		Debug("New connections!");
		
		foreach (producer in producers) {
			// Debug("Serviced producer: " + producer.GetName() + producer.GetProduction() + " units of " + producer.GetCargoLabel());
			foreach (consumer in consumers) {
				local cargoID = producer.GetCargo();
				if (consumer.AcceptsCargo(cargoID)) {
					// Debug("Serviced consumer: " + consumer.GetName() + " accepts " + producer.GetCargoLabel());
					local key = Key(producer, consumer);
					if (key in connections) {
						local network = connections[key];
						local route = Route(producer, consumer, network);
						if (route.IsBuildable()) {
							//newRouteHeap.Insert(route, -route.GetRouteValue());
							newRouteList.append(route);
						}
					}
				}
			}
			
			ManageVehicles();
		}
		
		tick = GetTick();
		// consider two connected towns two routes, A->B and B->A
		foreach (town in towns) {
			foreach (town2 in towns) {
				if (town != town2) {
					local key = Key(town, town2);
					if (key in connections) {
						local network = connections[key];
						// add routes for each city district
						foreach (district in town.GetDistricts()) {
							local dropoff = FindClosestDropoff(district, town2);
							local route = Route(district, dropoff, network);
							if (route.IsBuildable()) {
								//newRouteHeap.Insert(route, -route.GetRouteValue());
								newRouteList.append(route);
							}
						}
					}
				}
			}
			
			ManageVehicles();
		}
		Debug("Finding connected towns took " + (GetTick() - tick) + " ticks");
		
		routeList = newRouteList;
		networks = newNetworks;
	}
	
	function Key(destination1, destination2) {
		return destination1.GetName() + "-" + destination2.GetName();
	}
	
	/**
	 * Returns an unordered list of routes.
	 */
	function GetRouteList() {
		return routeList;
	}
	
	function SetRouteList(routeL) {
		routeList = routeL;
	}
	
	function GetNetworks() {
		return networks;
	}
	
	/**
	 * Follow the roads from all stations at a destination
	 * and determine which stations are connected.
	 */
	function Trace(networks, destination) {
		local stations = destination.GetStations();
		foreach (station in stations) {
			if (GetNetwork(networks, station) == null) {
				local newNetwork = FindConnectedStations(station);
				networks.append(newNetwork);
			}
		}
	}
	
	/**
	 * Return a network in the list of networks that contains the given station,
	 * or null if there isn't one.
	 */
	function GetNetwork(networks, station) {
		foreach (network in networks) {
			if (network.Contains(station.stationID)) return network;
		}
		
		return null;
	}
	
	/**
	 * Find the dropoff in the town that is closest to the given district.
	 */
	function FindClosestDropoff(destination, town) {
		local closest = null;
		local distance = 1000000;
		foreach (dropoff in town.dropoffs) {
			local newDistance = AIMap.DistanceManhattan(destination.GetLocation(), dropoff.GetLocation());
			if (newDistance < distance) {
				closest = dropoff;
				distance = newDistance;
			}
		}
		
		return closest;
	}
	
	/**
	 * See if a route exists between two cargo-matched industries.
	 * @return the (first) network in which they are connected
	 */
	function FindConnectingNetwork(networks, producer, consumer) {
		//Debug("Looking for route between " + producer.GetName() + " and " + consumer.GetName());
		local sourceStations = producer.GetStations();
		local sinkStations = consumer.GetStations();
		
		foreach (sourceStation in sourceStations) {
			// Debug("Source " + sourceStation.stationID + " @ " + producer.GetName());
			local network = GetNetwork(networks, sourceStation);
			if (!network) continue;
			
			foreach (sinkStation in sinkStations) {
				if (network.Contains(sinkStation.stationID)) {
					// Sign(sinkStation.GetLocation(), sinkStation.stationID + " accepts " + producer.GetCargoLabel());
					// Debug(sinkStation.stationID + " at " + consumer.GetName() + " connects to " + sourceStation.stationID + " at " + producer.GetName());
					return network;
				}
			}
		}
		
		return null;
	}
	
	/**
	 * Return a Network starting at the given station.
	 */
	function FindConnectedStations(station) {
		local stationTile = station.GetLocation();
		local network = Network();
		network.stations.append(station.stationID);
		Follow(stationTile, network.tiles, network.stations);
		foreach (stationID in network.stations) {
			// if a station is built between looking for serviced destinations and coming here,
			// there won't be a Station object for it 
			local station = Station.GetStation(stationID);
			if (station) network.destinations.append(station.GetDestination());
		}
		
		return network;
	}
	
	function Follow(tile, visitedTiles, stationIDs) {
		// manage vehicles every N tiles
		if (visitedTiles.Count() % 100 == 0) {
			ManageVehicles();
		}
		
		local roads = ConnectedRoads(tile, visitedTiles, stationIDs);
		if (roads.Count() == 0) return;
		
		for (local road = roads.Begin(); roads.HasNext(); road = roads.Next()) {
			visitedTiles.AddTile(road);
			Follow(road, visitedTiles, stationIDs);
		}
	}
	
	function ConnectedRoads(tile, visitedTiles, stationIDs) {
		local connected = AITileList();
		local adjacent = AITileList();
		adjacent.AddTile(tile - AIMap.GetTileIndex(1,0));
		adjacent.AddTile(tile - AIMap.GetTileIndex(0,1));
		adjacent.AddTile(tile - AIMap.GetTileIndex(-1,0));
		adjacent.AddTile(tile - AIMap.GetTileIndex(0,-1));
		
		for (local neighbourTile = adjacent.Begin(); adjacent.HasNext(); neighbourTile = adjacent.Next()) {
			if (visitedTiles.HasItem(neighbourTile)) continue;
			
			if (RoadsConnected(tile, neighbourTile)) {
				connected.AddTile(neighbourTile);
			} else if (TunnelOrBridgeConnected(tile, neighbourTile)) {
				connected.AddTile(GetOtherEnd(neighbourTile));
			}
			
			if (StationConnected(tile, neighbourTile)) {
				local stationID = AIStation.GetStationID(neighbourTile);
				stationIDs.append(stationID);
				if (!connected.HasItem(neighbourTile)) connected.AddTile(neighbourTile);
			}
		}
		
		return connected;
	}
	
	function RoadsConnected(tile, neighbourTile) {
		return AIRoad.IsRoadTile(neighbourTile) && AIRoad.AreRoadTilesConnected(tile, neighbourTile);
	}
	
	function TunnelOrBridgeConnected(tile, neighbourTile) {
		return (AITunnel.IsTunnelTile(neighbourTile) || AIBridge.IsBridgeTile(neighbourTile)) &&
			AIRoad.AreRoadTilesConnected(tile, neighbourTile);
	}
	
	function GetOtherEnd(tile) {
		if (AITunnel.IsTunnelTile(tile)) return AITunnel.GetOtherTunnelEnd(tile);
		if (AIBridge.IsBridgeTile(tile)) return AIBridge.GetOtherBridgeEnd(tile);
		throw "not a bridge or tunnel";
	}
	
	/**
	 * See if a station is connected: the road tile is connected to the
	 * front or back station tile.
	 */
	function StationConnected(roadTile, stationTile) {
		return AIRoad.IsRoadStationTile(stationTile) && (
			AIRoad.GetRoadStationFrontTile(stationTile) == roadTile || 
			AIRoad.IsDriveThroughRoadStationTile(stationTile) && AIRoad.GetDriveThroughBackTile(stationTile) == roadTile
		);
	}
}

/**
 * A network is a set of connected road tiles and the stations and destinations they connected.
 */
class Network {
	
	tiles = null;
	stations = null;
	destinations = null;
	
	constructor() {
		tiles = AITileList();
		stations = [];
		destinations = [];
	}
	
	function Contains(stationID) {
		return IndexOf(stations, stationID) != -1;
	}
}

class MapObject {
	
	location = 0;	// location TileIndex
	
	constructor(location) {
		this.location = location;
	}
	
	function GetLocation() {
		return location;
	}
}

class Sign extends MapObject {
	
	id = 0;
	
	constructor(location, text) {
		MapObject.constructor(location);
		id = AISign.BuildSign(location, text);
	}
}

/**
 * A source or destination for cargo (including passengers).
 */
class Destination extends MapObject {
	
	cargoID = 0;
	myStation = null;
	stations = null;
	
	constructor(location, cargoID) {
		MapObject.constructor(location);
		this.cargoID = cargoID;
		this.myStation = null;
		this.stations = [];
	}
	
	/**
	 * Destination name.
	 */
	function GetName();
	
	/**
	 * Return an AITileList of tiles where cargo can be picked up.
	 */
	function GetPickupArea();
	
	/**
	 * Return an AITileList of tiles where cargo can be dropped off.
	 */
	function GetDropoffArea();
	
	/**
	 * Return an AITileList of the area in which we may find stations for this destination.
	 */
	function GetStationArea();
	
	/**
	 * Station type: AIStation.STATION_BUS_STOP or AIStation.STATION_TRUCK_STOP.
	 */
	function GetStationType();
	
	/**
	 * Returns whether this destination still exists, since industries can disappear (esp. oil wells).
	 */
	function Exists() {
		// default to true; overridden in Industry
		return true;
	}
	
	function GetCargo() {
		return cargoID;
	}
	
	function GetCargoLabel() {
		return AICargo.GetCargoLabel(cargoID);
	}
	
	function GetStations() {
		FindStationsInternal();
		// Debug(GetName() + " has " + stations.len() + " stations: " + ArrayToString(stations));
		return stations;
	}
	
	function GetMyStation() {
		// see if GetStations() already found it
		if (!myStation) FindStationsInternal();
		return myStation;
	}
	
	function SetMyStation(station) {
		this.myStation = station;
	}
	
	function GetMyDepot() {
		return Depot.GetDepot(this);
	}
	
	function FindStationsInternal() {
		stations.clear();
		myStation = null;
		
		local area = GetStationArea();
		area.Valuate(AIRoad.IsRoadStationTile);
		area.KeepValue(1);
		
		for (local tile = area.Begin(); area.HasNext(); tile = area.Next()) {
			local stationID = AIStation.GetStationID(tile);
			local station = Station(stationID, tile, this);
			stations.append(station);
			
			// if we find multiple stations of our own, use the closest
			if (AICompany.IsMine(AITile.GetOwner(tile)) && AIStation.HasStationType(stationID, GetStationType())) {
				if (myStation == null || IsCloser(station))
					myStation = station;
			}
		}
	}
	
	/**
	 * See if this station is closer than myStation (must not be null).
	 */
	function IsCloser(station) {
		local newDistance = AIMap.DistanceManhattan(station.GetLocation(), this.GetLocation());
		local currentDistance = AIMap.DistanceManhattan(myStation.GetLocation(), this.GetLocation());
		return newDistance < currentDistance;
	}
}


class Town extends Destination {
	
	townID = 0;
	dropoffs = null;
	districts = null;
	
	constructor(townID, cargoID) {
		if (!AITown.IsValidTown(townID)) throw "Invalid townID";
		Destination.constructor(AITown.GetLocation(townID), cargoID);
		this.townID = townID;
		this.dropoffs = [];
		this.districts = [];
	}
	
	function GetName() {
		return RemoveTrailingNull(AITown.GetName(townID)) + " " + townID;
	}
	
	/**
	 * Calculate an estimated town radius based on population.
	 */
	function GetRadius() {
		return Sqrt(GetPopulation()/100) + 3;
	}
	
	function GetStationArea() {
		local area = AITileList();
		SafeAddRectangle(area, AITown.GetLocation(townID), GetRadius());
		return area;
	}
	
	function GetStationType() {
		return AIStation.STATION_BUS_STOP;
	}
	
	function IsServiced() {
		return GetStations().len() > 0;
	}
	
	function GetPopulation() {
		return AITown.GetPopulation(townID);
	}
	
	function GetProduction() {
		return AITown.GetMaxProduction(townID, GetCargo());
	}
	
	function GetDistricts() {
		return districts;
	}
	
	/**
	 * Create a list of serviced towns.
	 */
	static function FindServicedTowns() {
		local towns = [];
		local list = AITownList();
		for (local townID = list.Begin(); list.HasNext(); townID = list.Next()) {
			local town = Town(townID, PASSENGERS);
			if (town.IsServiced()) {
				towns.append(town);
				local center = District(town, PASSENGERS, Districts.CENTER);
				
				foreach (sector in [Dropoffs.NORTH_WEST, Dropoffs.NORTH_EAST, Dropoffs.SOUTH_WEST, Dropoffs.SOUTH_EAST]) {
					local dropoff = PassengerDropoff(town, sector);
					if (dropoff.GetDropoffArea().Count() > 0) town.dropoffs.append(dropoff);
				}
				
				town.districts = [];
				
				// (only) if there are no dedicated dropoffs available, drop people off at the city center
				// in this case, the town is too small to generate passengers; just use it as a dropoff
				if (town.dropoffs.len() == 0) {
					town.dropoffs.append(center);
				} else {
					// we have at least one dedicated dropoff
					town.districts.append(center);
					
					local sectors = [];
					if (town.GetPopulation() > 1000) {
						sectors.append(Districts.NORTH);
					}
					if (town.GetPopulation() > 2000) {
						sectors.append(Districts.SOUTH);
					}
					if (town.GetPopulation() > 3000) {
						sectors.append(Districts.WEST);
					}
					if (town.GetPopulation() > 4000) {
						sectors.append(Districts.EAST);
					}
					
					foreach (sector in sectors) {
						local district = District(town, PASSENGERS, sector);
						if (district.GetPickupArea().Count() > 0) town.districts.append(district);
					}
				}
			}
		}
		
		return towns;
	}
}

class District extends Destination {
	
	town = null;
	district = null;
	
	constructor(town, cargoID, district) {
		local direction;
		switch (district) {
			case Districts.CENTER: direction = [ 0,  0]; break;
			case Districts.NORTH:  direction = [-1, -1]; break;
			case Districts.SOUTH:  direction = [ 1,  1]; break;
			case Districts.WEST:   direction = [ 1, -1]; break;
			case Districts.EAST:   direction = [-1,  1]; break;
		}
		
		local location = town.GetLocation() + AIMap.GetTileIndex(direction[0] * DISTRICT_DISTANCE, direction[1] * DISTRICT_DISTANCE);
		Destination.constructor(location, cargoID);
		this.town = town;
		this.district = district;
	}
	
	function GetDistrict() {
		return district;
	}
	
	function GetName() {
		local suffix;
		switch (district) {
			case Districts.CENTER: suffix = "Center"; break;
			case Districts.NORTH: suffix = "North"; break;
			case Districts.SOUTH: suffix = "South"; break;
			case Districts.WEST: suffix = "West"; break;
			case Districts.EAST: suffix = "East"; break;
		}
		
		return town.GetName() + " " + suffix;
	}
	
	function GetStationArea() {
		local area = AITileList();
		SafeAddRectangle(area, GetLocation(), DISTRICT_RADIUS);
		return area;
	}
	
	function GetStationType() {
		return AIStation.STATION_BUS_STOP;
	}
	
	function GetPopulation() {
		return town.GetPopulation() / town.GetDistricts().len();
	}
	
	function GetProduction() {
		return town.GetProduction() / town.GetDistricts().len();
	}
	
	function GetPickupArea() {
		local area = AITileList();
		
		local roads = AITileList();
		SafeAddRectangle(roads, GetLocation(), DISTRICT_RADIUS);
		roads.Valuate(AIRoad.IsRoadTile);
		roads.KeepValue(1);
		area.AddList(roads);
		
		local clear = AITileList();
		SafeAddRectangle(clear, GetLocation(), DISTRICT_RADIUS);
		clear.AddList(GetStationArea());
		clear.Valuate(AITile.IsBuildable);
		clear.KeepValue(1);
		area.AddList(clear);
		
		// for suburbs, only consider tiles that capture passengers from several tiles
		if (district != Districts.CENTER) {
			area.Valuate(AITile.GetCargoProduction, PASSENGERS, 1, 1, BUS_STATION_RADIUS);
			area.KeepAboveValue(12);	// out of a 49 tile area
		}
		
		return area;
	}
	
	function GetDropoffArea() {
		return GetPickupArea();
	}
}

class PassengerDropoff extends Destination {
	
	town = null;
	sector = null;
	
	constructor(town, sector) {
		local direction;
		switch (sector) {
			case Dropoffs.NORTH_EAST: direction = [-1, 0]; break;
			case Dropoffs.NORTH_WEST: direction = [ 0,-1]; break;
			case Dropoffs.SOUTH_EAST: direction = [ 0, 1]; break;
			case Dropoffs.SOUTH_WEST: direction = [ 1, 0]; break;
		}
		
		// NOTE: stay out of the areas where we might find pickup stations or we'd confuse the two types
		local location = town.GetLocation() + AIMap.GetTileIndex(direction[0] * DROPOFF_DISTANCE, direction[1] * DROPOFF_DISTANCE);
		
		Destination.constructor(location, PASSENGERS);
		this.town = town;
		this.sector = sector;
	}
	
	function GetName() {
		local suffix;
		switch (sector) {
			case Dropoffs.NORTH_EAST: suffix = "NE"; break;
			case Dropoffs.NORTH_WEST: suffix = "NW"; break;
			case Dropoffs.SOUTH_EAST: suffix = "SE"; break;
			case Dropoffs.SOUTH_WEST: suffix = "SW"; break;
		}
		
		return town.GetName() + " " + suffix + " Dropoff";
	}
	
	function GetStationArea() {
		local area = AITileList();
		SafeAddRectangle(area, GetLocation(), DROPOFF_RADIUS);
		//for (local tile = area.Begin(); area.HasNext(); tile = area.Next()) Sign(tile, GetName());
		return area;
	}
	
	function GetStationType() {
		return AIStation.STATION_BUS_STOP;
	}
	
	function GetProduction() {
		return 0;
	}
	
	function GetPickupArea() {
		return AITileList();
	}
	
	function GetDropoffArea() {
		local area = AITileList();
		local roads = GetStationArea();
		roads.Valuate(AIRoad.IsRoadTile);
		roads.KeepValue(1);
		area.AddList(roads);
		
		local clear = AITileList();
		clear.AddList(GetStationArea());
		clear.Valuate(AITile.IsBuildable);
		clear.KeepValue(1);
		area.AddList(clear);
		
		// keep tiles that accept enough passengers
		// an acceptance of 8 is required to accept a cargo,
		// and we don't want the first house torn down making our station useless
		area.Valuate(AITile.GetCargoAcceptance, PASSENGERS, 1, 1, BUS_STATION_RADIUS);
		area.KeepAboveValue(16);
		return area;
	}
}

class Industry extends Destination {
	
	industryID = 0;
	numStations = 0;
	
	constructor(industryID, cargoID) {
		if (!AIIndustry.IsValidIndustry(industryID)) throw "Invalid industryID";
		Destination.constructor(AIIndustry.GetLocation(industryID), cargoID);
		this.industryID = industryID;
		this.numStations = 0;
	}
	
	function GetName() {
		return RemoveTrailingNull(AIIndustry.GetName(industryID)) + " " + industryID;
	}
	
	function GetStationArea() {
		local area = AITileList();
		area.AddList(AITileList_IndustryAccepting(industryID, TRUCK_STATION_RADIUS));
		area.AddList(AITileList_IndustryProducing(industryID, TRUCK_STATION_RADIUS));
		return area;
	}
	
	function GetStationType() {
		return AIStation.STATION_TRUCK_STOP;
	}
	
	function Exists() {
		return AIIndustry.IsValidIndustry(industryID);
	}
	
	/**
	 * Return whether this industry has stations.
	 */
	function IsServiced() {
		return GetStations().len() > 0;
	}
	
	function GetProduction() {
		return AIIndustry.GetLastMonthProduction(industryID, cargoID);
	}
	
	function AcceptsCargo(cargoID) {
		return AIIndustry.IsCargoAccepted(industryID, cargoID);
	}
	
	/**
	 * Returns an AITileList of tiles at which a road station can be built to pick up cargo.
	 */
	function GetPickupArea() {
		return KeepBuildableArea(AITileList_IndustryProducing(industryID, TRUCK_STATION_RADIUS));
	}

	/**
	 * Returns an AITileList of tiles at which a road station can be built to deliver cargo.
	 */
	function GetDropoffArea() {
		return KeepBuildableArea(AITileList_IndustryAccepting(industryID, TRUCK_STATION_RADIUS));
	}
	
	function _tostring() {
		return GetName();
	}

	/**
	 * Create a list of all serviced cargo producing industries.
	 */
	static function FindServicedProducers() {
		local industries = AIIndustryList();
		local cargoes = AICargoList();
		local producing = [];
		
		for (local i = industries.Begin(); industries.HasNext(); i = industries.Next()) {
			for (local c = cargoes.Begin(); cargoes.HasNext(); c = cargoes.Next()) {
				local production = AIIndustry.GetLastMonthProduction(i, c);
				if (production > 0) {
					//Debug(AIIndustry.GetName(i) + " produces " + production + " units of " + AICargo.GetCargoLabel(c));
					local industry = Industry(i, c);
					if (industry.IsServiced()) producing.append(industry);
				}
			}
		}
		
		return producing;
	}
	
	/**
	 * Create a list of all cargo accepting industries.
	 */
	static function FindServicedConsumers() {
		local industries = AIIndustryList();
		local cargoes = AICargoList();
		local accepting = [];
		
		for (local i = industries.Begin(); industries.HasNext(); i = industries.Next()) {
			for (local c = cargoes.Begin(); cargoes.HasNext(); c = cargoes.Next()) {
				if (AIIndustry.IsCargoAccepted(i, c)) {
					// Debug(AIIndustry.GetName(i) + " accepts " + AICargo.GetCargoLabel(c));
					local industry = Industry(i, c);
					if (industry.IsServiced()) accepting.append(industry);
				}
			}
		}
		
		return accepting;
	}
}

class Station extends MapObject {
	
	static stations = {};	// maps station IDs to Station instances
	
	static function GetStation(stationID) {
		return stationID in Station.stations ? Station.stations[stationID] : null;
	}
	
	stationID = 0;
	destination = null;

	/**
	 * Create a Station. We can't ask for the location of a
	 * competitor's station, so we use the tile we found it at.
	 */	
	constructor(stationID, location, destination) {
		MapObject.constructor(location);
		this.stationID = stationID;
		this.destination = destination;
		Station.stations[stationID] <- this;
	}
	
	function GetStationID() {
		return stationID;
	}
	
	function GetName() {
		return AIStation.GetName(stationID);
	}
	
	function GetDestination() {
		return destination;
	}
	
	function AcceptsCargo(cargoID) {
		return AITile.GetCargoAcceptance(GetLocation(), cargoID, 1, 1, TRUCK_STATION_RADIUS) >= 8;
	}
	
	function _tostring() {
		return stationID.tostring();
	}
	
	function IsOccupied() {
		// check for vehicles currently loading at the station
		local vehicles = AIVehicleList_Station(stationID);
		vehicles.Valuate(AIVehicle.GetLocation);
		vehicles.KeepValue(GetLocation());
		vehicles.Valuate(AIVehicle.GetState);
		vehicles.KeepValue(AIVehicle.VS_AT_STATION);
		if (vehicles.Count() > 0) return true;
		
		// check for vehicles on their way to load at the station
		vehicles = AIVehicleList_Station(stationID);
		vehicles.Valuate(AIOrder.ResolveOrderPosition, AIOrder.ORDER_CURRENT);
		vehicles.KeepValue(0);
		vehicles.Valuate(AIOrder.GetOrderDestination, AIOrder.ORDER_CURRENT);
		vehicles.KeepValue(GetLocation());
		vehicles.Valuate(DistanceValuator, GetLocation());
		vehicles.KeepBelowValue(DEPOT_RANGE + 1);
		
		return vehicles.Count() > 0;
	}
	
	function DistanceValuator(vehicle, stationTile) {
		return AIMap.DistanceManhattan(AIVehicle.GetLocation(vehicle), stationTile);
	}
}

class Depot extends MapObject {
	
	// keep track of our depots by destination name
	static depots = {}
	
	static function GetDepot(destination) {
		return destination.GetName() in Depot.depots ? Depot.depots[destination.GetName()] : null;
	}
	
	static function SetDepot(destination, depot) {
		depots[destination.GetName()] <- depot;
	}
	
	constructor(location, destination) {
		MapObject.constructor(location);
		depots[destination.GetName()] <- this;
	}
	
	function BuildVehicle(cargoID) {
		local vehicle = AIVehicle.BuildVehicle(location, GetBestEngine(cargoID));
		AIVehicle.RefitVehicle(vehicle, cargoID);
		return vehicle;
	}
	
	function CloneVehicle(vehicleID) {
		return AIVehicle.CloneVehicle(location, vehicleID, true);
	}
}


class Vehicle extends MapObject {
	
	static vehicles = {};	// maps vehicle IDs to Vehicle instances
	
	static function GetVehicle(vehicleID) {
		return vehicleID in Vehicle.vehicles ? Vehicle.vehicles[vehicleID] : null;
	}
	
	vehicleID = 0;
	route = null;
	ageOffset = 0;
	isSubsidyRunner = false;
	headingForDepot = false;
	hinUndWieder = false;
	
	constructor(vehicleID, route) {
		this.vehicleID = vehicleID;
		this.route = route;
		this.ageOffset = 0;
		this.isSubsidyRunner = false;
		this.headingForDepot = false;
		this.hinUndWieder = false;
		vehicles[vehicleID] <- this;
	}
	
	function GetName() {
		return AIVehicle.GetName(vehicleID);
	}
	
	/**
	 * Returns the current destination of a vehicle.
	 */
	function GetCurrentDestination() {
		return AIOrder.GetOrderDestination(vehicleID, AIOrder.ORDER_CURRENT);
	}
	
	function GoToDepot() {
		if (headingForDepot) {
			// already heading to depot
			return true;
		} else {
			if (AIVehicle.SendVehicleToDepot(vehicleID)) {
				headingForDepot = true;
				return true;
			}
		}
	}
	
	function IsHeadingForDepot() {
		return headingForDepot;
	}
	
	/**
	 * A vehicle is a bus if it holds passengers.
	 */
	function IsBus() {
		return AIVehicle.GetCapacity(vehicleID, PASSENGERS) > 0;
	}

	/**
	 * A vehicle is at its destination if its location is the same as its second order's destination.
	 */
	function IsAtDestination() {
		return AIVehicle.GetLocation(vehicleID) == AIOrder.GetOrderDestination(vehicleID, 1);
	}
	
	/**
	 * A vehicle is loading if it is VS_AT_STATION at its first order's destination.
	 */
	function IsLoading() {
		return AIVehicle.GetLocation(vehicleID) == AIOrder.GetOrderDestination(vehicleID, 0) &&
			AIVehicle.GetState(vehicleID) == AIVehicle.VS_AT_STATION;
	}
	
	/**
	 * A vehicle is loading if it is VS_AT_STATION at its second order's destination.
	 */
	function IsUnloading() {
		return AIVehicle.GetLocation(vehicleID) == AIOrder.GetOrderDestination(vehicleID, 1) &&
			AIVehicle.GetState(vehicleID) == AIVehicle.VS_AT_STATION;
	}
	
	/**
	 * Set the number of days to subtract from the vehicle's age when selling it.
	 */
	function SetAgeOffset(ageOffset) {
		this.ageOffset = ageOffset;
	}
	
	function SetHinUndWieder(hinUndWieder) {
		this.hinUndWieder = hinUndWieder;
		local loadFlag = hinUndWieder ? AIOrder.AIOF_NONE : AIOrder.AIOF_NO_LOAD;
		AIOrder.SetOrderFlags(vehicleID, 1, loadFlag | AIOrder.AIOF_NON_STOP_INTERMEDIATE);
	}
	
	function IsHinUndWieder() {
		return hinUndWieder;
	}
	
	function Sell() {
		local route = GetRoute();
		if (route && !isSubsidyRunner) {
			local age = AIVehicle.GetAge(vehicleID) - ageOffset;
			local vehicleCost = AIEngine.GetPrice(AIVehicle.GetEngineType(vehicleID)) - AIVehicle.GetCurrentValue(vehicleID);
			local profit = AIVehicle.GetProfitThisYear(vehicleID) + AIVehicle.GetProfitLastYear(vehicleID) - vehicleCost;
			//Debug("Update " + route + ": triptime = " + age + ", profit = " + profit.tofloat()/age + " G/day");
			// can't happen, but I do see routes with infinite value, so check for zero age -- Michiel
			if (age > 0) route.Update(age, profit.tofloat()/age);
		}
		
		local sold = AIVehicle.SellVehicle(vehicleID);
		if (sold) vehicles[vehicleID] <- null;
		return sold;
	}
	
	/**
	 * Find this vehicle's route in the current list of routes.
	 */
	function GetRoute() {
		return route;
	}
	
	/**
	 * Return whether the vehicle is currently empty.
	 */
	function IsEmpty() {
		local list = AICargoList();
		for (local cargoID = list.Begin(); list.HasNext(); cargoID = list.Next()) {
			if (AIVehicle.GetCargoLoad(vehicleID, cargoID) > 0) return false;
		}
		
		return true;
	}
	
	/**
	 * Return whether the vehicle is currently full.
	 */
	function IsFull() {
		local list = AICargoList();
		for (local cargoID = list.Begin(); list.HasNext(); cargoID = list.Next()) {
			if (AIVehicle.GetCargoLoad(vehicleID, cargoID) < AIVehicle.GetCapacity(vehicleID, cargoID)) return false;
		}
		
		return true;
	}
}
