/**
 * Class that handles nodes "spawning".
 */
class AISpawnNodesAction extends TriggerAction
{
/* public */
	/**
	 * Create AISpawnNodesAction object.
	 * @param period Time to sleep between big nodes sections handling.
	 */
	constructor(period)
	{
		::TriggerAction.constructor();
		this.period = period;
		this.regions = clone (PlayableRegions.Get().map_regions_to_play);
		if (this.regions.len() == 0) this.SelfRemove();
	}

	function GetName()
	{
		return "Spawning Nodes";
	}

/* protected */
	function IsReady()
	{
		return this.respawn_date <= AIDate.GetCurrentDate();
	}

	function Execute()
	{
		local t = AIController.GetTick();
		local region = this.regions.pop();
		AICompany.SetLoanAmount(AICompany.GetLoanInterval());

		local rgn_str = "map region #" + region;
		local start_msg = "Announce information about nodes inside " + rgn_str;
		CodeUtils.Log(start_msg + " ...", 2);

		local nodes_repository = NodesRepository.Get();
		if (!nodes_repository.IsFilled()) {
			nodes_repository.FillSelf();
		}
		this.FilterNodes(region);

		foreach (node_type_id, type_nodes in nodes_repository.nodes) {
			foreach (id, node in type_nodes) {
				if (node.GetRegion() == region) NewNodeEvent.Fire(node);
			}
		}
		RegionScanFinishedEvent.Fire(region);

		local finish_msg = " nodes information announced"
		t = AIController.GetTick() - t;
		CodeUtils.Log("... " + rgn_str + finish_msg + "  , ticks " + t, 2);

		if (this.regions.len() == 0) {
			this.SelfRemove();
			CodeUtils.Log("All nodes revealed!", 2);
			//AIController.Sleep(100000);
		} else {
			respawn_date = AIDate.GetCurrentDate() + this.period;
		}
		return 0;
	}

/* private */
	/** Time to sleep between big nodes sections handling */
	period = 0;

	/** Regions whose nodes to spawn */
	regions = null;

	/** Date of next spawn wave */
	respawn_date = -1000;

	/**
	 * Necessary last-moment actions with created raw nodes.
	 */
	function FilterNodes(region_id)
	{
		this.FindHeapNodes(region_id);
	}

	/**
	 * Finds "heap nodes" and removes their sub-nodes from public access.
	 * Thus those sub-nodes will not participate in regular routes creation
	 *  and only routes leading into heap center will exist for "heap" parts.
	 */
	function FindHeapNodes(region_id)
	{
		local types = TransportSchema.Get().node_types;
		local checked = {}, new_ = [], to_ignore = {};
		local nodes_repository = NodesRepository.Get();

		if (SaveLoadUtils.WasLoaded()) {
			foreach (node_type_id, type_nodes in nodes_repository.nodes) {
				foreach (dummy_id, node in type_nodes) {
					local runtime_id = node.GetID();
					if (runtime_id in to_ignore) continue;
					if (node.GetClassName() == HeapNode.GetClassName()) {
						to_ignore[runtime_id] <- 1;
						foreach (dummy, sub_node in node.secondary_nodes) {
							to_ignore[sub_node.GetID()] <- 1;
						}
					}
				}
			}
		}

		foreach (node_type_id, type_nodes in nodes_repository.nodes) {
			if (
				node_type_id == NodeTypeID.NT_PAX_TOWN ||
				node_type_id == NodeTypeID.NT_RAW_TOWN ||
				!(node_type_id in types) ||
				!(types[node_type_id].raw_production)
			) {
				continue;
			}

			foreach (id, node in type_nodes) {
				if (node.GetID() in to_ignore || node.node_region_id != region_id) continue;

				local heap = this.FindHeapAroundNode(node, checked, to_ignore);
				if (heap == null) continue;

				local central = heap.heap_nodes[heap.lead_node_id];
				foreach (runtime_id, node_in_local_heap in heap.heap_nodes) {
					checked[runtime_id] <- node_in_local_heap;
				}
				heap.heap_nodes.rawdelete(central.GetID());

				local c_ids = {};
				foreach (cx, dummy in central.type.production) {
					c_ids[cx] <- cx;
				}

				new_.append(HeapNode(central, c_ids, [TransitType.TT_TRUCK], heap.heap_nodes));
			}
		}

		foreach (dummy_id, node in checked) {
			nodes_repository.RemoveNode(node.type_id, node.node_id);
			node.is_closed = false;
		}
		foreach (dummy_id, node in new_) {
			nodes_repository.AddNode(node);
		}

		checked.clear();
		new_.clear();
		to_ignore.clear();
	}

	/**
	 * Finds "heap nodes"(if it exist) including the given node.
	 */
	function FindHeapAroundNode(node, checked, to_ignore)
	{
		local tmp_node_id = node.GetID();
		if (tmp_node_id in checked || tmp_node_id in to_ignore) return null;
		if (node.is_built_on_water) return null;

		local nodes_repository = NodesRepository.Get();
		local r_id = node.node_region_id;
		if (!(r_id in nodes_repository.region_grids)) return null;

		local r = MagicNumbers.heap_node_radius;

		/*
		 * We only need to find nodes of specific types, which produce(raw)
		 *  same cargo as the given node.
		 * So we get cargo ids produced by the given node...
		 */
		local c_ids = {};
		foreach (cx, dummy in node.type.production) {
			c_ids[cx] <- cx;
		}
		if (c_ids.len() == 0) return null;

		/* ...and get types which are raw producing it */
		local good_types = {[node.type_id] = node.type};
		foreach (type_id, type in TransportSchema.Get().node_types) {
			if (!type.raw_production) continue;
			foreach (c, dummy in type.production) {
				if (c in c_ids) good_types[type_id] <- type;
			}
		}

		/*
		 * Given node can be positioned inside heap(here: just set of nodes)
		 *  built around every nearby(distance <= given r) node.
		 * That's why we find heaps which can be built around all nearby nodes.
		 * Than we compare them and choose best heap.
		 */
		local t = node.location;

		local cells = nodes_repository.region_grids[r_id].GetCellsNearTile(t, 2 * r);
		local close_nodes = {};
		local everything_around = {};
		foreach (dummy_id, cell_objects in cells) {
			foreach (dummy_dummy_id, nearby_node in cell_objects) {
				if (!(nearby_node.type_id in good_types)) continue;
				if (nearby_node.GetID() in checked) continue;
				local d = AIMap.DistanceManhattan(t, nearby_node.location);
				if (d <= r) {
					close_nodes[nearby_node.GetID()] <- nearby_node;
				}
				if (d <= 2 * r) {
					everything_around[nearby_node.GetID()] <- nearby_node;
				}
			}
		}

		/*
		 * For each nearby node build its heap,
		 *  and index all heaps built by their center tiles.
		 * Such indexing needed to remove dublicates(sets with same elements),
		 *  plus we would need centers later anyway.
		 */
		local local_heaps = {};
		foreach (runtime_id, nearby_node in close_nodes) {
			local t = nearby_node.location;
			local local_heap = {};
			local x_sum = 0, y_sum = 0;
			foreach (dummy_dummy_id, x in everything_around) {
				local d = AIMap.DistanceManhattan(t, x.location);
				if (d <= r) {
					x_sum += AIMap.GetTileX(x.location);
					y_sum += AIMap.GetTileY(x.location);
					local_heap[x.GetID()] <- x;
				}
			}
			local n = local_heap.len();
			if (n > 1) {
				local_heaps[AIMap.GetTileIndex(x_sum / n, y_sum / n)] <- local_heap;
			}
		}
		if (local_heaps.len() == 0) return null;

		/*
		 * If we've found several nodes sets select the one with highest summary
		 *  cargo production.
		 */
		local f = function(node) : (c_ids) {
			local p = 0;
			foreach (c, dummy in c_ids) {
				p += node.GetFreeProduction(c);
			}
			return p;
		}
		local p_max = -1, best_heap = {};
		foreach (heap_center, heap in local_heaps) {
			local p = 0;
			foreach (runtime_id, node in heap) {
				p += f(node);
			}

			if (p > p_max) {
				p_max = p;
				t = heap_center;
			}
		}

		foreach (dummy_id, node in local_heaps[t]) {
			best_heap[node.GetID()] <- node;
		}

		/*
		 * Last: select one of the nodes to be resource gatherer.
		 * If heap consist of 3 or more nodes => choose "central" node;
		 *  lesser(exactly 2) => the one with higher production.
		 */
		if (best_heap.len() > 2) {
			f = function(node) : (t, best_heap) {
				local p = 0;
				foreach (dummy_id, item in best_heap) {
					p += AIMap.DistanceManhattan(node.location, item.location);
				}
				return -p;
			}
		}

		p_max = -best_heap.len() * (AIMap.GetMapSizeX() + AIMap.GetMapSizeY());
		local result = {lead_node_id = -1, heap_nodes = best_heap, c_ids = c_ids};
		foreach (node_runtime_id, node in best_heap) {
			local p = f(node);
			if (p > p_max) {
				p_max = p;
				result.lead_node_id = node.GetID();//node_runtime_id;
			}
		}

		return result;
	}
}
