/**
 * Class that represents vehicles handling.
 * One instance of this class will look after the one trade route
 *  and initiate vehicles buy/sell for it.
 */
class AIManageVehiclesAction extends Terron_AbstractAction
{
/* public */
	/**
	 * Creates new AIManageVehiclesAction action.
	 * @param trade_route Trade route to manage.
	 */
	constructor(trade_route)
	{
		::Terron_AbstractAction.constructor();

		this.trade_route = trade_route;
		this.popup_task = DisableTask(this);

		this.priority = Property(this.Update.bindenv(this), 45 * GameTime.DAY);
		this.priority.SetName("\"" + this.GetName() + "\" action priority update");

		TradeRouteClosedEvent.AddListener(this);
	}

	function GetName()
	{
		return "Managing " + this.trade_route.GetName();
	}

	function GetPriority()
	{
		return this.priority.Get();
	}

	function Execute()
	{
		if (this.trade_route.IsEnabled()) {
			while (this.n > 0) {
				if (!CorporationUtils.GetMoney(this.cost)) return -2;
				if (this.trade_route.BuyVehicle() == -1) return -1;
				this.n--;
			}
			if (this.n == 0) this.priority.ForceUpdate();
		} else {
			this.SelfRemove();
		}
		return 0;
	}

/* protected */
	/**
	 * Check trade route needs: buy, sell vehicles, or nothing.
	 */
	function Update()
	{
		local trade_route = this.trade_route;
		if (!trade_route.IsEnabled()) {
			this.popup_task.DisableAction(3 * GameTime.MONTH + GameTime.DAY);
			return APriority.BAD;
		}

		local previous_n = this.n;

		/*
		 * n <= 0 cause AI to ignore this action, and
		 *  n > 0 only when new vehicle required. 
		 *  So action will be executed only when new vehicle required. 
		 *  So Execute function must only buy vehicles!
		 */
		n = trade_route.EstimateVehiclesNeeds();
		if (n > 0) {
			local e = trade_route.GetBestEngineToBuy();
			if (!AIEngine.IsBuildable(e)) {
				CodeUtils.Log("No valid engine types [" + trade_route.GetName() + "]", 2);
				local t = GameTime.ENGINE_REVIEW_INTERVAL + GameTime.DAY;
				trade_route.Disable(t);
				this.popup_task.DisableAction(t + GameTime.DAY);
				return APriority.BAD;
			}

			/* +1000 to handle inflation and refit price with care */
			this.cost = AIEngine.GetPrice(e) + 1000;
			local income = trade_route.GetOneVehicleProfit();
			return ActionPriority.CalculatePriority(0, this.cost, income, 0);
		}

		local t = trade_route.GetManageInterval(n);

		/* If need to sell, we can sell right now */
		if (n < 0) {
			trade_route.SendVehiclesToSell(-n);
		} else {
			t = trade_route.GetManageInterval(previous_n);
		}

		if (this.popup_task != null) {
			this.popup_task.DisableAction(t + GameTime.DAY);
		} else {
			/* can happen when trade is closing */
			this.SelfRemove();
		}
		return APriority.BAD;
	}

/* private */
	/** Trade route to manage */
	trade_route = null;

	/** This action priority */
	priority = null;

	/** Cost of one vehicle to buy */
	cost = null;

	/** Amount of vehicles to buy */
	n = -1;

	/** This task should add/remove this action to/from action repository */
	popup_task = null;

/* public */
	/**
	 * Stop self when host trade route is closed.
	 * @param closed_trade_route Just closed trade route.
	 */
	function OnTradeRouteClose(closed_trade_route)
	{
		if (this.trade_route.object_id == closed_trade_route.object_id) {
			TradeRouteClosedEvent.RemoveListener(this);
			_ai_clock.RemoveTask(this.popup_task);

			this.priority.Destroy();
			this.trade_route = null;
			this.popup_task.manage_action = null;
			this.popup_task = null;
			this.SelfRemove();
		}
	}
}

/**
 * Class that removes manage action from the active actions list
 *  when it(manage action) must sleep, and returns to the active actions list
 *  when it(manage action) must work.<p>
 * Should save a couple of ticks. 
 */
class AIManageVehiclesAction.DisableTask extends Terron_AbstractTask
{
/* public */
	/**
	 * Creates new AIManageVehiclesAction.DisableTask task.
	 * @param manage_action Action to disable/enable.
	 */
	constructor(manage_action)
	{
		::Terron_AbstractTask.constructor(manage_action.GetID());
		this.manage_action = manage_action;
		this.SetName("Pop \"" + manage_action.GetName() + "\" action");
	}

	/**
	 * Remove assosciated action from the actions repository
	 *  for the given period of time.
	 * @param time "Disable" time. AI will not even see assosciated action,
	 *  within this period of time.
	 */
	function DisableAction(time)
	{
		this.next_run_date = AIDate.GetCurrentDate() + time;
		_ai_clock.AddTask(this);
		this.manage_action.SelfRemove();
	}

	function Run(current_date)
	{
		Corporation.Get().AddAction(this.manage_action);
	}

/* private */
	/** Action to handle */
	manage_action = null;
}
