/**
 * Calculate the action's priority.
 * @param startup_cost Initial action cost
 *  (action can have several steps, this is cost of first).
 * @param full_cost Full action cost. Negative number mean "selling".
 * @param income Direct month income AI will begin to receive right after action.
 * @param future_income Possible month income AI can receive in future
 *  (if this action will be completed).
 * @return Number that indicates action's desireability for AI.
 */
/* static */ function ActionPriority::CalculatePriority(startup_cost, full_cost, income, future_income)
{
	/* Set max priority for "sell something" actions */
	if (full_cost < 0) return APriority.MAX;

	/*
	 * Set priority above average for the actions which make money,
	 * (or triggers money making mechanism).
	 * Example: Action that buy new truck.
	 */
	if (income > 0) {
		return full_cost != 0 ? (APriority.HALF + 10.0 * income / full_cost) : APriority.MAX; 
	}

	if (future_income <= 0) return APriority.BAD;

	local time_limit = 60; // 5 years
	local money = AICompany.GetBankBalance(AICompany.COMPANY_SELF) +
		AICompany.GetMaxLoanAmount() - AICompany.GetLoanAmount();
	local company_income = 2 + abs(CorporationUtils.month_income.value);
	local future_cost = 1 + full_cost - startup_cost;

	if (money >= 0.95 * full_cost) {
		local profit = time_limit * future_income - full_cost;
		if (profit <= 0) return APriority.BAD;
		local x = 10.0 * startup_cost / profit;
		local p = APriority.HALF - x;
		return p > APriority.BAD ? p : APriority.BAD;
	}
 
 	money += company_income / 2;
	local good_money = full_cost - 4 * future_cost / 5;
	if (money < good_money) money = good_money;

	local startup_income = ((money - startup_cost).tofloat() / future_cost) * future_income;
	local income_to_gain = future_income - startup_income;
	local di = 1 + income_to_gain.tofloat() / (startup_income + company_income);
	// 't' is time in months required to complete action/project,
	//   before 't' it will be giving income lesser than 'future_income' var
	local t = log(di) * future_cost / future_income;

	if (t >= time_limit) return APriority.BAD;

	local pt = time_limit.tofloat() - t;
	local profit = pt * future_income - money;
	if (profit <= 0) return APriority.BAD;
	local x = 10.0 * startup_cost / profit;
	// x represent income from each $ "buried into ground"(e.g.
	//  into roads, airports, something that not bring money by itself like
	//  ships or planes), and will be more accurate then similar value in the
	// "Simle" function.

	local p = APriority.HALF - x;
	return p > APriority.BAD ? p : APriority.BAD;
}
