/**
 * Wrapper for squirrel table with built-in save/load support.
 */
TableContainer <- {
/* public */
	/**
	 * Get class name.
	 * @return unique class name.
	 */
	/* static */ function GetClassName()
	{
		return "TableContainer";
	}

	/**
	 * Constructor for the TableContainer.
	 * @param name container name. Will be used as container id.
	 */
	/* static */ function new(name)
	{
		return TableContainer.new_custom(name, TableContainer.ItemContext);
	}

	/**
	 * Constructor for the TableContainer.
	 * @param name container name. Will be used as container id.
	 */
	/* static */ function new_custom(name, context)
	{
		local new_container = clone ::TableContainer;
		new_container.table_name = name;
		new_container.context = context;
		return delegate new_container : {}; // returns "{}"
	}

	/**
	 * Says that type of TableContainer is "instance"(even if it is not).
	 * @note For save/load, again.
	 */
	function _typeof()
	{
		return "instance";
	}

	/**
	 * Same as AddItem.
	 * Used for save/load compatibility with array
	 * (to handle loaded items insertion in one manner).
	 */
	function append(item)
	{
		return this[this.context.ResolveObjectID(item)] <- item;
	}

	/**
	 * Get ID;
	 * @return container id.
	 */
	function GetID()
	{
		return this.table_name;
	}

	/**
	 * Get object name.
	 * @return short string - name of the object.
	 */
	function GetName()
	{
		return this.table_name;
	}

	/**
	 * Add given object into container.
	 * @param object Object to add. Must have GetID() method.
	 */
	function AddItem(object)
	{
		return this[this.context.ResolveObjectID(object)] <- object;
	}

	/**
	 * Get item with given id from the container.
	 * @param item_id Id of item to retrieve.
	 * @return Container item with desired id, or null if no such item.
	 */
	function GetItem(item_id)
	{
		return item_id in this ? this[item_id] : null;
	}

	/**
	 * Check if container already has item with given id.
	 * @param item_id Id to check.
	 * @return True is item present, false else.
	 */
	function HasItem(item_id)
	{
		return item_id in this;
	}

	/**
	 * Remove item with given id(if any) from the container.
	 * @param item_id Id of item to remove.
	 */
	function RemoveItem(item_id)
	{
		return item_id in this ? this.rawdelete(item_id) : null;
	}

	/**
	 * Get best item from container.
	 * What is "best" defined by this container item's context.
	 *  Greater result => better item.
	 * @return The table with the best item, and it's rank.
	 */
	function GetBestItem()
	{
		return this.context.GetBestItem(this);
	}

	/**
	 * Get memento.
	 * @return repository that hold restoration data for this table.
	 */
	function GetMemento()
	{
		local memento = {table_name = this.table_name, items = array(this.len())};
		local i = 0;
		foreach (dummy_id, item in this) {
			memento.items[i] = item;
			i++;
		}
		return memento;
	}

	/**
	 * Create new object, restored from memento.
	 * @param memento result of this.GetMemento().
	 * @return restored object.
	 */
	function Restore(memento)
	{
		local restored = TableContainer.new(memento.table_name);
		foreach (dummy_id, item in memento.items) {
			// will cause errors if item is raw sq type(e.g. integer)
			// to correctly save such stuff must use CodeUtils.TableToArray
			restored.AddItem(item);
		}
		return restored;
	}

	/**
	 * Items description.
	 */
	context = null

	/** Table name */
	table_name = "Squirrel table wrapper"
};

/**
 * Base class for custom item types injections into TableContainer.
 */
class TableContainer.ItemContext extends Terron_Object
{
/* public */
	constructor()
	{
		::Terron_Object.constructor(null);
	}

	/**
	 * This function must return unique(inside container) ID of the given object
	 * @param object Some object.
	 * @return Unique ID of the given object.
	 */
	function ResolveObjectID(object)
	{
		return object.GetID();
	}

	/**
	 * Get the best item from container.
	 * What is "best" defined by this.GetObjectRank function.
	 * @param container Container for search in.
	 * @return Table with the best item, and it's rank.
	 */
	function GetBestItem(container)
	{
		local result = {};
		result.best_item <- null;
		result.best_item_rank <- -0x0FFFFFFF;

		foreach (id, item in container) {
			local rank = this.GetObjectRank(item);
			if (rank > result.best_item_rank) {
				result.best_item_rank = rank;
				result.best_item = item;
			}
		}
		return result;
	}

/* protected */
	/**
	 * This function must rate given object.
	 * @param object Some object.
	 * @return Number, representing "rating", of the given object.
	 */
	function GetObjectRank(object);
}

Terron_ClassTable.RegisterClass(TableContainer);
