/**
 * Fibonacci heap.
 *  This heap is heavily optimized for the Insert and Pop functions.
 *  Peek and Pop always return the current lowest value in the list.
 *  Insert is implemented as a lazy insert, as it will simply add the new
 *  node to the root list. Sort is done on every Pop operation.
 * My fibonacci heap. A bit faster than the library(v 2) one.
 */
class Terron_FibonacciHeap {
/* public */
	/**
	 * Create a new fibonacci heap.
	 * http://en.wikipedia.org/wiki/Fibonacci_heap
	 */
	constructor() {
		this.root_list = [];
	}

	/**
	 * Insert a new entry in the heap.
	 *  The complexity of this operation is O(1).
	 * @param object The object to add to the list.
	 * @param priority The priority this object has.
	 */
	function Insert(object, priority)
	{
		count++;
		local node = clone ::Terron_FibonacciHeap.MyNode;
		node.item = object;
		node.childs = [];
		node.priority = priority;

		local root = this.root_list;
		root.append(node);
		if (priority >= this.min_priority) return;
		this.min_priority = priority;
		this.min_item = object;
		this.min_node_index = root.len() - 1;
	}

	/**
	 * Check wheteher the heap is empty.
	 * @return True if and only if the heap has no items.
	 */
	function IsEmpty()
	{
		return this.min_node_index < 0;
	}

	/**
	 * Pop the first entry of the list.
	 *  This is always the item with the lowest priority.
	 *  The complexity of this operation is O(ln n).
	 * @return The item of the entry with the lowest priority.
	 */
	function Pop()
	{
		if (this.min_node_index < 0) return null;
		count--;

		/*
		 * Bring variables from the class scope to this scope explicitly to
		 * optimize variable lookups by Squirrel.
		 */
		local root = this.root_list;
		local z = root[this.min_node_index];

		/* Remove the minimum node from the rootList. */
		root.remove(this.min_node_index);
		/* If there are any childrens, bring them all to the root level. */
		root.extend(z.childs);

		local root_cache = {};

		/*
		 * Now we decrease the number of nodes on the root level by
		 * merging nodes which have the same degree. The node with
		 * the lowest priority value will become the parent.
		 */
		foreach (x in root) {
			/* See if we encountered a node with the same degree already. */
			while (x.degree in root_cache) {
				local y = delete root_cache[x.degree];

				/* Check the priorities. */
				if (y.priority < x.priority) {
					local tmp = x;
					x = y;
					y = tmp;
				}

				/* Make y a child of x. */
				x.childs.append(y);
				x.degree++;
			}

			root_cache[x.degree] <- x;
		}

		/*
		 * The root_cache contains all the nodes which will form the
		 *  new rootList. We reset the priority to the maximum number
		 *  to find the new minumum.
		 */
		root.resize(root_cache.len());

		/* Now we need to find the new minimum among the root nodes. */
		local i = 0, min_index = -1, p = 0x6FFFFFFF;
		foreach (node in root_cache) {
			if (node.priority < p) {
				p = node.priority;
				min_index = i;
			}

			root[i++] = node;
		}

		this.min_node_index = min_index;
		this.min_priority = p;
		this.min_item = min_index == -1 ? null : root[min_index].item;

		return z.item;
	}

	/**
	 * Get current number of items in this container.
	 */
	function Count()
	{
		return count;
	}

	/**
	 * Peek the first entry of the list.
	 *  This is always the item with the lowest priority.
	 *  The complexity of this operation is O(1).
	 * @return The item of the entry with the lowest priority.
	 */
	function Peek()
	{
		return this.min_item;
	}

	/**
	 * Peek priority of the first entry of the list.
	 * @return The lowest priority among all items(if any).
	 */
	function PeekPriority()
	{
		if (this.min_node_index < 0) return -0x6FFFFFFF;
		return this.root_list[this.min_node_index].priority;
	}

/* private */
	/** Current min item */
	min_item = null;

	/** Current min item's index */
	min_node_index = -1;

	/** Current min priority index */
	min_priority = 0x6FFFFFFF;

	/** private */
	root_list = null;

	/** Current number of items in this container. */
	count = 0;
}

/**
 * Basic class for nodes the fibonacci heap is composed of.
 */
Terron_FibonacciHeap.MyNode <- {
	/** item to store */
	item = null,

	/** item's "degree" */
	degree = 0,

	/** item's childs */
	childs = null,

	/** item's priority */
	priority = null,
}
