/**
 * 2D Grid data structure rotated by 45 degrees.
 */
class Terron_ObliqueGrid
{
/* public */
	/**
	 * Creates Terron_ObliqueGrid object which can keep data about the
	 *  given square area(SQA).
	 * @param cell_radius Desired radus of the grid cell.
	 * @param map_region_size Side length the SQA to keep in grid.
	 * @param x_shift Minimal possible x coordinate for tiles inside SQA.
	 * @param y_shift Minimal possible y coordinate for tiles inside SQA.
	 */
	constructor(cell_radius, map_region_size, x_shift, y_shift)
	{
		this.x_shift = x_shift;
		this.y_shift = y_shift;
		this.grid_size = map_region_size;
		this.cell_radius = cell_radius;
		local d = 2 * cell_radius;
		this.n = map_region_size / cell_radius;

		/*
		 * We'll keep grid as 2d array which basically looks like this:
		 * x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x
		 * x       x       x       x     / x \     x       x       x       x
		 * x       x       x       x [0,7] x [1,7] x       x       x       x
		 * x       x       x       x /     x     \ x       x       x       x
		 * x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x
		 * x       x       x     / x       x       x \     x       x       x
		 * x       x       x [0,6] x [1,6] x [2,6] x [3,6] x [4,6] x       x
		 * x       x       x /     x       x       x     \ x       x       x
		 * x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x
		 * x       x     / x       x       x       x       x \     x       x
		 * x       x [0,5] x [1,5] x [2,5] x [3,5] x [4,5] x [5,5] x       x
		 * x       x /     x       x       x       x       x     \ x       x
		 * x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x
		 * x     / x       x       x       x       x       x       x \     x
		 * x [0,4] x [1,4] x [2,4] x [3,4] x [4,4] x [5,4] x [6,4] x [7,4] x
		 * x /     x       x       x       x       x       x       x     \ x
		 * x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x
		 * x \     x       x       x       x       x       x       x     / x
		 * x [0,3] x [1,3] x [2,3] x [3,3] x [4,3] x [5,3] x [6,3] x [7,3] x
		 * x     \ x       x       x       x       x       x       x /     x
		 * x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x
		 * x       x \     x       x       x       x       x     / x       x
		 * x       x [0,2] x [1,2] x [2,2] x [3,2] x [4,2] x [5,2] x       x
		 * x       x     \ x       x       x       x       x /     x       x
		 * x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x
		 * x       x       x \     x       x       x     / x       x       x
		 * x       x       x [0,1] x [1,1] x [2,1] x [3,1] x [4,1] x       x
		 * x       x       x     \ x       x       x /     x       x       x
		 * x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x
		 * x       x       x       x \     x     / x       x       x       x
		 * x       x       x       x [0,0] x [1,0] x       x       x       x
		 * x       x       x       x     \ x /     x       x       x       x
		 * x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x
		 * It's important: game map there^^ is oblique square and
		 *  we do not store big square, but only its intersection with map.
		 * In usual perspective picture above looks like this:
		 *
		 *  - - - - - - - - - - - - - - - -x- - - - - - - - - - - - - - - - 
		 * |                             /   \                             |
		 *                             /       \                           
		 *                           /           \                          
		 *                         /               \                       
		 * |                     /                   \                     |
		 *                     x       x       x       x                   
		 *                   /   \   /   \   /   \   /   \                  
		 *                 x--0,4--x--0,5--x--0,6--x--0,7--x               
		 * |             / | \   / | \   / | \   / | \   / | \             |
		 *             x  0,3  x  1,4  x  1,5  x  1,6  x       \           
		 *           /   \ | /   \ | /   \ | /   \ | /     |     \          
		 *         /       x--1,3--x--2,4--x--2,5--x--- ---        \       
		 * |     /       / | \   / | \   / | \   / |       |         \     |
		 *     /       x  0,2  x  2,3  x  3,4  x                       \   
		 *   /           \ | /   \ | /   \ | /     |       |             \  
		 *  x               --1,2-- --3,3-- --- --- --- ---               x 
		 *   \           / | \   / | \   / |       |       |             / 
		 *     \       x  0,1  x  2,2  x                               /   
		 * |     \       \ | /   \ | /     |       |       |         /     |
		 *         \       x--1,1-- --- --- --- --- --- ---        /       
		 *           \   / | \   / |       |       |       |     /         
		 *             x  0,0  x                               /           
		 * |             \ | /     |       |       |       | /             |
		 *                 \--- --- --- --- --- --- --- ---/               
		 *                   \      ^^ GAME MAP ^^       /                 
		 *                     \   WITH HALF OF THE    /                   
		 * |                     \ GRID AND INDEXES  /                     |
		 *                         \               /                       
		 *                           \           /                         
		 *                             \       /                           
		 * |                             \   /                             |
		 *  --- --- --- --- --- --- --- ---x--- --- --- --- --- --- --- ---
		 *
		 * Such approach requires minor magic with indexes calculation but
		 *  because we are using DistanceManhattan and because
		 *  L1 balls are essentially rectangles in such grid =>
		 *  we do not receive excessive cells during GET_ALL_OBJECTS_NEAR_POINT
		 *  operation, which makes it very effective.
		 *
		 * Actually i've made some tests and it seems that
		 *  effectiveness(compared to regular grid) is not all that great.
		 * Time to separate excessive cells in regular grid is compensated
		 *  by time to calculate indexes. Of course it depends on required
		 *  ball radius which makes this grid preferable, but still... 
		 */
		this.grid = array(this.n);
		for (local y = 0; y < n / 2; y++) {
			this.grid[y] = array(2 * (y + 1));
			this.grid[this.n - y - 1] = array(2 * (y + 1));
			for (local x = 0; x < 2 * (y + 1); x++) {
				this.grid[y][x] = {};
				this.grid[this.n - y - 1][x] = {};
			}
		}
	}

	function InsertObjectFromTile(object_id, object, object_location)
	{
		local tx = AIMap.GetTileX(object_location) - this.x_shift;
		local ty = AIMap.GetTileY(object_location) - this.y_shift;

		if (min(tx, ty) < 0 || max(tx, ty) >= this.grid_size) return;

		local cell = this.TransformXY(tx, ty);
		local cell_y = (cell.x + cell.y) / 2;
		local cell_x = (cell.y - cell.x) / 2 + this.cell_radius;
		if (cell_y < n / 2) {
			cell_x = 1 + cell_y - n / 2 + cell_x;
		} else {
			cell_x = cell_x - cell_y + n / 2;
		}
		/*local m = this.n / 2 - cell_y;
		cell_x += m > 0 ? 1 - m : m;*/

		this.grid[cell_y][cell_x][object_id] <- object;
	}

	function RemoveObject(object_id, object_location)
	{
		local tx = AIMap.GetTileX(object_location) - this.x_shift;
		local ty = AIMap.GetTileY(object_location) - this.y_shift;

		if (min(tx, ty) < 0 || max(tx, ty) >= this.grid_size) return;

		local cell = this.TransformXY(tx, ty);
		local cell_y = (cell.x + cell.y) / 2;
		local cell_x = (cell.y - cell.x) / 2 + this.cell_radius;
		if (cell_y < n / 2) {
			cell_x = 1 + cell_y - n / 2 + cell_x;
		} else {
			cell_x = cell_x - cell_y + n / 2;
		}

		if (object_id in this.grid[cell_y][cell_x]) {
			local object = this.grid[cell_y][cell_x][object_id];
			delete this.grid[cell_y][cell_x][object_id];
		}
	}

	function GetCellObjectsFromTile(t)
	{
		local tx = AIMap.GetTileX(t) - this.x_shift;
		local ty = AIMap.GetTileY(t) - this.y_shift;

		if (min(tx, ty) < 0 || max(tx, ty) >= this.grid_size) return [];

		local cell = this.TransformXY(tx, ty);
		local cell_y = (cell.x + cell.y) / 2;
		local cell_x = (cell.y - cell.x) / 2 + this.cell_radius;
		if (cell_y < n / 2) {
			cell_x = 1 + cell_y - n / 2 + cell_x;
		} else {
			cell_x = cell_x - cell_y + n / 2;
		}

		return this.grid[cell_y][cell_x];
	}

	function PrintCell(b, a)
	{
		local x0 = (b) * this.cell_radius;
		local y0 = (a) * this.cell_radius;
		AILog.Info("x0 = " + x0);
		AILog.Info("y0 = " + y0);
		local t = AIMap.GetTileIndex(x0, y0);
		for (local i = 0; i < cell_radius; i++) {
			//AILog.Info("x = " + AIMap.GetTileX(t + AIMap.GetTileIndex(1, 0) * i));
			AISign.BuildSign(t + AIMap.GetTileIndex(1, 1) * i, "+");
			AISign.BuildSign(t + AIMap.GetTileIndex(-1, 1) * i, "+");
		}
		for (local i = 0; i < cell_radius; i++) {
			AISign.BuildSign(t + AIMap.GetTileIndex(0, 2) * cell_radius + AIMap.GetTileIndex(1, -1) * i, "-");
			AISign.BuildSign(t + AIMap.GetTileIndex(0, 2) * cell_radius + AIMap.GetTileIndex(-1, -1) * i, "-");
		}
		local sy = (b + a) / 2;
		local sx = (a - b) / 2 + this.cell_radius;
		if (sy < n / 2) {
			sx = 1 + sy - n / 2 + sx;
		} else {
			sx = sx - sy + n / 2;
		}
		local p = 0;
		foreach (object_id, object in this.grid[sy][sx]) {
			AISign.BuildSign(AIMap.GetTileIndex(x0 + 2 + p, y0 + 8 + p), object);
			p++;
		}	
	}

	function GetCellsNearTile(t, r)
	{
		local tx = AIMap.GetTileX(t) - this.x_shift;
		local ty = AIMap.GetTileY(t) - this.y_shift;

		local top_cell = this.TransformXY(tx, ty + r);
		local top_y = min((top_cell.x + top_cell.y) / 2, this.n - 1);
		local bottom_cell = this.TransformXY(tx, ty - r);
		local bottom_y = max((bottom_cell.x + bottom_cell.y) / 2, 0);

		local result = [];
		if (top_y < 0 || bottom_y > this.n) return result;

		local top_x = (top_cell.y - top_cell.x) / 2 + this.cell_radius;
		top_x = this.MakeSafeCoordinate(top_x);
		if (top_y >= this.n) top_y = this.n - 1;

		local bottom_x = (bottom_cell.y - bottom_cell.x) / 2 + cell_radius;
		bottom_x = this.MakeSafeCoordinate(bottom_x);
		if (bottom_y < 0) bottom_y = 0;

		for (local y = bottom_y; y <= min(this.n / 2 - 1, top_y); y++) {
			local l = max(0, 1 + bottom_x + y - n / 2);
			local r = min (this.grid[y].len() - 1, 1 + top_x + y - n / 2);
			for (local x = l; x <= r; x++) {
				result.append(this.grid[y][x]);
			}
		}

		for (local y = max(this.n / 2, bottom_y); y <= top_y; y++) {
			local l = max(0, bottom_x - y + n / 2);
			local r = min(this.grid[y].len() - 1, top_x - y + n / 2);
			for (local x = l; x <= r; x++) {
				result.append(this.grid[y][x]);
			}
		}

		return result;
	}

/* private */
	/**
	 * Length of array with grid cells.
	 */
	n = 0;

	/**
	 * 2D Array that contains grid cells.
	 */
	grid = null;

	/**
	 * Length of side of the map square which must be represented by the grid. 
	 */
	grid_size = 0;

	/**
	 * Radius(in tiles) of a single grid cell.
	 */
	cell_radius = null;

	/**
	 * Rounds given cell ID to closest value inside grid specific limits.
	 * @param value Cell ID in question.
	 * @return Given value if it is in allowed range, or 0 if it is negative,
	 *  or cells array length - 1 if it is greater than cells array length - 1.
	 */
	function MakeSafeCoordinate(value)
	{
		if (value < 0) return 0;
		if (value >= this.n) return this.n - 1;
		return value;
	}

	/**
	 * Helper to find position of the grid cell which contain given tile.
	 * @param x The tile x coordinate.
	 * @param y The tile y coordinate.
	 * @return Relative coordinates of the rhombus containing given tile.
	 * @note Returned values still need to be modified to find actual cell's
	 *  position.
	 */
	function TransformXY(x, y)
	{
		local r = this.cell_radius;
		local d = 2 * r;
		local dx = x % d;
		local dy = y % d;
		local a = y - dy;
		local b = x - dx;

		/*
		 * At picture:
		 * Square: cell of the regular grid containing given tile(would we
		 *  use such grid).
		 * 4 Triangles(marked q1, ..., q4):
		 *  Those are not triangles but halves of the rhombuses.
		 *  Such rhombus is square rotated by 45 degrees and is our grid cell.
		 *  Thus given tile can fall into one of those cells.
		 *  We'll return bottom corner of the rhombus containing tile(which
		 *   is position of the grid cell in 'regular'(*) map coordinates)
		 *
		 * (*) Therefore there is no transformation and function name is a bit
		 *  inaccurate. To receive proper x and y(or i and j whatever) values
		 *  to adress cell inside our 2D array two more operations must be done:
		 *   a) actual transformation - 45 degree rotation.
		 *   b) 2D array indexes rescale - because grid is packed array.
		 * Such operations are trivial (and temporary result of this function
		 *  has a value by itself) so i decided not to write special wrapper
		 *  for them. 
		 *
		 *                                  * ___ line [y = x];
		 *  d* * * * * * * * * * * * * * * *
		 *   * *                         * *
		 *   *   *         q3          *   *
		 *   *     *                 *     *
		 *   *       *             *       *
		 *   *         *         *         *
		 *   *           *     *           *
		 *   *    q4       * *       q2    *
		 *   *             * *             *
		 *   *           *     *           *
		 *   *         *         *         *
		 *   *       *             *       *
		 *   *     *       q1        *     *
		 *   *   *                     *   *
		 *   * *                         * *
		 *  0* * * * * * * * * * * * * * * *
		 *   0              r              d *
		 *                                    * ___ line [y = d - x];
		 */
		local q23 = (dy >= d - dx);
		local q34 = (dy >= dx);
		local q12 = !q34;
		local q41 = !q23;

		if (q23) {
			if (q34) {
				//q3
				a += r;
				b += r;
			} else /* if (q12) */ {
				//q2
				b += d;
			}
		} else /* if (q41) */ {
			if (q12) {
				//q1
				a -= r;
				b += r;
			} else /* if (q34) */ {
				//q4
			} 
		}

		return {x = b / this.cell_radius, y = a / this.cell_radius};
	}

/* public */
	/** X shift from the map [X == 0] edge. */
	x_shift = 0;

	/** Y shift from the map [Y == 0] edge. */
	y_shift = 0;
}
