/**
 * Region quadtree data structure.<p>
 */
/*
 * It's current implementation is kinda complex,
 *  but this version is 3x - 4x times faster then naive recursion realisation I wrote first.
 */
class QuadTree
{
/* public */
	/**
	 * Pyramid with tree nodes saved as arrays.<p>
	 * Level 0 contains one node - full region square,<p>
	 * Level 1 already four,<p>
	 * Level 2 has 16 items, <p>
	 * ..e.t.c until last level's leaves will be reached.
	 * @note most of items will be null, though.
	 */
	full_tree = null;

	constructor(min_resolution, max_depth, data_read_callback, precision_black, precision_white)
	{
		this.max_depth = max_depth;
		this.min_resolution = min_resolution;
		this.full_tree = array(max_depth + 1, null);

		local min_size = 1 << min_resolution;
		this.min_cell_size = min_size;
		local max_size = min_size << max_depth;
		local tt = AIController.GetTick();
		CodeUtils.Log("Creating a " + max_size + "x" + max_size + " quadtree...", 1);
 
		for (local h = max_depth, n = 1 << h; h >= 0; h--, n = n >> 1) {
			this.full_tree[h] = array(n, null);
			for (local i = 0; i < n; i++) {
				this.full_tree[h][i] = array(n, null);
			}
		}

		local max = QuadTree.max_value;
		this.precision_black = precision_black;
		this.precision_white = precision_white;
		this.real_black_max = (max * precision_black).tointeger();
		this.real_white_min = (max * (1 - precision_white)).tointeger();

		local big = MakeNode(0, 0, max_depth, max_size, data_read_callback);
		if (big != -1){
			 this.full_tree[0][0][0] = QTNode(0, 0, max_size, big);
		}

		for (local level = 0, len = 1; level < max_depth; len *= 2, level++) {
			local last_level_len = 1 << (max_depth - level);
			for (local i = 0; i < len; i++)
			for (local j = 0; j < len; j++) {
				local qt_node = this.full_tree[level][i][j];
				if (qt_node == null) continue;

				/*
				 * For each "big" square go to last level - it smallest parts,
				 *  and make them point to this big square.
				 * It grants that when we'll try to access max depth element =>
				 *  we'll be accessing true leaves. And due to qt making func
				 *  qt leaves are those not null elements in the '.full_tree'
				 *  Hence qt_node != null.
				 * Such approach allows us to work with qt like it is grid, and
				 *  access squad containing given point in O(1) time.
				 * Extra memory usage and code simplicity is price though.
				 */
				local x0 = i * (1 << (max_depth - level));
				local y0 = j * (1 << (max_depth - level));
				for (local x = x0; x < x0 + last_level_len; x++)
					for (local y = y0; y < y0 + last_level_len; y++) {
					this.full_tree[max_depth][x][y] = qt_node.weakref();
				}
			}
		}

		CodeUtils.Log("... complete, ticks " + (AIController.GetTick() - tt), 1);
	}

	function GetLength()
	{
		return this.min_cell_size << this.max_depth;
	}

	function GetAdjacent(qt_node)
	{
		local max_id = 1 << this.max_depth;
		local min_size = this.min_cell_size;

		local left   = qt_node.x_min / min_size;
		local bottom = qt_node.y_min / min_size;
		local right  = left + qt_node.edge_len / min_size;
		local top    = bottom + qt_node.edge_len / min_size;

		local qt = this.full_tree[this.max_depth];
		local result = [];
		if (left >= 1) {
			for (local y = max(bottom, 0); y < min(top, max_id);) {
				local qt_node = qt[left - 1][y];
				result.append(qt_node);
				y = (qt_node.y_min + qt_node.edge_len) / min_size;
			}
		}

		if (right < max_id) {
			for (local y = max(bottom, 0); y < min(top, max_id);) {
				local qt_node = qt[right][y];
				result.append(qt_node);
				y = (qt_node.y_min + qt_node.edge_len) / min_size;
			}
		}

		if (bottom >= 1) {
			for (local x = max(left, 0); x < min(right, max_id);) {
				local qt_node = qt[x][bottom - 1];
				result.append(qt_node);
				x = (qt_node.x_min + qt_node.edge_len) / min_size;
			}
		}

		if (top < max_id) {
			for (local x = max(left, 0); x < min(right, max_id);) {
				local qt_node = qt[x][top];
				result.append(qt_node);
				x = (qt_node.x_min + qt_node.edge_len) / min_size;
			}
		}

		return result;
	}

	function GetDiagAdjacent(qt_node)
	{
		local max_id = 1 << this.max_depth;
		local min_size = this.min_cell_size;

		local left   = qt_node.x_min / min_size - 1;
		local bottom = qt_node.y_min / min_size - 1;
		local right  = left + qt_node.edge_len / min_size + 1;
		local top    = bottom + qt_node.edge_len / min_size + 1;

		local qt = this.full_tree[this.max_depth];
		local tmp = [];
		if (bottom >= 0) {
			if (left >= 0) tmp.append(qt[left][bottom]);
			if (right < max_id) tmp.append(qt[right][bottom]);
		}
		if (top < max_id) {
			if (left >= 0) tmp.append(qt[left][top]);
			if (right < max_id) tmp.append(qt[right][top]);
		}

		local result = [];
		local points = {};
		foreach (dummy_id, qt_node in tmp) {
			local x = qt_node.x_min;
			if (x in points && qt_node.y_min in points[x]) continue;
			if (!(x in points)) points[x] <- {};
			points[x][qt_node.y_min] <- 1;
			result.append(qt_node);
		}
		return result;
	}

	function GetIntersection(x0, y0, x1, y1, v)
	{
		local min_size = this.min_cell_size;
		local meaningfullbits_mask = ~(min_size - 1);
		local left, right, top, bottom;

		x0 = x0 < this.x_shift ? 0 : x0 - this.x_shift;
		y0 = y0 < this.y_shift ? 0 : y0 - this.y_shift;
		x1 = x1 < this.x_shift ? 0 : x1 - this.x_shift;
		y1 = y1 < this.y_shift ? 0 : y1 - this.y_shift;

		if (x0 < x1) {
			left = (x0 & meaningfullbits_mask) / min_size;
			right = 1 + (x1 & meaningfullbits_mask) / min_size;
		} else {
			left = (x1 & meaningfullbits_mask) / min_size;
			right = 1 + (x0 & meaningfullbits_mask) / min_size;
		}
		if (y0 < y1) {
			bottom = (y0 & meaningfullbits_mask) / min_size;
			top = 1 + (y1 & meaningfullbits_mask) / min_size;
		} else {
			bottom = (y1 & meaningfullbits_mask) / min_size;
			top = 1 + (y0 & meaningfullbits_mask) / min_size;
		}

		local max_id = 1 << this.max_depth;
		if (left > max_id || bottom > max_id) return [];
		if (right > max_id) right = max_id;
		if (top > max_id) top = max_id;

		/* Following 4 sections just align search rectangle */
		/* It'll have even length then(measured in the qt blocks of min size) */
		local qt = this.full_tree[this.max_depth];
		local result = [];
		if (left % 2 == 1) {
			local new_left = (left + 1) * min_cell_size;
			local thin_rect = (left == right - 1);
			for (local y = bottom; y < top;) {
				local qt_node = this.full_tree[this.max_depth][left][y];
				if (qt_node.edge_len == min_size) {
					if (qt_node.value > v) result.append(qt_node);
					y++;
				} else {
					if (qt_node.x_min + qt_node.edge_len == new_left || thin_rect) {
						if (qt_node.value > v) result.append(qt_node);
					}
					y = (qt_node.y_min + qt_node.edge_len) / min_size;
				}
			}
			left = left + 1;
		}

		if (bottom % 2 == 1) {
			local new_bottom = (bottom + 1) * min_cell_size;
			local thin_rect = (bottom == top - 1);
			for (local x = left; x < right;) {
				local qt_node = this.full_tree[this.max_depth][x][bottom];
				if (qt_node.edge_len == min_size) {
					if (qt_node.value > v) result.append(qt_node);
					x++;
				} else {
					if (qt_node.y_min + qt_node.edge_len == new_bottom || thin_rect) {
						if (qt_node.value > v) result.append(qt_node);
					}
					x = (qt_node.x_min + qt_node.edge_len) / min_size;
				}
			}
			bottom = bottom + 1;
		}

		if (right % 2 == 1) {
			local new_right = (right - 1) * min_cell_size;
			local thin_rect = (left == right - 1);
			for (local y = bottom; y < top;) {
				local qt_node = this.full_tree[this.max_depth][right - 1][y];
				if (qt_node.edge_len == min_size) {
					if (qt_node.value > v) result.append(qt_node);
					y++;
				} else {
					if (qt_node.x_min == new_right || thin_rect) {
						if (qt_node.value > v) result.append(qt_node);
					}
					y = (qt_node.y_min + qt_node.edge_len) / min_size;
				}
			}
			right = right - 1;
		}

		if (top % 2 == 1) {
			local new_top = (top - 1) * min_cell_size;
			local thin_rect = (bottom == top - 1);
			for (local x = left; x < right;) {
				local qt_node = this.full_tree[this.max_depth][x][top - 1];
				if (qt_node.edge_len == min_size) {
					if (qt_node.value > v) result.append(qt_node);
					x++;
				} else {
					if (qt_node.y_min == new_top || thin_rect) {
						if (qt_node.value > v) result.append(qt_node);
					}
					x = (qt_node.x_min + qt_node.edge_len) / min_size;
				}
			}
			top = top - 1;
		}

		/* Here begins line-by-line rectangle scan */
		// actually it checks 2 lines together, but this is just
		//  minor optimization and does not change the idea of the method.
		// Just draw the result and it'll show how it works
		//  (because result array saves order in which squares are added).
		// should be O(m), where m is number of squares in the final result.
		result.extend(this.ParseRect2(left, bottom, right, top, v));
		return result;
	}

/* private */
	/*
	 * This function grants zero dublicate nodes creation, mean
	 *  it'll read each map tile/region only once.
	 */
	function MakeNode(x, y, level, side_len, data_read_callback)
	{
		if (level == 0) {
			return data_read_callback(x * side_len, y * side_len, side_len);
		}

		x = x * 2;
		y = y * 2;
		side_len = side_len >> 1;
		level = level - 1;

		/*
		 * As current node/map looks from player(in game) view.
		 * Divide into 4 subnodes, and repeat process for each.
		 * ***************
		 * *      *      *
		 * *  01  *  00  *
		 * *      *      *
		 * ***************
		 * *      *      *
		 * *  11  *  10  *
		 * *      *      *
		 * ***************
		 */
		local zero_zero = MakeNode(x, y, level, side_len, data_read_callback);
		local zero_one = MakeNode(x + 1, y, level, side_len, data_read_callback);
		local one_zero = MakeNode(x, y + 1, level, side_len, data_read_callback);
		local one_one = MakeNode(x + 1, y + 1, level, side_len, data_read_callback);

		/* Unite childs when their values are close to each other. */
		if (zero_zero != -1 && zero_one != -1 && one_zero != -1 && one_one != -1) {
			local max = QuadTree.max_value;
			local avg_value = (zero_zero + zero_one + one_zero + one_one) / 4;

			local allowed_max = (max * this.precision_black).tointeger();
			local can_unite_black = true;
			can_unite_black = can_unite_black && (zero_zero >= allowed_max);
			can_unite_black = can_unite_black && (zero_one >= allowed_max);
			can_unite_black = can_unite_black && (one_zero >= allowed_max);
			can_unite_black = can_unite_black && (one_one >= allowed_max);
			if (can_unite_black) return avg_value;

			local white_err = 1 - this.precision_white;
			local zzzz = (max * white_err).tointeger();
			local can_unite_white = true;
			can_unite_white = can_unite_white && (zero_zero <= zzzz);
			can_unite_white = can_unite_white && (zero_one <= zzzz);
			can_unite_white = can_unite_white && (one_zero <= zzzz);
			can_unite_white = can_unite_white && (one_one <= zzzz);
			if (can_unite_white) return avg_value;

			local tmp = [zero_zero, zero_one, one_zero, one_one];
			for (local i = 0; i <= 3; i++) {
				// left and right borders
				// when within, the values can be considered close enough
				local l = (tmp[i] * this.precision_black).tointeger();
				local r = (tmp[i] + (max - tmp[i]) * white_err).tointeger();
				for (local j = i + 1; j <= 3; j++) {
					if (tmp[j] > r || tmp[j] < l) {
						tmp[0] = -1;
						i = 4;
						break;
					}
				}
			}
			if (tmp[0] != -1) return avg_value;
		}

		if (zero_zero == zero_one && one_one == one_zero && zero_zero == one_one) {
			return zero_zero;
		}

		local xx = x * side_len;
		local yy = y * side_len;

		if (zero_zero != -1) {
			local qt_node = QTNode(xx, yy, side_len, zero_zero);
			this.full_tree[this.max_depth - level][x][y] = qt_node;
		}

		if (zero_one != -1) {
			local qt_node = QTNode(xx + side_len, yy, side_len, zero_one);
			this.full_tree[this.max_depth - level][x + 1][y] = qt_node;
		}

		if (one_zero != -1) {
			local qt_node = QTNode(xx, yy + side_len, side_len, one_zero);
			this.full_tree[this.max_depth - level][x][y + 1] = qt_node;
		}

		if (one_one != -1) {
			local qt_node = QTNode(xx + side_len, yy + side_len, side_len, one_one);
			this.full_tree[this.max_depth - level][x + 1][y + 1] = qt_node;
		}
		return -1;
	}

	function ParseCorner(left, bottom, right, top, start_x, end_y, h, v)
	{
		local result = [];
		local qt = this.full_tree[this.max_depth];

		for (local y = bottom; y < end_y; y = y + 2) {
			for (local x = start_x; x < right; x = x + 2) {
				local qt_node = this.full_tree[this.max_depth][x][y];
				if (qt_node.value > v) result.append(qt_node);

				if (qt_node.edge_len == this.min_cell_size) {
					if (qt[x + 1][y].value > v) result.append(qt[x + 1][y]);
					if (qt[x][y + 1].value > v) result.append(qt[x][y + 1]);
					if (qt[x + 1][y + 1].value > v) result.append(qt[x + 1][y + 1]);
				} else if (qt_node.edge_len > 2 * this.min_cell_size) {
					local qt_left = qt_node.x_min >> this.min_resolution;
					local cell_scale_size = qt_node.edge_len / this.min_cell_size;
					local qt_top = cell_scale_size + qt_node.y_min / this.min_cell_size;
					local qt_right = qt_left + cell_scale_size;
					local min_top = min(top, qt_top);
					local min_right = min(right, qt_right);
					if (cell_scale_size > h) {
						if (qt_left > start_x && end_y > (y + 2)) {
							result.extend(this.ParseRect2(start_x, y + 2, qt_left, end_y, v));
						}
						if (qt_left > left && min_top > end_y) {
							result.extend(this.ParseRect2(left, end_y, qt_left, min_top, v));
						}
						h = cell_scale_size;
						start_x = min_right;
						end_y = min_top;
						y = y - 2;
					} else if (cell_scale_size == h) {
						if (qt_left > start_x && min_top > (y + 2)) {
							result.extend(this.ParseRect2(start_x, y + 2, qt_left, min_top, v));
						}
						start_x = min_right;
						y = y - 2;
					} else {
						if (qt_left > start_x && min_top > (y + 2)) {
							result.extend(this.ParseRect2(start_x, y + 2, qt_left, min_top, v));
						}
						local zzz = qt_right % h;
						local aligned_right = zzz == 0 ? qt_right : h + qt_right - qt_right % h;
						result.extend(this.ParseCorner(start_x, y, min(right, aligned_right), end_y, qt_right, min_top, cell_scale_size, v));
						start_x = aligned_right;
						y = y - 2;
					}
					break;
				}
			}
		}

		result.extend(this.ParseRect2(left, end_y, right, top, v));
		return result;
	}

	function ParseRect2(left, bottom, right, top, v)
	{
		local result = [];
		local qt = this.full_tree[this.max_depth];

		for (local y = bottom; y < top; y = y + 2) {
			for (local x = left; x < right; x = x + 2) {
				local qt_node = qt[x][y];
				if (qt_node.value > v) result.append(qt_node);

				if (qt_node.edge_len == this.min_cell_size) {
					if (qt[x + 1][y].value > v) result.append(qt[x + 1][y]);
					if (qt[x][y + 1].value > v) result.append(qt[x][y + 1]);
					if (qt[x + 1][y + 1].value > v) result.append(qt[x + 1][y + 1]);
				} else if (qt_node.edge_len > 2 * this.min_cell_size) {
					local qt_left = qt_node.x_min / this.min_cell_size;
					local cell_scale_size = qt_node.edge_len / this.min_cell_size;
					local qt_top = cell_scale_size + qt_node.y_min / this.min_cell_size;
					local qt_right = qt_left + cell_scale_size;
					local min_top = min(top, qt_top);
					if (left < qt_left && min_top > (y + 2)) {
						result.extend(this.ParseRect2(left, y + 2, qt_left, min_top, v));
					}
					if (right > qt_right) {
						if (min_top < top) {
							result.extend(this.ParseCorner(left, y, right, top, qt_right, min_top, cell_scale_size, v));
							return result;
						} else {
							left = qt_right;
							x = left - 2;
						}
					} else {
						y = qt_top - 2;
						break;
					}
				}
			}
		}

		return result;
	}

	function SetShift(dx, dy)
	{
		this.x_shift = dx;
		this.y_shift = dy;
	}

/* private */
	static max_value = 100;

	max_depth = null;

	min_cell_size = null;

	min_resolution = null;

	precision_black = 1.0;

	precision_white = 1.0;

	real_black_max = 0.0;

	real_white_min = 0.0;

/* public */
	/**
	 * X shift of the stored region's bottom left corner
	 *  from the map's bottom left corner.
	 */
	x_shift = 0;

	/**
	 * Y shift of the stored region's bottom left corner
	 *  from the map's bottom left corner.
	 */
	y_shift = 0;
}

/**
 * Class that represent square in quadtree.
 */
class QTNode
{
/* public */
	/**
	 * Square's left coordinate.
	 * @note In game it looks "right" actually.
	 */
	x_min = null;

	/**
	 * Square's bottom coordinate.
	 * @note In game it looks "top" actually.
	 */
	y_min = null;

	/** Square's edge length. */
	edge_len = null;

	/** Square's color - integer in range from 0("white") to 100("black"). */
	value = null;

	/** Table to place user data. */
	custom_data = null;

	/**
	 * Creates a node to represent square inside quadtree.
	 * @param x_min The square's bottom left corner's X coordinate.
	 * @param y_min The square's bottom left corner's Y coordinate.
	 * @param edge_len The square's edge length.
	 * @param value Integer between 0 and 100.
	 */
	constructor(x_min, y_min, edge_len, value) {
		if (value < 0 || value > 100) assert(null);
		this.x_min = x_min;
		this.y_min = y_min;
		this.edge_len = edge_len;
		this.value = value;
		this.custom_data = {};
	}

	/**
	 * Draw self with signs. 
	 * @note For debug only.
	 */
	function PrintSelf(x_shift, y_shift)
	{
		local x_min = this.x_min + x_shift;
		local y_min = this.y_min + y_shift;
		local half = this.edge_len / 2;
		local center = AIMap.GetTileIndex(x_min + half, y_min + half);
		AISign.BuildSign(center, "value = " + this.value);

		for (local i = 0; i <= this.edge_len; i++) {
			AISign.BuildSign(AIMap.GetTileIndex(x_min, y_min + i), "x");
		}
		for (local x_max = x_min + this.edge_len, i = 0; i <= this.edge_len; i++) {
			AISign.BuildSign(AIMap.GetTileIndex(x_max, y_min + i), "x");
		}
		for (local i = 0; i <= this.edge_len; i++) {
			AISign.BuildSign(AIMap.GetTileIndex(x_min + i, y_min), "x");
		}
		for (local y_max = y_min + this.edge_len, i = 0; i <= this.edge_len; i++) {
			AISign.BuildSign(AIMap.GetTileIndex(x_min + i, y_max), "x");
		}
	}
}
