/**
 * Class to generate AI's event system.<p>
 * To use it first must be imported, like this:<p>
 *  <code>import("AILib.EventBase", "MyEventBaseImportedName", 1);</code>
 *
 * Then base class must be created:<p>
 *  <code>class MyAI_Event extends MyEventBaseImportedName {}</code>
 *
 * An event is the class inherited from the base we've just created.<p>
 * <code>
 *  class MyAIStartedEvent extends MyAI_Event
 *  {
 * 		static reaction_method_name = "OnAIStarted";
 *  }
 * </code>
 * The event class must have only one static field "reaction_method_name".<p>
 * The "reaction_method_name" must be a string, like "OnAIStarted".<p> 
 * Interface to interact with event class (calls must be static):<p>
 *  * function AddListener(listener)<p>
 *  * function RemoveListener(listener)<p>
 *  * function Fire(event_data)<p>
 * Each listener must have unique(among other listeners at least) ID,
 * accessable with GetID() method.<p>
 * I think methods appointment is easy enough, however:<p>
 *  When event will happen (triggered by Fire method),
 *  method named as value of event class reaction_method_name field
 *  will be called for all event listeners (objects added with AddListener,
 *  and yet not removed with RemoveListener) with Fire method param.<p>
 * If listener does not have proper method to handle event, there will be assert
 *  during AddListener call.<p>
 * To completely remove listener from all event classes use
 *  function WipeOutListener(listener)<p>
 */
class EventBase
{
	/**
	 * See Squirrel manual.
	 */
	function _inherited(attributes)
	{
		local base_event_class = this;
		this["event_classes"] <- [];

		this["_inherited"] <- function(attributes):(base_event_class) {
			/* (!) THIS IS THE SQ CLASS OBJECT FOR THE EVENT CLASS */
			/* "EVENT CLASS" mean any class derived from this class */
			local event_class_obj = this;

			/* Remember new class as unique event */
			base_event_class.event_classes.append(event_class_obj);

			/* Create a fields and methods for new class */
			/* listeners field */
			this["listeners"] <- {};

			/* AddListener method */
			this["AddListener"] <- function(listener):(event_class_obj) {
				if (!(event_class_obj.reaction_method_name in listener)) assert(null);
				event_class_obj.listeners[listener.GetID()] <- listener;
			}

			/* Fire method */
			this["Fire"] <- function(event_data):(event_class_obj) {
				local listeners = event_class_obj.listeners;
				local reaction_method_name = event_class_obj.reaction_method_name;
				foreach (dummy_id, listener in listeners) {
					listener[reaction_method_name](event_data);
				}
			}

			/* RemoveListener method */
			this["RemoveListener"] <- function(listener):(event_class_obj) {
				event_class_obj.listeners.rawdelete(listener.GetID());
			}
		}

		/* Function to remove the certain listener from all known events */
		this["WipeOutListener"] <- function(listener):(base_event_class) {
			local id = listener.GetID();
			foreach (dummy_id, class_obj in base_event_class.event_classes) {
				if (id in class_obj.listeners) delete class_obj.listeners[id];
			}
		}

		this["CheckAPIEvents"] <- function():(base_event_class) {
			while (AIEventController.IsEventWaiting()) {
				local e = AIEventController.GetNextEvent();
				switch (e.GetEventType()) {
					case AIEvent.ET_COMPANY_BANKRUPT:
						local id = AIEventCompanyBankrupt.Convert(e).GetCompanyID();
						base_event_class.CompanyBankrupt.Fire(id);
						break;
					case AIEvent.ET_COMPANY_IN_TROUBLE:
						local id = AIEventCompanyInTrouble.Convert(e).GetCompanyID();
						base_event_class.CompanyInTrouble.Fire(id);
						break;
					case AIEvent.ET_COMPANY_MERGER:
						local ec = AIEventCompanyMerger.Convert(e);
						local new_id = ec.GetNewCompanyID();
						local old_id = ec.GetOldCompanyID();
						local param = {new_company_id = new_id, old_company_id = new_id};
						base_event_class.CompanyMerger.Fire(param);
						break;
					case AIEvent.ET_COMPANY_ASK_MERGER:
						local ec = AIEventCompanyAskMerger.Convert(e);
						base_event_class.CompanyAskMerger.Fire(ec);
						break;
					case AIEvent.ET_COMPANY_NEW:
						local id = AIEventCompanyNew.Convert(e).GetCompanyID();
						base_event_class.CompanyNew.Fire(id);
						break;

					case AIEvent.ET_ENGINE_PREVIEW:
						base_event_class.EnginePreview.Fire(AIEventEnginePreview.Convert(e));
						break;
					case AIEvent.ET_INDUSTRY_OPEN:
						local id = AIEventIndustryOpen.Convert(e).GetIndustryID();
						base_event_class.IndustryOpen.Fire(id);
						break;
					case AIEvent.ET_INDUSTRY_CLOSE:
						local id = AIEventIndustryClose.Convert(e).GetIndustryID();
						base_event_class.IndustryClosed.Fire(id);
						break;

					case AIEvent.ET_SUBSIDY_AWARDED:
						local id = AIEventSubsidyAwarded.Convert(e).GetSubsidyID();
						base_event_class.SubsidyAwarded.Fire(id);
						break;
					case AIEvent.ET_SUBSIDY_EXPIRED:
						local id = AIEventSubsidyExpired.Convert(e).GetSubsidyID();
						base_event_class.SubsidyExpired.Fire(id);
						break;
					case AIEvent.ET_SUBSIDY_OFFER:
						local id = AIEventSubsidyOffer.Convert(e).GetSubsidyID();
						base_event_class.SubsidyOffer.Fire(id);
						break;
					case AIEvent.ET_SUBSIDY_OFFER_EXPIRED:
						local id = AIEventSubsidyOfferExpired.Convert(e).GetSubsidyID();
						base_event_class.SubsidyOfferExpired.Fire(id);
						break;

					case AIEvent.ET_ENGINE_AVAILABLE:
						local e = AIEventEngineAvailable.Convert(e).GetEngineID();
						switch (AIEngine.GetVehicleType(e)) {
							case AIVehicle.VT_AIR: base_event_class.AirEngineAvailable.Fire(e); break;
							case AIVehicle.VT_RAIL: base_event_class.RailEngineAvailable.Fire(e); break;
							case AIVehicle.VT_ROAD: base_event_class.RoadEngineAvailable.Fire(e); break;
							case AIVehicle.VT_WATER: base_event_class.WaterEngineAvailable.Fire(e); break;
						}
						break;
					case AIEvent.ET_VEHICLE_CRASHED:
						local v = AIEventVehicleCrashed.Convert(e).GetVehicleID();
						switch (AIVehicle.GetVehicleType(v)) {
							case AIVehicle.VT_AIR: base_event_class.AircraftCrashed.Fire(v); break;
							case AIVehicle.VT_RAIL: base_event_class.TrainCrashed.Fire(v); break;
							case AIVehicle.VT_ROAD: base_event_class.RoadVehicleCrashed.Fire(v); break;
							case AIVehicle.VT_WATER: base_event_class.ShipCrashed.Fire(v); break;
						}
						break;
					case AIEvent.ET_VEHICLE_LOST:
						local v = AIEventVehicleLost.Convert(e).GetVehicleID();
						switch (AIVehicle.GetVehicleType(v)) {
							case AIVehicle.VT_AIR: base_event_class.AircraftLost.Fire(v); break;
							case AIVehicle.VT_RAIL: base_event_class.TrainLost.Fire(v); break;
							case AIVehicle.VT_ROAD: base_event_class.RoadVehicleLost.Fire(v); break;
							case AIVehicle.VT_WATER: base_event_class.ShipLost.Fire(v); break;
						}
						break;
					case AIEvent.ET_VEHICLE_UNPROFITABLE:
						local v = AIEventVehicleUnprofitable.Convert(e).GetVehicleID();
						switch (AIVehicle.GetVehicleType(v)) {
							case AIVehicle.VT_AIR: base_event_class.AircraftUnprofitable.Fire(v); break;
							case AIVehicle.VT_RAIL: base_event_class.TrainUnprofitable.Fire(v); break;
							case AIVehicle.VT_ROAD: base_event_class.RoadVehicleUnprofitable.Fire(v); break;
							case AIVehicle.VT_WATER: base_event_class.ShipUnprofitable.Fire(v); break;
						}
						break;

					case AIEvent.ET_STATION_FIRST_VEHICLE:
						local ec = AIEventStationFirstVehicle.Convert(e);
						local s = ec.GetStationID();
						local v = ec.GetVehicleID();
						local param = {station_id = s, vehicle_id = v};
						base_event_class.StationFirstVehicle.Fire(param);
						break;

					case AIEvent.ET_VEHICLE_WAITING_IN_DEPOT:
						local v = AIEventVehicleWaitingInDepot.Convert(e).GetVehicleID();
						base_event_class.VehicleWaitingInDepot.Fire(v);
						break;
 
					case AIEvent.ET_DISASTER_ZEPPELINER_CLEARED:
						local ec = AIEventDisasterZeppelinerCleared.Convert(e);
						local station_id = ec.GetStationID();
						base_event_class.ZeppelinerCleared.Fire(station_id);
						break;
					case AIEvent.ET_DISASTER_ZEPPELINER_CRASHED:
						local ec = AIEventDisasterZeppelinerCrashed.Convert(e);
						local station_id = ec.GetStationID();
						base_event_class.ZeppelinerCrashed.Fire(station_id);
						break;
					case AIEvent.ET_AIRCRAFT_DEST_TOO_FAR:
						local ec = AIEventAircraftDestTooFar.Convert(e);
						local v = ec.GetVehicleID();
						base_event_class.AircraftDestTooFar.Fire(v);
						break;
					default: break;
				}
			}
		}

		/*  Wrappers for standart API events */
		base_event_class.CompanyBankrupt <- class extends base_event_class {
			static reaction_method_name = "OnCompanyBankrupt";
		}
		base_event_class.CompanyInTrouble <- class extends base_event_class {
			static reaction_method_name = "OnCompanyInTrouble";
		}
		base_event_class.CompanyMerger <- class extends base_event_class {
			static reaction_method_name = "OnCompanyMerger";
		}
		base_event_class.CompanyAskMerger <- class extends base_event_class {
			static reaction_method_name = "OnCompanyAskMerger";
		}
		base_event_class.CompanyNew <- class extends base_event_class {
			static reaction_method_name = "OnCompanyNew";
		}

		base_event_class.IndustryClosed <- class extends base_event_class {
			static reaction_method_name = "OnIndustryClose";
		}
		base_event_class.IndustryOpen <- class extends base_event_class {
			static reaction_method_name = "OnIndustryOpen";
		}

		base_event_class.SubsidyAwarded <- class extends base_event_class {
			static reaction_method_name = "OnSubsidyAwarded";
		}
		base_event_class.SubsidyExpired <- class extends base_event_class {
			static reaction_method_name = "OnSubsidyExpired";
		}
		base_event_class.SubsidyOffer <- class extends base_event_class {
			static reaction_method_name = "OnSubsidyOffer";
		}
		base_event_class.SubsidyOfferExpired <- class extends base_event_class {
			static reaction_method_name = "OnSubsidyOfferExpired";
		}

		base_event_class.EnginePreview <- class extends base_event_class {
			static reaction_method_name = "OnEnginePreview";
		}
		base_event_class.AirEngineAvailable <- class extends base_event_class {
			static reaction_method_name = "OnAirEngineAvailable";
		}
		base_event_class.RailEngineAvailable <- class extends base_event_class {
			static reaction_method_name = "OnRailEngineAvailable";
		}
		base_event_class.RoadEngineAvailable <- class extends base_event_class {
			static reaction_method_name = "OnRoadEngineAvailable";
		}
		base_event_class.WaterEngineAvailable <- class extends base_event_class {
			static reaction_method_name = "OnWaterEngineAvailable";
		}

		base_event_class.AircraftCrashed <- class extends base_event_class {
			static reaction_method_name = "OnAircraftCrash";
		}
		base_event_class.TrainCrashed <- class extends base_event_class {
			static reaction_method_name = "OnTrainCrash";
		}
		base_event_class.RoadVehicleCrashed <- class extends base_event_class {
			static reaction_method_name = "OnRoadVehicleCrash";
		}
		base_event_class.ShipCrashed <- class extends base_event_class {
			static reaction_method_name = "OnShipCrash";
		}

		base_event_class.AircraftLost <- class extends base_event_class {
			static reaction_method_name = "OnAircraftLost";
		}
		base_event_class.TrainLost <- class extends base_event_class {
			static reaction_method_name = "OnTrainLost";
		}
		base_event_class.RoadVehicleLost <- class extends base_event_class {
			static reaction_method_name = "OnRoadVehicleLost";
		}
		base_event_class.ShipLost <- class extends base_event_class {
			static reaction_method_name = "OnShipLost";
		}

		base_event_class.AircraftUnprofitable <- class extends base_event_class {
			static reaction_method_name = "OnAircraftUnprofitable";
		}
		base_event_class.TrainUnprofitable <- class extends base_event_class {
			static reaction_method_name = "OnTrainUnprofitable";
		}
		base_event_class.RoadVehicleUnprofitable <- class extends base_event_class {
			static reaction_method_name = "OnRoadVehicleUnprofitable";
		}
		base_event_class.ShipUnprofitable <- class extends base_event_class {
			static reaction_method_name = "OnShipUnprofitable";
		}

		base_event_class.StationFirstVehicle <- class extends base_event_class {
			static reaction_method_name = "OnStationFirstVehicle";
		}

		base_event_class.VehicleWaitingInDepot <- class extends base_event_class {
			static reaction_method_name = "OnVehicleWaitingInDepot";
		}

		base_event_class.ZeppelinerCleared <- class extends base_event_class {
			static reaction_method_name = "OnZeppelinerCleared";
		}
		base_event_class.ZeppelinerCrashed <- class extends base_event_class {
			static reaction_method_name = "OnZeppelinerCrash";
		}

		base_event_class.AircraftDestTooFar <- class extends base_event_class {
			static reaction_method_name = "OnAircraftDestTooFar";
		}
	}
}
