/**
 * Base class for AI transport systems.
 */
class AbstractTransportSystem extends Terron_SaveableObject
{
/* public */
	/**
	 * Constructor for AIAbstractTransportSystem.
	 */
	constructor()
	{
		::Terron_SaveableObject.constructor(null);
		this.subsystems = TableContainer.new("Subsystems container");
	}

	/**
	 * Accept a visitor.
	 * @param visitor - visitor object.
	 */
	function Accept(visitor)
	{
		visitor.Visit(this);
	}

	/**
	 * Add a subsystem for this system.
	 * @param subsystem - Subsystem to add.
	 */
	function AddSubsystem(subsystem)
	{
		this.subsystems[subsystem.GetID()] <- subsystem;
	}

	/**
	 * Close self.
	 * Make all the things required to remove this system from the game world
	 *  (like vehicles selling or stations demolishing).
	 * @return True if was able to close self, false if need retry.
	 */
	function Close()
	{
		return true;
	}

	/**
	 * Make transport system "disabled" for a given period of time.
	 * @param time time to sleep.
	 * @note "disabled" means AI will ignore this transport system.
	 */
	function Disable(time)
	{
		this.is_enabled = false;
		this.onDisable(time);
		local t = AIDate.GetCurrentDate() + time;
		this.enable_date = max(this.enable_date, t);
		EnableTask(this, t);
	}

	/**
	 * Check whether the transport system is "enabled".
	 * @note "disabled" means AI will ignore this transport system.
	 */
	function IsEnabled()
	{
		return this.is_enabled;
	}	

	/**
	 * Get this transport system subsystems.
	 * @return Container with subsystems.
	 */
	function GetSubsystems()
	{
		return this.subsystems;
	}

	/**
	 * Find best action within transport system (including subsystems).
	 * @return best available AI action (or null if nothing available). 
	 */
	function GetBestAction()
	{
		if (!this.is_enabled) return null;

		local best_action = this.doGetBestAction();
		local best_action_priority = (best_action == null) ?
			APriority.BAD : best_action.GetPriority();
		if (best_action_priority >= APriority.MAX) return best_action;

		foreach (id, subsystem in this.subsystems) {
			local a = subsystem.GetBestAction();
			if (a == null) continue;

			local p = a.GetPriority();
			if (p > best_action_priority) {
				best_action_priority = p;
				best_action = a;
				if (p >= APriority.MAX) return best_action;
			}
		}

		return best_action;
	}

/* protected */
	/**
	 * Find best action within transport system (not including subsystems).
	 * @return best available AI action (or null if nothing available).
	 */
	function doGetBestAction() { return null;}

	/** Make last desired operations when it's time to sleep. */
	function onDisable(time) {}

/* private */
	/** Transport system subsystems. */
	subsystems = null;

	/** Flag, showing if this system is enabled or disabled. */
	is_enabled = true;

	/** Enable date of this system. */
	enable_date = 0;
}

/**
 * Class that handles transport system enabling.
 */
class AbstractTransportSystem.EnableTask extends Terron_AbstractTask
{
/* public */
	/**
	 * Creates a new task.
	 * @param system Instance to "enable".
	 * @param enable_date Enable date.
	 */
	constructor(system, enable_date)
	{
		::Terron_AbstractTask.constructor(null);
		this.next_run_date = enable_date;
		this.system_to_enable = system;
		this.SetName("Enable system \"" + system.GetName() + "\"");
		_ai_clock.AddTask(this);
	}

	function Run(current_date)
	{
		local sys = this.system_to_enable;
		sys.is_enabled = (sys.enable_date <= current_date);
	}

/* private */
	/** Instance to "enable" */
	system_to_enable = null;
}
