/**
 * Class that describes AI "corporation".<p>
 * All the AI can build assumed to be a property of corporaion.<p>
 * All other transport systems assumed to be a part of corporation.<p>
 * It is assumed that AI can control only one corporation.<p>
 */
class Corporation extends AbstractTransportSystem
{
/* public */
	static function GetClassName()
	{
		return "Corporation";
	}

	/**
	 * Get the instance of the Corporation singleton.
	 */
	static function Get()
	{
		if (Corporation._instance[0] == null) Corporation();
		return Corporation._instance[0];
	}

	static function Restore(memento)
	{
		local restored = Corporation();
		restored.subsystems = memento.subsystems;
		restored.closing_systems = memento.closing_systems;
		return restored;
	}

	function GetName()
	{
		return "AI Corporation";
	}

	/**
	 * Add new action to possible corporation actions pool.
	 * @param a Action to add.
	 */
	function AddAction(a)
	{
		this.actions.AddItem(a);
	}

	/**
	 * Run one operating cycle.
	 */
	function Work()
	{
		local info = this.actions.GetBestItem();
		local a = info.best_item_rank <= APriority.BAD ? null : info.best_item;
		switch (a == null ? null : a.Try()) {
			case null :
				local tick = AIController.GetTick();
				CodeUtils.Log("Choosing action...", 0);
				local a = this.GetBestAction();
				CodeUtils.Log("... ticks: " + (AIController.GetTick() - tick), 0);
				if (a != null) {
					CodeUtils.Log("New action choosen for work pool: " + a.GetName(), 0);
					this.actions.AddItem(a);
					info = this.actions.GetBestItem();
					if (info.best_item_rank > APriority.BAD) info.best_item.Try();
					break;
				}
			case -2:
				/* Sleep when nothing to do, or not enough money to act */
				AIController.Sleep(this.sleep_time *= 2);
				if (this.sleep_time >= 32) break;
				
				FreeTimeEvent.Fire(null);
			default:
				this.sleep_time = 1;
				break;
		}
	}

	/**
	 * Cease loan repayment.
	 * At the begining of a game loan repayment can save couple of thousands
	 *  $ per year, but later on such operation become useless because of
	 *  huge(millions and millions) year income. Therefore even 10000 interests
	 *  is practically nothing.
	 * @note I wouldn't care, but AI goes bankrupt because funds
	 *  dive into red zone too fast(in seconds) with high expenses,
	 *  and even force loan stop helping.
	 */
	function StopLoanOperations()
	{
		local repay_action_name = AIRepayLoanAction().GetName();
		local to_delete = [];
		foreach (dummy_id, a in this.actions) {
			if (a.GetName() == repay_action_name) to_delete.append(a);
		}
		foreach (dummy_id, a in to_delete) {
			this.actions.RemoveItem(a.GetID());
		}
	}

	/**
	 * Add self to array of save-needed objects.
	 * @param arr Array with save-needed objects.
	 */
	function OnSave(arr)
	{
		arr.append(this);
	}

	/**
	 * Final initialization.
	 */
	function OnAIStarted(is_game_loaded)
	{
		local current_date = AIDate.GetCurrentDate();
		ClosePendingSystemsTask(current_date + GameTime.DAY, GameTime.MONTH);
		SellLostVehiclesTask(current_date + GameTime.DAY, 2 * GameTime.MONTH);
	}

/* private */
	/** Container with all actions available for the AI at moment */
	actions = null;

	/** Single instance of this class */
	static _instance = [null];

	/** Container with currently closing transport systems */
	</ must_save = true />
	closing_systems = null;

	/** Amount of time(in ticks) for the AI to sleep next time */
	sleep_time = 1;

	/**
	 * Creates the AI corporation.
	 */
	constructor()
	{
		::AbstractTransportSystem.constructor();
		if (Corporation._instance[0] != null) assert(null);
		Corporation._instance[0] = this;

		this.actions = TableContainer.new_custom("AI Actions", ActionsItems());
		this.actions.AddItem(AIRepayLoanAction());
		this.actions.AddItem(AIForceLoanAction());
		this.actions.AddItem(AIProcessEventsAction());

		if (!SaveLoadUtils.IsLoading()) {
			this.OnAIStarted(false);

			this.closing_systems = TableContainer.new("Stuff do close");

			this.actions.AddItem(AISpawnNodesAction(GameTime.YEAR));
			this.actions.AddItem(AISetCompanyNameAction());
			this.actions.AddItem(AIBuildCompanyHQAction());

			this.AddSubsystem(RoadDepartment.Get());
			this.AddSubsystem(ShipDepartment.Get());
			this.AddSubsystem(AirDepartment.Get());

			this.AddSubsystem(CentralTownPoliticsHandlingSection());
		} else {
			this.actions.AddItem(AISpawnNodesAction(0));
			AIStartedEvent.AddListener(this);
		}

		AISaveEvent.AddListener(this);
		CodeUtils.Log("Corporation control established...", 2);
	}
}

/** Class that defines "actions" items context */
class Corporation.ActionsItems extends TableContainer.ItemContext
{
/* protected */
	function GetObjectRank(object)
	{
		return object.GetPriority();
	}
}

/**
 * Task that periodically attempts to close all stuff
 *  that must be closed, but wasn't closed earlier.
 */
class Corporation.ClosePendingSystemsTask extends Terron_Task
{
/* protected */
	function Execute()
	{
		local closing = Corporation.Get().closing_systems;
		local closed = {};
		foreach (id, item in closing) {
			if (!item.Close()) continue;
			closed[id] <- item;
		}

		foreach (id, item in closed) {
			closing.RemoveItem(id);
			TransportSystemClosedEvent.Fire(item);
			Terron.UpdateMemento();
		}
		closed.clear();
	}
}

Corporation.setattributes("subsystems", {must_save = true});
Terron_ClassTable.RegisterClass(Corporation);
