/**
 * Class that provides methods for save/load.
 */
class SaveLoadUtils
{
/* public */
	/**
	 * Save given object into a valid structure.
	 * @param item Object to be saved.
	 * @param item_name For external call set to "root".
	 * @return Array of strings, similar to XML, that describes given object.
	 */
	static function Save(item, item_name)
	{
		local type = (typeof(item) == "instance") ? item.GetClassName() : typeof(item);
		local result = ["<" + item_name + " type=\"" + type + "\">"];
		local xxx = false; // strange, error without it
		switch (type)
		{
			//case "null"    : assert(null); break; // do not save null
			case "float"   : result.append(item); break;
			case "integer" : result.append(item); break;
			case "string"  : result.append(item); break;
			case "bool"    : result.append(item == true ? 1 : 0); break;
			case "array"   :
				local tmp_save = [];
				foreach (id, sub_item in item) {
					result.extend(SaveLoadUtils.Save(sub_item, "item"));
				}
				result.extend(tmp_save);
				break;
			default :
				//AILog.Warning("Saving " + item.GetName());
				foreach (id, sub_item in item.GetMemento()) {
					result.extend(SaveLoadUtils.Save(sub_item, id));
				}
		}
		result.append("</" + item_name + ">");
		return result;
	}

	/**
	 * Load an object from save structure.
	 * @param document Result of Save.
	 * @return loaded object.
	 */
	static function Load(document)
	{
		if (typeof(document) != typeof([])) {
			return SaveLoadUtils.Error("Can't load, invalid save file");
		}
		if (document.len() == 0) {
			return SaveLoadUtils.Error("Can't load, nothing is saved");
		}

		AILog.Warning("Loading...");
		SaveLoadUtils.save.document = document;
		for (local i = 0, data_stack = [], tags_stack = []; i < document.len(); i++) {
			if (!Tag.IsTag(document[i])) {
				return SaveLoadUtils.Error("Invalid tag: " + document[i]);
			}

			/*
			 * Don't forget:
			 * -----------------------------------------------------------------
			 * 1) Tag consist of "tag" and "type".
			 * Tag.tag is a label that defines how to to handle loaded stuff:
			 *  keyword "item" mean: stuff must be an item in some container
			 *  any other value mean: stuff must be an item in some other save
			 *  data - memento - and must be written into the memento[tag.tag].
			 * Tag.type mean type of the loaded stuff - and it is the type name.
			 *  For Squirrel types it is result of the typeof() function;
			 *  For classes types it is class name (every Terron class must have
			 *  static GetClassName() method).
			 * (!) "Loaded stuff" in previos text is memento too, don't forget.
			 * 2) Saved data is one big memento, consisting of smaller mementos.
			 * 3) For primitives(e.g. integer) "memento" is saved value itself.
			 * 4) For non-primitives(Terron classes) "memento" is a table,
			 *  result of the some_object.GetMemento method.
			 * 5) Squirrel tables should never be saved as tables.
			 *  There is a "TableContainer" for save-needed tables.
			 * -----------------------------------------------------------------
			 */
			local tag = Tag.GetStartTag(document[i]);
			if (tag != null) {
				tags_stack.append(tag);
				switch (tag.type) {
					case "float" :
						data_stack.append(document[++i].tofloat());
						break;
					case "integer" :
						data_stack.append(document[++i].tointeger());
						break;
					case "string" :
						data_stack.append(document[++i]);
						break;
					case "bool" :
						data_stack.append(document[++i] == 0 ? false : true);
						break; 
					case "array" : data_stack.append([]); break;
					default : data_stack.append({});
				}
				continue;
			}
			/*
			 * Tag.GetStartTag(document[i]) == null => end/close tag => 
			 * all information for one item to load received.
			 */
			/* Pop open tag for just read close tag */
			tag = tags_stack.pop();
			/* Pop last loaded object */
			local memento = data_stack.pop();
			local loaded_object = memento;

			/*
			 * Memento(type of memento) can be: table, array, primitives(e. g. integer)
			 * Memento is table <==> class method Restore must be used to load object
			 */
			if (typeof(memento) == "table") {
				local _class = Terron_ClassTable.GetClass(tag.type);
				if (_class == null) {
					return SaveLoadUtils.Error("Unknown class: " + tag.type);
				}

				AILog.Info("Loading object of class \"" + tag.type + "\"");
				loaded_object = _class.Restore(memento);

				if (loaded_object == null) {
					return SaveLoadUtils.Error("Failed to restore object of class " + tag.type);
				} else {
					AILog.Info("\"" + loaded_object.GetName() + "\" loaded");
				}
			}

			/* Stop when we found close tag for first(must be open) tag in save document */
			if (data_stack.len() == 0) {
				SaveLoadUtils.save.document = null;
				SaveLoadUtils.was_loaded.value = true;
				AILog.Warning("Data loading complete");
				return loaded_object;
			}

			/*
			 * Do not stop? => not all loaded yet.
			 * => we must use just loaded object in some other memento.
			 * Due to save structure and load sequence
			 *  "we know" that "some other" is the data stack top.
			 */
			if (tag.tag == "item") {
				/*
			 	 * If tag itself is the word "item" =>
			 	 *  => loaded object must be an item in the container, so
			 	 *  we add it to the previously created (due to right tags 
			 	 *  sequence) container.
			 	 */
				data_stack[data_stack.len() - 1].append(loaded_object);
			} else {
				/*
				 * If tag itself is not the word "item" =>
				 * => it's a field name(must be) at the parent memento
				 *  and loaded object must be the data under this field pointer
				 *  (parent_memento[Tag.tag] = loaded object) =>
				 * => we add loaded_object to the parent memento.
				 */
				data_stack[data_stack.len() - 1][tag.tag] <- loaded_object;
			}
		}

		/* Should never be here: mean save document had wrong tag sequence */
		return SaveLoadUtils.Error("Wrong tag sequence in save document");
	}

	/**
	 * Ckeck if AI is loading a save data.
	 * @return True if AI is restoring own save, false else.
	 */
	static function IsLoading()
	{
		return SaveLoadUtils.save.document != null;
	}

	/**
	 * Ckeck AI was restarted after game load.
	 * @return True if initial AI state was restored from game save, false else.
	 */
	static function WasLoaded()
	{
		return SaveLoadUtils.was_loaded.value;
	}

/* private */
	/** Save document */
	static save = {document = null};

	/** Shows wheteher AI was loaded or not */
	static was_loaded = {value = false};

	/**
	 * Error during loading happen. Finalize load process.
	 * @param log_text Error message.
	 * @return null.
	 */
	static function Error(log_text)
	{
		AILog.Error("Error while loading...");
		AILog.Error(log_text);
		AILog.Error("Loading failed");
		SaveLoadUtils.save.document = null;
		return null;
	}
}
