/**
 * Color definitions for different game situations.<p>
 *
 * "Links" represent possible connections between specific structures/towns on
 *  the game map.<p>
 * And the LinkColor represent "how possible" actually these connections are.<p>
 *
 * Remember that link created only when two nodes are close to each other
 *  (within some "AI can connect" range).<p>
 *
 * Here are some color explanations with examples:<p>
 * a) Coal from a coal mine X to a power plant Y must have GREEN color.<p>
 *    Because X is raw producer, and Y is end-point of coal transport chain,
 *    ai always can try to handle such situation.<p>
 * b) GRAY means that there is no "full transport chain" from raw producer to
 *    some end-point-endless-consumer.<p>
 *    If some factory Z does not have access(mean too far away) to towns =>
 *    => ALL LINKS AND LINK CHAINS THAT INCLUDE Z MUST BE GRAY.
 * c) Consider situation when Z have access to some local town T.<p>
 *    Then link [Z-T] must be GREEN <=> Z IS SUPPLIED(with steel for example)
 *    Until then(while Z is not supplied) [Z-T] color must be RED.<p>
 * d) If AI is transporting coal from X to Y => [X-Y] must be YELLOW.<p>
 * In other words the link colors mean:
 *  GREEN  - AI can use this link(establish trade) right now<p>
 *  YELLOW - AI can't use this link, it's already in use<p>
 *  RED    - AI can't use this link yet, must provide supplies to start<p>
 *  GRAY   - AI can't use this link, the result considered unefficient anyway<p>
 *  BLUE, WHITE - use it as you wish.<p>
 */
enum LinkColor
{
	LC_GREEN
	LC_YELLOW
	LC_RED
	LC_GRAY
	LC_BLUE
	LC_WHITE
};

/**
 * Well, that one is serious.
 * We don't actually need such super container to run the AI,
 *  but it is needed to run the AI fast(or, at least faster).
 * Should provide access to important stuff by color.
 */
class GlobalTransportMap extends Terron_Object
{
/* public */
	/**
	 * Get the instance of the GlobalTransportMap singleton.
	 */
	static function Get()
	{
		return GlobalTransportMap._instance[0] == null ?
			GlobalTransportMap() : GlobalTransportMap._instance[0];
	}

	/**
	 * Add new transport node into this container.
	 * @param transport_node The TransportNode object (to add).
	 * @return Not used.
	 */
	function AddTransportNode(transport_node)
	{
		local class_name = transport_node.GetClassName();
		if (!(class_name in this.structured_transport_nodes)) {
			this.SetUpNewTransportNodeClass(class_name);
		}

		this.transport_nodes_map.AddItem(transport_node);

		local type_id = transport_node.node.GetTypeID();
		local class_nodes = this.structured_transport_nodes[class_name];
		class_nodes[type_id][transport_node.node.node_id] <- transport_node;
	}

	/**
	 * Removes a transport node from the AI scope.
	 * @param transport_node_id Runtime ID of the transport node.
	 * @return Not used.
	 */
	function RemoveTransportNode(transport_node_id)
	{
		if (transport_node_id in this.transport_nodes_map) {
			local x = this.transport_nodes_map[transport_node_id];
			delete transport_nodes_map[transport_node_id];
			RemoveTransportNodeEx(x.GetClassName(), x.node.GetTypeID(), x.node_id);
		}
	}

	/**
	 * Removes transport node from the AI scope.
	 * @param class_name Transport node's class name.
	 * @param type_id Transport node's "raw" node type ID.
	 * @param node_id Transport node's "raw" node ID.
	 * @return Not used.
	 */
	function RemoveTransportNodeEx(class_name, type_id, node_id)
	{
		local map = this.structured_transport_nodes[class_name];
		if (type_id in map) {
			local nodes = map[type_id];
			if (!(node_id in nodes)) return;
			CorporationUtils.CloseSystem(nodes[node_id]);
			delete nodes[node_id];
		}
	}

	/**
	 * Register a link.
	 * @param link The link object to register.
	 * @return Not used.
	 */
	function AddLink(link)
	{
		this.link_map[link.nodes_class_name][link.color][link.GetID()] <- link;
	}

	/**
	 * Retrieve a transport node object by ID.
	 * @param runtime_transport_node_id The transport node's runtime ID.
	 * @return Object with the given ID, or null if such object unknown.
	 */
	function GetTransportNodeByID(runtime_transport_node_id)
	{
		return this.transport_nodes_map.GetItem(runtime_transport_node_id);
	}

	/**
	 * Retrieve a transport node object by its class name and base node.
	 * @param transport_class_name The transport node's class name.
	 * @param raw_name The base(town, industry e.t.c) node.
	 * @return Object with the given class name and base node, or null if such
	 *  object unknown.
	 */
	function GetTransportNode(transport_class_name, raw_node)
	{
		local map = this.structured_transport_nodes[transport_class_name];
		local type_id = raw_node.GetTypeID();
		if (!(type_id in map)) return null
		local nodes = map[type_id];
		if (!(raw_node.node_id in nodes)) return null;
		return nodes[raw_node.node_id];
	}

/* private */
	/** Single instance of this class */
	static _instance = [null];

	/**
	 * "Structured" container with all AI transport nodes.
	 * Has following structure:
	 * level 1 - transport class name
	 * level 2 - node type id
	 * level 3 - node's constant id
	 * @note differs from transport_nodes_map: node key is node.node_id
	 */
	structured_transport_nodes = null;

	/**
	 * "Unstructured" container(just '{}' actually) with all AI transport nodes.
	 * Key - Transport node runtime ID.
	 * Value - Transport node itself.
	 * @note ID is not equal to ID of the same node in "structured" container.
	 */
	transport_nodes_map = null;

	/**
	 * Container with all links.
	 * Has following structure:
	 * Map level 1: Key - transport node class names, value - level 2.
	 * Map level 2: Key - link colors(from the LinkColor enum), value - level 3.
	 * Map level 3: Key - Link object ID, value - link object.
	 */
	link_map = null;

	constructor()
	{
		::Terron_Object.constructor(null);

		/* Make sure we'll have only one instance of this class */
		if (GlobalTransportMap._instance[0] != null) assert(null);
		GlobalTransportMap._instance[0] = this;

		this.structured_transport_nodes = {};
		this.transport_nodes_map = TableContainer.new("All transport nodes");
		this.link_map = {};
	}

	/**
	 * Removes a link from the global links container.
	 * @note This is just small helper function,
	 *  "syntactic sugar"(in a manner of speaking),
	 *  to safely (and completely) remove the link object from the AI,
	 *  must use Link.Terminate.
	 */
	function RemoveLink(link)
	{
		if (link.color == null) assert(link.color);
		this.link_map[link.nodes_class_name][link.color].rawdelete(link.GetID());
	}

	/**
	 * Print information about all known transport node links.
	 */
	function PrintAllLinks()
	{
		local c = function(color) {
			local result = "NONE";
			switch (color) {
				case LinkColor.LC_GREEN 	: result = "GREEN"; break;
				case LinkColor.LC_YELLOW	: result = "YELLOW"; break;
				case LinkColor.LC_RED		: result = "RED"; break;
				case LinkColor.LC_GRAY		: result = "GRAY"; break;
				case LinkColor.LC_BLUE		: result = "BLUE"; break;
				case LinkColor.LC_WHITE		: result = "WHITE"; break;
			}
			return result;
		}

		foreach (class_name, link_map in this.link_map) {
			foreach (color, links in link_map) {
				CodeUtils.Log("-==" + c(color) + "==-", 1);
				foreach (dummy_id, link in links) {
					foreach (dummy_id, route in link.routes) {
						CodeUtils.Log("    " + route.GetName(), 1);
					}
				}
			}
		}
	}

	/**
	 * Create a placehold for nodes of new transport class inside this object.
	 * @param class_name New transport class name.
	 */
	function SetUpNewTransportNodeClass(class_name)
	{
		this.structured_transport_nodes[class_name] <- {};
		foreach (type_id, dummy in TransportSchema.Get().node_types) {
			this.structured_transport_nodes[class_name][type_id] <- {};
		}

		this.link_map[class_name] <- {};
		this.link_map[class_name][LinkColor.LC_GREEN] <- {};
		this.link_map[class_name][LinkColor.LC_YELLOW] <- {};
		this.link_map[class_name][LinkColor.LC_RED] <- {};
		this.link_map[class_name][LinkColor.LC_GRAY] <- {};
		this.link_map[class_name][LinkColor.LC_BLUE] <- {};
		this.link_map[class_name][LinkColor.LC_WHITE] <- {};
	}
}
