/**
 * Class that represent connection between two transport nodes.<p>
 * It's important: objects of this class can have different "color",
 *  and can can change color as well. See LinkColor.<p>
 * It is assumed, that links are only possible between transport nodes
 *  belonging to one transport class, same for the pair of nodes.<p>
 * @note "Connection" in this case is a virtual possibility to connect
 *  nodes, nothing more.
 */
class GlobalTransportMap.Link extends Terron_Object
{
/* public */
	/**
	 * Create a new link between pair of nodes.
	 * @param a Transport node, A point.
	 * @param b Transport node, B point.
	 * @param transport_politics Custom nodes transport politics.
	 * @param direction Integer value which describes link direction.
	 *  Nonzero value mean A -> B, zero A <-> B.
	 */
	static function Create(a, b, transport_politics, direction, cargo_ids)
	{
		if (cargo_ids.len() == 0) return null;

		/* Make always [Start point node ID <= End point node ID] */
		if (direction == 0 && a.node.node_id > b.node.node_id) {
			local tmp = a;
			a = b;
			b = tmp;
		}

		local a_id = a.GetID();
		local b_id = b.GetID();
		/* Do not create dublicate links in cases when A <-> B */
		if (direction == 0 && a_id in b.links_out) return b.links_out[a_id];

		local new_link = GlobalTransportMap.Link(a_id, b_id, direction, a.GetClassName());
		new_link.routes = transport_politics.GetRoutes(a, b, cargo_ids);
		new_link.aux = TransportSchema.IsAuxShipment(a.node.type, b.node.type);

		a.links_out[new_link.to_id] <- new_link;
		b.AddLink_In(new_link);

		GlobalTransportMap.Get().AddLink(new_link);

		return new_link;
	}

	/**
	 * "Destructor" to remove link(and all related stuff) from the AI safely.
	 * @param link The link to terminate. 
	 */
	static function Terminate(link)
	{
		/* Close all related routes */
		foreach (dummy_route_id, route in link.routes) {
			foreach (dummy_id, trade_route in route.trade_routes) {
				trade_route.Close();
			}
		}
		//link.routes = null;
		link.routes.clear();

		GlobalTransportMap.Get().RemoveLink(link);

		/* Remove link from owners local link maps */
		local nodes = GlobalTransportMap.Get().transport_nodes_map;
		nodes[link.to_id].RemoveLink_In(link);
		nodes[link.from_id].RemoveLink_Out(link);
		//nodes[link.to_id].links_in.rawdelete(link.from_id);
		//nodes[link.from_id].links_out.rawdelete(link.to_id);

		local s = nodes[link.from_id];
		local e = nodes[link.to_id];
		CodeUtils.Log("Terminating link between " + s.GetName() + " and " + e.GetName(), 1);
	}

	/**
	 * Get the link start point.
	 * @return Transport node object, representing link begining.
	 */
	function GetSource()
	{
		return GlobalTransportMap.Get().GetTransportNodeByID(this.from_id);
	}

	/**
	 * Get the link end point.
	 * @return Transport node object, representing link end.
	 */
	function GetTarget()
	{
		return GlobalTransportMap.Get().GetTransportNodeByID(this.to_id);
	}

	/**
	 * Link start point(transport node).
	 */
	from_id = null;

	/**
	 * Link end point(transport node).
	 */
	to_id = null;

	/**
	 * True mean that cargo can be transported only in one direction.
	 */
	one_way_link = true;

	/**
	 * True mean that link is not necessary for endpoint production.<p>
	 */
	aux = false;

	/**
	 * Current link color.
	 */
	color = null;

	/**
	 * Routes possible within this link.
	 */
	routes = null;

	/**
	 * Set link color.
	 * @param new_color Color to set.
	 * @return None.
	 */
	function SetColor(new_color)
	{
		if (new_color == this.color) return;

		local global = GlobalTransportMap.Get();

		global.RemoveLink(this);
		local old_color = this.color;
		this.color = new_color;
		global.AddLink(this);

		local from = global.GetTransportNodeByID(this.from_id);
		if (from != null) from.OnOutLinkColorChange(this, old_color);

		local to = global.GetTransportNodeByID(this.to_id);
		if (to != null) to.OnInLinkColorChange(this, old_color);
	}

/* private */
	/** Class name of nodes which this link connects. */
	nodes_class_name = null;

	/**
	 * Link private constructor.
	 * @param from_id ID of the link start transport node.
	 * @param from_id ID of the link end transport node.
	 * @param direction Integer value which describes link direction.
	 *  Zero means A <-> B, else A -> B.
	 * @param class_name Transport nodes class name.
	 */
	constructor(from_id, to_id, direction, class_name)
	{
		::Terron_Object.constructor(null);
		this.from_id = from_id;
		this.to_id = to_id;
		if (direction == 0) this.one_way_link = false;
		this.color = LinkColor.LC_GRAY;
		this.nodes_class_name = class_name;
		this.routes = {};
	}
}
