/**
 * Class that describes "node type".
 * Contains data(sometimes dublicate, but stored differently) about
 *  node type and it's place among/[link with] other types.<p>
 *  Similar to the API AIIndustryType class, but not limited only by industries,
 *  and can describe other entities(e. g. "Town").<p>
 * @note The class is not equivalent to "node type", it's instances must.
 */
class NodeType extends Terron_Object
{
/* public */
	/**
	 * Manually assignable(via constructor) type ID.
	 * @note Should watch out for possible clashes with several types ID.
	 */
	type_id = null;

	/** String - name of this type */
	type_name = null;

	/**
	 * Table describing this type's production.<p>
	 * Item key - ID of the produced cargo.<p>
	 * Item value - array with IDs of types that consume the produced cargo.<p>
	 */
	production = null;

	/**
	 * Table describing this type's consumption.<p>
	 * Item key - ID of the consumed(accepted) cargo.<p>
	 * Item value - array with IDs of supplier types.<p>
	 */
	consumption = null;

	/**
	 * Table describing this type's supplier types.<p>
	 * Item key - ID of the supplier node type.<p>
	 * Item value - Array of supplier's cargo IDs consumed by this type.<p>
	 */
	supplier_node_types = null;

	/**
	 * Table describing this type's production consumer types.<p>
	 * Item key - ID of the production consumer node type.<p>
	 * Item value - Array of this type's cargo IDs consumed by consumer type.<p>
	 */
	consumer_node_types = null;

	/** Flag - same meaning as AIIndustryType::IsRawIndustry API function */
	raw_production = false;

	/**
	 * Flag indicating that this node type is "endless" consumer.
	 * True when this node type considered as cargo-chain end point
	 *  (or root of the consumption tree from other point of view).
	 *  Mean that nodes of this type can <b>consume</b> (not process into other)
	 *  practicly illimitable amount of cargo.<p>
	 * Examples: "Power station", "town".
	 * However, "cargo-chain end point" is the most accurate definiton,
	 *  and, for example, I consider vehicles factories from the newgrfs as
	 *  raw consumer too, as well as other similar industry types.
	 */
	raw_consumption = false;

	/**
	 * Type specific cargo production rules.
	 */
	recipes = null;

	/**
	 * "Secondary" suppliers for this type.
	 * With newgrfs it is possible to have not tree-like shipments from raw
	 *  producers to this type. There can be cycles between production types,
	 *  e.g. A => B => C => A. This "secondary" suppliers list is used to
	 *  break cycles and be able to make a tree (e.g. A => B => C + C helps A).
	 */
	helpers = null;

	/**
	 * Full description of this type's consumption map.<p>
	 * Tree structure that must describe shipments
	 *  from raw producers(tree leaves) to this type(tree root).<p>
	 */
	supply_tree = null;

	/**
	 * Creates new node type.
	 * @param primitive_node_type Primitive that'll be the base for this type.
	 * @param primitive_node_types_table Table with all primitives.
	 * @note supply_tree will not be initialized.
	 */
	constructor(primitive_node_type, primitive_node_types_table)
	{
		::Terron_Object.constructor(null);
		local s = "Creating \"" + primitive_node_type.type_name + "\" node type...";
		CodeUtils.Log(s, 0);
		this.InitFromPrimitive(primitive_node_type);
		this.FillDirectCrossTypeLinks(primitive_node_types_table);
		this.SetAllTypeHelpers(primitive_node_types_table);
		CodeUtils.Log("... basic initialization complete", 0);
	}

	function GetName() { return this.type_name;}

	/**
	 * Get cargo production from the given cargo input.
	 * @param input_cargo_ids Table with input cargo IDs.
	 * @return Table that contains cargo IDs that can be received from
	 *  the given input.
	 */
	function GetOutput(input_cargo_ids)
	{
		return this.recipes.GetOutput(input_cargo_ids);
	}

	/**
	 * Finish the initialization of the node type by creating its full supply tree.
	 * @param node_types Table containing all existing node types.
	 */
	function BuildSupplyTree(node_types)
	{
		CodeUtils.Log("Building supply tree for " + this.GetName() + "...", 0);
		this.doFindCrossFeaders(this, {}, node_types);
		this.supply_tree = this.doBuildSupplyTree(this, {}, node_types);
	}

	/**
	 * Prints basic information about this node type.
	 * @param types Table with all node types.
	 */
	function PrintInfo(types)
	{
		/* local helpers to draw nice out text */
		local map_keys_to_string = function(map, to_string_callback) {
			local s = "", i = 0, n = map.len();
			foreach (key, dummy in map) {
				s += to_string_callback(key) + (n != ++i ? ", " : "");
			}
			return s;
		}
		local f = function(c) { return AICargo.GetCargoLabel(c);}
		local this_type = this;
		local g = function(i):(this_type, types) {
			local t = types[i];
			local appendix = this_type.type_id in t.helpers ? "(AUX)" : "";
			return t.type_name + appendix;
		}
		local h = function(i):(this_type, types) {
			local t = types[i];
			local appendix = t.type_id in this_type.helpers ? "(AUX)" : "";
			return t.type_name + appendix;
		}

		CodeUtils.Log(this.type_name, 2);
		local s = this.raw_production ? "Produce (RAW): " : "Produce: ";
		CodeUtils.Log(s + map_keys_to_string(this.production, f), 2);
		s = this.raw_consumption ? "Consume (RAW): " : "Consume: "; 
		CodeUtils.Log(s + map_keys_to_string(this.consumption, f), 2);

		CodeUtils.Log("Suppliers: " + map_keys_to_string(this.supplier_node_types, h), 2);
		CodeUtils.Log("Consumers: " + map_keys_to_string(this.consumer_node_types, g), 2);
		//CodeUtils.Log("Recipes: ", 2);
		//foreach (dummy_id, recipe in recipes) {
		//	recipe.Print();
		//}
	}

	/**
	 * Prints the supply tree for this node type.
	 * @param types Table with all node types.
	 */
	function PrintSupplyTree(types, log_priority)
	{
		assert(this.supply_tree);
		foreach (id, s in this.SupplyTreeToStrings(types, this.supply_tree)) {
			CodeUtils.Log(s, log_priority);
		}
	}

/* private */
	/**
	 * Set basic type's data from the given prototype.
	 * @param primitive_node_type This type's prototype.
	 */
	function InitFromPrimitive(primitive_node_type)
	{
		CodeUtils.Log("...copying primitive data...", 0);
		this.type_id = primitive_node_type.type_id;
		this.type_name = primitive_node_type.type_name;
		this.raw_consumption = primitive_node_type.raw_consumption;
		this.raw_production = primitive_node_type.raw_production;
		this.production = {};
		this.consumption = {};
		this.consumer_node_types = {};
		this.supplier_node_types = {};
		this.supply_tree = {};
		this.recipes = {};
		this.helpers = {};
		foreach (c, dummy in primitive_node_type.production) {
			this.production[c] <- [];
		}
		foreach (c, dummy in primitive_node_type.consumption) {
			this.consumption[c] <- [];
		}
		this.recipes = primitive_node_type.recipes;
	}

	/**
	 * Set basic type's data from the given prototype.
	 */
	function FillDirectCrossTypeLinks(primitive_node_types_table)
	{
		CodeUtils.Log("...searching for cross type connections...", 0);
		foreach (id, type in primitive_node_types_table) {
			foreach (c, dummy in this.production) {
				if (!(c in type.consumption)) continue;
				this.production[c].append(id);
				if (!(id in this.consumer_node_types)) {
					this.consumer_node_types[id] <- {};
				}
				this.consumer_node_types[id][c] <- null;
			}
			foreach (c, dummy in this.consumption) {
				if (!(c in type.production)) continue;
				this.consumption[c].append(id);
				if (!(id in this.supplier_node_types)) {
					this.supplier_node_types[id] <- {};
				}
				this.supplier_node_types[id][c] <- null;
			}
		}
	}

	/**
	 * Type's aux suppliers search and initialization.
	 */
	function SetAllTypeHelpers(primitive_node_types_table)
	{
		CodeUtils.Log("...defining aux suppliers...", 0);
		/* All that supplies raw industries is helpers for raw industries */
		if (this.raw_production) {
			foreach (supplier_id, cargo_ids in this.supplier_node_types) {
				this.helpers[supplier_id] <- clone cargo_ids;
			}
		}

		/* Set correct raw_consumption flag */
		foreach (consumer_id, dummy in this.consumer_node_types) {
			/* Can supply self(e.g. Town -> Town) => mark as raw consumer */
			if (consumer_id == this.type_id) {
				this.raw_consumption = true;
				break;
			}
			if (primitive_node_types_table[consumer_id].raw_production) {
				//primitive_node_types_table[consumer_id].raw_consumption = true;
				this.raw_consumption = true;
			}
		}
	}

	/**
	 * Circular references in type's supply handling.
	 */
	function doFindCrossFeaders(root, used, node_types)
	{
		local root_id = root.type_id;
		used[root_id] <- root_id;
		foreach (id, cargo_ids in root.supplier_node_types) {
			if (root_id == id) continue;
			local s = node_types[id]; // s == supplier
			if (id in used || (s.raw_consumption && !s.raw_production)) {
				root.helpers[id] <- cargo_ids;
			}
		}

		foreach (id, cargo_ids in root.supplier_node_types) {
			if (id in root.helpers) continue;
			local s = node_types[id]; // s == supplier
			if (s.raw_consumption || s.supplier_node_types.len() == 0) continue;
			this.doFindCrossFeaders(s, clone used, node_types);
		}
	}

	/**
	 * Build full supply tree for this type.
	 */
	function doBuildSupplyTree(root, already_used, node_types)
	{
		local types = node_types;
		local supply = {root_id = root.type_id, tree = {}};
		local used = clone already_used;
		used[root.type_id] <- root.type_id;

		/* All suppliers except the "helpers" ones must be in supply tree */
		local helpers = types[root.type_id].helpers;
		foreach (id, cargo_ids in root.supplier_node_types) {
			if (id in helpers) continue;
			supply.tree[id] <- {cargos = cargo_ids, sub_tree = null};
		}
		foreach (id, tree in supply.tree) {
			/*
			 * Stop when the supplier already in tree, or is raw consumer( =>
			 *  => don't produce indispensable cargo for other node types).
			 */
			tree.sub_tree = (
				id in used || (types[id].raw_consumption && id != root.type_id)
			) ? {root_id = id, tree = {}} : this.doBuildSupplyTree(types[id], used, node_types);
		}
		return supply;
	}

	/**
	 * Converts some supply tree to array of XML-formated strings.
	 */
	function SupplyTreeToStrings(types, root_tree, ...)
	{
		local print_c = function(arr)
		{
			local s = "", i = 0, n = arr.len();
			foreach (c, dummy_value in arr) {
				s += AICargo.GetCargoLabel(c) + (n != ++i ? " AND " : "");
			}
			return s;
		}

		local paths = [];
		local root_name = types[root_tree.root_id].type_name;
		if (vargc == 0) paths.append("<" + root_name + ">");
		if (types[root_tree.root_id].helpers.len() > 0) {
			paths.append("<helpers>");
			foreach (id, c in types[root_tree.root_id].helpers) {
				local s = print_c(c) + " from " + types[id].type_name;
				s = CodeUtils.MakeNoSpacesString(s);
				if (types[id].raw_production) s += "_RAW_PRODUCTION";
				paths.append("<"+ s +"/>");
			}
			paths.append("</helpers>");
		}
		foreach (id, tree in root_tree.tree) {
			local s = print_c(tree.cargos) + " from " + types[id].type_name;
			s = CodeUtils.MakeNoSpacesString(s);
			if (types[id].raw_production) s += "_RAW_PRODUCTION";
			local tmp_arr = this.SupplyTreeToStrings(types, tree.sub_tree, 1);
			if (tmp_arr.len() == 0) {
				paths.append("<"+ s +"/>");
			} else {
				paths.append("<"+ s +">");
				paths.extend(tmp_arr);
				paths.append("</"+ s +">");
			}
		}
		if (vargc == 0) paths.append("</" + root_name + ">");
		return paths;
	}
}
