/**
 * Class that handles towns as nodes.
 */
class TownNode extends Node
{
/* public */
	static function GetClassName()
	{
		return "TownNode";
	}

	static function Restore(memento)
	{
		local town_type = TransportSchema.Get().node_types[memento.type_id];
		local restored = TownNode(memento.town_id, town_type);
		restored.stations = memento.stations;
		return restored;
	}

	/** Corresponding town ID. */
	</ must_save = true />
	town_id = null;

	/** Tiles inside and near the town. */
	town_tiles = null;

	/** Currently not used. */
	accepting_now = null;

	/**
	 * Creates new TownNode.
	 * @param town_id Base town id.
	 * @param town_type Node type.
	 */
	constructor(town_id, town_type)
	{
		local name = AITown.GetName(town_id);		
		local location = AITown.GetLocation(town_id);
		::Node.constructor(town_id, town_type, name, location);

		this.town_id = town_id;
		this.town_tiles = Property(this.doGetTownTiles.bindenv(this), 6 * GameTime.YEAR / 5);
		this.town_tiles.SetName("Rechecking town " + name + " borders");
		this.accepting_now = Property(this.doGetCurrentAcceptance.bindenv(this), 3 * GameTime.MONTH);
		this.town_tiles.SetName("Rechecking town " + name + " acceptance");
	}

	/**
	 * Get town id.
	 * @return ID of node base town.
	 */
	function GetTownID()
	{ 
		return this.town_id;
	}

	/**
	 * Get town population.
	 * @return Population of node base town.
	 */
	function GetPopulation()
	{
		return AITown.GetPopulation(this.town_id);
	}

	/**
	 * Checks whether this town has a statue of the AI company.
	 * @return True if and only if the town has AI company statue.
	 */
	function HasStatue()
	{
		return AITown.HasStatue(this.town_id);
	}

	/**
	 * Checks whether this town node represent town as
	 *  passengers generator(opposite to cargo consumer - this is another
	 *  town node type).
	 * @return True if and only if the town node belongs to NT_PAX_TOWN type.
	 */
	function IsPaxTown()
	{
		return this.type_id == NodeTypeID.NT_PAX_TOWN;
	}

	/** Currently not used. */
	function IsAcceptingRightNow(c)
	{
		local acc = this.accepting_now.Get();
		return c in acc ? acc[c] : false;
	}

	function GetLastMonthProduction(c)
	{
		return AITown.GetLastMonthProduction(this.town_id, c);
	}

	function GetLastMonthTransported(c)
	{
		return max(AITown.GetLastMonthSupplied(this.town_id, c), 0);
	}

	function AITileList_NodeAccepting(radius, cargo_id)
	{
		local list = AITileList();
		list.AddList(this.town_tiles.Get());

		list.Valuate(AITile.GetCargoAcceptance, cargo_id, 1, 1, radius);
		list.RemoveBelowValue(8);

		return list;
	}

	function AITileList_NodeProducing(radius, ...)
	{
		if (vargc != 1) {
			local c = CorporationUtils.pass_cargo_id.value;
			return this.IsProducing(c) ?
				this.AITileList_NodeAccepting(radius, c) : AITileList();
		}
		return this.IsProducing(vargv[0]) ?
			this.AITileList_NodeAccepting(radius, vargv[0]) : AITileList();
	}

	function GetAmountOfOpponentsStationsAround()
	{
		return this.GetOpponentsCount();
	}

	/**
	 * Get AI rating at the node.
	 * @return AI company rating in base node town.
	 */
	function GetMyRating()
	{
		local rating = AITown.GetRating(this.town_id, AICompany.COMPANY_SELF);
		return rating == AITown.TOWN_RATING_NONE ? AITown.TOWN_RATING_VERY_GOOD : rating;
	}

	/**
	 * Try to make an advertise campaign.
	 * @param period Time that should have passed
	 *  since the last advertizing campaign.
	 * @return True if and only if advertise campaign succeed.
	 */
	function TryAdvertiseCampaign(period)
	{
		if (this.type_id != NodeTypeID.NT_PAX_TOWN) return false;

		local date = AIDate.GetCurrentDate();
		if (date - this.last_advertizing_campaign_date < period) return false;

		local medium = AITown.TOWN_ACTION_ADVERTISE_MEDIUM;
		if (AITown.IsActionAvailable(this.town_id, medium)) {
			if (AITown.PerformTownAction(this.town_id, medium)) {
				this.last_advertizing_campaign_date = date;
				return true;
			}
		}

		local small = AITown.TOWN_ACTION_ADVERTISE_SMALL;
		if (AITown.IsActionAvailable(this.town_id, small)) {
			if (AITown.PerformTownAction(this.town_id, small)) {
				this.last_advertizing_campaign_date = date;
				return true;
			}
		}

		return false;
	}

/* private */
	/** Date of the the last successful advertizing campaign in the node's town. */
	last_advertizing_campaign_date = -1000;
}

/**
 * Get list with town owned tiles for this town.
 */
function TownNode::doGetTownTiles()
{
	local list = AITileList();

	/* Estimate town borders distnace from town center */
	local r = 10;
	local population = AITown.GetPopulation(this.town_id);
	if (population > 1500) r += population / 1500;

	/* Find square area around town */
	TileUtils.AddSimmetricRectangleSafe(list, this.location, r, r);

	/* Remove not town related tiles */
	list.Valuate(AITile.IsWithinTownInfluence, this.town_id);
	list.RemoveValue(0);

	return list;
}

/** Currently not used. */
function TownNode::doGetCurrentAcceptance()
{
	local result = {};
	local list = this.town_tiles.Get();
	foreach (c, dummy in AICargoList()) {
		local tmp_list = AITileList();
		tmp_list.AddList(list);

		tmp_list.Valuate(AITile.GetCargoAcceptance, c, 1, 1, 3);
		tmp_list.RemoveBelowValue(8);

		result[c] <- (tmp_list.Count() > 2);
	}
	return result;
}

TownNode.setattributes("type_id", {must_save = true});
Terron_ClassTable.RegisterClass(TownNode);
