/**
 * Class that describes set of transport nodes which can feed the given node.<p>
 * This is basically one of the graph implementations, though with a condition
 *  that we build quite special graph, and de facto reduce it to tree<p>.
 * The given node can be then considered root.<p>
 */
class SupplyContingent extends Terron_Object
{
/* public */
	static function GetClassName()
	{
		return "SupplyContingent";
	}

	/**
	 * Array that represents supply graph.<p>
	 * Each item is a map with the description of the graph's "level", where
	 *  "level" mean set of the graph's vertices equidistant from root.<p>
	 * Item 'k' describes vertices on distance k from the root.<p>
	 * Items keys are transport node's runtime IDs.<p>
	 * Items values are arrays with the links
	 *  incoming into the node(which ID == item's key) from next graph level.<p>
	 * @note "Distance" means amount of edges in shortest path
	 *  between a vertex and the root.<p>
	 */
	levels = null;

	/**
	 * Full index of the graph's edges.<p>
	 * This is a map where keys are link IDs and values are empty tables.<p>
	 */
	links = null;

	function GetName()
	{
		return this.name;
	}

	function Clone()
	{
		return this.doCloning(SupplyContingent(null));
	}

	/**
	 * @param links_to_remove Table with links to remove, but only keys - link
	 *  ids matter, values are ignored.
	 */
	function RemoveLinks(links_to_remove)
	{
		foreach (link_id, dummy in links_to_remove) {
			if (link_id in this.links) delete this.links[link_id];
		}

		local nodes_with_out_links = {};
		foreach (link_id, link_info in this.links) {
			nodes_with_out_links[link_info.link.from_id] <- 1;
		}

		local nodes_to_remove = {};
		foreach (level, level_nodes in this.levels) {
			foreach (node_id, links in level_nodes) {
				local to_remove = [];
				foreach (link_id, dummy in links) {
					if (link_id in links_to_remove) to_remove.append(link_id);
				}
				foreach (dummy_id, link_id in to_remove) {
					delete links[link_id];
				}

				if (level > 0 && !(node_id in nodes_with_out_links)) {
					nodes_to_remove[node_id] <- 1;
				}
			}
		}

		if (nodes_to_remove.len() > 0) this.RemoveNodes(nodes_to_remove);
	}

	function RemoveNodes(nodes_to_remove)
	{
		local links_to_remove = {};
		foreach (link_id, link_info in this.links) {
			local from_id = link_info.link.from_id;
			local to_id = link_info.link.to_id;
			if (from_id in nodes_to_remove || to_id in nodes_to_remove) {
				links_to_remove[link_id] <- 1;
			}
		}

		for (local level = 1; level < this.levels.len(); level++) {
			local to_remove = [];
			foreach (node_id, links in this.levels[level]) {
				if (node_id in nodes_to_remove) {
					to_remove.append(node_id);
					/*foreach (link_id, dummy in links) {
						links_to_remove[link_id] <- 1;
					}*/
				}
			}

			foreach (dummy_id, node_id in to_remove) {
				delete this.levels[level][node_id];
			}
		}

		if (links_to_remove.len() > 0) this.RemoveLinks(links_to_remove);
	}

	function Print()
	{
		local global = GlobalTransportMap.Get();
		foreach (level, level_nodes in this.levels) {
			CodeUtils.Log("------------ Level " + level + " ------------", 2);
			foreach (node_id, links in level_nodes) {
				local to = global.GetTransportNodeByID(node_id);
				CodeUtils.Log(" <* " + to.GetName(), 2);
				foreach (link_id, link in links) {
					local from = global.GetTransportNodeByID(link.link.from_id);
					CodeUtils.Log(from.GetName() + " - " + to.GetName(), 2);
				}
				CodeUtils.Log(to.GetName() + " *> ", 2);
			}
		}
	}

/* protected */
	/**
	 * Object's name.
	 */
	name = null;

	/**
	 * Creates a new SupplyContingent object.
	 * @param root_node Transport node that ought to be "root" of the new graph.
	 */
	constructor(root_node)
	{
		if (root_node == null) return;

		this.name = "Supply Contingent for \"" + root_node.GetName() + "\"";
		this.links = {};
		this.levels = [{}];
		this.levels[0][root_node.GetID()] <- {};

		CodeUtils.Log("Creating " + this.name + " ...", 0);

		local node_types = TransportSchema.Get().node_types;
		if (node_types[root_node.node.type_id].raw_production) {
			CodeUtils.Log(name + " is raw production node!", 0);
			CodeUtils.Log("... aborting supply contingent creation.", 0);
			return;
		}

		local tt = AIController.GetTick();
		local global = GlobalTransportMap.Get();

		local available_suppliers = [{node = root_node, level = 0}];
		local paths_info = {};

		/* While stack is not empty we have nodes to check */
		for ( ; available_suppliers.len() > 0;) {
			/* Check them one by one */
			local stack_top = available_suppliers.pop();

			/*
			 * Transport node that can supply(maybe indirectly) the given root
			 * On step 0 this is the root itself obviously.
			 */
			local supplier = stack_top.node;
			local supplier_id = supplier.GetID();
			local supplier_level = stack_top.level;

			/* extend levels container if needed */
			if (this.levels.len() <= supplier_level) this.levels.append({});

			/* If information about supplier was not stored yet */
			if (!(supplier_id in this.levels[supplier_level])) {
				/* Create a record about the supplier */
				this.levels[supplier_level][supplier_id] <- {};
			}

			/* Check links going into the supplier */
			foreach (from_id, link in supplier.links_in) {
				/* If link already encountered than just proceed to next one */
				local link_id = link.GetID();
				if (link_id in this.links) continue;

				this.links[link_id] <- {link = link};
				this.levels[supplier_level][supplier_id][link_id] <- this.links[link_id].weakref();

				/* If link can not lead to self */
				// 'aux' marks links that cause nodes circular dependency
				// by refusing to follow such links we guarantee loop break 
				if (!link.aux) {
					/* Add link source as a next level supplier */
					available_suppliers.push({
							node = global.GetTransportNodeByID(from_id),
							level = supplier_level + 1
					});
				}
			}
		}

		tt = AIController.GetTick() - tt;
		CodeUtils.Log("... " + this.name + " complete, ticks: " + tt, 0);
	}

	function GetNodeLevel(node_id)
	{
		for (local level = this.levels.len() - 1; level >= 0; level--) {
			if (node_id in this.levels[level]) return level;
		}
		return -1;
	}

	function doCloning(dummy_clone)
	{
		dummy_clone.name = this.name;

		dummy_clone.links = {}
		foreach (id, link_info in this.links) {
			dummy_clone.links[id] <- {link = link_info.link};
		}

		dummy_clone.levels = [];
		foreach (level, level_nodes in this.levels) {
			local level_nodes_copy = {};
			foreach (node_id, links in level_nodes) {
				local links_copy = {};
				foreach (link_id, dummy in links) {
					links_copy[link_id] <- dummy_clone.links[link_id].weakref();
				}
				level_nodes_copy[node_id] <- links_copy;
			}
			dummy_clone.levels.append(level_nodes_copy);
		}

		return dummy_clone;
	}
}
