enum LinkStatus
{
	LS_FREE = 0
	LS_MARKED = 1
	LS_ABANDONED = 2
};

enum NodeStatus
{
	NS_RAW = 0
	NS_FREE = 1
	NS_SUPPLY_REQUESTED = 2
	NS_SUPPLIED = 3
	NS_REJECTED = 4
};

enum NodeUtilization
{
	NU_NOT_SET = 0
	NU_ZERO = 1
	NU_PARTIAL = 2
	NU_FULL = 3
};

/**
 * Class that searches and creates supply chain/net for the given node.
 */
class SupplyNetBuilder
{
/* public */
	/**
	 * Find links network that can be used to supply the given node.
	 * @param supply_contingent SupplyContingent class instance - sets up root
	 *  node to feed and all of it's possible feeder nodes as well.
	 * @param supply_lines_wanted Desired (max)amount of different links
	 *  directly feeding root node in the result net.
	 * @param must_ignore_links Table with links to ignore(only keys == link ids
	 *  matters)
	 * @note This is wrapper for ExcreteNet function.
	 * @return Table with pairs [link_id] - [link].
	 */
	/* static */ function ExcreteMaxNet(supply_contingent, supply_lines_wanted, must_ignore_links)
	{
		return SupplyNetBuilder.ExcreteNet(supply_contingent, supply_lines_wanted, must_ignore_links, 0);
	}

/* private */
	/**
	 * Find links network that can be used to supply the given node.
	 * @param supply_contingent SupplyContingent class instance - describes root
	 *  node to feed and all of it's possible feeder nodes as well.
	 * @param supply_lines_wanted Desired (max)amount of different links
	 *  directly feeding root node in the result net.
	 * @param must_ignore_links Table with links to ignore(only keys - link ids
	 *  matters)
	 * @param n Recursion calls depth for this function.
	 * @note Must use ExcreteMaxNet instead of this.
	 * @return Table with pairs [link_id] - [link].
	 */
	/* static */ function ExcreteNet(supply_contingent, supply_lines_wanted, must_ignore_links, n)
	{
		/* This is a recursive function */
		/* And this is recursion stop clause */
		if (supply_lines_wanted <= 0) return {};

		/*
		 * Create a local working copy of the given data to avoid situation
		 *  when different recursive calls mess with each other.
		 */
		local copy = supply_contingent.Clone();
		/* Remove links that should be ignored(if any) */
		copy.RemoveLinks(SupplyNetBuilder.GetLinksExtension(must_ignore_links, copy));

		/*
		 * On first run(zero recursion counter) clean up unproductive nodes.
		 *  "Unproductive" mean nodes without route to raw production nodes,
		 *  thus we'll cut hanging supply tree leaves that do not produce cargo.
		 * This reduces amount of possible graph decisions and speed up further
		 *  calculations without impact on result tree quality.
		 */
		/*
		 * Explicit first call handling can be done outside function body, but
		 *  requires additional supply_contingent.Clone call, which is costly.
		 */
		if (n == 0) {
			local global = GlobalTransportMap.Get();
			while (true) {
				/* here we'll store unproductive nodes */
				local dummy_nodes = {};
				/*
				 * Scan graph nodes starting from lowest level(closest to root)
				 *  and remember nodes without links to next level
				 *  (=> no incoming supplies) as useless.
				 */
				for (local level = 1; level < copy.levels.len(); level++) {
					foreach (node_id, links in copy.levels[level]) {
						local transport_node = global.GetTransportNodeByID(node_id);
						// should handle possible bad data without ai crash
						if (transport_node == null) {
							dummy_nodes[node_id] <- 1;
							continue;
						}

						// nodes producing raw materials are good ones
						if (transport_node.node.type.raw_production) continue;

						// others are bad...
						local is_good_node = false;
						foreach (dummy_from_id, link in transport_node.links_in) {
							// ...unless they have links inside graph
							if (link.GetID() in copy.links) {
								is_good_node = true;
								break;
							}
						}
						if (!is_good_node) dummy_nodes[node_id] <- 1;
					}
				}

				/* Repeat if at least one hanging node was found */
				if (dummy_nodes.len() == 0) break;

				copy.RemoveNodes(dummy_nodes);
			}
		}

		/*
		 * Run a working cycle with the graph(copy) to receive one(or more)
		 *  supply net decisions for it.
		 */
		local actual_extractor = SupplyNetBuilder(copy);
		local result = actual_extractor.doExcreteNet();

		/* If we've found something... */
		if (result.len() > 0) {
			/* Find out how many solution links directly coming into root... */
			local k = supply_lines_wanted;
			foreach (link_id, link in result) {
				if (link.to_id == actual_extractor.root.GetID()) k--;
			}

			/*
			 * ...and run self again with solution links as ignored links
			 * until either
			 * a) requirements for supply_lines_wanted are met
			 * or
			 * b) no further net expansion can be found.
			 */
			local further_result = SupplyNetBuilder.ExcreteNet(copy, k, result, n + 1);

			/* Anyway, add further recursion calls result to initial solution */
			foreach (link_id, link in further_result) {
				result[link_id] <- link;
			}

		}

		/*
		 * I've made this stuff as additional layer of protection
		 *  against memory leaks
		 */
		actual_extractor.CleanUp();

		/*
		 * Solution tree can be improved because additional function calls
		 *  do not "see" earlier results(but just ignore(!) them instead),
		 *  thus we may miss several routes: e.g. leading from
		 *  first call result subtree to second, or vice versa.
		 */
		// (!) alg can be expanded here
		if (n == 0) {
			local additional = SupplyNetBuilder.FindMissedCrossLinks(result);
			foreach (id, item in additional) {
				result[id] <- item;
			}
		}

		return result;
	}

	/**
	 * Improve the given net by adding any missed routes.
	 * @param net Table with pairs [link_id] - [link], describing links net.
	 * @return New table with pairs [link_id] - [link], representing improved
	 *  links net.
	 */
	/* static */ function FindMissedCrossLinks(net)
	{
		local result = {};

		/*
		 * This table will describe nodes supplied with resources, but not fully
		 *  used - such nodes can have out routes to other net nodes
		 *  which(routes) are missed and which we are searching.
		 * Because we'll store ids of all net points here, we also will be using
		 *  it for fast index-based checks to answer question
		 *  "whether node X belongs to the net?".
		 */
		local not_used_production_nodes = {};

		/*
		 * In this first part of the algorithm we must properly find
		 *  not fully used nodes. => next part becomes trivial:
		 * we just need to cycle through all such nodes
		 *  and find best route leading to any other net section for each.
		 */
		foreach (link_id, link in net) {
			// copy net into result
			result[link_id] <- link;

			// create net nodes index
			if (!(link.from_id in not_used_production_nodes)) {
				not_used_production_nodes[link.from_id] <- {};
			}

			local info = not_used_production_nodes[link.from_id];
			foreach (dummy_id, route in link.routes) {
				// on first encounter with next net node
				if (info.len() == 0) {
					// write down all known output cargo ids for this node type
					foreach (c, dummy in route.GetStart().type.production) {
						if (!(c in info)) info[c] <- 1;
					}
				}

				/*
				 * Erase records about all cargo ids that are being exported
				 *  from the node within the given net.
				 * Thus cargo ids records that will not be erased are
				 *  "spare"(or missed) production we are looking for.
				 */
				local c = route.GetCargoID();
				if (c in info) delete info[c];
			}
		}

		/*
		 * Now trivial second part: cycling through all nodes with spare
		 *  production records and adding links providing such production into
		 *  the the given net. 
		 */
		local global = GlobalTransportMap.Get();
		foreach (node_id, cargo_ids in not_used_production_nodes) {
			if (cargo_ids.len() == 0) continue;

			local transport_node = global.GetTransportNodeByID(node_id);
			if (transport_node == null) continue;

			foreach (c, dummy in cargo_ids) {
				// we need just one link per cargo, let it be the shortest one
				local best_link = null;
				local best_length = 10000;
				foreach (link_id, link in transport_node.links_out) {
					if (!(link.to_id in not_used_production_nodes)) continue;

					foreach (dummy_route_id, route in link.routes) {
						if (route.GetCargoID() == c && route.length < best_length) {
							best_length = route.length;
							best_link = link;
						}
					}
				}

				if (best_link != null) result[best_link.GetID()] <- best_link;
			}
		}

		return result;
	}

/* private */
	/**
	 * The transport node to supply.
	 * Entire purpose of this class is to find/create supply network for this
	 *  particular node.
	 */
	root = null;

	/**
	 * Structure that should reflect algorithm progress up to current moment.
	 */
	workspace = null;

	/**
	 * Actual algorithm: it does one subnet extraction at time.
	 * No params, all should be set up in the constructor.
	 * @return Table with pairs [link_id] - [link] representing found subnet(or
	 *  empty table if no solution was found).
	 */
	function doExcreteNet()
	{
		/*
		 * Here we'll store decisions of the greedy link picking algorithm.
		 */
		local choosen_stack = Terron_PriorityQueue();

		/* can happen if root is closed industry */
		if (this.root == null) return {};
		if (this.root.node.is_closed) return {};

		for (local steps = 0, root_id = this.root.GetID(); ; steps--) {
			// root should always be free for additional supplying
			this.SetNodeStatus(root_id, NodeStatus.NS_FREE);

			/*
			 * Greedy pick one of the currently available links.
			 * Initially this is only links leaving from raw production
			 *  industries, but with the resulting decision state progress
			 *  a new possibilities will pop up.
			 */
			local choosen_link = this.ChooseLink();
			// if we've found a good link
			if (choosen_link.best_link != null) {
				// remember it as algorithm step
				choosen_stack.Insert(choosen_link, steps);
				// renaming for convinience
				choosen_link = choosen_link.best_link;

				//AILog.Warning("Link Choosen");
				/*foreach (dummy_id, route in choosen_link.routes) {
					AILog.Info("     : " + route.GetName());
				}*/

				/*
				 * Modify algorithm's links/nodes database to reflect the
				 *  choice made and its influence on the set of possible further
				 *  decisions.
				 */
				this.SetProperNodeUtilization(choosen_link.from_id);
				this.ModifyCandidates(choosen_link);

				// if link is leading into root
				if (choosen_link.to_id == root.GetID()) {
					/*
					 * Push all decision links from "choosen_stack" into
					 *  more convinient table structure
					 */
					local pairs = {};
					while (choosen_stack.Count() > 0) {
						local link = choosen_stack.Pop().best_link;
						pairs[link.from_id] <- {to_id = link.to_id, link = link};
					}

					/*
					 * Get rid of the routes leading "outside" of decision set.
					 * Put the rest into new "initial_result" table.
					 */
					local initial_result = {};
					foreach (dummy, link_info in pairs) {
						local to_id = link_info.to_id;
						if (to_id != root_id && !(to_id in pairs)) continue;
						local link = link_info.link;
						initial_result[link.GetID()] <- link;
					}

					/*
					 * Now we've found what can be called "decision skeleton".
					 * This is valid tree, but in practice it usually does not
					 *  have enough cargo produced inside to maintain constant
					 *  flow. Vehicles will stuck at higher tier industries
					 *  waiting for sparse raw resources to be delivered and
					 *  processed.
					 * Thus before returning we push several additional branches
					 *  into our decision tree/net. New links maybe not as
					 *  good from cost/income perspective as initial ones,
					 *  but they provide(or at least should provide in theory)
					 *  constant cargo flow for stable and nice looking system.
					 */
					return this.EnrichNet(
						MagicNumbers.supply_net_target_raw_production_sum,
						initial_result
					);
				}
			} else if (!choosen_stack.IsEmpty()) {
				/*
				 * If we didn't found a good link to add into decision set,
				 *  but we have links selected before => abandon one of them
				 *  Thus alg will try to search alternative branch, maybe it
				 *   will lead to root unlike the currently choosen links set.
				 */
				 this.StepBack(choosen_stack.Pop());
			} else {
				/*
				 * If we didn't found a good link to add into decision set
				 *  and decision set is empty => stop, alg failed
				 */
				return {};
			}
		}

		// should not be here
		assert(null);
	}

	/**
	 * Memory deallocation.
	 * @note Even though this is garbage collector's work, it behaves strange
	 *  and there are enough forum posts about memory leaks, thus i try to
	 *  manually free memory in certain "suspective" cases.
	 */
	function CleanUp()
	{
		this.root = null;
		foreach (node_id, node_info in this.workspace.all_nodes) {
			node_info.links_in.clear();
			node_info.links_out.clear();
		}
		this.workspace.all_nodes.clear();
	}

	function DeleteNodeInfo(node_id)
	{
		local global = GlobalTransportMap.Get();
		CodeUtils.Log("Erasing node: " + global.GetTransportNodeByID(node_id), 0);
		local links_in = this.workspace.all_nodes[node_id].links_in;
		foreach (link_id, link_info in links_in) {
			local from_info = this.workspace.all_nodes[link_info.link.from_id];
			from_info.links_out.rawdelete(link_id);
		}

		local links_out = this.workspace.all_nodes[node_id].links_out;
		foreach (link_id, link_info in links_out) {
			local to_info = this.workspace.all_nodes[link_info.link.to_id];
			to_info.links_in.rawdelete(link_id);
		}

		this.workspace.all_nodes[node_id].links_in.clear();
		this.workspace.all_nodes[node_id].links_out.clear();
		this.workspace.all_nodes[node_id].clear();

		foreach (dummy_id, sorted_nodes in this.workspace.sorted_nodes) {
			if (node_id in sorted_nodes) delete sorted_nodes[node_id];
		}

		delete this.workspace.all_nodes[node_id];
	}

/* private */
	/**
	 * This function represents "virtual" run of the greedy route
	 *  choosing algorithm. It takes current network state coded in the instance
	 *  fields and returns a link with best cost/income ratio. But it should be
	 *  noted that function chooses links with respect to specific order which
	 *  in theory must enforce correct tree-like structure of the final result.
	 */
	function ChooseLink()
	{
		local sorted_nodes = this.workspace.sorted_nodes;

		/* Selection order is quite simple actually */
		/*
		 * If there is a node which need supplying => make a choice from pool
		 *  of links leading into such node(s).
		 */
		if (sorted_nodes[NodeStatus.NS_SUPPLY_REQUESTED].len() > 0) {
			return {
				best_link = this.ChooseLink_MustSupplyDirective(),
				direction = -1,
			}
		}

		/*
		 * And if there is not a single node which need supplying => choose
		 * "best cost/income" link exiting from
		 *  a) already supplied(and not raw production) nodes
		 *  b) which are currently closest to root
		 *
		 * In other words: must select link exiting from the lowest level node
		 *  with NodeStatus.NS_SUPPLIED current status.
		 */
		if (sorted_nodes[NodeStatus.NS_SUPPLIED].len() > 0) {
			foreach (node_id, node_info in sorted_nodes[NodeStatus.NS_SUPPLIED]) {
				if (node_info.utilization == NodeUtilization.NU_FULL) continue;
				if (node_info.current_status == NodeStatus.NS_REJECTED) continue;
				return {
					best_link = this.ChooseLink_ForwardSupplyDirective(),
					direction = 1
				};
			}
		}

		/*
		 * Neither supplied nor need supplying nodes?
		 * Than just choose best link exiting from raw producer.
		 */
		return {best_link = ChooseLink_FreeSearchDirective(), direction = 1};
	}

	function StepBack(last_step_info)
	{
		local dir = last_step_info.direction;
		local choosen_link = last_step_info.best_link;
		local step_back_link_id = choosen_link.GetID();
		/*AILog.Warning("Link abandoned");
		foreach (dummy_id, route in choosen_link.routes) {
			AILog.Info("     : " + route.GetName());
		}*/

		if (!(choosen_link.from_id in this.workspace.all_nodes)) {
			if (choosen_link.to_id in this.workspace.all_nodes) {
				local to_info = this.workspace.all_nodes[choosen_link.to_id];
				if (step_back_link_id in to_info.links_in) {
					to_info.links_in[step_back_link_id].link_status = LinkStatus.LS_ABANDONED;
				}
			}
			return;
		}

		if (!(choosen_link.to_id in this.workspace.all_nodes)) {
			if (choosen_link.from_id in this.workspace.all_nodes) {
				local from_info = this.workspace.all_nodes[choosen_link.from_id];
				if (step_back_link_id in from_info.links_out) {
					from_info.links_out[step_back_link_id].link_status = LinkStatus.LS_ABANDONED;
				}
			}
			return;
		}

		local from_info = this.workspace.all_nodes[choosen_link.from_id];
		local to_info = this.workspace.all_nodes[choosen_link.to_id];
		local global = GlobalTransportMap.Get();
		local from = global.GetTransportNodeByID(choosen_link.from_id);
		local to = global.GetTransportNodeByID(choosen_link.to_id);
		to_info.links_in[step_back_link_id].link_status = LinkStatus.LS_ABANDONED;

		local must_clean_up = this.MustCleanUp(last_step_info);
		if (must_clean_up) {
			//AILog.Warning("*********MUST CLEAN UP**********");
			foreach (node_id, node_info in this.workspace.all_nodes) {
				if (node_info.level > to_info.level || node_info.level == 0) continue;
				local xnode = global.GetTransportNodeByID(node_id);
				foreach (dummy_id, link_info in node_info.links_out) {
					/*if (link_info.link_status != LinkStatus.LS_FREE) {
						AILog.Warning("Link cleansed");
						foreach (dummy_id, route in link_info.link.routes) {
							AILog.Info("     : " + route.GetName());
						}
					}*/
					link_info.link_status = LinkStatus.LS_FREE;
				}
				if (node_info.current_status != NodeStatus.NS_RAW) {
					this.SetNodeStatus(node_id, NodeStatus.NS_FREE);
				}
			}

			to_info.links_in[choosen_link.GetID()].link_status = LinkStatus.LS_ABANDONED;
			delete this.workspace.all_nodes[choosen_link.from_id].links_out[choosen_link.GetID()];
			delete this.workspace.all_nodes[choosen_link.to_id].links_in[choosen_link.GetID()];

			local nodes_to_delete = [];
			foreach (node_id, node_info in this.workspace.all_nodes) {
				if (node_info.level > to_info.level || node_info.level == 0) continue;

				local must_not_delete = false;
				foreach (link_id, link_info in node_info.links_in) {
					if (link_info.link_status == LinkStatus.LS_MARKED) {
						must_not_delete = true;
						break;
					}
				}
				if (node_info.links_out.len() == 0 && !must_not_delete) {
					nodes_to_delete.append(node_id);
				}
			}

			foreach (dummy_id, node_id in nodes_to_delete) {
				this.DeleteNodeInfo(node_id);
			}
		}

		local reject_from = true;
		local reject_to = true;
		if (from_info.len() != 0){
			foreach (dummy_id, link_info in from_info.links_out) {
				if (link_info.link_status != LinkStatus.LS_ABANDONED) {
					reject_from = false;
					break;
				}
			}
			if (reject_from) {
				//AILog.Info("REJECT 'FROM' : " + from.GetName());
				this.SetNodeStatus(from.GetID(), NodeStatus.NS_REJECTED);
			} else {
				this.SetProperNodeUtilization(from.GetID());
				if (dir == -1 && from_info.current_status != NodeStatus.NS_RAW) {
					//AILog.Info("REJECT 'FROM' DUE TO -1: " + from.GetName());
					local delivered_cargo_ids = this.GetDeliveredCargoIDs(from.GetID());
					local n = from.node.type.consumption.len() > 1 ? 2 : 1;
					local t = from_info.total_cargo_ids_accepted;
					local k = delivered_cargo_ids.len();
					if (n > t && t == 1) {
						k = 0;
						foreach (c, amount in delivered_cargo_ids) {
							if (c in TransportSchema.Get().production_supply_cargo_ids) continue;
							k += amount;
						}
					}
					//AILog.Warning("k == " + k + ", n == " + n)
					local new_status = k >= n ?	NodeStatus.NS_SUPPLIED : NodeStatus.NS_FREE;
					this.SetNodeStatus(from.GetID(), new_status);
					//this.SetNodeStatus(from.GetID(), NodeStatus.NS_REJECTED);
					return
				}
			}
		}

		if (must_clean_up) return;

		foreach (dummy_id, link_info in to_info.links_out) {
			if (link_info.link_status != LinkStatus.LS_ABANDONED) {
				if (this.workspace.all_nodes[link_info.link.to_id].current_status == NodeStatus.NS_REJECTED) continue;
				//AILog.Info("Good link to " + global.GetTransportNodeByID(link_info.link.to_id).GetName());
				reject_to = false;
				break;
			}
		}

		if (reject_to) {
			//AILog.Info("REJECT 'TO' : " + to.GetName());
			this.SetNodeStatus(to.GetID(), NodeStatus.NS_REJECTED);
		} else {
			//AILog.Info("MUST NOT REJECT 'TO' : " + to.GetName());
			local delivered_cargo_ids = this.GetDeliveredCargoIDs(to.GetID());
			local n = to.node.type.consumption.len() > 1 ? 2 : 1;
			local t = to_info.total_cargo_ids_accepted;
			local k = delivered_cargo_ids.len();
			if (n > t && t == 1) {
				k = 0;
				foreach (c, amount in delivered_cargo_ids) {
					if (c in TransportSchema.Get().production_supply_cargo_ids) continue;
					k += amount;
				}
			}
			//AILog.Warning("k == " + k + ", n == " + n)
			local new_status = k >= n ?	NodeStatus.NS_SUPPLIED : NodeStatus.NS_SUPPLY_REQUESTED;
			this.SetNodeStatus(to.GetID(), new_status);
		}
	}

	function MustCleanUp(last_step_info)
	{
		if (last_step_info.direction == -1) {
			//AILog.Warning("*********MUST CLEAN CONDITION FALSE DUE TO -1**********");
			return false;
		}

		//local from_info = this.workspace.all_nodes[choosen_link.from_id];
		local global = GlobalTransportMap.Get();
		//local from = global.GetTransportNodeByID(choosen_link.from_id);

		local last_choosen_link = last_step_info.best_link;
		local to_info = this.workspace.all_nodes[last_choosen_link.to_id];
		local stss = [NodeStatus.NS_SUPPLY_REQUESTED, NodeStatus.NS_SUPPLIED];
		foreach (id, status in stss) {
			foreach (node_id, node_info in this.workspace.sorted_nodes[status]) {
				if (node_id == last_choosen_link.to_id) continue;
				if (node_info.utilization == NodeUtilization.NU_NOT_SET) continue;
				if (node_info.level <= to_info.level) {
					local xnode = global.GetTransportNodeByID(node_id);
					local to = global.GetTransportNodeByID(last_choosen_link.to_id);
					//AILog.Warning("*********" + xnode.GetName() + " has > level than " + to.GetName() + " **********");
					return false;
				}
			}
		}

		//AILog.Warning("*********MUST CLEAN CONDITION 0**********");
		foreach (dummy_id, link_info in to_info.links_out) {
			if (link_info.link_status == LinkStatus.LS_MARKED) {
				//AILog.Warning("*********HAS GOOD OUT LINK**********");
				return false;
			}
		}

		//AILog.Warning("*********MUST CLEAN CONDITION 1**********");
		foreach (link_id, link_info in to_info.links_in) {
			if (last_choosen_link.GetID() == link_id) continue;
			if (link_info.link_status == LinkStatus.LS_MARKED) {
				//AILog.Warning("*********NOT THE LAST LINK INCOMING THIS LEVEL**********");
				return false;
			}
		}

		return true;
	}

	/**
	 * Safely change the status of the given node.
	 * @param node_id ID of the network's node which status must be changed.
	 * @param new_status Desired node's status.
	 * @return Not used.
	 */
	function SetNodeStatus(node_id, new_status)
	{
		local node_info = this.workspace.all_nodes[node_id];
		if (new_status == node_info.current_status) return;

		/* Relocate the node info into the status based container */
		this.workspace.sorted_nodes[node_info.current_status].rawdelete(node_id);
		this.workspace.sorted_nodes[new_status][node_id] <- node_info;

		/* Set new status */
		this.workspace.all_nodes[node_id].current_status = new_status;
	}

	function ModifyCandidates(choosen_link)
	{
		/*AILog.Warning("MODIFY");
		foreach (dummy_id, route in choosen_link.routes) {
			AILog.Info("     : " + route.GetName());
		}*/

		local to_id = choosen_link.to_id;
		local to_info = this.workspace.all_nodes[to_id];
		local node_to = GlobalTransportMap.Get().GetTransportNodeByID(to_id);

		to_info.links_in[choosen_link.GetID()].link_status = LinkStatus.LS_MARKED;

		if (to_info.current_status != NodeStatus.NS_SUPPLY_REQUESTED) {
			local delivered_cargo_ids = this.GetDeliveredCargoIDs(to_id);
			local n = node_to.node.type.consumption.len() > 1 ? 2 : 1;
			local t = to_info.total_cargo_ids_accepted;
			local k = delivered_cargo_ids.len();
			if (n > t && t == 1) {
				k = 0;
				foreach (c, amount in delivered_cargo_ids) {
					if (c in TransportSchema.Get().production_supply_cargo_ids) continue;
					k += amount;
				}
			}
			//AILog.Warning("k == " + k + ", n == " + n)
			local new_status = k >= n ?	NodeStatus.NS_SUPPLIED : NodeStatus.NS_SUPPLY_REQUESTED;
			this.SetNodeStatus(to_id, new_status);
			return;
		}

		local from_id = choosen_link.from_id;
		local from_status = this.workspace.all_nodes[from_id].current_status;
		if (from_status != NodeStatus.NS_RAW && from_status != NodeStatus.NS_SUPPLIED) {
			this.SetNodeStatus(from_id, NodeStatus.NS_SUPPLY_REQUESTED);
			return;
		}

		local n = node_to.node.type.consumption.len() > 1 ? 2 : 1;
		n = min(n, to_info.total_cargo_ids_accepted);
		local delivered = this.GetDeliveredCargoIDs(to_id);
		if (n > delivered.len()) return;
		if (delivered.len() == 1) {
			if (n == 2) return;
			foreach (c, dummy in delivered) {
				if (c in TransportSchema.Get().production_supply_cargo_ids) return;
			}
		}

		this.SetNodeStatus(to_id, NodeStatus.NS_SUPPLIED);

		foreach (link_id, link_info in to_info.links_out) {
			if (link_info.link_status == LinkStatus.LS_MARKED) {
				this.ModifyCandidates(link_info.link);
				//break;
			}
		}
	}

	function ChooseLink_MustSupplyDirective()
	{
		//AILog.Warning("!1");
		local nodes = this.workspace.sorted_nodes[NodeStatus.NS_SUPPLY_REQUESTED];

		local max_level = 0;
		foreach (node_id, node_info in nodes) {
			if (node_info.level > max_level) max_level = node_info.level;
		}

		local best_link = null;
		local best_link_value = 0;
		local global = GlobalTransportMap.Get();
		foreach (node_id, node_info in nodes) {
			if (node_info.level != max_level) continue;

			local xnode = global.GetTransportNodeByID(node_id);
			local delivered_cargo_ids = this.GetDeliveredCargoIDs(node_id);
			//AILog.Warning("!must supply " + xnode.GetName());
			foreach (link_id, link_info in node_info.links_in) {
				if (link_info.value <= best_link_value) continue;
				if (link_info.link_status != LinkStatus.LS_FREE) continue;

				local from_info = this.workspace.all_nodes[link_info.link.from_id];
				if (from_info.utilization == NodeUtilization.NU_FULL) continue;
				if (from_info.current_status == NodeStatus.NS_REJECTED) continue;

				foreach (dummy_route_id, route in link_info.link.routes) {
					if (route.GetCargoID() in delivered_cargo_ids) {
						if (node_info.total_cargo_ids_accepted > 1) continue;
					}

					best_link = link_info.link;
					best_link_value = link_info.value;
					break;
				}
			}
		}

		return best_link;
	}

	function ChooseLink_ForwardSupplyDirective()
	{
		//AILog.Warning("!2");
		local nodes = this.workspace.sorted_nodes[NodeStatus.NS_SUPPLIED];

		local min_level = 1024;
		foreach (node_id, node_info in nodes) {
			if (node_info.level < min_level) min_level = node_info.level;
		}

		local best_link = null;
		local best_link_value = 0;
		local global = GlobalTransportMap.Get();
		foreach (node_id, node_info in this.workspace.sorted_nodes[NodeStatus.NS_REJECTED]) {
			local xnode = global.GetTransportNodeByID(node_id);
			//AILog.Warning("!rejected " + xnode.GetName());
		}

		//AILog.Warning("!level " + min_level);
		foreach (node_id, node_info in nodes) {
			if (node_info.level != min_level) continue;

			local xnode = global.GetTransportNodeByID(node_id);
			//AILog.Warning("!check " + xnode.GetName());

			if (node_info.utilization == NodeUtilization.NU_FULL) continue;
			//AILog.Warning("marked: " + node_info.utilization);

			//AILog.Warning("!supplied " + xnode.GetName());
			local already_trade_away_cargo_ids = node_info.cargo_ids_out;

			foreach (link_id, link_info in node_info.links_out) {
				if (link_info.value <= best_link_value) continue;
				//AILog.Warning("!");
				//AILog.Warning("!" + link_info.link_status);
				if (link_info.link_status != LinkStatus.LS_FREE) continue;
				//AILog.Warning("!");

				local to_info = this.workspace.all_nodes[link_info.link.to_id];
				if (link_info.link.aux && to_info.level >= node_info.level) continue;
				if (to_info.utilization != NodeUtilization.NU_NOT_SET) continue;
				if (to_info.current_status == NodeStatus.NS_REJECTED) continue;

				foreach (dummy_route_id, route in link_info.link.routes) {
					if (route.GetCargoID() in already_trade_away_cargo_ids) continue;
					best_link = link_info.link;
					best_link_value = link_info.value;
					break;
				}
			}
		}

		//AILog.Warning("!result id: " + (best_link == null ? "null" : best_link.GetID()));
		return best_link;
	}

	function ChooseLink_FreeSearchDirective()
	{
		//AILog.Warning("!3");
		local nodes = this.workspace.sorted_nodes[NodeStatus.NS_RAW];

		local best_link = null;
		local best_link_value = 0;
		local global = GlobalTransportMap.Get();
		foreach (node_id, node_info in nodes) {
			local xnode = global.GetTransportNodeByID(node_id);
			//AILog.Warning(xnode.GetName());
			foreach (link_id, link_info in node_info.links_out) {
				local ynode = global.GetTransportNodeByID(link_info.link.to_id);
				//AILog.Warning("         " + ynode.GetName() + ": " + link_info.value);
				if (link_info.value <= best_link_value) continue;
				if (link_info.link_status != LinkStatus.LS_FREE) continue;

				local to_info = this.workspace.all_nodes[link_info.link.to_id];
				if (link_info.link.aux && to_info.level >= node_info.level) continue;
				if (to_info.utilization != NodeUtilization.NU_NOT_SET) continue;
				if (to_info.current_status == NodeStatus.NS_REJECTED) continue;

				best_link = link_info.link;
				best_link_value = link_info.value;
			}
		}

		return best_link;
	}

	function GetDeliveredCargoIDs(node_id)
	{
		local result = {};

		local node_info = this.workspace.all_nodes[node_id];
		foreach (link_id, link_info in node_info.links_in) {
			if (link_info.link_status != LinkStatus.LS_MARKED) continue;
			foreach (dummy_route_id, route in link_info.link.routes) {
				local c = route.GetCargoID();
				if (!(c in result)) result[c] <- 0;
				result[c] = result[c] + 1;
			}
		}

		return result;
	}

	function SetProperNodeUtilization(node_id)
	{
		local node_info = this.workspace.all_nodes[node_id];

		node_info.cargo_ids_out.clear();

		local net_cargo_ids = {};
		foreach (link_id, link_info in node_info.links_out) {
			if (net_cargo_ids.len() < node_info.total_cargo_ids_produced) {
				foreach (dummy_route_id, route in link_info.link.routes) {
					net_cargo_ids[route.GetCargoID()] <- 1;
				}
			}
			if (link_info.link_status == LinkStatus.LS_MARKED) {
				foreach (dummy_route_id, route in link_info.link.routes) {
					node_info.cargo_ids_out[route.GetCargoID()] <- 1;
				}
			}
		}

		local n = net_cargo_ids.len(), k = node_info.cargo_ids_out.len();
		node_info.utilization = k == 0 ? NodeUtilization.NU_ZERO :
			(k >= n ? NodeUtilization.NU_FULL : NodeUtilization.NU_PARTIAL);
	}

	static function GetLinksExtension(base_links_set, supply_contingent)
	{
		/* must test effect */
		//return base_links_set;

		local result = {};

		foreach (base_link_id, dummy in base_links_set) {
			if (!(base_link_id in supply_contingent.links)) continue;

			local base_link = supply_contingent.links[base_link_id].link;
			local cargo_ids = {};
			foreach (dummy_route_id, route in base_link.routes) {
				cargo_ids[route.GetCargoID()] <- 1;
			}

			foreach (link_id, link_info in supply_contingent.links) {
				if (link_info.link.from_id != base_link.from_id) continue;
				foreach (dummy_route_id, route in link_info.link.routes) {
					if (route.GetCargoID() in cargo_ids) result[link_id] <- 1;
				}
			}
		}

		return result;
	}

	function GetIncomingProductionRate(node_id, nodes)
	{
		local result = 0;
		local node_info = nodes[node_id];
		foreach (link_id, link_info in node_info.links_in) {
			if (link_info.link_status != LinkStatus.LS_MARKED) continue;

			local from_info = nodes[link_info.link.from_id];
			if (from_info.current_status == NodeStatus.NS_RAW) {
				foreach (dummy_id, route in link_info.link.routes) {
					result += route.GetFreeProduction();
				}
				continue;
			}

			result += GetIncomingProductionRate(link_info.link.from_id, nodes);
		}

		return result;
	}

	function EnrichNet(desired_transport_rate, initial_net_links)
	{
		local nodes = this.workspace.all_nodes;
		local to_enrich = {};
		foreach (link_id, link in initial_net_links) {
			if (nodes[link.from_id].current_status == NodeStatus.NS_RAW) {
				to_enrich[link.to_id] <- nodes[link.to_id];
			}
		}

		local result = {};
		foreach (link_id, link in initial_net_links) {
			result[link_id] <- link;
		}

		while (true) {
			local rich_nodes = [];
			foreach (node_id, node_info in to_enrich) {
				if (this.GetIncomingProductionRate(node_id, nodes) > desired_transport_rate) {
					rich_nodes.append(node_id);
				}
			}
			foreach (dummy_id, rich_node_id in rich_nodes) {
				delete to_enrich[rich_node_id];
			}

			local best_link_info = null;
			local best_link_value = 0;
			foreach (node_id, node_info in to_enrich) {
				foreach (link_id, link_info in node_info.links_in) {
					if (link_info.link_status != LinkStatus.LS_FREE) continue;
					if (link_info.value > best_link_value) {
						best_link_info = link_info;
						best_link_value = link_info.value;
					}
				}
			}

			if (best_link_info == null) break;
			best_link_info.link_status = LinkStatus.LS_MARKED;
			result[best_link_info.link.GetID()] <- best_link_info.link;
		}

		/*AILog.Info("Enriched Net Now: ");
		foreach (link_id, link in result) {
			foreach (dummy_id, route in link.routes) {
				AILog.Info("#####: " + route.GetName());
			}
		}*/
		return result;
	}
}

/**
 * Initializes inner data structures(zero step) for the supply
 *  chain search algorithm.
 * @param modified_supply_contingent SupplyContingent class instance which
 *  describes net to "solve" - find good enough subnet to supply the root.
 */
function SupplyNetBuilder::constructor(modified_supply_contingent)
{
	/* Proper initial(zero step) alg setup is all this constructor does */

	/* Let's remember root node for fast access */
	local global = GlobalTransportMap.Get();
	foreach (node_id, dummy in modified_supply_contingent.levels[0]) {
		// there should be only one level 0 node - "root" node
		this.root = global.GetTransportNodeByID(node_id);
		break;
	}

	/* Algorithm's state description structure */
	this.workspace = {
		// information about all nodes within the given net we can work with
		// indexed by node id
		// further description below
		all_nodes = {},
		/*
		 * Here we store nodes ordered by their current "state" within alg,
		 *  e.g raw production node first will be in ***[NodeStatus.NS_RAW]
		 *  container, and if we find out that it is useless we can move it
		 *  into ***[NodeStatus.NS_REJECTED] and do not bother anymore.
		 * Array is used instead of flag field description because it allows
		 *  fast access to desired nodes without checking other ones.
		 */
		sorted_nodes = [{}, {}, {}, {}, {}],
	};

	local all_links = modified_supply_contingent.links;

	local n = modified_supply_contingent.levels.len();
	for (local level = 0; level < n; level++) {
		local level_nodes = modified_supply_contingent.levels[level];
		foreach (node_id, links in level_nodes) {
			local transport_node = global.GetTransportNodeByID(node_id);
			if (transport_node == null) return null;
			if (node_id in this.workspace.all_nodes) continue;

			// fill in info about each node
			this.workspace.all_nodes[node_id] <- {
				// node's "level" within structure describing the given net
				level = modified_supply_contingent.GetNodeLevel(node_id),

				// unlike node's own links_in/out table(which
				// contain ALL node's links) this ones
				// will hold info only about links WITHIN the given net
				links_in = {},
				links_out = {},

				// table with cargo ids(keys, values do not matter) produced
				// by the node which can be transported to other net's nodes
				cargo_ids_out = {},

				// number of different cargo ids produced/accepted
				// by the node WITHIN the given net
				total_cargo_ids_accepted = 0,
				total_cargo_ids_produced = 0,

				// node's current status and "usage" -
				// see NodeStatus and NodeUtilization enums description
				// as well as alg description 
				current_status = transport_node.node.type.raw_production ?
					NodeStatus.NS_RAW : NodeStatus.NS_FREE,
				utilization = NodeUtilization.NU_NOT_SET,
			}

			// actually write down info about incoming links
			local links_in = this.workspace.all_nodes[node_id].links_in;
			foreach (dummy_from_id, link in transport_node.links_in) {
				local id = link.GetID();
				if (id in all_links) {
					this.workspace.all_nodes[node_id].links_in[id] <- all_links[id];
					this.workspace.all_nodes[node_id].links_in[id].link_status <- LinkStatus.LS_FREE;
				}
			}
		}
	}

	// two things here:
	foreach (node_id, info in this.workspace.all_nodes) {
		/*
		 * First: correctly fill "sorted_nodes" array by putting ref
		 *  to each node's info into corresponding cell
		 */
		this.workspace.sorted_nodes[info.current_status][node_id] <- info.weakref();

		/*
		 * Second: correctly fill "links_out" table by referencing it as
		 *  corresponding "links_in" table (based on fact that
		 *  "in" for the end is the same as "out" for the start)
		 */
		foreach (link_id, link_info in info.links_in) {
			local from_id = link_info.link.from_id;
			this.workspace.all_nodes[from_id].links_out[link_id] <- link_info;
		}
	}

	// finnaly account amount of different cargo ids produced/accepted by
	//  each node within the given net
	foreach (node_id, info in this.workspace.all_nodes) {
		local cargo_ids = {};
		foreach (link_id, link_info in info.links_out) {
			foreach (dummy_id, route in link_info.link.routes) {
				local c = route.GetCargoID();
				if (!(c in cargo_ids)) cargo_ids[c] <- 1;
			}
		}
		this.workspace.all_nodes[node_id].total_cargo_ids_produced = cargo_ids.len();

		cargo_ids.clear();
		foreach (link_id, link_info in info.links_in) {
			foreach (dummy_id, route in link_info.link.routes) {
				local c = route.GetCargoID();
				if (!(c in cargo_ids)) cargo_ids[c] <- 1;
			}
		}
		this.workspace.all_nodes[node_id].total_cargo_ids_accepted = cargo_ids.len();
	}

	//modified_supply_contingent.Print()
}
