/**
 * Class that describes AI corporation "air department".
 */
class AirDepartment extends AbstractTransportSystem
{
/* public */
	static function GetClassName()
	{
		return "AirDepartment";
	}

	/**
	 * Get the AirDepartment.
	 */
	static function Get()
	{
		if (AirDepartment.instance[0] == null) AirDepartment();
		return AirDepartment.instance[0];
	}

	static function Restore(memento)
	{
		local result = AirDepartment();
		if ("progress_sign_id" in memento) {
			result.progress_sign_id = memento.progress_sign_id;
		}
		return result;
	}

	function GetName()
	{
		return "Airlines section";
	}

	/**
	 * Check whether air department is in passive mode now.
	 * @return True <=> air department is in "passive" mode.
	 */
	function IsPassive()
	{
		return this.is_in_passive_mode.Get();
	}
/* protected */

	function onDisable(time)
	{
		foreach (id, sys in this.GetSubsystems()) {
			sys.Disable(time);
		}
	}

/* private */
	/** The single instance of the class AirSettings. */
	static instance = [null];

	/** True <=> air department is in "passive" mode. */
	is_in_passive_mode = true;

	/** Action object that handles airports replacement. */
	replace_airport_action = null;

	/** ID of the sign with air progress (will be near HQ) */
	</ must_save = true />
	progress_sign_id = -1;

	/** How many times this department can requst highest priority. */
	exclusive_control_requests = 0;

	/**
	 * Creates the AirDepartment.
	 */
	constructor()
	{
		::AbstractTransportSystem.constructor();

		if (AirDepartment.instance[0] != null) assert(null); 
		AirDepartment.instance[0] = this;

		this.is_in_passive_mode = Property(this.IsPassive_Check.bindenv(this), GameTime.YEAR);
 
		if (AirSettings.Get().planes_limit != 0) {
			this.exclusive_control_requests = 3;
			this.AddSubsystem(AirHubsControl());

			if (!SaveLoadUtils.IsLoading()) {
				this.replace_airport_action = ReplaceAirportAction();
			} else {
				AIStartedEvent.AddListener(this);
			}
		}
	}

	/**
	 * Place a sign with air progress near HQ.
	 * @param air_progress Float between 0 and 1 - percentage.
	 */
	function ShowProgress(air_progress)
	{
		if (this.progress_sign_id != -1) {
			AISign.RemoveSign(this.progress_sign_id);
		}
		if (AirSettings.Get().planes_limit == 0) return;

		local t = AICompany.GetCompanyHQ(AICompany.COMPANY_SELF);
		if (AIMap.IsValidTile(t)) {
			t += AIMap.GetTileIndex(1, 1);
			local p = (air_progress * 100).tointeger();
			local text = air_progress >= 1.0 ? "Airports construction complete!" :
				"Air expansion: " + p + "%";
			local id = AISign.BuildSign(t, text);
			if (AISign.IsValidSign(id)) this.progress_sign_id = id
		}
	}

	/**
	 * Checks if air department must become passive.
	 * @return True if and only if the department must not build new airports.
	 */
	function IsPassive_Check()
	{
		local v_list = AIVehicleList();
		local n = v_list.Count();

		v_list.Valuate(AIVehicle.GetVehicleType);
		v_list.KeepValue(AIVehicle.VT_AIR);

		local planes_total = v_list.Count();
		/*if (planes_total == 0 && CorporationUtils.month_income.value > 30000) {
			if (this.exclusive_control_requests > 0 &&
				AirSettings.Get().GetBestAirportType() != AIAirport.AT_INVALID) {
				this.exclusive_control_requests--;
				DepartmentRequestsPriorityEvent.Fire(this);
				return false;
			}
		}*/

		if (n < AirConstants.Get().min_vehicles_to_enable_planes) {
			this.ShowProgress(0);
			return true;
		}

		/* Disable if AI already has too many planes */
		local p = 1.0 * planes_total / (AirSettings.Get().planes_limit + 1);
		this.ShowProgress(p);
		if (p >= 0.5) AirSettings.Get().greedy_strategy = false;
		return (p >= 1) ? true : false;
	}

	/**
	 * Air department reaction on AI start.
	 */
	function OnAIStarted(is_game_loaded)
	{
		if (this.replace_airport_action == null) {
			this.replace_airport_action = ReplaceAirportAction();
		}
	}
}

Terron_ClassTable.RegisterClass(AirDepartment);
