/**
 * Class that handles air routes.
 * @note Only for towns.
 */
class AirRoute extends SimpleRoute
{
/* public */
	static function GetClassName()
	{
		return "AirRoute";
	}

	/**
	 * Creates a new air route. 
	 * @param start_node first town node
	 * @param end_node second town node
	 * @param cargo_id id of cargo to transport
	 */
	constructor(start_node, end_node, cargo_id)
	{
		::SimpleRoute.constructor(start_node, end_node, cargo_id, false);

		local s = start_node.location;
		local e = end_node.location;
		local dx = abs(AIMap.GetTileX(s) - AIMap.GetTileX(e));
		local dy = abs(AIMap.GetTileY(s) - AIMap.GetTileY(e));
		this.length = (min(dx, dy) * 1.41).tointeger() + abs(dx - dy);
		this.length_manhattan = dx + dy;

		this.ev = Property(this.doGetEstimatedVehicles.bindenv(this), 2 * GameTime.MONTH);
		CodeUtils.Log("Air route " + this.name + " revealed", 1);
	}

	function GetVehiclesCost()
	{
		local n = this.ev.Get();
		return n == 0 ? 0 : this.GetFreePlanes(n) * this.best_engine_price;
	}

	function GetEstimatedIncome()
	{
		local n = this.ev.Get();
		return n == 0 || this.vehicle_profit < 150 ? 0 :
			this.GetFreePlanes(n) * this.vehicle_profit;
	}

	/**
	 * Get estimated number of vehicles for this route.
	 * @return Estimated number of vehicles required for this route.
	 */
	function GetEstimatedVehicles()
	{
		return this.ev.Get();
	}

/* proteted */
	/**
	 * Get number of "free" planes for this route.
	 * "Free" mean max number of planes that can be added to route.
	 * @note Limited by the capacity and arrival frequency of the route airports.
	 */
	function GetFreePlanes(max_planes)
	{
		if (this.frequency == null) return 0;

		local s = Airport.GetAirport(this.start_node);
		local e = Airport.GetAirport(this.end_node);
		local n = (s == null) ? max_planes :
			min(max_planes, s.GetFreePlanes(this.frequency));

		return (e == null) ? n : min(n, e.GetFreePlanes(this.frequency));
	}

/* private */
	/** Price of the best engine. */
	best_engine_price = 0;

	/** Estimated vehicles. */
	ev = null;

	/** Estimated arrival frequency for planes assigned to this route. */
	frequency = null;

	/** Distance manhattan between start and end nodes. */
	length_manhattan = 0;

	/** Transport capacity of one route vehicle (cargo amount per month). */
	transport_rate = null;

	/** Estimated month profit from one vehicle. */
	vehicle_profit = null;

	/**
	 * Aux function to select best aircraft engine for this route.
	 * @return Engine ID considered to be the best for this route.
	 */
	function ChooseBestEngine()
	{
		local best_airport_type = AirSettings.Get().GetBestAirportType();
		if (best_airport_type == AIAirport.AT_INVALID) return -1;

		local s = Airport.GetAirport(this.start_node);
		local e = Airport.GetAirport(this.end_node);
		local s_airport_type = s == null ? best_airport_type : s.GetAirportType();
		local e_airport_type = e == null ? best_airport_type : e.GetAirportType();

		local s_pop = this.start_node.GetPopulation();
		local e_pop = this.end_node.GetPopulation();
		local pop_valuator = AirlineUtils.TownPopulationValuator;
		local pop = min(s_pop, e_pop);
		foreach (dummy_id, e_id in PlaneEnginesRoster.Get().GetRoster()) {
			if (!pop_valuator(e_id, pop)) continue;
			if (!AirlineUtils.AirportSizeValuator(e_id, s_airport_type)) continue;
			if (!AirlineUtils.AirportSizeValuator(e_id, e_airport_type)) continue;

			local d_max = AIEngine.GetMaximumOrderDistance(e_id);
			if (d_max == 0) return e_id;

			local st = s != null ? s.GetLocation() : this.start_node.location;
			local et = e != null ? e.GetLocation() : this.end_node.location;
			if (AIOrder.GetOrderDistance(AIVehicle.VT_AIR, st, et) < d_max) return e_id;
		}

		return -1;
	}

/* public */
	function doGetEngine()
	{
		/* Set correct values for the case when there is no valid engine */
		this.vehicle_profit = 0;
		this.transport_rate = 0;
		this.frequency = null;

		local e = this.ChooseBestEngine();
		if (e == -1) return -1;

		local stats = EnginesInfo.stats[e];
		local e_price = stats[EngineStat.ES_PRICE];
		local e_capacity = stats[EngineStat.ES_CAPACITY];

		local t = AirlineUtils.CalculateAlmostFullTravelTime(e, this.length).tointeger();
		t += (e_capacity / 50);

		local ci = AICargo.GetCargoIncome(this.cargo_id, this.length_manhattan, t / 2);
		local mail_ci = 0;
		local mail_capacity = 0;

		if (this.cargo_id == CorporationUtils.pass_cargo_id.value) {
			local roster = PlaneEnginesRoster.Get();
			local mail = CorporationUtils.mail_cargo_id.value;

			mail_capacity = roster.GetMailCapacity(e);
			if (mail_capacity == -1 || mail == -1) {
				/* 1.4 multiplier because of mail cargo */
				/* Mail capacity usually about 1/5 of pass, but mail 2 times more profitable */
				ci = 1.4 * ci
				mail_capacity = 0;
			} else {
				mail_ci = AICargo.GetCargoIncome(mail, this.length_manhattan, t / 2)
			}
		}

		local tx = (GameTime.MONTH).tofloat() / t;
		this.transport_rate = e_capacity * tx;

		local profit = (e_capacity * ci + mail_capacity * mail_ci) * tx;
	
		if (!this.one_way) profit = 1.9 * profit;
		profit -= 2 * stats[EngineStat.ES_MONTH_RUNNING_COST] / 3;
		//profit -= e_price * GameTime.MONTH / (2 * stats[EngineStat.ES_MAX_AGE]);

		/* penalty for small towns */
		local s_pop = this.start_node.GetPopulation();
		local e_pop = this.end_node.GetPopulation();
		//local bonus = min(s_pop, e_pop) * ci * GameTime.MONTH / (100 * t);
		profit += min((min(s_pop, e_pop) - 2000), 0) * ci * GameTime.MONTH / (100 * t);

		/*
		 * All above give 20-30% higher numbers than in game performance
		 * Thus we just reduce estimation by 15-16%
		 * (dirty trick, but works just fine)
		 */
		profit = profit - profit / 7;
		/*
		 * Also use known profit drop for engine type
		 */
		profit = profit - profit * stats[EngineStat.ES_AVG_PROFIT_DROP] / 100;

		this.frequency = 1.0 / t;
		this.vehicle_profit = profit;
		this.best_engine_price = e_price;
		return e;
	}

	/**
	 * Get estimated number of vehicles for this route.
	 * @return Estimated number of vehicles required for this route.
	 */
	function doGetEstimatedVehicles()
	{
		local e = this.GetEngine();
		if (e == -1) return 0;
		//local x = AirConstants.good_corporation_income_for_full_scale_expansion;
		//local full_scale = CorporationUtils.month_income.value > x && !AirSettings.Get().greedy_strategy;
		local n = this.GetFreePlanes(12);
		return n == 0 ? 0 :
			min(n, (2 * this.GetFreeProduction() / (3 * this.transport_rate)).tointeger());
	}
}
