/**
 * Class that represent air related AI settings.
 */
class AirSettings extends Terron_Settings
{
/* public */
	static function GetClassName()
	{
		return "AirSettings";
	}

	static function Restore(memento)
	{
		local restored = AirSettings();
		restored.max_route_length = memento.max_route_length;
		restored.min_route_length = memento.min_route_length;
		restored.planes_limit = memento.planes_limit;
		return restored;
	}

	/** Array with allowed airports' types */
	allowed_airport_types = null;

	/**
	 * Defines how AI will manage airlines.
	 * True mean hard limit for planes per airport.
	 */
	greedy_strategy = true;

	/** Is greedy airports spam aloowed? */
	is_greedy_strategy_allowed = true;

	/** Defines max airline length allowed */
	</ must_save = true />
	max_route_length = null;

	/** Defines min airline length allowed */
	</ must_save = true />
	min_route_length = null;

	/** The amount of planes when AI must cease new airports construction */
	</ must_save = true />
	planes_limit = null;

	/** Magic number representing base airport maintenance cost */
	airport_base_maintenance_cost = 0;

	/**
	 * Get approximate cost of the airport.
	 * @param airport_type Desired airport's type.
	 * @return Approximate construction cost for the airport of desired type,
	 *  or -1 if the given type unknown/not allowed.
	 */
	function GetAirportCost(airport_type)
	{
		local costs = this.airports_cost.Get();
		return (airport_type in costs) ? costs[airport_type] :
			AirConstants.infinite_airport_cost;
	}

	/**
	 * Get the best airport type available.
	 * @return Airport type currently considered as the best.
	 */
	function GetBestAirportType()
	{
		foreach (dummy_id, type in this.allowed_airport_types) {
			if (AIAirport.IsValidAirportType(type)) return type;
		}
		return AIAirport.AT_INVALID;
	}

/* protected */
	/**
	 * Set air settings up.
	 */
	constructor()
	{
		::Terron_Settings.constructor();
		this.allowed_airport_types = [AIAirport.AT_INTERCON, AIAirport.AT_INTERNATIONAL,
									  AIAirport.AT_METROPOLITAN, AIAirport.AT_LARGE];
		if (AIController.GetSetting("use_small_airports") > 0) {
			this.allowed_airport_types.append(AIAirport.AT_COMMUTER);
			this.allowed_airport_types.append(AIAirport.AT_SMALL);
		}

		this.max_route_length = AIController.GetSetting("max_air_route_length");
		if (this.max_route_length == -1) this.max_route_length = 350;
		this.min_route_length = (5 + this.max_route_length / 4).tointeger();

		this.airport_base_maintenance_cost = 50;
		if (AIGameSettings.IsValid("economy.infrastructure_maintenance")) {
			if (AIGameSettings.GetValue("economy.infrastructure_maintenance") > 0) {
				this.airport_base_maintenance_cost = 625;
			}
		}
		if (AIController.GetSetting("airport_base_maintenance_cost") > 0) {
			this.airport_base_maintenance_cost = 78;
		}

		/*foreach (dummy_id, airport_type in this.allowed_airport_types) {
			local factor = AIAirport.GetMaintenanceCostFactor(airport_type);
			if (!AIAirport.IsValidAirportType(airport_type)) continue;
			AILog.Warning("factor = " + factor);
			AILog.Warning("monthly maintanence = " + this.airport_base_maintenance_cost * factor);
		}
		assert(null);*/

		/* "greedy" behavior deprecated now */
		this.greedy_strategy = false;
		this.is_greedy_strategy_allowed = false;

		this.construction_cost_multiplier = 1.0;
		local hills = AIGameSettings.IsValid("difficulty.terrain_type") ?
			AIGameSettings.GetValue("difficulty.terrain_type") : 2;
		switch (hills) {
			case 0 : this.construction_cost_multiplier += 0.2; break;
			case 1 : this.construction_cost_multiplier += 0.4; break;
			case 2 : this.construction_cost_multiplier += 0.6; break;
			case 3 : this.construction_cost_multiplier += 0.8; break;
			default : this.construction_cost_multiplier += 0.4;
		}

		this.airports_cost = Property(this.doGetAirportsCost.bindenv(this), 8 * GameTime.MONTH);

		if (AIGameSettings.IsDisabledVehicleType(AIVehicle.VT_AIR)
		|| AIController.GetSetting("PlaneModule") <= 0
		|| !AICargo.IsValidCargo(CorporationUtils.pass_cargo_id.value)) {
			this.planes_limit = 0;
			return;
		}
		this.planes_limit = AIGameSettings.IsValid("vehicle.max_aircraft") ?
			3 * AIGameSettings.GetValue("vehicle.max_aircraft") / 5 : 80;
	}

	/**
	 * Calculate price for building single airport of each of allowed types.
	 * @return Array with airports cost.
	 */
	function doGetAirportsCost()
	{
		local result = {};
		foreach (dummy_id, airport_type in this.allowed_airport_types) {
			if (AIAirport.IsValidAirportType(airport_type)) {
				local cost = AIAirport.GetPrice(airport_type);
				result[airport_type] <- this.construction_cost_multiplier * cost;
			} else {
				result[airport_type] <- AirConstants.infinite_airport_cost;
			}
		}
		return result;
	}

/* private */
	/** Different airports cost stored here. */
	airports_cost = null;

	/**	Cost multiplier to reflect difficult terrain terraforming cost. */
	construction_cost_multiplier = 1.0;
}

Terron_ClassTable.RegisterClass(AirSettings);
