/**
 * Class that provides base settings and utility service functions for AI air section.
 */
class AirlineUtils
{
/* public */
	/**
	 * Get the time required for plane to travel between two airports.
	 * @param e The plane engine ID. Must be valid.
	 * @param one_way_len Distance between airports.
	 * @return The amount of days from flight on at departure airport
	 *  till landing at <b>departure</b> airport (visiting arrival airport in middle).
	 */
	function CalculateAlmostFullTravelTime(e, one_way_len) {
		local avg_speed = PlaneEnginesRoster.Get().GetEngineAvgSpeed(e);

		/* +40 tiles to approach airports and maneuver around it */
		/*
		 * +50 days to land manevour inside both airports
		 */
		// let's hope mistake will be small enough
		//return 668 * (2 * one_way_len + 40) / (24 * avg_speed) + 50;
		return 60 + 28 * (2 * one_way_len + 40) / avg_speed;
	}

	/**
	 * Checks whether the given aircraft can land to the given airport's type.<p>
	 * It is better to forbid landing of big airplanes to small airports.
	 * @param engine_id The aircraft's engine ID to check.
	 * @param airport_type The airport type to check.
	 * @return True if and only if pair [aiplane size, airport size] is safe.
	 */
	static function AirportSizeValuator(engine_id, airport_type)
	{
		if (airport_type == AIAirport.AT_INVALID) return false;
		if (airport_type != AIAirport.AT_SMALL && airport_type != AIAirport.AT_COMMUTER) return true;
		return AIEngine.GetPlaneType(engine_id) == AIAirport.PT_SMALL_PLANE;
	}

	/**
	 * Checks whether the aircraft is suitable for a town with given population.
	 * @param engine_id The aircraft's engine ID to check.
	 * @param population The town population.
	 * @return True if and only if engine is good enough.
	 */
	static function TownPopulationValuator(engine_id, population)
	{
		local capacity = AIEngine.GetCapacity(engine_id);
		if (population < 800) return capacity <= 130;
		if (population < 1000) return capacity <= 150;
		if (population < 1250) return capacity <= 170;
		if (population < 1400) return capacity <= 200;
		if (population < 1550) return capacity <= 240;
		if (population < 1750) return 100 <= capacity && capacity <= 290;
		if (population < 2000) return 100 <= capacity && capacity <= 350;
		if (population < 2250) return 100 <= capacity && capacity <= 400;
		if (population < 2500) return 120 <= capacity && capacity <= 440;
		if (population < 2750) return 120 <= capacity && capacity <= 470;
		if (population >= 2750) return capacity >= 120;
	}
}
