/**
 * Class that handles proper plane engine selection.
 */
class PlaneEnginesRoster extends Terron_Settings
{
/* public */
	static function GetClassName()
	{
		return "PlaneEnginesRoster";
	}

	/**
	 * Get the max amount of mail which the given engine's vehicle can load.
	 * @param engine_id The engine ID.
	 * @return The engine's mail capacity, or -1 if such value is not set yet.
	 */
	function GetMailCapacity(engine_id)
	{
		if (!(engine_id in this.mail_capacity)) return -1;
		return this.mail_capacity[engine_id];
	}

	/**
	 * Set the max amount of mail which the given engine's vehicle can load.
	 * @param engine_id The engine ID.
	 * @param amount The engine's mail capacity.
	 */
	function SetMailCapacity(engine_id, amount)
	{
		this.mail_capacity[engine_id] <- amount;
	}

	function EngineReport(engine_id, supposed_profit, actual_profit)
	{
		if (supposed_profit <= 1) return;
		if (!(engine_id in this.engine_reports)) {
			this.engine_reports[engine_id] <- [];
		}
		this.engine_reports[engine_id].append({
			supposed_profit = supposed_profit, actual_profit = actual_profit,
		});
	}

	/**
	 * Get the roster with air engines.
	 * @return Sorted array with the IDs of an engines, first - best.
	 */
	function GetRoster()
	{
		return this.roster.Get();
	}

	/**
	 * Accept good air engines.
	 */
	function OnEnginePreview(ec)
	{
		if (ec.GetVehicleType() != AIVehicle.VT_AIR) return;

		ec.AcceptPreview();

		AIController.Sleep(1);

		local new_engine_name = ec.GetName();
		foreach (e, dummy_value in AIEngineList(AIVehicle.VT_AIR)) {
			if (AIEngine.GetName(e) == new_engine_name) {
				this.OnAirEngineAvailable(e);
				break;
			}
		}
	}

	/**
	 * Properly handle newly available engine so it will receive reliability
	 *  bonus.
	 */
	function OnAirEngineAvailable(e)
	{
		if (AIEngine.GetPlaneType(e) == AIAirport.PT_HELICOPTER) return;

		this.roster.ForceUpdate();
	}

	/**
	 * Update this plane roster.
	 */
	function doGetRoster()
	{
		/* Update reliability bonus for new engines */
		local current_date = AIDate.GetCurrentDate();

		/* Make a list sorted by engine's rating */
		local list = AIEngineList(AIVehicle.VT_AIR);

		list.Valuate(AIEngine.GetPlaneType);
		list.RemoveValue(AIAirport.PT_HELICOPTER);

		list.Valuate(AIEngine.IsBuildable);
		list.RemoveValue(0);

		local all_engines_stats = EnginesInfo.stats;
		foreach (e, dummy_value in list) {
			if (!(e in all_engines_stats)) EnginesInfo.CalculateEngineStats(e);
			this.avg_engines_speed[e] <- this.doGetEngineAvgSpeed(e);
		}

		foreach (engine_id, reports in this.engine_reports) {
			local sum = 0;
			foreach (dummy_id, report in reports) {
				local s = report.supposed_profit;
				sum += ((s - report.actual_profit).tofloat() / s) * 100;

				CodeUtils.Log(
					"s = " + s + ", a = " + report.actual_profit +
					", e = " + AIEngine.GetName(engine_id), 0
				);
			}

			local e_stats = all_engines_stats[engine_id];
			if (sum != 0) {
				local mod = sum.tointeger() / (reports.len() + 1);
				e_stats[EngineStat.ES_AVG_PROFIT_DROP] = mod;
			}
			if (e_stats[EngineStat.ES_AVG_PROFIT_DROP] != 0) {
				local e_name = AIEngine.GetName(engine_id);

				CodeUtils.Log(
					e_name + " rating drop based on planes performance: -" +
					e_stats[EngineStat.ES_AVG_PROFIT_DROP] + "%", 2
				);
			}
		}

		foreach (e, dummy_value in list) {
			list.SetValue(e, this.GetEngineRating(e).tointeger());
		}
		list.Sort(AIList.SORT_BY_VALUE, false);

		foreach (engine_id, reports in this.engine_reports) {
			reports.clear();
		}

		return CodeUtils.ListToArray(list);
	}

	/**
	 * Get the rating of the given engine.
	 * @param e Engine ID.
	 * @return The engine rating, more => better.
	 */
	function GetEngineRating(e)
	{
		if (!AIEngine.IsBuildable(e)) return 0;
		if (AIEngine.GetVehicleType(e) != AIVehicle.VT_AIR) return 0;
		if (AIEngine.GetPlaneType(e) == AIAirport.PT_HELICOPTER) return 0;

		local s = this.GetEngineAvgSpeed(e);
		local c = AIEngine.GetCapacity(e);
		local mail_capacity = this.GetMailCapacity(e);
		if (mail_capacity != -1) c += 2 * mail_capacity;

		local e_stats = EnginesInfo.stats[e];
		local s0 = s - s * e_stats[EngineStat.ES_AVG_PROFIT_DROP] / 100;

		local rating = s0 * c * (1 + c.tofloat() / (c + s));
		rating = rating * AIEngine.GetReliability(e) / 100;
		return rating * rating / (10 * AIEngine.GetPrice(e));
	}

	/**
	 * Get the reliability of the given engine.
	 * @param e Engine ID.
	 * @return Reliability the engine has.
	 */
	function GetEngineReliability(e)
	{
		local e_stats = EnginesInfo.stats[e];
		local r = e_stats[EngineStat.ES_RELIABILITY]; 
		local year = AIDate.GetYear(AIDate.GetCurrentDate());
		local design_year = e_stats[EngineStat.ES_DESIGN_YEAR];
		return (year > design_year + 12) ? r :
			min(100, r + (13.0 / max(1, year - design_year + 1)).tointeger());
	}

	/**
	 * Get average speed for the given engine.
	 * @param e Engine ID. Must be valid.
	 * @return Average speed for this engine.
	 */
	function GetEngineAvgSpeed(e)
	{
		if (e in this.avg_engines_speed) return this.avg_engines_speed[e];
		if (!(e in EnginesInfo.stats)) EnginesInfo.CalculateEngineStats(e);

		local result = this.doGetEngineAvgSpeed(e);
		this.avg_engines_speed[e] <- result;
		return result;
	}

	/**
	 * Calculate average speed for the given engine.
	 * @param e Engine ID. Must be valid.
	 * @return Average speed for this engine.
	 */
	function doGetEngineAvgSpeed(e)
	{
		local s = EnginesInfo.stats[e][EngineStat.ES_MAX_SPEED];
		/* happens when vehicles do not broke down */
		if (this.broken_plane_speed == 0) return s;

		local broken_speed = s > this.broken_plane_speed ?
			this.broken_plane_speed : s;

		/* Make the engine reliability a serious factor */
		local rl = this.GetEngineReliability(e).tofloat() / 100;
		// it is decreasing during flight
		rl = rl - 0.03;
		if (rl < 0.7) return broken_speed;

		return s * broken_speed / (s * (1 - rl) + broken_speed * rl);
	}

/* protected */
	constructor()
	{
		::Terron_Settings.constructor();

		AISettingsSaveEvent.RemoveListener(this);

		this.avg_engines_speed = {};
		this.mail_capacity = {};
		this.roster = Property(
			this.doGetRoster.bindenv(this),
			GameTime.ENGINE_REVIEW_INTERVAL + GameTime.DAY
		);

		local v_psf = AIGameSettings.IsValid("vehicle.plane_speed_factor") ?
						AIGameSettings.GetValue("vehicle.plane_speed_factor") : 4;
		local v_bkdn = AIGameSettings.IsValid("difficulty.vehicle_breakdowns") ?
						AIGameSettings.GetValue("difficulty.vehicle_breakdowns") : 2;
		if (v_bkdn > 0) {
			this.broken_plane_speed = AirConstants.broken_plane_speed / v_psf;
		}

		this.engine_reports = {};

		Terron_Event.EnginePreview.AddListener(this);
		Terron_Event.AirEngineAvailable.AddListener(this);
	}

/* private */
	/** Table with pairs [engine_id, avg_engine_speed]. */
	avg_engines_speed = null;

	/** Table with engines mail capacity. */
	mail_capacity = null;

	/** Sorted air engines. */
	roster = null;

	/** Stored report about engines performance. */
	engine_reports = null;

	/** Actual speed ot the broken airplanes. */
	broken_plane_speed = 0;
}
