/**
 * Class that manages air hubs.
 */
class AirHubsControl extends AbstractTransportSystem
{
/* public */
	static function GetClassName()
	{
		return "AirHubsControl";
	}

	static function Restore(memento)
	{
		return AirHubsControl();
	}

	constructor()
	{
		::AbstractTransportSystem.constructor();

		local c_name = AirHub.GetClassName();
		GlobalTransportMap.Get().SetUpNewTransportNodeClass(c_name);
		this.transport_politics = PlanesTransportPolitics();

		NewNodeEvent.AddListener(this);

		TrafficReallocationTask(this, GameTime.HALF_YEAR);
	}

	function GetName()
	{
		return "AirHubsControl";
	}

	/**
	 * Intgrate new node into air network.
	 * @param node Newly found node.
	 */
	function OnNewNode(node)
	{
		local new_air_hub = this.transport_politics.EnableTransportNode(node);
		if (new_air_hub != null) {
			//new_air_hub.Reshake();
			this.AddSubsystem(new_air_hub);
		}
	}

/* private */
	/** Custom air transport politics. */
	transport_politics = null;
}

/**
 * Class that handles air traffic reallocation.
 */
class AirHubsControl.TrafficReallocationTask extends Terron_Task
{
/* public */
	/**
	 * Creates AirHubsControl.TrafficReallocationTask object.
	 * @param air_hubs_control AirHubsControl instance.
	 * @param period Execution period (e.g. 17 mean once in 17 days).
	 */
	constructor(air_hubs_control, period)
	{
		::Terron_Task.constructor(AIDate.GetCurrentDate() + GameTime.DAY, period);
		this.air_hubs_control = air_hubs_control;
	}

/* protected */
	function Execute()
	{
		local planes_roster = PlaneEnginesRoster.Get().GetRoster();
		if (planes_roster.len() == 0) return;

		local tt = AIController.GetTick();
		local c = CorporationUtils.pass_cargo_id.value;
		local cfg = AirSettings.Get();
		local date = AIDate.GetCurrentDate();
		local target_rating = AirConstants.low_airport_rating_border;
		local hubs_class_name = AirHub.GetClassName();
		local very_best_engine = planes_roster[0];
		local department_is_active = !AirDepartment.Get().IsPassive();

		local airports_count = 0;
		local good_towns_count = 0;
		foreach (dummy_id, air_hub in this.air_hubs_control.GetSubsystems()) {
			if (air_hub.GetClassName() != hubs_class_name) continue;
			if (!air_hub.IsEnabled()) continue;
			if (!(air_hub.node.GetID() in ForceBuildAirportAction.to_ignore)) {
				good_towns_count++;
			}

			local airport = Airport.GetAirport(air_hub.node);
			if (airport != null) {
				airports_count++;
				local a_id = airport.GetStationID();
				if (AIStation.GetCargoRating(a_id, c) > target_rating) {
					if (!cfg.greedy_strategy) {
						TransitWantedEvent.Fire([airport, [TransitType.TT_BUS], {[c] = 1}]);
					}
					continue;
				}

				/*
				 * Close trade route when it's dead - 0 planes at servise there -
				 *  and new planes can't be added due to intense air traffic.
				 */
				foreach (dummy_link_id, link in air_hub.links_out) {
					foreach (dummy_route_id, route in link.routes) {
						if (route.GetEngine() == very_best_engine) continue;
						if (route.trade_routes.len() == 0) continue;
						if (route.GetFreePlanes(16) > 0) continue;
						foreach (dummy_id, trade_route in route.trade_routes) {
							if (trade_route.GetVehicles().Count() == 0) {
								CorporationUtils.CloseSystem(trade_route);
							}
						}
					}
				}
			}

			if (airport != null || department_is_active) {
				air_hub.Reshake();
				air_hub.CreateBestAirTrade();
			}
		}

		CodeUtils.Log("Air hubs reshake, ticks " + (AIController.GetTick() - tt), 1);

		if (!cfg.is_greedy_strategy_allowed) return;

		if (department_is_active) {
			if (airports_count > 5 * cfg.planes_limit / 12 ||
				airports_count.tofloat() / good_towns_count > 0.8) {
					cfg.greedy_strategy = false;
					ForceBuildAirportAction.to_ignore.clear();
				} else {
					cfg.greedy_strategy = true;
					local m = AirConstants.good_corporation_income_for_full_scale_expansion;
					if (CorporationUtils.month_income.value > 2 * m) {
						Corporation.Get().AddAction(ForceBuildAirportAction(this.air_hubs_control.GetSubsystems()));
					}
				}
		} else {
			AirSettings.Get().greedy_strategy = false;
		}
	}

/* private */
	/** Hubs control object. */
	air_hubs_control = null;
}

/**
 * Action to build more and more airports. 
 */
class ForceBuildAirportAction extends InstantAction
{
/* public */
	/** Ids of towns to ignore */
	static to_ignore = {};

	constructor(air_hubs)
	{
		::InstantAction.constructor();

		this.air_hubs = air_hubs;
		this.limit_date = AIDate.GetCurrentDate() + 3 * GameTime.MONTH / 2;
	}

	function GetName()
	{
		return "Building an airport somewhere";
	}

	function Execute()
	{
		local result = this.doForceBuild();
		if (result != -1) return result;

		return this.doForceBuild();
	}

	function HandleResult(result)
	{
		return result == -2 ? result : 0;
	}

/* private */
	/** Container with air hubs */
	air_hubs = null;

	/** Date to limit building in time */
	limit_date = null;

	/**
	 * Aux function that actually builds.
	 */
	function doForceBuild()
	{
		if (AIDate.GetCurrentDate() > this.limit_date) return -1;

		local cfg = AirSettings.Get();
		local best_airport_type = cfg.GetBestAirportType();
		if (best_airport_type == AIAirport.AT_INVALID) return -1;

		local cost = cfg.GetAirportCost(best_airport_type);
		if (!CorporationUtils.GetMoney(cost)) return -2;

		local big_town = null;
		local min_pop = 0;
		local hubs_class_name = AirHub.GetClassName();
		foreach (dummy_id, air_hub in this.air_hubs) {
			if (air_hub.GetClassName() != hubs_class_name) continue;

			local town = air_hub.node;
			if (town.GetID() in ForceBuildAirportAction.to_ignore) continue;

			if ((!air_hub.HasAirport()) && town.GetPopulation() > min_pop) {
				min_pop = town.GetPopulation();
				big_town = town;
			}
		}

		if (big_town == null) return -1;
		local result = AirHub.BuildAirport(big_town);
		if (result == -1) {
			ForceBuildAirportAction.to_ignore[big_town.GetID()] <- 1;
		}

		return result;
	}
}

Terron_ClassTable.RegisterClass(BusHubControlSection);
