/**
 * Class that handles air hubs/routes generation.
 */
class PlanesTransportPolitics extends TransportPolitics
{
/* public */
	constructor()
	{
		::TransportPolitics.constructor();

		local cfg = AirSettings.Get();
		this.d_min = cfg.min_route_length;
		this.d_max = cfg.max_route_length;
		this.pass_cargo_id = CorporationUtils.pass_cargo_id.value;
		this.shipments = TransportSchema.Get().GetShipmentsList(NodeTypeID.NT_PAX_TOWN, NodeTypeID.NT_PAX_TOWN);
		this.shipments = (this.pass_cargo_id in this.shipments) ?
			{[this.pass_cargo_id] = 1} : {};
	}

	function GetTransportNode(node)
	{
		if (!node.IsTown()) return null;
		if (!node.IsPaxTown()) return null;
		return node.GetPopulation() > AirConstants.lowest_population_limit_for_towns ?
			AirHub(node, this) : null;
	}

	function GetRoutes(from_transport_node, to_transport_node, cargo_ids)
	{
		local route = AirRoute(from_transport_node.node, to_transport_node.node, this.pass_cargo_id);
		return {[route.GetID()] = route};
	}

/* protected */
	function IntegrateNode(air_hub)
	{
		local air_hubs = GlobalTransportMap.Get().structured_transport_nodes["AirHub"][NodeTypeID.NT_PAX_TOWN];
		local t = air_hub.node.location;
		local LINK = GlobalTransportMap.Link;
		local distance_manhattan = AIMap.DistanceManhattan;
		foreach (dummy_id, e in air_hubs) {
			local d = distance_manhattan(t, e.node.location);
			if (d <= this.d_max && this.d_min <= d) {
				LINK.Create(e, air_hub, this, 0, this.shipments);
			}
		}

		/*
		 * Main goal is to make sure that we will not "lose" any
		 *  planes or entire trade airlines after loading.
		 * We can lose info about such things because we do not save everything,
		 *  just required miminum.
		 */
		local airport = Airport.GetAirport(air_hub.node);
		if (airport == null) return;

		// Only alternative to get here without loading is to create two
		// instances of "AirHub" class for single town, and this is atrocity

		NewAirportBuiltEvent.Fire(airport);

		/*
		 * We need to check both directions here because otherwise final result
		 *  will depend on hubs creation order, and by bidirectional check
		 *  we eliminate such dependency.
		 */
		foreach (dummy_id, link in air_hub.links_in) {
			foreach (dummy_route_id, air_route in link.routes) {
				this.ReCreateTrade(air_route);
			}
		}
		foreach (dummy_id, link in air_hub.links_out) {
			foreach (dummy_route_id, air_route in link.routes) {
				this.ReCreateTrade(air_route);
			}
		}

		air_hub.CreateBestAirTrade();
	}

/* private */
	/** Min allowed air routes' length */
	d_min = 0

	/** Max allowed air routes' length */
	d_max = 0;

	/** Pass cargo ID */
	pass_cargo_id = null;

	/** Table with town-to-town shipments */
	shipments = null;

	/**
	 * Restore air trade route from the given air route if it existed before
	 *  game load, and there are planes currently operational.
	 * @param air_route Some air route instance.
	 */
	function ReCreateTrade(air_route)
	{
		if (air_route.IsCompletelyUpgraded()) return;

		local s = Airport.GetAirport(air_route.GetStart());
		if (s == null) return;
		local e = Airport.GetAirport(air_route.GetEnd());
		if (e == null) return;
 
		local trade_route = AirTradeRoute(air_route, s, e);
		CodeUtils.Log(trade_route.GetName() + " restored", 2);

		if (trade_route.GetVehicles().Count() == 0) trade_route.Close();
	}
}
