/**
 * Class that searches road trade routes after game load.
 */
class AISearchRoadTradeRoutesAction extends InstantAction
{
/* public */
	constructor(to_restore_transport_class_name)
	{
		::InstantAction.constructor();
		this.transport_class_name = to_restore_transport_class_name;
		this.stations_with_checked_depots = {};
	}

	function GetName()
	{
		return "Seraching road trade routes";
	}

	function GetPriority()
	{
		/* We need this action AFTER those transport nodes will be created */
		/* And this is the simplest(yet bad) way to do so */
		return APriority.MAX - 1;
	}

/* protected */
	function Execute()
	{
		CodeUtils.Log("Restoring [" + this.transport_class_name + "] trade routes", 2);

		local sell_name = SellLostVehiclesTask.special_sell_name;
		local vehicles = AIVehicleList();
		vehicles.Valuate(AIVehicle.GetVehicleType);
		vehicles.KeepValue(AIVehicle.VT_ROAD);
		vehicles.Valuate(AIVehicle.GetAgeLeft);
		foreach (v, age_left in vehicles) {
			if (age_left < 3 * GameTime.YEAR) {
				AIVehicle.SetName(v, (sell_name + v));
			}
		}

		this.stations_with_checked_depots.clear();

		/* Start station pairs search */
		local structured_nodes = GlobalTransportMap.Get().structured_transport_nodes;
		local map = structured_nodes[this.transport_class_name];
		local stop;
		do {
			stop = true;
			foreach (dummy_type_id, type_nodes in map) {
				foreach (dummy_node_id, node in type_nodes) {
					foreach (dummy_id, link in node.links_out) {
						if (link.color != LinkColor.LC_GREEN) continue;
						foreach (dummy_id, road_route in link.routes) {
							if (this.TryToRestoreTrade(road_route)) stop = false;
						}
					}
				}
			}
		} while (!stop);
		foreach (dummy_type_id, type_nodes in map) {
			foreach (dummy_node_id, node in type_nodes) {
				foreach (dummy_id, link in node.links_out) {
					foreach (dummy_id, road_route in link.routes) {
						this.TryToRestoreTrade(road_route); 
					}
				}
			}
		}

		this.stations_with_checked_depots.clear();
		CodeUtils.Log("Trade routes restoration complete", 2);
		return 0;
	}

/* private */
	/** Helper to avoid depots checking multimple times */
	stations_with_checked_depots = null;

	/**
	 * Creates a new trade route from the given route.
	 * @param road_route Road route.
	 * @return True if and only if new trade route has been created.
	 */
	function TryToRestoreTrade(road_route)
	{
		local result = false;
		local roadmap = RoadDepartment.Get().GetStationsRoadMap();
		local max_level = max(3, road_route.GetMaxRouteLevel());
		local heap_class = HeapNode.GetClassName();
		for (local l = road_route.GetCurrentRouteLevel(); l <= max_level; l++) {
			local s_table = road_route.GetStartStations(l);
			local e_table = road_route.GetEndStations(l);

			foreach (dummy_s_id, s in s_table) {
				local s_id = s.GetStationID();
				if (!(s_id in this.stations_with_checked_depots)) {
					this.stations_with_checked_depots[s_id] <- s.CheckDepots();
				}
				if (!this.stations_with_checked_depots[s_id]) continue;

				foreach (dummy_e_id, e in e_table) {
					if (!roadmap.HasRoad(s_id, e.GetStationID())) continue;
					local e_id = e.GetStationID();
					if (!(e_id in this.stations_with_checked_depots)) {
						this.stations_with_checked_depots[e_id] <- e.CheckDepots();
					}
					if (!this.stations_with_checked_depots[e_id]) continue;

					local route_start = road_route.GetStart();
					if (route_start.GetClassName() == heap_class) {
						local tmp = GlobalTransportMap.Get().GetTransportNode(
							this.transport_class_name, route_start
						);
						if (tmp == null) continue;

						local c = road_route.GetCargoID();
						local ci = road_route.normal_cargo_income;
						RoadTransitHandler.Get().EnableHeapNodeGathering(tmp, s_id, c);
						TransitIncomeCalculatedEvent.Fire(
							{station_id = s_id, cargo_id = c, cargo_income = ci}
						);
					}

					local restored = RoadTradeRoute(road_route, s, e);
					CodeUtils.Log("[" + restored.GetName() + "] restored", 2);
					result = true; 
				}
			}
		}

		return result;
	}

	/** private */
	transport_class_name = null;
}
