/**
 * Class that handles road stations building near industries.
 */
class IndustryStationBuilder
{
/* public */
	/**
	 * Build new road station near the given industry node.
	 * @param node Industry node where to build.
	 * @param c Cargo ID, new station must handle this cargo.
	 * @param other_route_end Used to calculate correct station orientation.
	 * @param is_load_station Flag, defines if this is load station,
	 *  or cargo drop station.
	 * @return ID of new station, or error code if fails:
	 *  -1 Unknown error
	 *  -2 Not enough money
	 *  -3 Impossible to build(local authority refuses, low acceptance, e.t.c) 
	 */
	static function BuildStation(node, c, other_route_end, is_load_station)
	{
		if (IndustryStationBuilder.direction_based_offsets.len() == 0) {
			IndustryStationBuilder.SetOffsetsUp();
		}

		local old_road_type = RoadUtils.SetRoadType(AIRoad.ROADTYPE_ROAD);

		local is_transit = node.type_id == NodeTypeID.NT_TRANSIT_DROP;
		if (is_transit) {
			local id = node.s_id;
			local tl = AITileList_StationType(id, RoadUtils.GetStationType(c));
			if (tl.Count() > 2) return node.s_id;
		}

		local t = other_route_end;
		local dx = AIMap.GetTileX(node.location) - AIMap.GetTileX(t);
		local dy = AIMap.GetTileY(node.location) - AIMap.GetTileY(t);
		if (t == node.location) dx = 1; // should never happen

		/* Define how station can be rotated (3 of 4 directions) */
		local dir = abs(dx) > abs(dy) ? 1 + dx / abs(dx) : 2 + dy / abs(dy);
		local offsets = IndustryStationBuilder.direction_based_offsets[dir];

		local list = IndustryStationBuilder.GetTilesWhereCanBuild(node, c, is_load_station, other_route_end, offsets);
		local v_type = AIRoad.GetRoadVehicleTypeForCargo(c);
		local id = IndustryStationBuilder.BuildIndustryStation(list, v_type, offsets, AIStation.STATION_NEW);
		RoadUtils.SetRoadType(old_road_type);

		if (id > 0 && !is_load_station) {
			local location = AIStation.GetLocation(id);
			local tiles = AITileList();
			TileUtils.AddSimmetricRectangleSafe(tiles, location, 3, 3);
			local danger_map = RoadDepartment.Get().dangerous_station_locations;
			foreach (t, dummy in tiles) {
				danger_map[t] <- t;
			}
		}

		return id;
	}

/* private */
	/**
	 * Array describing station orientations.
	 * ID (0..3 - array len is 4) coincides to the OpenTTD diagdir.
	 * Values: Another arrays - let's say A0..A3 - with 3 offsets each.
	 * A0[0] offset corresponding to the diagdir 0,
	 * A1[0] offset corresponding to the diagdir 1,
	 * A2[0] offset corresponding to the diagdir 2,
	 * A3[0] offset corresponding to the diagdir 3.
	 * Other values (A0[1], A0[2], A1[1], ... , A3[1], A3[2])
	 *  are side offsets (to left, and to right from the "front" Ax[0]).
	 */
	static direction_based_offsets = [];

	/**
	 * Fill IndustryStationBuilder.direction_based_offsets array with correct data.
	 */
	static function SetOffsetsUp()
	{
		local diag_dir_to_offset = function(dir) {
			return AIMap.GetTileIndex((dir - 1) % 2, (dir - 2) % 2);
		}

		IndustryStationBuilder.direction_based_offsets.clear();
		IndustryStationBuilder.direction_based_offsets.extend([[], [], [], []]);
		local arr = IndustryStationBuilder.direction_based_offsets;
		for (local dir = 0; dir < 4; dir++) {
			arr[dir].append(diag_dir_to_offset(dir));
			arr[dir].append(diag_dir_to_offset((dir + 1) % 4));
			arr[dir].append(diag_dir_to_offset((dir - 1) % 4));
		}
	}

	/**
	 * Calculate desireability of the given tile as new station location.
	 * @param tile Tile to check.
	 * @param offsets Array with preferable station orientation offsets.
	 * @param r Station acceptance radius.
	 * @param avoid_danger_positions Flag = true if we must keep distance
	 *  with already built stations.
	 * @return Integer number, higher value indicates better tile.
	 */
	static function RankLocation(tile, offsets, r, avoid_danger_positions)
	{
		local result = 0;
		local nearest_tiles = AITileList();
		/* Prefer location with flat tiles around */
		TileUtils.AddSimmetricRectangleSafe(nearest_tiles, tile, 4, 4);
		foreach (t, dummy in nearest_tiles) {
			if (AITile.IsStationTile(t)) {
				result -= 14;
				result += (AIMap.DistanceSquare(t, tile)).tointeger();
			}
			if (!AITile.IsBuildable(t)) {
				result -= AIRoad.IsRoadTile(t) ? 2 : 4;
			}
			local s = AITile.GetSlope(t);
			if (TileUtils.IsSlopeFlatOrWithThreeCornersRaised(s)) result += 2;
		}
		nearest_tiles.Clear();

		if (avoid_danger_positions) {
			local danger_map = RoadDepartment.Get().dangerous_station_locations;
			if (tile in danger_map) result -= 12;
		}
		/* Follow in most possible stattion exit direction... */
		for (local i = 1; i < r; i++) {
			nearest_tiles.AddTile(tile - r * offsets[0]);
		}
		/* ...and make sure there is no obstacles */
		foreach (t, dummy in nearest_tiles) {
			local s = AITile.GetSlope(t);
			if (!TileUtils.IsSlopeFlatOrWithThreeCornersRaised(s)) result--;
			if (AITile.IsBuildable(t) || AIRoad.IsRoadTile(t)) continue;
			result -= 6;
			break;
		}
		return result;
	}

	/**
	 * Build nice road loop around drive through station.
	 */
	static function BuildDriveThroughNiceLoop(station_tile, to_exit_offset, side_offset) 
	{
		local loop_tiles = [];

		loop_tiles.append(station_tile);
		loop_tiles.append(station_tile - to_exit_offset);
		loop_tiles.append(station_tile - to_exit_offset - side_offset);
		loop_tiles.append(station_tile - side_offset);
		loop_tiles.append(station_tile + to_exit_offset - side_offset);
		loop_tiles.append(station_tile + to_exit_offset);
		loop_tiles.append(station_tile);
		foreach (dummy_id, t in loop_tiles) {
			if (AIRoad.IsDriveThroughRoadStationTile(t)) continue;
			//if (!AITile.IsBuildable(t) && !AIRoad.IsRoadTile(t)) return false;
			if (AITile.IsWaterTile(t)) {
				if (AITile.GetMaxHeight(t) != 1) return false;
				if (!TileUtils.IsSlopeFlatOrWithThreeCornersRaised(AITile.GetSlope(t))) return false;
			}
		}

		local h = AITile.GetMaxHeight(station_tile);
		foreach (dummy_id, t in loop_tiles) {
			if (TileUtils.IsSlopeFlatOrWithThreeCornersRaised(AITile.GetSlope(t))) {
				if (AITile.GetMaxHeight(t) == h) continue;
			}
			if (!TileUtils.LevelTile(t, h)) return false;
		}

		for (local i = 0; i < loop_tiles.len() - 1; i++) {
			local t1 = loop_tiles[i];
			local t2 = loop_tiles[i + 1]
			if (!AIRoad.AreRoadTilesConnected(t1, t2)) {
				AIRoad.BuildRoad(t1, t2);
				//if (!AIRoad.BuildRoad(t1, t2)) return false;
			}
		}
		return true;
	}
}

/**
 * Get sorted AITileList with possible locations for new road station.
 * @param node The node where to build new station.
 * @param c ID of the cargo station must deal with. 
 * @param is_load_station Flag, defines if this is load station,
 *  or cargo drop station.
 * @param opposite_node_location Used to define correct station orientation.
 * @param offsets Array with preferable station orientation offsets.
 */
function IndustryStationBuilder::GetTilesWhereCanBuild(node, c, is_load_station, opposite_node_location, offsets)
{
	local station_type = RoadUtils.GetStationType(c);
	local r = AIStation.GetCoverageRadius(station_type);
	local list = is_load_station ?
			node.AITileList_NodeProducing(r, c) :
			node.AITileList_NodeAccepting(r, c);

	/* Remove water and non-buildable tiles */
	list.Valuate(AITile.IsWaterTile);
	list.KeepValue(0);
	foreach (t, dummy in list) {
		local is_good = AITile.IsBuildable(t) || AIRoad.IsRoadTile(t);
		list.SetValue(t, is_good ? 1 : 0);
	}
	list.RemoveValue(0);

	local opposite_height = AITile.GetMaxHeight(opposite_node_location);
	foreach (t, dummy in list) {
		local rank = IndustryStationBuilder.RankLocation(t, offsets, r, is_load_station);
		if (AITile.GetMaxHeight(t) < opposite_height) {
			rank += is_load_station ? -1 : 1;
		}
		list.SetValue(t, rank);
	}

	list.Sort(AIList.SORT_BY_VALUE, false);

	return list;
}

/**
 * Try to build a road station.
 * @param tiles List of "candidate" tiles, where we should try to place the station.
 * @param vehicle_type Either AIRoad.ROADVEHTYPE_BUS or AIRoad.ROADVEHTYPE_TRUCK.
 *  Defines if bus or truck facilities must be built.
 * @param allowed_offsets One of the IndustryStationBuilder.direction_based_offsets
 *  array items. Defines how station should be oriented(rotated).
 * @param station_type AIStation.STATION_NEW or AIStation.STATION_JOIN_ADJACENT
 *  or ID of the station to join.
 */
function IndustryStationBuilder::BuildIndustryStation(tiles, vehicle_type, allowed_offsets, station_type)
{
	if (tiles.Count() == 0) return -3;

	local err_auth = AIError.ERR_LOCAL_AUTHORITY_REFUSES;
	local err_cash = AIError.ERR_NOT_ENOUGH_CASH;
	local offsets = [AIMap.GetTileIndex(0, -1), AIMap.GetTileIndex(-1, 0),
						AIMap.GetTileIndex(0, 1), AIMap.GetTileIndex(1, 0)];
	local s_type = station_type;
	local v_type = vehicle_type;
	local truck_station = (v_type == AIRoad.ROADVEHTYPE_TRUCK);

	foreach (offset_id, offset in allowed_offsets) {
		foreach (t, dummy in tiles) {
			local exit = t - offset;
			local enter = t + offset;
			if (!AITile.IsBuildable(exit) && !AIRoad.IsRoadTile(exit)) continue;
			if (!AITile.IsBuildable(enter) && !AIRoad.IsRoadTile(enter)) continue;

			/* Do not build station if it front tile leads to a "dead end" */
			local blocks = 0;
			foreach (o in offsets) {
				local z = exit - o;
				if (AITile.IsStationTile(z)) blocks++;
				if  ( (!TileUtils.IsRoad(z)) && (!AITile.IsBuildable(z)) ) blocks++;
			}
			if (blocks >= 3) continue;

			{
				local test_mode = AITestMode();
				if (!AITile.DemolishTile(t)) continue;
			}
			if (!AIRoad.IsRoadTile(t)) {
				if (!AITile.DemolishTile(t)) continue;
			}

			local h = AITile.GetMaxHeight(t);
			local exit_slope = AITile.GetSlope(exit);
			local enter_slope = AITile.GetSlope(enter);
			if (!TileUtils.IsSlopeFlatOrWithThreeCornersRaised(exit_slope)) {
				TileUtils.LevelTile(exit, h);
				exit_slope = AITile.GetSlope(exit);
				if (!TileUtils.IsSlopeFlatOrWithThreeCornersRaised(exit_slope)) continue;
			}
			if (!TileUtils.IsSlopeFlatOrWithThreeCornersRaised(enter_slope)) {
				TileUtils.LevelTile(enter, h);
				enter_slope = AITile.GetSlope(enter);
				if (!TileUtils.IsSlopeFlatOrWithThreeCornersRaised(enter_slope)) continue;
			}

			{
				local test_mode = AITestMode();
				if (!AIRoad.AreRoadTilesConnected(t, exit)) {
					if (!AIRoad.BuildRoad(t, exit)) continue;
				}
				if (!AIRoad.AreRoadTilesConnected(t, enter)) {
					if (!AIRoad.BuildRoad(t, enter)) continue;
				}
				if (!AIRoad.BuildDriveThroughRoadStation(t, exit, v_type, s_type)) continue;
			}

			if (!AIRoad.AreRoadTilesConnected(t, exit)) {
				if (!AIRoad.BuildRoad(t, exit)) continue;
			}
			if (!AIRoad.AreRoadTilesConnected(t, enter)) {
				if (!AIRoad.BuildRoad(enter, t)) continue;
			}

			if (AIRoad.BuildDriveThroughRoadStation(t, exit, v_type, s_type)) {
				local side_offset = allowed_offsets[(offset_id + 1) % 3];
				IndustryStationBuilder.BuildDriveThroughNiceLoop(t, offset, side_offset);
				IndustryStationBuilder.BuildDriveThroughNiceLoop(t, offset, -side_offset);
				return AIStation.GetStationID(t);
			}
			local err = AIError.GetLastError();
			if (err == err_auth || err == err_cash) return err == err_cash ? -2 : -1;
		}
	}
	return -1;
}

/**
 * Build additional station tiles to expand already existing station.
 * @param terminal Road station terminal, must be drive through only.
 * @param v_type Either AIRoad.ROADVEHTYPE_BUS or AIRoad.ROADVEHTYPE_TRUCK.
 * @return -2 when not enough money;
 *         -1 when failed to build;
 *         Expansion tile ID (in the case of success).
 */
function IndustryStationBuilder::ExpandDriveThroughStation(terminal, v_type)
{
	if (IndustryStationBuilder.direction_based_offsets.len() == 0) {
		IndustryStationBuilder.SetOffsetsUp();
	}
	local old_road_type = RoadUtils.SetRoadType(AIRoad.ROADTYPE_ROAD);

	/* Despite name, just a random tile from the station to upgrade */
	local core = null, core_front = null;
	/* Make expansion only at the sides of original drive through */
	local hot_spot = AIList();
	local to_join_id = AIStation.STATION_JOIN_ADJACENT;
	foreach (dummy_id, t in terminal.core.tiles) {
		if (AIRoad.IsDriveThroughRoadStationTile(t)) {
			core = t;
			to_join_id = AIStation.GetStationID(t);
			core_front = AIRoad.GetRoadStationFrontTile(t);
			local front_offset = t - core_front;
			for (local k = -2; k < 2; k++) {
				local good_t = t - AIMap.GetTileIndex(k % 2, (k + 1) % 2);
				if (AITile.IsWaterTile(good_t)) continue;
				if (AITile.IsStationTile(good_t)) continue;
				if (!AITile.IsBuildable(good_t) && !AIRoad.IsRoadTile(good_t)) continue;
				hot_spot.AddItem(good_t, front_offset);
			}
		}
	}

	if (core == null) {
		RoadUtils.SetRoadType(old_road_type);
		return -1;
	}

	foreach (dummy_id, t in terminal.core.tiles) {
		local front_offset = t - AIRoad.GetRoadStationFrontTile(t);
		hot_spot.RemoveItem(t - front_offset);
		hot_spot.RemoveItem(t + front_offset);
	}

	/* Select correct offset set */
	local dx = AIMap.GetTileX(core) - AIMap.GetTileX(core_front);
	local dy = AIMap.GetTileY(core) - AIMap.GetTileY(core_front);
	local dir = abs(dx) > abs(dy) ? 1 + dx / abs(dx) : 2 + dy / abs(dy);
	local offsets = IndustryStationBuilder.direction_based_offsets[dir];

	local id = IndustryStationBuilder.BuildIndustryStation(hot_spot, v_type, offsets, to_join_id);
	RoadUtils.SetRoadType(old_road_type);

	if (id >= 0) {
		foreach (t, dummy in hot_spot) {
			if (AIStation.GetStationID(t) == id) {
				return t;
			}
		}
	}
	return id;
}

/**
 * Build additional station tiles to expand already existing station.
 * @param terminal Road station terminal, intended to be NOT drive through.
 * @param vehicle_type Either AIRoad.ROADVEHTYPE_BUS or AIRoad.ROADVEHTYPE_TRUCK.
 * @return -2 when not enough money;
 *         -1 when failed to build;
 *         Expansion tile ID in the case of success.
 * TODO: rewrite from scratch
 */
function IndustryStationBuilder::ExpandNormalRoadStation(terminal, v_type)
{
	local station_id = terminal.station_id;
	local tmp = AITileList_StationType(station_id, AIStation.STATION_TRUCK_STOP);
	tmp.AddList(AITileList_StationType(station_id, AIStation.STATION_BUS_STOP));
	local list = AITileList();
	foreach (tile, dummy in tmp) {
		TileUtils.AddSimmetricRectangleSafe(list, tile, 4, 4);
	}
	tmp.Clear();

	list.Valuate(AIRoad.IsRoadDepotTile);
	list.KeepValue(0);
	list.Valuate(AITile.IsWaterTile);
	list.KeepValue(0);

	/* Station "facilities" (small, one-tile stops) */
	local facilities = {};

	/*
	 * List containing some info about entrances to station
	 * Item:  tile index - tile which is entrance to road station(front tile)
	 * Value: number of station facilities, having Item = tile index as front tile
	 */
	local entrances = {};

	/* Fill facilities and entrance lists */
	list.Valuate(AIRoad.IsRoadStationTile);
	foreach (t, is_station in list) {
		if (!is_station) {
			if (!AIRoad.IsDriveThroughRoadStationTile(t)) continue;
		}
		if (AIStation.GetStationID(t) != station_id) continue;
		facilities[t] <- 0;
		local front = AIRoad.GetRoadStationFrontTile(t);
		if (front in entrances) {
			entrances[front]++;
		} else {
			entrances[front] <- 1;
		}
	}

	/* Cut off tiles, that is not neighbour to station */
	local expand_tiles = {};
	foreach (t1, is_station in list) {
		if (is_station || (t1 in entrances)) continue;
		if (AIRoad.IsDriveThroughRoadStationTile(t1)) continue;
		if (AITile.IsSteepSlope(AITile.GetSlope(t1))) continue;
		foreach (t2, dummy2 in facilities) {
			if (AIMap.DistanceMax(t1, t2) != 1) continue;
			expand_tiles[t1] <- 0;
			break;
		}
	}

	local slope_offsets = {};
	slope_offsets[AITile.SLOPE_NE] <- AIMap.GetTileIndex(-1, 0);
	slope_offsets[AITile.SLOPE_SE] <- AIMap.GetTileIndex(0, 1);
	slope_offsets[AITile.SLOPE_SW] <- AIMap.GetTileIndex(1, 0);
	slope_offsets[AITile.SLOPE_NW] <- AIMap.GetTileIndex(0, -1);

	local is_correct_slope = function(station_tile, station_exit_tile) : (slope_offsets) {
		local slope = AITile.GetSlope(station_tile);
		if (slope in slope_offsets) {
			return station_exit_tile == station_tile + slope_offsets[slope];
		}
		return (slope == AITile.SLOPE_ELEVATED ||
			TileUtils.IsSlopeFlatOrWithThreeCornersRaised(slope));
	}

	/* Build roads for future station expansion */
	foreach (t, dummy1 in expand_tiles) {
		local t_top = AITile.GetMaxHeight(t);
		local t_bottom = AITile.GetMinHeight(t);
		local t_slope = AITile.GetSlope(t);
		foreach (s, dummy2 in entrances) {
			if (AIMap.DistanceManhattan(t, s) != 1) continue;
			local s_slope = AITile.GetSlope(s);
			if (!TileUtils.IsSlopeFlatOrWithThreeCornersRaised(s_slope)) continue;

			local can_build_station = true;
			{
				local test_mode = AITestMode();
				can_build_station = AIRoad.BuildRoadStation(t, s, v_type, AIStation.STATION_JOIN_ADJACENT);
			}
			if (!can_build_station) {
				local s_top = AITile.GetMaxHeight(s);
				if (t_top > s_top) AITile.LowerTile(t, t_slope);
				if (t_bottom < s_top) AITile.RaiseTile(t, AITile.GetComplementSlope(t_slope));
			}

			AIRoad.BuildRoad(t, s);
			//AITile.RaiseTile(s, AITile.GetComplementSlope(s_slope));
		}
	}

	/* Find all roads, that can used (to serve as future entrance) for new station expansion */
	//list.Valuate(AIRoad.IsRoadTile);
	while (true) {
		local is_nothing_to_add = true;
		foreach (t, is_station in list) {
			if (is_station || (t in entrances)) continue;
			if (AIRoad.IsDriveThroughRoadStationTile(t)) continue;
			foreach (s, dummy2 in entrances) {
				if (!AIRoad.AreRoadTilesConnected(s, t)) continue;
				entrances[t] <- 0;
				is_nothing_to_add = false;
				break;
			}
		}
		if (is_nothing_to_add == true) break;
	}

	/*
	 * Keep only tiles that can directly used for station expansion
	 * If we build new station on theese tiles,
	 * this new station become facility of old big station we expanding,
	 * and will be connected with road to other facilities
	 * Other words: here were cut tiles that although can be used as new facility,
	 * but requires additional road connection to big station
	 */
	list.Clear();
	foreach (t, dummy1 in expand_tiles) {
		expand_tiles[t] = [AITile.GetMaxHeight(t), AITile.GetMinHeight(t)];
		if (AIRoad.IsRoadStationTile(t)) continue;
		foreach (s, dummy2 in entrances) {
			if (AIMap.DistanceMax(t, s) == 1 && entrances[s] < 2) {
				list.AddTile(t);
				break;
			}
		}
	}

	if (!terminal.UnloadOnly()) {
		list.Valuate(AITile.GetCargoAcceptance, terminal.cargo_id, 1, 1, 3);
		list.Sort(AIList.SORT_BY_VALUE, false);
	}

	/* Trying to build new station facility at one of the previously found tiles */
	foreach (s, dummy2 in entrances) {
		if (entrances[s] >= 2) continue;
		local n_close_occlusions = 0;
		for (local k = -2; k < 2; k++) {
			local t = s - AIMap.GetTileIndex( k % 2,  (k + 1) % 2);
			if (TileUtils.IsRoad(t)) continue;
			if (!AITile.IsBuildable(t)) n_close_occlusions++;
		}
		if (n_close_occlusions >= 3) continue;

		foreach (t, height in expand_tiles) {
			if (AIMap.DistanceManhattan(t, s) != 1) continue;
			if (AITile.IsStationTile(t)) continue;
			//if (AIRoad.GetNeighbourRoadCount(t) > 1) continue;

			local t_top = height[0];
			local t_bottom = height[1];
			if (!AIRoad.AreRoadTilesConnected(t, s)) {
				AITile.RaiseTile(s, AITile.GetComplementSlope(AITile.GetSlope(s)));
				if (is_correct_slope(t, s)) {
					if (!AIRoad.IsRoadTile(t)) {
						if (!AITile.DemolishTile(t)) continue;
					}
					if (!AIRoad.BuildRoad(t, s)) continue;
					if (!AIRoad.BuildRoadStation(t, s, v_type, AIStation.STATION_JOIN_ADJACENT)) continue;
					AITile.RaiseTile(t, AITile.GetComplementSlope(AITile.GetSlope(t)));
					return t;
				}

				if (!AIRoad.IsRoadTile(t)) {
					if (!AITile.DemolishTile(t)) continue;
				}
				if (t_top > AITile.GetMaxHeight(s)) {
					AITile.LowerTile(t, AITile.GetSlope(t));
				}
				if (t_bottom < AITile.GetMaxHeight(s)) {
					AITile.RaiseTile(t, AITile.GetComplementSlope(AITile.GetSlope(t)));
				}
				if (!is_correct_slope(t, s)) continue;
				{
					local test_mode = AITestMode();
					if (!AIRoad.BuildRoadStation(t, s, v_type, AIStation.STATION_JOIN_ADJACENT)) continue;
				}

				if (!AIRoad.BuildRoad(t, s)) continue;
			}

			if (!AIRoad.BuildRoadStation(t, s, v_type, AIStation.STATION_JOIN_ADJACENT)) continue;
			//if (!AIRoad.BuildDriveThroughRoadStation(t, s, v_type, AIStation.STATION_JOIN_ADJACENT)) continue;
			AITile.RaiseTile(t, AITile.GetComplementSlope(AITile.GetSlope(t)));
			return t;
		}
	}
	return -1;
}
